package org.seasar.nazuna;

import java.math.BigDecimal;

import org.seasar.expr.ExprUtil;
import org.seasar.util.Assertion;
import org.seasar.util.Conversion;
import org.seasar.util.MathUtil;
import org.seasar.util.SeasarException;

public abstract class GroupValue {

    private GroupElement _groupElement;

    public GroupValue(final GroupElement groupElement) {
    	Assertion.assertNotNull("groupElement", groupElement);

        _groupElement = groupElement;
    }

    public GroupElement getGroupElement() {
        return _groupElement;
    }

    public abstract void calculate(final Object item) throws SeasarException;

    public abstract Object getValue();

    public static final class CountValue extends GroupValue {

        private int _count = 0;

        public CountValue(final GroupElement groupElement) {
            super(groupElement);
        }

        public void calculate(final Object item) throws SeasarException {
            String groupedAttributeName = getGroupElement().getGroupedPropertyName();
            if (groupedAttributeName == null) {
                ++_count;
            } else {
                Object val = ExprUtil.getProperty(item, groupedAttributeName);
                System.out.println(val);
                if (val != null) {
                    ++_count;
                }
            }
        }

        public Object getValue() {
            return new Integer(_count);
        }
    }

    public static final class SumValue extends GroupValue {

        private BigDecimal _value = MathUtil.ZERO_BIGDECIMAL;

        public SumValue(final GroupElement groupElement) {
            super(groupElement);
        }

        public void calculate(final Object item) throws SeasarException {
            String groupedAttributeName = getGroupElement().getGroupedPropertyName();
            BigDecimal val = Conversion.toBigDecimal(ExprUtil.getProperty(item, groupedAttributeName));
            _value = MathUtil.sum(_value, val);
        }

        public Object getValue() {
            return _value;
        }
    }

    public static final class MaxValue extends GroupValue {

        private Comparable _value = null;

        public MaxValue(final GroupElement groupElement) {
            super(groupElement);
        }

        public void calculate(final Object item) throws SeasarException {
            String groupedAttributeName = getGroupElement().getGroupedPropertyName();
            Comparable val = (Comparable) ExprUtil.getProperty(item, groupedAttributeName);
            _value = MathUtil.max(_value, val);
        }

        public Object getValue() {
            return _value;
        }
    }

    public static final class MinValue extends GroupValue {

        private Comparable _value = null;

        public MinValue(final GroupElement groupElement) {
            super(groupElement);
        }

        public void calculate(final Object item) throws SeasarException {
            String groupedAttributeName = getGroupElement().getGroupedPropertyName();
            Comparable val = (Comparable) ExprUtil.getProperty(item, groupedAttributeName);
            _value = MathUtil.min(_value, val);
        }

        public Object getValue() {
            return _value;
        }
    }

    public static final class AvgValue extends GroupValue {

        private GroupValue _sumValue;
        private GroupValue _countValue;

        public AvgValue(final GroupElement groupElement) {
            super(groupElement);
            _sumValue = new SumValue(groupElement);
            _countValue = new CountValue(groupElement);
        }

        public void calculate(final Object item) throws SeasarException {
            _sumValue.calculate(item);
            _countValue.calculate(item);
        }

        public Object getValue() {
            double sum = Conversion.toBigDecimal(_sumValue.getValue()).doubleValue();
            double count = Conversion.toBigDecimal(_countValue.getValue()).doubleValue();
            if (count != 0) {
                return new BigDecimal(sum / count);
            } else {
                return null;
            }
        }
    }
}