package org.seasar.util;

import java.io.Serializable;
import java.sql.Connection;
import java.util.Map;
import java.util.Properties;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NameNotFoundException;
import javax.naming.NamingException;
import javax.sql.DataSource;
import javax.transaction.Status;
import javax.transaction.SystemException;
import javax.transaction.UserTransaction;

import org.xml.sax.Attributes;

public final class SeasarContext implements Serializable {

    final static long serialVersionUID = 5048121277903501233L;
    private static final EMap _seasarContextMap = new EMap();
    private static XMLHandlerRule _xmlHandlerRule = new XMLHandlerRule();

    private String _name;
    private int _hashCode;
    private Properties _jndiProperties = new Properties();
    private String _dataSourceJNDIName;
    private String _userTransactionJNDIName;
	private String _hostName = null;
	private int _port = 0;
    private transient Map _jndiMap;
    private transient InitialContext _initialContext;
    private transient DataSource _dataSource;
    private transient UserTransaction _userTransaction;

    static {
        configure();
    }

    private SeasarContext(final String name) {
        _name = name;
        _hashCode = name.hashCode();
    }

    public static SeasarContext getInstance() {
        return (SeasarContext) _seasarContextMap.get(0);
    }

    public static SeasarContext getInstance(final String name) {
        SeasarContext churaContext = (SeasarContext) _seasarContextMap.get(name);
        if (churaContext != null) {
            return churaContext;
        } else {
            throw new SeasarRuntimeException("ESSR0001", new Object[]{name});
        }
    }

    public String getName() {
        return _name;
    }
    
    public final String getHostName() {
    	return _hostName;
    }
    
    public final int getPort() {
    	return _port;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || !(o instanceof SeasarContext)) {
            return false;
        }
        return _name.equals(((SeasarContext) o)._name);
    }

    public int hashCode() {
        return _hashCode;
    }

    public InitialContext getInitialContext() throws SeasarException {
        if (_initialContext != null) {
            return _initialContext;
        }
        synchronized (this) {
            if (_initialContext == null) {
                try {
                    _initialContext = new InitialContext(_jndiProperties);
                } catch (Exception ex) {
                    throw new SeasarException("ESSR0333", new Object[]{ex}, ex);
                }
            }
        }
        return _initialContext;
    }

    public Object lookup(String jndiName) throws SeasarException {
        Map jndiMap = getJndiMap();
        Object o = jndiMap.get(jndiName);
        if (o != null) {
            return o;
        }
        synchronized (this) {
            if (o == null) {
                o = lookupNoCache(jndiName);
                jndiMap.put(jndiName, o);
            }
        }
        return o;
    }

    public Object lookupNoCache(String jndiName) throws SeasarException {
        Context context = getInitialContext();
        try {
            return context.lookup(jndiName);
        } catch (NameNotFoundException ex) {
            try {
                return context.lookup(jndiName.replace('.', '/'));
            } catch (NamingException ex2) {
                throw new SeasarException("ESSR0328", new Object[]{jndiName, ex2}, ex2);
            }
        } catch (Throwable t) {
            init();
            try {
            	return getInitialContext().lookup(jndiName);
        	} catch (NamingException ex3) {
                throw new SeasarException("ESSR0328", new Object[]{jndiName, ex3}, ex3);
            }
        }
    }

    public DataSource getDataSource() throws SeasarException {
        if (_dataSource != null) {
            return _dataSource;
        }
        synchronized (this) {
            if (_dataSource == null) {
                _dataSource = (DataSource) lookupNoCache(_dataSourceJNDIName);
            }
        }
        return _dataSource;
    }

    public Connection getConnection() throws SeasarException {
        DataSource ds = getDataSource();
        try {
            return ds.getConnection();
        } catch (Throwable t) {
        	synchronized (this) {
            	_dataSource = (DataSource) lookupNoCache(_dataSourceJNDIName);
            }
            try {
            	return _dataSource.getConnection();
            } catch (Exception ex) {
            	throw new SeasarException("ESSR0329", new Object[]{ex}, ex);
            }
        }
    }

    public UserTransaction getUserTransaction() throws SeasarException {
        if (_userTransaction != null) {
            return _userTransaction;
        }
        synchronized (this) {
            if (_userTransaction == null) {
                _userTransaction = (UserTransaction) lookupNoCache(_userTransactionJNDIName);
            }
        }
        return _userTransaction;
    }
    
	public boolean hasTransaction() throws SeasarException {
		UserTransaction ut = getUserTransaction();
		try {
			return ut.getStatus() != Status.STATUS_NO_TRANSACTION;
		} catch (SystemException ex) {
			throw SeasarException.convertSeasarException(ex);
		}
	}


    public void begin() throws SeasarException {
        UserTransaction userTransaction = getUserTransaction();
        try {
            userTransaction.begin();
        } catch (Exception ex) {
            throw new SeasarException("ESSR0330", new Object[]{ex}, ex);
        }
    }

    public void commit() throws SeasarException {
        UserTransaction userTransaction = getUserTransaction();
        try {
            userTransaction.commit();
        } catch (Exception ex) {
            throw new SeasarException("ESSR0331", new Object[]{ex}, ex);
        }
    }

    public void rollback() throws SeasarException {
        UserTransaction userTransaction = getUserTransaction();
        try {
            userTransaction.rollback();
        } catch (Exception ex) {
            throw new SeasarException("ESSR0332", new Object[]{ex}, ex);
        }
    }

    private Map getJndiMap() {
        if (_jndiMap != null) {
            return _jndiMap;
        } else {
            synchronized (this) {
                if (_jndiMap == null) {
                    _jndiMap = new EMap();
                }
            }
            return _jndiMap;
        }
    }

    private static void configure() {
        setupXMLHandlerRule();
        XMLHandlerParser.parse("/seasar-context.xml", _xmlHandlerRule);
        _seasarContextMap.put(null, _seasarContextMap.get(0));
    }

    private static void setupXMLHandlerRule() {
        _xmlHandlerRule.addElementHandler("/seasarContext/context",
            new ElementHandler() {
                public void start(XMLHandler xmlHandler, Attributes attributes) {
                    String name = attributes.getValue("name");
                    SeasarContext seasarContext = new SeasarContext(name);
                    _seasarContextMap.put(seasarContext.getName(), seasarContext);
                    xmlHandler.push(seasarContext);
                }
                public void end(final XMLHandler xmlHandler, final String body) {
                    SeasarContext seasarContext = (SeasarContext) xmlHandler.pop();
                    seasarContext.setupHostNameAndPort();
                    
                }
            });
        _xmlHandlerRule.addElementHandler("/seasarContext/context/jndiProperties/property",
            new ElementHandler() {
            	public void start(XMLHandler xmlHandler, Attributes attributes) {
                    String name = attributes.getValue("name");
                    String value = attributes.getValue("value");
                    SeasarContext seasarContext = (SeasarContext) xmlHandler.peek();
                    seasarContext._jndiProperties.setProperty(name, value);
                }
            });
        _xmlHandlerRule.addElementHandler("/seasarContext/context/dataSource",
            new ElementHandler() {
                public void start(XMLHandler xmlHandler, Attributes attributes) {
                    SeasarContext seasarContext = (SeasarContext) xmlHandler.peek();
                    seasarContext._dataSourceJNDIName = attributes.getValue("jndiName");

                }
            });
        _xmlHandlerRule.addElementHandler("/seasarContext/context/userTransaction",
            new ElementHandler() {
                public void start(XMLHandler xmlHandler, Attributes attributes) {
                    SeasarContext seasarContext = (SeasarContext) xmlHandler.peek();
                    seasarContext._userTransactionJNDIName = attributes.getValue("jndiName");
                }
            });
    }

	private void setupHostNameAndPort() {
        String url = (String) _jndiProperties.get(Context.PROVIDER_URL);
        if (url == null) {
            return;
        }
        int colonPos = url.lastIndexOf(':');
        int slashPos = url.lastIndexOf('/');
        if (colonPos > 0) {
            _port = Integer.parseInt(url.substring(colonPos + 1));
            if (slashPos > 0) {
                _hostName = url.substring(slashPos + 1, colonPos);
            } else {
                _hostName = url.substring(0, colonPos);
            }
        }
    }
    
    private synchronized void init() {
    	_dataSource = null;
    	_userTransaction = null;
    	_jndiMap = null;
    	_initialContext = null;
    }

}
