package org.seasar.extension.jta;

import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.Status;
import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;
import javax.transaction.xa.XAException;

import org.seasar.framework.exception.SIllegalStateException;
import org.seasar.framework.exception.SNotSupportedException;
import org.seasar.framework.exception.SSystemException;

public final class TransactionManagerImpl implements TransactionManager {

	private ThreadLocal _threadAttachTx = new ThreadLocal();

	public TransactionManagerImpl() {
	}

	public void begin() throws NotSupportedException, SystemException {
		TransactionImpl tx = getCurrent();
		if (tx != null) {
			throw new SNotSupportedException("ESSR0316", null);
		}
		tx = attachTransaction();
		tx.begin();
	}

	public void commit()
		throws
			RollbackException,
			HeuristicMixedException,
			HeuristicRollbackException,
			SecurityException,
			IllegalStateException,
			SystemException {

		TransactionImpl tx = getCurrent();
		if (tx == null) {
			throw new SIllegalStateException("ESSR0311", null);
		}
		try {
			tx.commit();
		} finally {
			detachTransaction(tx);
		}
	}

	public Transaction suspend() throws SystemException {
		TransactionImpl tx = getCurrent();
		if (tx == null) {
			throw new SIllegalStateException("ESSR0311", null);
		}
		try {
			tx.suspend();
		} catch (XAException ex) {
			throw new SSystemException("ESSR0363", new Object[]{ex}, ex);
		}
		
		setCurrent(null);
		return tx;
	}

	public void resume(Transaction resumeTx)
		throws InvalidTransactionException, IllegalStateException, SystemException {

		TransactionImpl tx = getCurrent();
		if (tx != null) {
			throw new SIllegalStateException("ESSR0317", null);
		}
		setCurrent((TransactionImpl) resumeTx);
		try {
			((TransactionImpl) resumeTx).resume();
		} catch (XAException ex) {
			throw new SSystemException("ESSR0364", new Object[]{ex}, ex);
		}
	}

	public void rollback()
		throws IllegalStateException, SecurityException, SystemException {

		TransactionImpl tx = getCurrent();
		if (tx == null) {
			throw new SIllegalStateException("ESSR0311", null);
		}
		try {
			tx.rollback();
		} finally {
			detachTransaction(tx);
		}
	}

	public void setRollbackOnly()
		throws IllegalStateException, SystemException {

		Transaction tx = getTransaction();
		if (tx == null) {
			throw new SIllegalStateException("ESSR0311", null);
		}
		tx.setRollbackOnly();
	}

	public void setTransactionTimeout(final int timeout)
		throws SystemException {
	}

	public int getStatus() {
		TransactionImpl tx = getCurrent();
		if (tx != null) {
			return tx.getStatus();
		} else {
			return Status.STATUS_NO_TRANSACTION;
		}
	}

	public Transaction getTransaction() {
		return getCurrent();
	}
	
	private TransactionImpl getCurrent() {
		TransactionImpl tx = (TransactionImpl) _threadAttachTx.get();
		if (tx != null && tx.getStatus() == Status.STATUS_NO_TRANSACTION) {
			return null;
		}
		return tx;
	}

	private void setCurrent(TransactionImpl current) {
		_threadAttachTx.set(current);
	}

	private TransactionImpl attachTransaction() {
		TransactionImpl tx = (TransactionImpl) _threadAttachTx.get();
		if (tx != null) {
			tx.init();
		} else {
			tx = new TransactionImpl();
			setCurrent(tx);
		}
		return tx;
	}

	private void detachTransaction(TransactionImpl tx) {
		if (tx != null) {
			tx.destroy();
		}
	}
}