/**
 * Copyright (C) 2006 Henning Norén
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, 
 * USA.
 */

#include <stdbool.h>
#include <string.h>
#include <stdlib.h>
#include <stdint.h>
#include "passwords.h"

#define PASSLENGTH 33

static FILE *wordList = NULL;
static const char *wordListName;
static bool wlMore;
static bool recovery = false;
static bool (*npw)() = NULL;
static unsigned int (*spw)(uint8_t *outbuf) = NULL;
static passwordMethod pwMethod;

bool
nextPassword() { return npw(); }

unsigned int
setPassword(uint8_t *outbuf) { return spw(outbuf); }

static bool
wlNextPassword() { return wlMore; }

static unsigned int 
wlSetPassword(uint8_t *outbuf) {
  int ch;
  unsigned int passlength;

  ch = 0;
  passlength = 0;

  ch = getc(wordList);
  while(ch != '\n' && ch != EOF && passlength < 32) {
    outbuf[passlength++] = ch;
    ch = getc(wordList);
  }

  /** clean up garbage of passwords longed than 32 chars */
  if(unlikely(passlength == 32))
    while(ch != '\n' && ch != EOF)
      ch = getc(wordList);

  if(unlikely(ch == EOF))
    wlMore = false;

  return passlength; 

}

static void
setWordList(FILE *file, const char *wl) {
  wordList = file;
  wordListName = wl;
  npw = &wlNextPassword;
  spw = &wlSetPassword;
  wlMore = true;
}

static const uint8_t
stdchars[] = {"abcdefghijklmnopqrstuvwxyz"
	      "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
	      "0123456789"};

static const uint8_t *charset = stdchars;
static unsigned int charsetLen;
static unsigned int maxPasswordLen;
static int password[PASSLENGTH];

static unsigned int
genSetPassword(uint8_t *outbuf) {
  unsigned int i;

  for (i=0;i<maxPasswordLen && password[i] != -1;i++)
    outbuf[i] = charset[password[i]];

  return i;
}


static bool
genNextPassword() {
  unsigned int i = 0;

  while(++password[i] == (int)charsetLen)
    password[i++] = 0;
 
  return (i != maxPasswordLen);
}

static void
setCharset(const char *cs, const unsigned int minPw, 
	   const unsigned int maxPw) {
  int i;
  int minPasswordLen;

  if(cs)
    charset = (uint8_t*)cs;
  charsetLen = strlen((char*)charset);
  maxPasswordLen = maxPw;
  minPasswordLen = (int)minPw;
  npw = &genNextPassword;
  spw = &genSetPassword;

  if(!recovery) {
    for(i=0;i<(int)maxPasswordLen;i++) {
      if (i<minPasswordLen-1)
	password[i] = charsetLen-1;
      else
	password[i] = -1;
    }
  }
}

void
initPasswords(const passwordMethod pm, FILE *file, const char *wl,
	      const char *cs, const unsigned int minPw,
	      const unsigned int maxPw) {
  if(!recovery)
    pwMethod = pm;
  switch(pwMethod) {
  case Generative:
    setCharset(cs, minPw, maxPw);
    break;
  case Wordlist:
    setWordList(file, wl);
    break;
  default:
    break;
  }
}

void
pw_loadState(FILE *file, char **wl) {
  int pm;
  unsigned int i, len;
  char *string;
  fscanf(file, "PM: %d\n", &pm);
  if(pm == Generative) {
    fscanf(file, "MaxPWL: %d\nCharsetLength: %d\nCharset: ",
	   &maxPasswordLen,&charsetLen);
    string = malloc(sizeof(uint8_t)*charsetLen+1);
    for(i=0;i<charsetLen;i++)
      string[i] = getc(file);
    string[i] = '\0';
    charset = (uint8_t*)string;
    getc(file);
    fscanf(file, 
	   "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d "
	   "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d",
	   &password[ 0],&password[ 1],&password[ 2],&password[ 3],
	   &password[ 4],&password[ 5],&password[ 6],&password[ 7],
	   &password[ 8],&password[ 9],&password[10],&password[11],
	   &password[12],&password[13],&password[14],&password[15],
	   &password[16],&password[17],&password[18],&password[19],
	   &password[20],&password[21],&password[22],&password[23],
	   &password[24],&password[25],&password[26],&password[27],
	   &password[28],&password[29],&password[30],&password[31]);
  }
  else if(pm == Wordlist) {
    fscanf(file, "Wordlist(%d): ", &len);
    string = malloc(sizeof(char)*len+1);
    fscanf(file, "%[^\n]\n", string);
    string[len] = '\0';
    *wl = string;
    wordListName = string;
  }
  pwMethod = pm;
  recovery = true;
}

void
pw_saveState(FILE *file) {
  unsigned int i;
  fprintf(file, "PM: %d\n", pwMethod);
  if(pwMethod == Generative) {
    fprintf(file, "MaxPWL: %d\nCharsetLength: %d\nCharset: %s\n",
	    maxPasswordLen,charsetLen,charset);
    for(i=0;i<PASSLENGTH;i++)
      fprintf(file,"%d ", password[i]);
  }
  else if(pwMethod == Wordlist) {    
    fprintf(file, "Wordlist(%d): %s", strlen(wordListName), wordListName);
  }
  fprintf(file,"\n");
}
