#region PDFsharp - A .NET library for processing PDF
//
// Authors:
//   Stefan Lange (mailto:Stefan.Lange@pdfsharp.com)
//
// Copyright (c) 2005-2007 empira Software GmbH, Cologne (Germany)
//
// http://www.pdfsharp.com
// http://sourceforge.net/projects/pdfsharp
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included
// in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
// DEALINGS IN THE SOFTWARE.
#endregion

using System;
using System.Diagnostics;
using System.Collections;
using PdfSharp.Pdf.Internal;

namespace PdfSharp.Pdf.AcroForms
{
  /// <summary>
  /// Represents the radion button field.
  /// </summary>
  public sealed class PdfRadioButtonField : PdfButtonField
  {
    /// <summary>
    /// Initializes a new instance of PdfRadioButtonField.
    /// </summary>
    internal PdfRadioButtonField(PdfDocument document) : base(document)
    {
      this.document = document;
    }

    internal PdfRadioButtonField(PdfDictionary dict) : base(dict)
    {
    }

    /// <summary>
    /// Gets or sets the index of the selected radio button in a radio button group.
    /// </summary>
    public int SelectedIndex
    { 
      get
      {
        string value = Elements.GetString(Keys.V);
        return IndexInOptStrings(value);
      }
      set 
      {
        PdfArray opt = Elements[Keys.Opt] as PdfArray;
        if (opt != null)
        {
          int count = opt.Elements.Count;
          if (value < 0 || value >= count)
            throw new ArgumentOutOfRangeException("value");
          Elements.SetName(Keys.V, opt.Elements[value].ToString());
        }
      }
    }

    int IndexInOptStrings(string value)
    {
      PdfArray opt = Elements[Keys.Opt] as PdfArray;
      if (opt != null)
      {
        int count = opt.Elements.Count;
        for (int idx = 0; idx < count; idx++)
        {
          PdfItem item = opt.Elements[idx];
          if (item is PdfString)
          {
            if (item.ToString() == value)
              return idx;
          }
        }
      }
      return -1;
    }

    /// <summary>
    /// Predefined keys of this dictionary. 
    /// The description comes from PDF 1.4 Reference.
    /// </summary>
    public new class Keys : PdfButtonField.Keys
    {
      /// <summary>
      /// (Optional; inheritable; PDF 1.4) An array of text strings to be used in
      /// place of the V entries for the values of the widget annotations representing
      /// the individual radio buttons. Each element in the array represents
      /// the export value of the corresponding widget annotation in the
      /// Kids array of the radio button field.
      /// </summary>
      [KeyInfo(KeyType.Array | KeyType.Optional)]
      public const string Opt = "/Opt";

      /// <summary>
      /// Gets the KeysMeta for these keys.
      /// </summary>
      internal static DictionaryMeta Meta
      {
        get
        {
          if (Keys.meta == null)
            Keys.meta = CreateMeta(typeof(Keys));
          return Keys.meta;
        }
      }
      static DictionaryMeta meta;
    }

    /// <summary>
    /// Gets the KeysMeta of this dictionary type.
    /// </summary>
    internal override DictionaryMeta Meta
    {
      get {return Keys.Meta;}
    }
  }
}
