#region PDFsharp - A .NET library for processing PDF
//
// Authors:
//   Stefan Lange (mailto:Stefan.Lange@pdfsharp.com)
//
// Copyright (c) 2005-2007 empira Software GmbH, Cologne (Germany)
//
// http://www.pdfsharp.com
// http://sourceforge.net/projects/pdfsharp
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included
// in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
// DEALINGS IN THE SOFTWARE.
#endregion

using System;
using System.Diagnostics;
using System.Globalization;
using System.Collections;
using System.Text;
using System.IO;
using PdfSharp.Internal;
using PdfSharp.Pdf.IO;

namespace PdfSharp.Pdf
{
  /// <summary>
  /// Represents a direct unsigned integer value.
  /// </summary>
  [DebuggerDisplay("({Value})")]
  public sealed class PdfUInteger : PdfNumber, IConvertible
  {
    public PdfUInteger()
    {
    }

    public PdfUInteger(uint value)
    {
      this.value = value;
    }

    public uint Value
    {
      // This class must behave like a value type. Therefore it cannot be changed (like System.String).
      get { return this.value; }
    }
    uint value;

    /// <summary>
    /// Returns the unsigned integer as string.
    /// </summary>
    public override string ToString()
    {
      return this.value.ToString();
    }

    /// <summary>
    /// Writes the integer as string.
    /// </summary>
    internal override void WriteObject(PdfWriter writer)
    {
      writer.Write(this);
    }

    #region IConvertible Members

    public ulong ToUInt64(IFormatProvider provider)
    {
      return Convert.ToUInt64(this.value);
    }

    public sbyte ToSByte(IFormatProvider provider)
    {
      throw new InvalidCastException();
    }

    public double ToDouble(IFormatProvider provider)
    {
      return value;
    }

    public DateTime ToDateTime(IFormatProvider provider)
    {
      // TODO:  Add PdfUInteger.ToDateTime implementation
      return new DateTime();
    }

    public float ToSingle(IFormatProvider provider)
    {
      return value;
    }

    public bool ToBoolean(IFormatProvider provider)
    {
      return Convert.ToBoolean(this.value);
    }

    public int ToInt32(IFormatProvider provider)
    {
      return Convert.ToInt32(this.value);
    }

    public ushort ToUInt16(IFormatProvider provider)
    {
      return Convert.ToUInt16(this.value);
    }

    public short ToInt16(IFormatProvider provider)
    {
      return Convert.ToInt16(this.value);
    }

    string System.IConvertible.ToString(IFormatProvider provider)
    {
      return value.ToString(provider);
    }

    public byte ToByte(IFormatProvider provider)
    {
      return Convert.ToByte(this.value);
    }

    public char ToChar(IFormatProvider provider)
    {
      return Convert.ToChar(this.value);
    }

    public long ToInt64(IFormatProvider provider)
    {
      return value;
    }

    public System.TypeCode GetTypeCode()
    {
      return TypeCode.Int32;
    }

    public decimal ToDecimal(IFormatProvider provider)
    {
      return value;
    }

    public object ToType(Type conversionType, IFormatProvider provider)
    {
      // TODO:  Add PdfUInteger.ToType implementation
      return null;
    }

    public uint ToUInt32(IFormatProvider provider)
    {
      return Convert.ToUInt32(this.value);
    }

    #endregion
  }
}
