/*
 * Copyright (C) 2011-2020 Gary Kramlich <grim@reaperworld.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, see <https://www.gnu.org/licenses/>.
 */

#if !defined(GPLUGIN_GLOBAL_HEADER_INSIDE) && !defined(GPLUGIN_COMPILATION)
#error "only <gplugin.h> may be included directly"
#endif

#ifndef GPLUGIN_MANAGER_H
#define GPLUGIN_MANAGER_H

#include <glib.h>
#include <glib-object.h>

#include <gplugin/gplugin-plugin.h>

G_BEGIN_DECLS

#define GPLUGIN_TYPE_MANAGER (gplugin_manager_get_type())
G_DECLARE_DERIVABLE_TYPE(
	GPluginManager,
	gplugin_manager,
	GPLUGIN,
	MANAGER,
	GObject);

typedef void (*GPluginManagerForeachFunc)(
	const gchar *id,
	GSList *plugins,
	gpointer data);

struct _GPluginManagerClass {
	/*< private >*/
	GObjectClass parent;

	/*< public >*/
	void (*append_path)(GPluginManager *manager, const gchar *path);
	void (*prepend_path)(GPluginManager *manager, const gchar *path);
	void (*remove_path)(GPluginManager *manager, const gchar *path);
	void (*remove_paths)(GPluginManager *manager);

	GList *(*get_paths)(GPluginManager *manager);

	gboolean (
		*register_loader)(GPluginManager *manager, GType type, GError **error);
	gboolean (*unregister_loader)(
		GPluginManager *manager,
		GType type,
		GError **error);
	GSList *(*get_loaders)(GPluginManager *manager);

	void (*refresh)(GPluginManager *manager);

	void (*foreach)(
		GPluginManager *manager,
		GPluginManagerForeachFunc func,
		gpointer data);

	GSList *(*find_plugins)(GPluginManager *manager, const gchar *id);
	GSList *(*find_plugins_with_state)(
		GPluginManager *manager,
		GPluginPluginState state);

	GList *(*list_plugins)(GPluginManager *manager);

	GSList *(*get_plugin_dependencies)(
		GPluginManager *manager,
		GPluginPlugin *plugin,
		GError **error);

	gboolean (*load_plugin)(
		GPluginManager *manager,
		GPluginPlugin *plugin,
		GError **error);
	gboolean (*unload_plugin)(
		GPluginManager *manager,
		GPluginPlugin *plugin,
		GError **error);

	/* signals */
	gboolean (*loading_plugin)(
		GPluginManager *manager,
		GPluginPlugin *plugin,
		GError **error);
	void (*loaded_plugin)(GPluginManager *manager, GPluginPlugin *plugin);
	void (*load_failed)(GPluginManager *manager, GPluginPlugin *plugin);
	gboolean (*unloading_plugin)(
		GPluginManager *manager,
		GPluginPlugin *plugin,
		GError **error);
	void (*unloaded_plugin)(GPluginManager *manager, GPluginPlugin *plugin);
	void (
		*unload_plugin_failed)(GPluginManager *manager, GPluginPlugin *plugin);

	/*< private >*/
	gpointer reserved[8];
};

void gplugin_manager_append_path(const gchar *path);
void gplugin_manager_prepend_path(const gchar *path);
void gplugin_manager_remove_path(const gchar *path);
void gplugin_manager_remove_paths(void);

void gplugin_manager_add_default_paths(void);
void gplugin_manager_add_app_paths(const gchar *prefix, const gchar *appname);

GList *gplugin_manager_get_paths(void);

gboolean gplugin_manager_register_loader(GType type, GError **error);
gboolean gplugin_manager_unregister_loader(GType type, GError **error);
GSList *gplugin_manager_get_loaders(void);

void gplugin_manager_refresh(void);

void gplugin_manager_foreach(GPluginManagerForeachFunc func, gpointer data);

GSList *gplugin_manager_find_plugins(const gchar *id);
GSList *gplugin_manager_find_plugins_with_version(
	const gchar *id,
	const gchar *op,
	const gchar *version);
GSList *gplugin_manager_find_plugins_with_state(GPluginPluginState state);

GPluginPlugin *gplugin_manager_find_plugin(const gchar *id);
GPluginPlugin *gplugin_manager_find_plugin_with_newest_version(const gchar *id);

GSList *gplugin_manager_get_plugin_dependencies(
	GPluginPlugin *plugin,
	GError **error);

gboolean gplugin_manager_load_plugin(GPluginPlugin *plugin, GError **error);
gboolean gplugin_manager_unload_plugin(GPluginPlugin *plugin, GError **error);

GList *gplugin_manager_list_plugins(void);

GPluginManager *gplugin_manager_get_instance(void);

G_END_DECLS

#endif /* GPLUGIN_MANAGER_H */
