#define _XOPEN_SOURCE 600

#include "global.h"
#include "struct.h"

#include "add_data.h"
#include "dir_traverse.h"
#include "same_or_different.h"

#include <stdio.h>
#include <stdlib.h>

#include <fcntl.h>
#include <sys/stat.h>
#include <unistd.h>
#include <glib.h>

// 関数プロトタイプ
UNList *  file_list_maker(UNList *cp_list,
					UNList *hdd_list,
					struct argument *arg_from,
					struct argument *arg_to,
					int argv_count,
					int optind,
					int argc,
					char *argv[]);
static inline __attribute__((always_inline)) void create_full_path(const char *tmp1, char *tmp2);
static UNList * link_to_target(UNList *cp_list,
								struct argument *arg_from,
								struct argument *arg_to,
								char c);
static UNList * file_to_target(UNList *cp_list,
								UNList *hdd_list,
								struct argument *arg_from,
								struct argument *arg_to,
								char c);

/*******************************************************************************
コピーするファイルのリストを作成する。
引数大杉。
*******************************************************************************/
UNList * file_list_maker(UNList *cp_list,
					UNList *hdd_list,
					struct argument *arg_from,
					struct argument *arg_to,
					int argv_count,
					int optind,
					int argc,
					char *argv[])
{
	// コピー元が単一の場合
	if(argv_count == 1)
	{
		// シンボリックリンクの場合
		if(g_file_test(argv[optind], G_FILE_TEST_IS_SYMLINK))
		{
			// Link to Dir
			if(g_file_test(argv[argc - 1], G_FILE_TEST_IS_DIR) == TRUE)
			{
				arg_from->path = argv[optind];
				arg_to->path = argv[argc - 1];
				cp_list = link_to_target(cp_list,
										arg_from,
										arg_to,
										DIRECTORY);
			}
			// Link to Nothing
			// Link to File
			else if(g_file_test(argv[argc - 1], G_FILE_TEST_EXISTS) == FALSE)
			{
				arg_from->path = argv[optind];
				arg_to->path = argv[argc - 1];
				cp_list = link_to_target(cp_list,
										arg_from,
										arg_to,
										SYMBOLIC);
			}
		}
		// コピー元がファイルの場合
		else if(g_file_test(argv[optind], G_FILE_TEST_IS_REGULAR))
		{
			create_full_path(argv[optind], arg_from->path);
			// File to Dir
			if(g_file_test(argv[argc - 1], G_FILE_TEST_IS_DIR) == TRUE)
			{
				create_full_path(argv[argc - 1], arg_to->path);
				cp_list = file_to_target(cp_list,
										hdd_list,
										arg_from,
										arg_to,
										DIRECTORY);
			}
			// File to Nothing
			else if(g_file_test(argv[argc - 1], G_FILE_TEST_EXISTS) == FALSE)
			{
				errno = 0;
				int ito;
				if((ito = open(argv[argc - 1], O_RDWR | O_CREAT, S_IRUSR | S_IWUSR)) == -1)
				{
					fprintf(stderr, "%s のオープンに失敗しました\n", argv[argc - 1]);
					fprintf(stderr, "errno %d\n", errno);
					exit(-1);
				}
				close(ito);
				create_full_path(argv[argc - 1], arg_to->path);
				unlink(argv[argc - 1]);
				cp_list = file_to_target(cp_list,
										hdd_list,
										arg_from,
										arg_to,
										REGULAR);
			}
			// File to File
			else
			{
				create_full_path(argv[argc - 1], arg_to->path);
				cp_list = file_to_target(cp_list,
										hdd_list,
										arg_from,
										arg_to,
										REGULAR);
			}
		}
		// コピー元がフォルダの場合
		else if(g_file_test(argv[optind], G_FILE_TEST_IS_DIR))
		{
			create_full_path(argv[optind], arg_from->path);
			// Dir to Dir
			if(g_file_test(argv[argc - 1], G_FILE_TEST_IS_DIR) == TRUE)
			{
				create_full_path(argv[argc - 1], arg_to->path);
				same_or_different(hdd_list, arg_from, arg_to);
				cp_list = dir_traverse(cp_list, arg_from->path, arg_to->path, EXISTS);
			}
			// Dir to Nothing
			else if(g_file_test(argv[argc - 1], G_FILE_TEST_EXISTS) == FALSE)
			{
				errno = 0;
				if(mkdir(argv[argc - 1], 0755) == -1)
				{
					fprintf(stderr, "%s の作成に失敗しました\n", argv[argc - 1]);
					fprintf(stderr, "%d\n", errno);
					exit(-1);
				}
				create_full_path(argv[argc - 1], arg_to->path);
				rmdir(arg_to->path);
				same_or_different(hdd_list, arg_from, arg_to);
				cp_list = dir_traverse(cp_list, arg_from->path, arg_to->path, NOT_EXISTS);
			}
			// Dir to File
			else
			{
				fprintf(stderr, "コピー先が不正です\n");
				exit(-1);
			}
		}
		// その他
		else
		{
			fprintf(stderr, "コピー元が不正です\n");
			exit(-1);
		}
	}
	// コピー元が複数の場合
	else
	{
		// コピー先がフォルダ、もしくは存在しない場合
		if((g_file_test(argv[argc - 1], G_FILE_TEST_IS_DIR) == TRUE) || (g_file_test(argv[argc - 1], G_FILE_TEST_EXISTS) == FALSE))
		{
			errno = 0;
			if(mkdir(argv[argc - 1], 0755) == -1)
			{
				if((errno != EEXIST) && (errno != 0))
				{
					fprintf(stderr, "%s の作成に失敗しました\n", argv[argc - 1]);
					fprintf(stderr, "%d\n", errno);
					exit(-1);
				}
			}
			create_full_path(argv[argc - 1], arg_to->path);

			for(; optind < (argc - 1); optind++)
			{
				// Link to Dir
				if(g_file_test(argv[optind], G_FILE_TEST_IS_SYMLINK))
				{
					arg_from->path = argv[optind];
					arg_to->path = argv[argc - 1];
					cp_list = link_to_target(cp_list,
											arg_from,
											arg_to,
											DIRECTORY);
				}
				// File to Dir
				else if(g_file_test(argv[optind], G_FILE_TEST_IS_REGULAR))
				{
					create_full_path(argv[optind], arg_from->path);
					cp_list = file_to_target(cp_list,
										hdd_list,
										arg_from,
										arg_to,
										DIRECTORY);
				}
				// Dir to Dir
				else if(g_file_test(argv[optind], G_FILE_TEST_IS_DIR))
				{
					create_full_path(argv[optind], arg_from->path);
					same_or_different(hdd_list, arg_from, arg_to);
					cp_list = dir_traverse(cp_list, arg_from->path, arg_to->path, EXISTS);
				}
				// Nothing to Dir
				else if(g_file_test(argv[optind], G_FILE_TEST_EXISTS) == FALSE)
				{
					fprintf(stderr, "%sが存在しません\n", argv[optind]);
				}
			}
		}
		else
		{
			fprintf(stderr, "コピー先が不正です\n");
			exit(-1);
		}
	}

	return cp_list;
}

/*******************************************************************************
絶対パスを作成する
*******************************************************************************/
static void create_full_path(const char *tmp1, char *tmp2)
{
	if((realpath(tmp1, tmp2) == NULL))
	{
		fprintf(stderr, "絶対パスの作成に失敗しました\n");
		exit(-1);
	}
}

/*******************************************************************************
シンボリックリンクの処理
*******************************************************************************/
static UNList * link_to_target(UNList *cp_list,
							struct argument *arg_from,
							struct argument *arg_to,
							char c)
{
	// ターゲットがフォルダの場合
	if(c == DIRECTORY)
	{
		char *from_basename = g_path_get_basename(arg_from->path);
		char *tmp = g_build_path(G_DIR_SEPARATOR_S, arg_to->path, from_basename, NULL);

		cp_list = add_data(cp_list, arg_from->path, tmp, SYMBOLIC);

		free(from_basename);
		free(tmp);
	}
	else
	{
		cp_list = add_data(cp_list, arg_from->path, arg_to->path, SYMBOLIC);
	}

	return cp_list;
}

/*******************************************************************************
ファイルをコピーする場合の処理
*******************************************************************************/
static UNList * file_to_target(UNList *cp_list,
							UNList *hdd_list,
							struct argument *arg_from,
							struct argument *arg_to,
							char c)
{
	same_or_different(hdd_list, arg_from, arg_to);

	// コピー先がフォルダの場合
	if(c == DIRECTORY)
	{
		char *from_basename = g_path_get_basename(arg_from->path);
		char *tmp = g_build_path(G_DIR_SEPARATOR_S, arg_to->path, from_basename, NULL);

		cp_list = add_data(cp_list, arg_from->path, tmp, REGULAR);

		free(from_basename);
		free(tmp);
	}
	else
	{
		cp_list = add_data(cp_list, arg_from->path, arg_to->path, REGULAR);
	}

	return cp_list;
}
