#define _XOPEN_SOURCE 500

#include "struct_OPArg.h"
#include "struct_SDir.h"
#include "xmalloc.h" // inline

#include <string.h>
#include <limits.h>

#include <glib.h>
#include <unistd.h>

#define INPUT_LEN 256
#define DEFAULT_BUFFER 32 * 1024 * 1024

#define HASH_CHECK_MODE_SET \
{\
	if(buf[2] == '1')\
	{\
		oparg->CHECK_MODE = BEFORE;\
	}\
	else if(buf[2] == '2')\
	{\
		oparg->CHECK_MODE = AFTER;\
	}\
	else \
	{\
		fprintf(stderr, "設定ファイルが正しくありません\n");\
		exit(EXIT_FAILURE);\
	}\
}

// 関数プロトタイプ
OPArg * read_conf(const SDir *sdir);

/*******************************************************************************
 * オプションの初期化を行った後、設定ファイルから設定を読み出す
*******************************************************************************/
OPArg * read_conf(const SDir *sdir)
{
	char buf[INPUT_LEN];
	OPArg *oparg = xmalloc(sizeof(OPArg));
	oparg->CHECK = NOT;
	oparg->I = NOT;
	oparg->L = NOT;
	oparg->M = NOT;
	oparg->V = NOT;
	oparg->EFFECT = NOT;

	oparg->buffer_size = DEFAULT_BUFFER;
	oparg->CHECK_MODE = NOT;
	oparg->WRITE_MODE = SIZE_OR_TIME;
	oparg->THREAD_MODE = DRIVE_AUTO;

	// そもそも~/.config/snowcpが存在するかが重要だ
	if(g_file_test(sdir->user_settings_dir, G_FILE_TEST_EXISTS) == TRUE)
	{
		if(chdir(sdir->user_settings_dir) == 0)
		{
			if(g_file_test(SNOWCP_CONF, G_FILE_TEST_EXISTS) == TRUE)
			{
				FILE *config = fopen(SNOWCP_CONF, "r");

				if(config == NULL)
				{
					fprintf(stderr, "設定ファイルのオープンに失敗しました\n");
					fprintf(stderr, "強制終了します\n");
					exit(EXIT_FAILURE);
				}

				while(fgets(buf, INPUT_LEN, config) != NULL)
				{
					switch(buf[0])
					{
					// 設定ファイルのバージョン
					case '!':
						// 改行文字をヌル文字に
						buf[strlen(buf) - 1] = '\0';
						char *ver = strtok(buf, "," );
						ver = strtok(NULL, "," );

						if(ver != NULL)
						{
							if(strcmp(SNOWCP_CONF_VER, ver) != 0)
							{
								fprintf(stderr, "設定ファイルに互換性がありません\n");
								fprintf(stderr, "-o オプションを使用して作成し直してください\n");
								exit(EXIT_FAILURE);
							}
						}
						else
						{
							fprintf(stderr, "設定ファイルが正しくありません\n");
							exit(EXIT_FAILURE);
						}
						break;
					// SHA1SUM
					case '1':
						oparg->CHECK = ONE;
						HASH_CHECK_MODE_SET
						break;
					// MD5SUM
					case '5':
						oparg->CHECK = FIVE;
						HASH_CHECK_MODE_SET
						break;
					// バッファサイズを指定
					case 'b':
						{
							// 改行文字をヌル文字に
							buf[strlen(buf) - 1] = '\0';
							char *cp = strtok(buf, "," );
							cp = strtok(NULL, "," );

							if(cp != NULL)
							{
								char *endptr = NULL;
								off_t b = strtoll(cp, &endptr, 10);

								if(b > INT_MAX)
								{
									fprintf(stderr, "バッファサイズの上限は2000MBです\n");
									exit(EXIT_FAILURE);
								}

								// サイズをメガバイト単位にする
								oparg->buffer_size = 1024 * 1024 * b;

								if(oparg->buffer_size <= 2)
								{
									fprintf(stderr, "バッファサイズが2以下です\n");
									exit(EXIT_FAILURE);
								}
							}
							else
							{
								fprintf(stderr, "設定ファイルが正しくありません\n");
								exit(EXIT_FAILURE);
							}
						}
						break;
					// コンペア
					case 'c':
						oparg->CHECK = MEMCMP;
						HASH_CHECK_MODE_SET
						break;
					// 上書き時に確認する
					case 'i':
						oparg->I = INTERACTIVE;
						oparg->WRITE_MODE = OVERWRITE;
						break;
					// コンペアのログを保存する
					case 'l':
						oparg->L = LOG;
						break;
					// 移動モード
					case 'm':
						oparg->M = MOVE;
						break;
					// ベリファイ（MD5）
					case 'r':
						oparg->CHECK = V_FIVE;
						HASH_CHECK_MODE_SET
						break;
					// スレッドモード
					case 't':
						switch(buf[2])
						{
						case '0':
							oparg->THREAD_MODE = DRIVE_AUTO;
							break;

						case '1':
							oparg->THREAD_MODE = DRIVE_SAME;
							break;

						case '2':
							oparg->THREAD_MODE = DRIVE_DIFFERENT;
							break;

						default:
							fprintf(stderr, "設定ファイルが正しくありません\n");
							exit(EXIT_FAILURE);
							break;
						}
						break;
					// 表示モード
					case 'v':
						oparg->V = VERBOS;
						break;
					// 上書きモード
					case 'w':
						oparg->I = NOT;

						switch(buf[2])
						{
						case '0':
							oparg->WRITE_MODE = NO_OVERWRITE;
							break;

						case '1':
							oparg->WRITE_MODE = SIZE_OR_TIME;
							break;

						case '2':
							oparg->WRITE_MODE = NEW_TIME;
							break;

						case '3':
							oparg->WRITE_MODE = OVERWRITE;
							break;

						default:
							fprintf(stderr, "設定ファイルが正しくありません\n");
							exit(EXIT_FAILURE);
							break;
						}
					}
				}

				fclose(config);
			}

			if(fchdir(sdir->current_fd) == -1)
			{
				fprintf(stderr, "カレントディレクトリへのリターンに失敗しました\n");
				exit(EXIT_FAILURE);
			}
		}
		else
		{
			fprintf(stderr, "設定フォルダにアクセスできませんでした\n");
			fprintf(stderr, "強制終了します\n");
			exit(EXIT_FAILURE);
		}
	}

	return oparg;
}
