/*
 * ファイルのコピーを行う。
 * 一部、copybench-1.0（New BSD License）を参考にした。
*/
#define _GNU_SOURCE

#include "ch_time_and_mod.h" // inline
#include "compare_hash.h"
#include "compare_memcmp.h"
#include "print_error.h" // inline
#include "struct_CPDD.h"
#include "struct_CPInfo.h"
#include "struct_OPArg.h"
#include "verify_hash.h"
#include "xmalloc.h"

#include "md5.h"

#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>

#include <errno.h>
#include <fcntl.h>
#include <linux/falloc.h>
#include <pthread.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <unistd.h>

#define TO_OPEN \
{\
	errno = 0;\
	if((ito = open(cpinfo->to, O_CREAT | O_NOATIME | O_NOFOLLOW | O_WRONLY, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH)) == -1)\
	{\
		switch(errno)\
		{\
		case EPERM:\
			errno = 0;\
\
			if((ito = open(cpinfo->to, O_CREAT | O_NOFOLLOW | O_WRONLY, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH)) == -1)\
			{\
				print_error("open", __FILE__, __LINE__, cpdd);\
			}\
			else \
			{\
				to_open = true;\
				cpdd->mk_file_count++;\
			}\
			break;\
\
		default:\
			print_error("open", __FILE__, __LINE__, cpdd);\
			break;\
		}\
	}\
	else \
	{\
		to_open = true;\
		cpdd->mk_file_count++;\
	}\
}

#define HASH_CHECK \
{\
	if((from_open == true) && (to_open == true))\
	{\
		compare_hash(oparg, cpinfo, sdir, cpdd);\
	}\
	else if(cpinfo->skip == false)\
	{\
		fprintf(stderr, "ファイルのオープンに失敗しているため、コンペアをスキップします\n");\
		fprintf(stderr, "from : %s\n", cpinfo->from);\
		fprintf(stderr, "to   : %s\n", cpinfo->to);\
	}\
}

struct verify_data
{
	md5_state_t *state;
	void *buffer;
	int len;
};

static void *buf[2];
static off_t from_size_local;
static off_t written_size[2];
static off_t next_write_size[2];
static off_t total_read_local;
static off_t total_write_local;
static off_t total_read_error; // スレッド用
static off_t total_write_error; // スレッド用
static int ifrom;
static int ito;
static int buffer_size;
static int buffer_size_half;
static int buffer_size_thread;
static int rbuf_num;
static int wbuf_num;
static long psize;
static _Bool from_open;
static _Bool from_stat;
static _Bool to_open;

// ※注意。グローバル変数。
char *write_now;

// 関数プロトタイプ
void read_write(const OPArg *oparg, CPInfo *cpinfo, SDir *sdir, CPDD *cpdd);
static void * read_thread(void *unknown);
static void * write_thread(void *unknown);
static void * do_md5_append(void *tmp);

/*******************************************************************************
*******************************************************************************/
void read_write(const OPArg *oparg, CPInfo *cpinfo, SDir *sdir, CPDD *cpdd)
{
	static struct stat stat_buf;
	static _Bool filesystem_check = false;
	static _Bool fallocate_support = false;
	static _Bool verify = false;
	static _Bool verify_option_check = false;

	from_stat = false;
	from_open = false;
	to_open = false;
	from_size_local = 0;
	written_size[0] = 0;
	written_size[1] = 0;
	next_write_size[0] = 0;
	next_write_size[1] = 0;
	total_read_local = 0;
	total_write_local = 0;
	total_read_error = 0;
	total_write_error = 0;
	ifrom = 0;
	ito = 0;
	rbuf_num = 0;
	wbuf_num = 0;

	if(buffer_size == 0)
	{
		buffer_size = oparg->buffer_size;
	}

	if(buffer_size_half == 0)
	{
		buffer_size_half = buffer_size / 2;
	}

	if(psize == 0)
	{
		psize = sysconf(_SC_PAGESIZE);
	}

	if(verify_option_check == false)
	{
		if(oparg->CHECK == V_FIVE)
		{
			verify = true;
		}

		verify_option_check = true;
	}

	cpinfo->from_size = 0;
	cpinfo->to_size = 0;

	// signalハンドラ用
	write_now = cpinfo->to;

	errno = 0;
	if((ifrom = open(cpinfo->from, O_RDONLY | O_NOATIME | O_NOFOLLOW)) == -1)
	{
		switch(errno)
		{
		case EPERM:
			errno = 0;

			if((ifrom = open(cpinfo->from, O_RDONLY | O_NOFOLLOW)) == -1)
			{
				print_error("open", __FILE__, __LINE__, cpdd);
			}
			else
			{
				from_open = true;
			}
			break;

		default:
			print_error("open", __FILE__, __LINE__, cpdd);
			break;
		}
	}
	else
	{
		from_open = true;
	}

	if(from_open == true)
	{
		errno = 0;
		if(fstat(ifrom, &stat_buf) == -1)
		{
			print_error("fstat", __FILE__, __LINE__, cpdd);
		}
		else
		{
			from_stat = true;
			from_size_local = stat_buf.st_size;
		}

		if((cpinfo->file_test == false) || (oparg->WRITE_MODE == OVERWRITE))
		{
			TO_OPEN
		}
		else if(from_stat == true)
		{
			struct stat stat_to;

			errno = 0;
			if(stat(cpinfo->to, &stat_to) == -1)
			{
				print_error("stat", __FILE__, __LINE__, cpdd);

				if(errno == ENOENT)
				{
					TO_OPEN
				}
			}
			else
			{
				if(oparg->WRITE_MODE == SIZE_OR_TIME)
				{
					if((stat_buf.st_size != stat_to.st_size) || (stat_buf.st_mtime != stat_to.st_mtime))
					{
						TO_OPEN
					}
					else
					{
						cpinfo->skip = true;
					}
				}
				else
				{
					if((stat_buf.st_mtime > stat_to.st_mtime))
					{
						TO_OPEN
					}
					else
					{
						cpinfo->skip = true;
					}
				}
			}
		}
		else
		{
			print_error("NODATA", __FILE__, __LINE__, cpdd);
			fprintf(stderr, "%s をコピーできません\n", cpinfo->from);
		}
	}

	if(filesystem_check == false)
	{
		char *tofs_tmp = cpinfo->tofs;

		if(tofs_tmp != NULL)
		{
			if
			(
				((tofs_tmp[0] == 'e') && (tofs_tmp[1] == 'x') && (tofs_tmp[2] == 't') && (tofs_tmp[3] == '4'))
				||
				((tofs_tmp[0] == 'x') && (tofs_tmp[1] == 'f') && (tofs_tmp[2] == 's'))
				||
				((tofs_tmp[0] == 'b') && (tofs_tmp[1] == 't') && (tofs_tmp[2] == 'r') && (tofs_tmp[2] == 'f') && (tofs_tmp[2] == 's'))
				||
				((tofs_tmp[0] == 'o') && (tofs_tmp[1] == 'c') && (tofs_tmp[2] == 'f') && (tofs_tmp[2] == 's') && (tofs_tmp[2] == '2'))
			)
			{
				fallocate_support = true;
			}
		}

		filesystem_check = true;
	}

	/*
	 * ファイルシステムがfallocateをサポートしている場合、それを実行して領域を確保する。
	 * 
	 * posix_fallocateはext4以外の、それをサポートしていないファイルシステムだとかなり遅くなる様子。
	 * ext4だと4秒弱で終わるフォルダのコピーが、
	 * ntfsだと12秒はかかる（何もしないと6秒弱、pwriteだと8秒ほど、ftruncateは何もしない場合とほぼ同じ）。
	*/
	if((from_size_local > 0) && (to_open == true) && (fallocate_support == true))
	{
		static int itmp = -1;

		// Ubuntu 11.04だとfallocateの第二引数は FALLOC_FL_KEEP_SIZE と FALLOC_FL_PUNCH_HOLE の二つ。
		// linux/falloc.h のincludeが必要。
		if((itmp = fallocate(ito, FALLOC_FL_KEEP_SIZE, 0, from_size_local)) != 0)
		{
			errno = itmp;
			print_error("fallocate", __FILE__, __LINE__, cpdd);
		}

/*
		if((itmp = posix_fallocate(ito, 0, from_size_local)) != 0)
		{
			errno = itmp;
			print_error("posix_fallocate", __FILE__, __LINE__, cpdd);
		}
*/
	}

	// コピー開始
	if((from_size_local > 0) && (to_open == true))
	{
		static md5_state_t state;
		static md5_byte_t digest[16];
		static struct verify_data thread_argument;
		static pthread_t pverify;

		if(verify == true)
		{
			md5_init(&state);
		}

		if(oparg->V == VERBOS)
		{
			printf("%s -> %s\n", cpinfo->from, cpinfo->to);
		}

		// 1048576 == 1024 * 1024 == 1MB
		if(1048576 > from_size_local)
		{
			errno = 0;
			buf[0] = mmap(NULL, from_size_local, PROT_READ, MAP_PRIVATE, ifrom, 0);

			if(buf[0] == MAP_FAILED)
			{
				perror("mmap");
			}
			else
			{
				total_read_local = from_size_local;

				errno = 0;
				if(madvise(buf[0], from_size_local, MADV_SEQUENTIAL) != 0)
				{
					print_error("madvise", __FILE__, __LINE__, cpdd);
				}

				if(verify == true)
				{
					thread_argument.state = &state;
					thread_argument.buffer = buf[0];
					thread_argument.len = from_size_local;

					pthread_create(&pverify, NULL, do_md5_append, (void *)&thread_argument);
				}

				char *tmp = buf[0];
				const char * const endp = tmp + from_size_local;

				while(tmp < endp)
				{
					errno = 0;
					if((written_size[0] = write(ito, tmp, endp - tmp)) == -1)
					{
						print_error("write", __FILE__, __LINE__, cpdd);
						break;
					}
					tmp += written_size[0];
					total_write_local += written_size[0];
				}

				if(verify == true)
				{
					pthread_join(pverify, NULL);
					md5_finish(&state, digest);
					char *hex_output = xmalloc((16 * 2) + 1);

					for(int di = 0; di < 16; ++di)
					{
						sprintf(hex_output + di * 2, "%02x", digest[di]);
					}

					cpinfo->verify_md5 = hex_output;
				}

				errno = 0;
				if(munmap(buf[0], from_size_local) != 0)
				{
					print_error("munmap", __FILE__, __LINE__, cpdd);
				}
			}
		}
		else if(cpinfo->copy_mode == DIFFERENT)
		{
			{
				VList *b_size_list = oparg->b_size_list;
				b_size_list = b_size_list->tail;

				int *i = (int *)b_size_list->data;
				int low = *i;

				b_size_list = b_size_list->prev;
				int high;

				for(;;)
				{
					i = (int *)b_size_list->data;
					high = *i;

					if(high > from_size_local)
					{
						buffer_size_thread = low;
						break;
					}

					if(b_size_list->prev == NULL)
					{
						buffer_size_thread = buffer_size_half;
						break;
					}
					else
					{
						low = high;
						b_size_list = b_size_list->prev;
					}
				}
			}

			rbuf_num = 0;

			pthread_t t1;
			pthread_t t2;
			pthread_create(&t1, NULL, read_thread, (void *)&rbuf_num);
			pthread_join(t1, NULL);

			for(;;)
			{
				rbuf_num = 1;
				wbuf_num = 0;

				if(verify == true)
				{
					thread_argument.state = &state;
					thread_argument.buffer = buf[wbuf_num];
					thread_argument.len = next_write_size[wbuf_num];

					pthread_create(&pverify, NULL, do_md5_append, (void *)&thread_argument);
				}

				pthread_create(&t1, NULL, read_thread, (void *)&rbuf_num);

				if(verify == true)
				{
					pthread_join(pverify, NULL);
				}

				pthread_create(&t2, NULL, write_thread, (void *)&wbuf_num);
				pthread_join(t1, NULL);
				pthread_join(t2, NULL);

				if((from_size_local <= total_write_local) || (written_size[wbuf_num] <= 0))
				{
					break;
				}

				rbuf_num = 0;
				wbuf_num = 1;

				if(verify == true)
				{
					thread_argument.state = &state;
					thread_argument.buffer = buf[wbuf_num];
					thread_argument.len = next_write_size[wbuf_num];

					pthread_create(&pverify, NULL, do_md5_append, (void *)&thread_argument);
				}

				pthread_create(&t1, NULL, read_thread, (void *)&rbuf_num);

				if(verify == true)
				{
					pthread_join(pverify, NULL);
				}

				pthread_create(&t2, NULL, write_thread, (void *)&wbuf_num);
				pthread_join(t1, NULL);
				pthread_join(t2, NULL);

				if((from_size_local <= total_write_local) || (written_size[wbuf_num] <= 0))
				{
					break;
				}
			}

			if(verify == true)
			{
				md5_finish(&state, digest);
				char *hex_output = xmalloc((16 * 2) + 1);

				for(int di = 0; di < 16; ++di)
				{
					sprintf(hex_output + di * 2, "%02x", digest[di]);
				}

				cpinfo->verify_md5 = hex_output;
			}
		}
		else
		{
			for(;;)
			{
				off_t read_size = from_size_local - total_read_local;

				if(read_size >= buffer_size)
				{
					read_size = buffer_size;
				}
				else
				{
					off_t tmp = read_size % psize;

					if(tmp != read_size)
					{
						read_size = read_size - tmp;
					}
				}

				errno = 0;
				buf[0] = mmap(NULL, read_size, PROT_READ, MAP_PRIVATE, ifrom, total_read_local);

				if(buf[0] == MAP_FAILED)
				{
					perror("mmap");
					break;
				}
				else
				{
					total_read_local += read_size;

					errno = 0;
					if(madvise(buf[0], read_size, MADV_SEQUENTIAL) != 0)
					{
						print_error("madvise", __FILE__, __LINE__, cpdd);
					}

					if(verify == true)
					{
						thread_argument.state = &state;
						thread_argument.buffer = buf[0];
						thread_argument.len = read_size;

						pthread_create(&pverify, NULL, do_md5_append, (void *)&thread_argument);
					}

					char *tmp = buf[0];
					const char * const endp = tmp + read_size;

					while(tmp < endp)
					{
						errno = 0;
						if((written_size[0] = write(ito, tmp, endp - tmp)) == -1)
						{
							print_error("write", __FILE__, __LINE__, cpdd);

							errno = 0;
							if(munmap(buf[0], read_size) != 0)
							{
								print_error("munmap", __FILE__, __LINE__, cpdd);
							}

							goto LOOP_END;
						}
						tmp += written_size[0];
						total_write_local += written_size[0];
					}

					if(verify == true)
					{
						pthread_join(pverify, NULL);
					}

					errno = 0;
					if(munmap(buf[0], read_size) != 0)
					{
						print_error("munmap", __FILE__, __LINE__, cpdd);
					}
				}

				if(total_read_local >= from_size_local)
				{
					break;
				}
			}
LOOP_END:
;
			if(verify == true)
			{
				md5_finish(&state, digest);
				char *hex_output = xmalloc((16 * 2) + 1);

				for(int di = 0; di < 16; ++di)
				{
					sprintf(hex_output + di * 2, "%02x", digest[di]);
				}

				cpinfo->verify_md5 = hex_output;
			}
		}

		/*
		 * 余計に書き込んだ部分を切り詰める。
		 * または大きなファイルを小さなファイルで上書きした場合のリサイズ。
		 * open時にO_TRUNCしてない場合、重要な処理。
		 * ここを弄ってバージョン0.1.2でバグを出したので、うかつに弄らない。
		*/
		{
			static struct stat to_stat_tmp;

			if(fstat(ito, &to_stat_tmp) == -1)
			{
				print_error("fstat", __FILE__, __LINE__, cpdd);
			}
			else if(from_size_local < to_stat_tmp.st_size)
			{
				if(ftruncate(ito, from_size_local) == -1)
				{
					print_error("ftruncate", __FILE__, __LINE__, cpdd);
				}
			}
		}
	}
	else if((from_size_local == 0) && (to_open == true))
	{
		if(ftruncate(ito, 0) == -1)
		{
			print_error("ftruncate", __FILE__, __LINE__, cpdd);
		}
	}

	if((from_stat == true) && (to_open == true))
	{
		ch_time_and_mod(&ito, &stat_buf, cpinfo, cpdd);
	}

/*
 * fdatasyncするようにしたら、30倍遅くなった。

	if((oparg->CHECK != NOT) && (fallocate_support == true))
	{
		if(fdatasync(ito) == -1)
		{
			print_error("fdatasync", __FILE__, __LINE__, cpdd);
		}
	}
*/

	if(from_open == true)
	{
		if(close(ifrom) == -1)
		{
			print_error("close", __FILE__, __LINE__, cpdd);
		}
	}

	if(to_open == true)
	{
		if(close(ito) == -1)
		{
			print_error("close", __FILE__, __LINE__, cpdd);
		}
	}

	if(total_read_local > 0)
	{
		cpinfo->from_size = from_size_local;
		cpdd->total_read += total_read_local;
	}

	if(total_write_local > 0)
	{
		cpinfo->to_size = total_write_local;
		cpdd->total_write += total_write_local;
	}

	if(total_read_error > 0)
	{
		cpdd->total_error += total_read_error;
	}

	if(total_write_error > 0)
	{
		cpdd->total_error += total_write_error;
	}

	write_now = NULL;

	if((from_open == true) && (to_open == true) && (from_size_local == total_write_local))
	{
		cpinfo->write = true;
	}
	else
	{
		cpinfo->write = false;
	}

	if(oparg->CHECK_MODE == BEFORE)
	{
		switch(oparg->CHECK)
		{
		case ONE:
			HASH_CHECK
			break;

		case FIVE:
			HASH_CHECK
			break;

		case MEMCMP:
			if((cpinfo->from_size > 0) && (cpinfo->to_size > 0))
			{
				compare_memcmp(oparg, cpinfo, sdir, cpdd);
			}
			else
			{
				fprintf(stderr, "ファイルサイズが0以下のため、コンペアをスキップします\n");
				fprintf(stderr, "from : %s\n", cpinfo->from);
				fprintf(stderr, "to   : %s\n", cpinfo->to);
			}
			break;

		case V_FIVE:
			if((cpinfo->verify_md5 != NULL) && (to_open == true))
			{
				verify_hash(oparg, cpinfo, sdir, cpdd);
			}
			else
			{
				if(from_size_local != 0)
				{
					fprintf(stderr, "ベリファイをスキップします\n");
					fprintf(stderr, "to : %s\n", cpinfo->to);
				}
			}
			break;

		default:
			fprintf(stderr, "コンペアオプションが不正です\n");
			break;
		}

		if(oparg->V == VERBOS)
		{
			puts("");
		}
	}
}

/*******************************************************************************
 * 読み取り関数。引数は使用しない。
*******************************************************************************/
static void * read_thread(void *unknown)
{
	int i = rbuf_num;
	unknown++;	// コンパイル時のwarningを出さないようにするため

	if(from_size_local <= total_read_local)
	{
		next_write_size[i] = 0;
		return NULL;
	}

	off_t read_size = from_size_local - total_read_local;

	if(read_size >= buffer_size_thread)
	{
		read_size = buffer_size_thread;
	}
	else
	{
		off_t tmp = read_size % psize;
		if(tmp != read_size)
		{
			read_size = read_size - tmp;
		}
	}

	errno = 0;
	buf[i] = mmap(NULL, read_size, PROT_READ, MAP_PRIVATE, ifrom, total_read_local);

	if(buf[i] == MAP_FAILED)
	{
		// このcpddに意味はない。print_errorの引数用。
		static CPDD cpdd;
		cpdd.total_error = 0;
		print_error("mmap", __FILE__, __LINE__, &cpdd);
		total_read_error++;

		next_write_size[i] = 0;
	}
	else
	{
		errno = 0;
		// マルチスレッドの場合、
		// MADV_SEQUENTIALよりもMADV_WILLNEEDの方が速い、ような気がする。
		if(madvise(buf[i], read_size, MADV_WILLNEED) != 0)
		{
			// このcpddに意味はない。print_errorの引数用。
			static CPDD cpdd;
			cpdd.total_error = 0;
			print_error("madvise", __FILE__, __LINE__, &cpdd);
			total_read_error++;
		}

		total_read_local += read_size;
		next_write_size[i] = read_size;
	}

	return NULL;
}

/*******************************************************************************
 * 書き込み関数。引数は使用しない。
*******************************************************************************/
static void * write_thread(void *unknown)
{
	int i = wbuf_num;
	unknown++;	// コンパイル時のwarningを出さないようにするため

	if(next_write_size[i] > 0)
	{
		char *tmp = (char *)buf[i];
		const char * const endp = tmp + next_write_size[i];

		while(tmp < endp)
		{
			errno = 0;
			if((written_size[i] = write(ito, tmp, endp - tmp)) == -1)
			{
				// このcpddに意味はない。print_errorの引数用。
				static CPDD cpdd;
				cpdd.total_error = 0;
				print_error("write", __FILE__, __LINE__, &cpdd);
				total_write_error++;
				break;
			}
			tmp += written_size[i];
			total_write_local += written_size[i];
		}

		errno = 0;
		if(munmap(buf[i], next_write_size[i]) == -1)
		{
			// このcpddに意味はない。print_errorの引数用。
			static CPDD cpdd;
			cpdd.total_error = 0;
			print_error("munmap", __FILE__, __LINE__, &cpdd);
			total_write_error++;
		}
	}
	else
	{
		written_size[i] = 0;
	}

	return NULL;
}

/*******************************************************************************
 * ベリファイ用
*******************************************************************************/
static void * do_md5_append(void *tmp)
{
	struct verify_data *tmp_vd = tmp;
	md5_state_t *state = tmp_vd->state;
	md5_byte_t *buffer = tmp_vd->buffer;
	int len = tmp_vd->len;

	md5_append(state, buffer, len);

	return NULL;
}
