package org.dyndns.nuda.mapper;

import java.io.InputStream;
import java.lang.reflect.Array;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.dyndns.nuda.mapper.annotation.JDBCQuery;
import org.dyndns.nuda.tools.util.ReflectUtil;
import org.dyndns.nuda.tools.util.StringUtil;

/**
 * JavaインタフェースクラスをSQLXMLと連動させるためのハンドラクラスです
 * 
 * @author nkoseki
 * 
 */
public class JDBCXMLInvocationHandler implements InvocationHandler {
	
	private List<InnerBean>	queryBeans			= new ArrayList<InnerBean>();
	
	private Connection		con					= null;
	
	private boolean			useAutoCommit		= true;
	
	private boolean			manualTransaction	= false;
	
	public static final int	FLAGS				= Pattern.DOTALL
														| Pattern.MULTILINE;
	
	/**
	 * コネクションオブジェクト・ハンドラクラスを指定してJDBCハンドラを初期化します
	 * 
	 * @param con
	 *            コネクションオブジェクト
	 * @param handlerClass
	 *            ハンドラクラス
	 * @throws Exception
	 *             例外が起きた場合にスローされます
	 */
	public JDBCXMLInvocationHandler(final Connection con,
			final Class<?> handlerClass) throws Exception {
		this.initContainer(con, handlerClass, true, false);
	}
	
	/**
	 * コネクションオブジェクト・ハンドラクラスを指定してJDBCハンドラを初期化します
	 * 
	 * @param con
	 *            コネクションオブジェクト
	 * @param handlerClass
	 *            ハンドラクラス
	 * @throws Exception
	 *             例外が起きた場合にスローされます
	 */
	public JDBCXMLInvocationHandler(final Connection con,
			final Class<?> handlerClass, final boolean useAutoCommit)
			throws Exception {
		this.initContainer(con, handlerClass, useAutoCommit, false);
	}
	
	/**
	 * コネクションオブジェクト・ハンドラクラスを指定してJDBCハンドラを初期化します
	 * 
	 * @param con
	 *            コネクションオブジェクト
	 * @param handlerClass
	 *            ハンドラクラス
	 * @throws Exception
	 *             例外が起きた場合にスローされます
	 */
	public JDBCXMLInvocationHandler(final Connection con,
			final Class<?> handlerClass, final boolean useAutoCommit,
			final boolean manualTransaction) throws Exception {
		this.initContainer(con, handlerClass, useAutoCommit, manualTransaction);
	}
	
	public Connection getConnection() {
		return this.con;
	}
	
	private void initContainer(final Connection con,
			final Class<?> handlerClass, final boolean useAutoCommit,
			final boolean manualTransaction) throws Exception {
		if (handlerClass == null) {
			return;
		}
		if (!handlerClass.isInterface()) {
			String format = "unsupported class {}. class is not a interface";
			String message = StringUtil.format(format, handlerClass);
			throw new Exception(message);
		}
		
		if (handlerClass.isAnnotationPresent(JDBCQuery.class)) {
			this.con = con;
			this.useAutoCommit = useAutoCommit;
			
			JDBCQuery query = handlerClass.getAnnotation(JDBCQuery.class);
			String xmlPath = query.sqlxml();
			
			QueryXMLReader reader = new QueryXMLReader();
			
			InputStream is = Thread.currentThread().getContextClassLoader()
					.getResourceAsStream(xmlPath);
			
			reader.read(is);
			
			List<QueryXMLBean> result = reader.getResult();
			
			for (QueryXMLBean xmlBean : result) {
				InnerBean inBean = this.convertSQLFromXML(xmlBean.getType(),
						xmlBean.getSql());
				inBean.id = xmlBean.getId();
				inBean.type = xmlBean.getType();
				this.queryBeans.add(inBean);
			}
			
		} else {
			String format = "unsupported class {}. class is not supported annotation<org.dyndns.nuda.repserv.datastore.annotation.JDBCQuery>";
			String message = StringUtil.format(format, handlerClass);
			
			throw new Exception(message);
		}
	}
	
	/**
	 * コネクションオブジェクト・ハンドラクラス・クラスローダを指定してJDBCハンドラを初期化します
	 * 
	 * @param con
	 *            コネクションオブジェクト
	 * @param handlerClass
	 *            ハンドラクラス
	 * @param loader
	 *            sqlxmlを参照可能なクラスローダ
	 * @throws Exception
	 *             例外が起きた場合にスローされます
	 */
	public JDBCXMLInvocationHandler(final Connection con,
			final Class<?> handlerClass, final ClassLoader loader)
			throws Exception {
		if (handlerClass == null) {
			return;
		}
		if (!handlerClass.isInterface()) {
			String format = "unsupported class {}. class is not a interface";
			String message = StringUtil.format(format, handlerClass);
			throw new Exception(message);
		}
		
		if (handlerClass.isAnnotationPresent(JDBCQuery.class)) {
			this.con = con;
			
			// SQLXMLの指定はもっとドラスティックに取得すべき
			// アノテーション指定だとクラス(インタフェース)との結合度が高くなるため
			// DIによる指定の方が良い？
			
			JDBCQuery query = handlerClass.getAnnotation(JDBCQuery.class);
			String xmlPath = query.sqlxml();
			
			QueryXMLReader reader = new QueryXMLReader();
			
			InputStream is = loader.getResourceAsStream(xmlPath);
			reader.read(is);
			
			List<QueryXMLBean> result = reader.getResult();
			
			is.close();
			
			for (QueryXMLBean xmlBean : result) {
				InnerBean inBean = this.convertSQLFromXML(xmlBean.getType(),
						xmlBean.getSql());
				inBean.id = xmlBean.getId();
				inBean.type = xmlBean.getType();
				this.queryBeans.add(inBean);
			}
		} else {
			String format = "unsupported class {}. class is not supported annotation<org.dyndns.nuda.repserv.datastore.annotation.JDBCQuery>";
			String message = StringUtil.format(format, handlerClass);
			
			throw new Exception(message);
		}
	}
	
	private static final String		QST			= "?";
	private static final String		SPACE		= " ";
	private static final String		BLANK		= "";
	
	// SELECT用
	private static final String		p1StrSub	= "(like|=|>|<|<=|>=)\\s+?(\\S+?\\s*?/\\*.+?\\*/)";
	private static final String		p2Str		= "(\\S+?\\s*/\\*\\s*?(.+?)\\s*?\\*/)";
	
	// INSERT用
	private static final String		p1InsStr	= "((\\d+|'.*?')\\s*?(/\\*.+?\\*/))";
	private static final Pattern	P1INS		= Pattern.compile(p1InsStr,
														FLAGS);
	
	private static final String		p2InsStr	= "((\\S+?\\s*|'.+')\\s*?/\\*\\s*?(.+?)\\s*?\\*/)";
	private static final Pattern	P2INS		= Pattern.compile(p2InsStr,
														FLAGS);
	
	// 第一正規表現(id = 1/* id */ のようなシーケンスにマッチする)
	private static final Pattern	P1			= Pattern.compile(p1StrSub);
	
	// 第二正規表現(1/* id */ のようなシーケンスにマッチする)
	private static final Pattern	P2			= Pattern.compile(p2Str, FLAGS);
	
	/**
	 * クエリタイプ・SQL文字列を指定して整形済みSQL文字列を含むJavaBeanを生成します
	 * 
	 * @param queryType
	 *            クエリタイプ(SELECT/INSERT/UPDATE/OTHER)
	 * @param queryStr
	 *            SQL文字列
	 * @return 整形済みSQL文字列を含むJavaBeanを生成します
	 */
	private InnerBean convertSQLFromXML(final String queryType,
			final String queryStr) {
		InnerBean bean = new InnerBean();
		
		if ("SELECT".equals(queryType)) {
			// 判定対象文字列(SQL)
			String source = queryStr;
			
			Matcher m = P1.matcher(source);
			
			String result0 = "";
			
			// '?'の位置(インデックス)とプロパティ名を対応させるためのマップ
			Map<Integer, String> matchMap = new HashMap<Integer, String>();
			int matchIndex = 0;
			
			// 第一正規表現でループさせる
			while (m.find()) {
				int startRegion = m.regionStart();
				int endRegion = m.regionEnd();
				
				// マッチした部分のサブシーケンスを取得する
				Matcher subMatcher = m.region(startRegion, endRegion);
				
				// 正規表現パターンを第二正規表現に変更する
				subMatcher.usePattern(P2);
				
				// 第二正規表現でループさせる
				while (subMatcher.find()) {
					
					// 1/* id */ のようなシーケンスのうち'id'の部分を抽出する
					String matchResult = subMatcher.group(2);
					
					// マッチ部分のインデックスとマッチ結果をマップに保存
					// PreparedStatementで使用するため1オリジンとする
					matchMap.put(matchIndex + 1,
							matchResult.replace(SPACE, BLANK));
					
					// マッチ部分を'?'にリプレースする(Javaで扱えるSQL文へ修正)
					String result2 = subMatcher.replaceFirst(QST);
					result0 = result2;
					
					// リプレース結果を新たなシーケンスとして設定し直す
					subMatcher.reset(result2);
					matchIndex++;
				}
			}
			
			if (result0.isEmpty()) {
				result0 = queryStr;
			}
			
			bean.sqlStr = result0;
			bean.map = matchMap;
		} else if ("INSERT".equals(queryType) || "UPDATE".equals(queryType)
				|| "DELETE".equals(queryType)) {
			// 判定対象文字列(SQL)
			String source = queryStr;
			
			Matcher m = P1INS.matcher(source);
			
			String result0 = "";
			
			// '?'の位置(インデックス)とプロパティ名を対応させるためのマップ
			Map<Integer, String> matchMap = new HashMap<Integer, String>();
			int matchIndex = 0;
			
			// 第一正規表現でループさせる
			while (m.find()) {
				int startRegion = m.regionStart();
				int endRegion = m.regionEnd();
				
				// マッチした部分のサブシーケンスを取得する
				Matcher subMatcher = m.region(startRegion, endRegion);
				
				// 正規表現パターンを第二正規表現に変更する
				subMatcher.usePattern(P2INS);
				
				// 第二正規表現でループさせる
				while (subMatcher.find()) {
					
					// 1/* id */ のようなシーケンスのうち'id'の部分を抽出する
					String matchResult = subMatcher.group(3);
					// matchResult = matchResult.trim();
					
					// マッチ部分のインデックスとマッチ結果をマップに保存
					// PreparedStatementで使用するため1オリジンとする
					matchMap.put(matchIndex + 1,
							matchResult.replace(SPACE, BLANK));
					
					// マッチ部分を'?'にリプレースする(Javaで扱えるSQL文へ修正)
					String result2 = subMatcher.replaceFirst(QST);
					result0 = result2;
					
					// リプレース結果を新たなシーケンスとして設定し直す
					subMatcher.reset(result2);
					matchIndex++;
				}
			}
			
			if (result0.isEmpty()) {
				result0 = queryStr;
			}
			
			bean.sqlStr = result0;
			bean.map = matchMap;
			
		} else {
			bean.sqlStr = queryStr;
			bean.map = null;
		}
		
		return bean;
		
	}
	
	/**
	 * 整形済みSQL文字列を含むJavaBeansです
	 * 
	 * @author nkoseki
	 * 
	 */
	class InnerBean {
		private String					id		= "";
		private String					type	= "";
		private String					sqlStr	= "";
		private Map<Integer, String>	map		= null;
		
		@Override
		public String toString() {
			return "InnerBean [id=" + this.id + ", type=" + this.type
					+ ", sqlStr=" + this.sqlStr + ", map=" + this.map + "]";
		}
		
	}
	
	/**
	 * 整形済みSQL文字列を含むJavaBeansを用いてStatementオブジェクトにパラメータを設定します
	 * 
	 * @param inBean
	 *            整形済みSQL文字列を含むJavaBeans
	 * @param pstmt
	 *            ステートメントオブジェクト
	 * @param parameter
	 *            ステートメントオブジェクトに設定するパラメータオブジェクト
	 * @throws Exception
	 *             例外が発生した場合にスローされます
	 */
	private void initStatement(final InnerBean inBean,
			final PreparedStatement pstmt, final Object parameter)
			throws Exception {
		
		//LoggerFactory.getLogger("A").info("{}, {}", inBean, parameter);
		
		try {
			if (inBean.map != null) {
				for (Entry<Integer, String> entry : inBean.map.entrySet()) {
					int index = entry.getKey();
					String propName = entry.getValue();
					if (parameter != null) {
						Object param01 = parameter;
						String realMethodName = ReflectUtil.PREFIX.GETTER
								.camelCaseTo(propName);
						
						Method getterMethod = null;
						
						try {
							getterMethod = param01.getClass()
									.getDeclaredMethod(realMethodName,
											new Class<?>[] {});
						} catch (Exception e) {
							//							LoggerFactory
							//									.getLogger("JDBCHandler#initStatement")
							//									.error("メソッがありません", e);
							throw e;
						}
						
						Class<?> paramType = getterMethod.getReturnType();
						
						Object paramValue = getterMethod.invoke(param01,
								new Object[] {});
						
						//LoggerFactory.getLogger("A").info("{}", paramValue);
						
						if (paramType == int.class) {
							// intの場合
							pstmt.setInt(index, Integer.class.cast(paramValue));
							
						} else if (paramType == String.class) {
							// Stringの場合
							pstmt.setString(index,
									String.class.cast(paramValue));
							
						} else if (paramType == java.util.Date.class) {
							// java.util.Dateの場合
							java.util.Date d = java.util.Date.class
									.cast(paramValue);
							
							pstmt.setDate(index, new java.sql.Date(d.getTime()));
						} else if (paramType == java.sql.Date.class) {
							// java.util.Dateの場合
							java.sql.Date d = java.sql.Date.class
									.cast(paramValue);
							
							pstmt.setDate(index, d);
							
						} else if (paramType == long.class) {
							// longの場合
							pstmt.setLong(index, Long.class.cast(paramValue));
							
						} else if (paramType == double.class) {
							// doubleの場合
							pstmt.setDouble(index,
									Double.class.cast(paramValue));
							
						} else if (paramType == BigInteger.class) {
							// BigIntegerの場合
							// 処理しない
							throw new Exception("unsupported type BigInteger");
							
						} else if (paramType == BigDecimal.class) {
							// BigDecimalの場合
							pstmt.setBigDecimal(index,
									BigDecimal.class.cast(paramValue));
							
						} else if (paramType == boolean.class) {
							// booleanの場合
							pstmt.setBoolean(index,
									Boolean.class.cast(paramValue));
							
						} else {
							// それ以外の場合(無視)
							
							throw new Exception("unsupported type " + paramType);
						}
					}
				}
			}
		} catch (Exception e) {
			throw e;
		}
		
	}
	
	private static Map<String, PreparedStatement>	prepareCache	= new HashMap<String, PreparedStatement>();
	
	@SuppressWarnings("unchecked")
	@Override
	public Object invoke(final Object proxy, final Method method,
			final Object[] args) throws Throwable {
		
		String methodName = method.getName();
		
		if (methodName.equals("toString")) {
			String result = this.queryBeans.toString();
			
			return result;
		}
		
		//		LoggerFactory.getLogger("JDBCXMLInvocationHandler").debug(
		//				"クエリの実行を開始します: {}", methodName);
		
		InnerBean inBean = null;
		
		// メソッド名を元にXMLよりクエリを取得する
		for (InnerBean aBean : this.queryBeans) {
			if (methodName.equals(aBean.id)) {
				inBean = aBean;
				break;
			}
		}
		
		if (inBean == null) {
			throw new Exception("QueryInterfaceを生成できませんでした");
		}
		
		String conditionedSQL = "";
		
		// コンディション設定
		// SQL編集機能はAutoResolverパターンを使うように変更すべき？
		if (args != null && args.length > 1) {
			Object obj = args[1];
			if (obj instanceof QueryCondition) {
				Pattern p = Pattern.compile("#@condition#");
				Matcher m = p.matcher(inBean.sqlStr);
				if (m.find()) {
					QueryCondition condition = (QueryCondition) obj;
					String conditionStr = condition.renderCondition();
					conditionedSQL = inBean.sqlStr.replace("#@condition#",
							conditionStr);
					
					//					LoggerFactory.getLogger("JDBCXMLInvocationHandler").debug(
					//							"SQLコンディションを設定しました\n{}", conditionedSQL);
				}
			}
		}
		
		if (conditionedSQL.isEmpty()) {
			conditionedSQL = inBean.sqlStr;
		}
		
		// プリペアキャッシュ
		PreparedStatement pstmt = null;
		
		if (prepareCache.containsKey(conditionedSQL)) {
			pstmt = prepareCache.get(conditionedSQL);
			
			if (pstmt.isClosed()) {
				pstmt = this.con.prepareStatement(conditionedSQL);
				prepareCache.put(conditionedSQL, pstmt);
			}
		} else {
			pstmt = this.con.prepareStatement(conditionedSQL);
			prepareCache.put(conditionedSQL, pstmt);
		}
		
		pstmt.clearParameters();
		
		//		LoggerFactory.getLogger("JDBCXMLInvocationHandler").debug("クエリ実行\n{}",
		//				inBean.sqlStr);
		
		if ("SELECT".equals(inBean.type)) {
			if (inBean.map != null) {
				try {
					
					if (args != null && args.length != 0 && args[0] != null) {
						this.initStatement(inBean, pstmt, args[0]);
					}
					
				} catch (Exception e) {
					//					LoggerFactory.getLogger("JDBCXMLInvocationHandler").error(
					//							"error", e);
				}
			}
			
			// SELECT句
			Object result0 = null;
			
			// 戻り値の型を判定する
			Class<?> cls = method.getReturnType();
			Class<?> returnType = null;
			
			if (cls.equals(List.class)) {
				// メソッド戻り値がListの場合(複数の結果を返す)
				ParameterizedType paramType = (ParameterizedType) method
						.getGenericReturnType();
				
				Type[] types = paramType.getActualTypeArguments();
				
				// Listの型パラメータを戻り値の型とする
				returnType = (Class<?>) types[0];
				
				if (returnType == null) {
					throw new Exception("query SELECT unknown return type");
				}
				
				@SuppressWarnings("rawtypes")
				List result = new ArrayList();
				
				ResultSet rs = pstmt.executeQuery();
				
				while (rs.next()) {
					// returnTypeはList<T> の型パラメータTと一致する
					Object obj = DaoHelper.convertResult(rs, returnType);
					
					// コンパイル時に型パラメータ情報は消えているため
					// Objectの実際の型がList<T>と一致していればOK
					result.add(obj);
				}
				
				try {
					rs.close();
					pstmt.close();
				} catch (Exception e) {
					
				}
				
				result0 = result;
			} else {
				// 単一の結果を返す
				ResultSet rs = pstmt.executeQuery();
				Object obj = null;
				
				Class<?> cls0 = null;
				Object errorResult = null;
				
				// プリミティブからオブジェクトに型変更
				if (cls == int.class) {
					// intの場合
					cls0 = Integer.class;
					errorResult = Integer.valueOf(0);
				} else if (cls == long.class) {
					// longの場合
					cls0 = Long.class;
					errorResult = Integer.valueOf(0);
					
				} else if (cls == double.class) {
					// doubleの場合
					cls0 = Double.class;
					
				} else if (cls == boolean.class) {
					// booleanの場合
					cls0 = Boolean.class;
					
				} else {
					// それ以外の場合(無視)
					cls0 = cls;
				}
				
				if (rs.next()) {
					obj = DaoHelper.convertResult(rs, cls0);
				} else {
					throw new Exception(
							"ResultSetによって0行が返却されたため、値の設定に失敗しました(値設定対象クラス:"
									+ cls.getCanonicalName() + ")");
				}
				
				try {
					rs.close();
					//pstmt.close();
				} catch (Exception e) {
				}
				
				result0 = obj;
			}
			
			return result0;
		} else if ("INSERT".equals(inBean.type) || "UPDATE".equals(inBean.type)
				|| "DELETE".equals(inBean.type)) {
			pstmt.clearParameters();
			// INSERT句
			Class<?>[] paramTypes = method.getParameterTypes();
			if (paramTypes.length > 0) {
				Class<?> paramType = paramTypes[0];
				if (List.class.equals(paramType)) {
					// 引数がリストの場合
					Object param = args[0];
					
					@SuppressWarnings("rawtypes")
					List paramList = (List) param;
					for (Object o : paramList) {
						if (inBean.map != null) {
							this.initStatement(inBean, pstmt, o);
							pstmt.addBatch();
							pstmt.clearParameters();
						}
					}
					
				} else if (paramType.isArray()) {
					// 引数が配列の場合
					int length = Array.getLength(args[0]);
					for (int i = 0; i < length; i++) {
						Object o = Array.get(args[0], i);
						if (inBean.map != null) {
							this.initStatement(inBean, pstmt, o);
							pstmt.addBatch();
							pstmt.clearParameters();
						}
					}
				} else {
					// 引数が単数オブジェクトの場合
					if (inBean.map != null) {
						this.initStatement(inBean, pstmt, args[0]);
						pstmt.addBatch();
						pstmt.clearParameters();
					}
				}
			}
			
			try {
				// トランザクションハンドラを用いた
				// イベント駆動トランザクション管理にすべき？
				if (!this.manualTransaction) {
					this.con.setAutoCommit(this.useAutoCommit);
				}
				
				pstmt.executeBatch();
				
				if (!this.manualTransaction) {
					if (!this.useAutoCommit) {
						this.con.commit();
					}
				}
				
			} catch (SQLException e) {
				//				LoggerFactory.getLogger("JDBCXMLInvocationHandler").error(
				//						"データベースへの書き込みに失敗しました", e);
				if (!this.manualTransaction) {
					if (!this.useAutoCommit) {
						this.con.rollback();
					}
				} else {
					throw e;
				}
				
			} finally {
				//pstmt.close();
			}
		} else if ("CREATE".equals(inBean.type)) {
			// CREATE TABLE/DROP TABLE
			pstmt.executeUpdate();
		} else {
			try {
				// truncate文など(パラメータを必要としないCRUD以外のSQL)
				if (!this.manualTransaction) {
					this.con.setAutoCommit(this.useAutoCommit);
				}
				
				pstmt.execute();
				
				if (!this.manualTransaction) {
					if (!this.useAutoCommit) {
						this.con.commit();
					}
				}
				
			} catch (SQLException e) {
				//				LoggerFactory.getLogger("JDBCXMLInvocationHandler").error(
				//						"データベースへの書き込みに失敗しました", e);
				
				if (!this.manualTransaction) {
					if (!this.useAutoCommit) {
						this.con.rollback();
					}
				} else {
					throw e;
				}
				
			} finally {
				//pstmt.close();
			}
		}
		
		return null;
	}
	
}