/*
 * [概要]
 * 
 * =====================================================================================
 * [履歴]
 * 2014-05-09 n.koseki 新規作成
 * 
 * =====================================================================================
 */
package org.dyndns.nuda.mapper;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

/**
 * JDBC結果セットからJavaBeans等への変換を行うためのユーティリティクラスです
 * 
 * @author nkoseki
 * 
 */
public class DaoHelper {
	private DaoHelper() {
		// インスタンス化させない
	}
	
	private static final String	SETTER_PREFIX	= "set";
	
	/**
	 * SQL結果セットをJavaBeanに変換します
	 * 
	 * @param <D>
	 *            SQL結果セット格納Beanクラス
	 * @param resultSet
	 *            SQL結果セット
	 * @param cls
	 *            SQL結果セット格納Beanクラスオブジェクト
	 * @return SQL結果セット格納Bean
	 */
	@SuppressWarnings("unchecked")
	public static <D> D convertResult(final ResultSet resultSet,
			final Class<D> cls) {
		
		try {
			
			// clsがプリミティブの場合
			if (cls == Integer.class || cls == int.class) {
				// intの場合
				Object result = resultSet.getInt(1);
				if (cls.isPrimitive()) {
					
					return (D) result;
				}
				D result0 = cls.cast(result);
				return result0;
				
			} else if (cls == String.class) {
				// Stringの場合
				Object result = resultSet.getString(1);
				D result0 = cls.cast(result);
				return result0;
				
			} else if (cls == java.util.Date.class) {
				// java.util.Dateの場合
				java.sql.Date result = resultSet.getDate(1);
				
				D result0 = cls.cast(new java.util.Date(result.getTime()));
				return result0;
			} else if (cls == java.sql.Date.class) {
				// java.util.Dateの場合
				Object result = resultSet.getDate(1);
				D result0 = cls.cast(result);
				return result0;
				
			} else if (cls == Long.class || cls == long.class) {
				// longの場合
				Object result = resultSet.getLong(1);
				
				if (cls.isPrimitive()) {
					
					return (D) result;
				}
				
				D result0 = cls.cast(result);
				return result0;
				
			} else if (cls == Double.class || cls == double.class) {
				// doubleの場合
				Object result = resultSet.getDouble(1);
				
				if (cls.isPrimitive()) {
					
					return (D) result;
				}
				
				D result0 = cls.cast(result);
				return result0;
				
			} else if (cls == BigInteger.class) {
				// BigIntegerの場合
				// 処理しない
				throw new SQLException("タイプ:BigIntegerはサポートされておりません");
			} else if (cls == BigDecimal.class) {
				// BigDecimalの場合
				Object result = resultSet.getBigDecimal(1);
				D result0 = cls.cast(result);
				
				return result0;
			} else if (cls == Boolean.class || cls == boolean.class) {
				// booleanの場合
				Object result = resultSet.getBoolean(1);
				
				if (cls.isPrimitive()) {
					
					return (D) result;
				}
				
				D result0 = cls.cast(result);
				return result0;
				
			} else {
				// それ以外の場合
				// DをJavaBeansとして処理する
				
				// カラムラベルの小文字化表現と
				// メソッド名から"set"を除いた部分の小文字化表現が一致した場合のみ
				// 自動変換を行う
				
				// ResultSetMetadataの取得
				ResultSetMetaData metadata = resultSet.getMetaData();
				List<String> columnList = new ArrayList<String>();
				for (int i = 1; i <= metadata.getColumnCount(); i++) {
					String columnLabel = metadata.getColumnLabel(i);
					//System.out.println(columnLabel);
					columnLabel = columnLabel.toLowerCase();
					columnList.add(columnLabel);
				}
				
				// publicなメソッドを抽出する
				Method[] methods = cls.getDeclaredMethods();
				D beans = cls.newInstance();
				for (Method aMethod : methods) {
					// メソッド名が「set」から始まっていた場合を対象とする
					String methodName = aMethod.getName();
					if (!methodName.startsWith(SETTER_PREFIX)) {
						continue;
					}
					
					// キャメルケース変換処理
					// メソッド名から先頭の「set」を取り除き
					// 先頭文字を小文字に変換する
					String operateParamName =
						methodName.substring(3, methodName.length());
					String startChar = operateParamName.substring(0, 1);
					startChar = startChar.toLowerCase();
					operateParamName =
						operateParamName
							.substring(1, operateParamName.length());
					
					String realName = startChar + operateParamName;
					
					int matchCount = 0;
					for (String aColumn : columnList) {
						if (realName.toLowerCase().equals(aColumn)) {
							matchCount++;
							break;
						}
					}
					
					if (matchCount == 0) {
						// JavaBeansの各フィールドがすべて大文字の場合
						for (String aColumn : columnList) {
							if (realName.toUpperCase().equals(aColumn)) {
								realName = realName.toUpperCase();
								matchCount++;
								break;
							}
						}
					}
					
					if (matchCount == 0) {
						continue;
					}
					if (!aMethod.isAccessible()
						&& aMethod.getModifiers() != Modifier.PUBLIC) {
						// メソッドにアクセスできない場合は
						// 処理をスキップ(private, protected, default)
						// (単純にメソッドのアクセス識別子がpublicでない場合に
						// スキップする)
						continue;
					}
					
					Class<?>[] paramClasses = aMethod.getParameterTypes();
					if (paramClasses != null && paramClasses.length == 1) {
						// パラメタが存在し、その個数が１の場合のみ処理の対象とする
						
						Class<?> targetClass = paramClasses[0];
						
						if (targetClass == int.class) {
							// intの場合
							aMethod.invoke(beans, resultSet.getInt(realName));
							
						} else if (targetClass == String.class) {
							// Stringの場合
							aMethod
								.invoke(beans, resultSet.getString(realName));
							
						} else if (targetClass == java.util.Date.class) {
							// java.util.Dateの場合
							
							java.sql.Date d = resultSet.getDate(realName);
							
							aMethod.invoke(
								beans,
								new java.util.Date(d.getTime()));
						} else if (targetClass == java.sql.Date.class) {
							// java.sql.Dateの場合
							
							java.sql.Date d = resultSet.getDate(realName);
							
							aMethod.invoke(beans, d);
						} else if (targetClass == long.class) {
							// longの場合
							aMethod.invoke(beans, resultSet.getLong(realName));
							
						} else if (targetClass == double.class) {
							// doubleの場合
							aMethod
								.invoke(beans, resultSet.getDouble(realName));
							
						} else if (targetClass == BigInteger.class) {
							// BigIntegerの場合
							// 処理しない
							throw new SQLException(
								"タイプ:BigIntegerはサポートされておりません");
						} else if (targetClass == BigDecimal.class) {
							// BigDecimalの場合
							aMethod.invoke(
								beans,
								resultSet.getBigDecimal(realName));
							
						} else if (targetClass == boolean.class) {
							// booleanの場合
							aMethod.invoke(
								beans,
								resultSet.getBoolean(realName));
							
						} else {
							// それ以外の場合(無視)
						}
					}
				}
				
				return beans;
			}
			
		} catch (InstantiationException e) {
			throw new SQLMapperException(e);
		} catch (IllegalAccessException e) {
			// IllegalArgumentExceptionがスローされる可能性
			// はない
			throw new SQLMapperException(e);
		} catch (IllegalArgumentException e) {
			// IllegalArgumentExceptionがスローされる可能性
			// はない
			throw new SQLMapperException(e);
		} catch (InvocationTargetException e) {
			Throwable cause = e.getCause();
			throw new SQLMapperException(cause);
		} catch (SQLException e) {
			throw new SQLMapperException(e);
		}
		
	}
}
