/*
 * SorMap
 * 
 * [history]
 * n.koseki 2013/09/02 V0.3.1.1 SQLインタフェースコンテナ制御機能
 * n.koseki 2014/01/31 V0.3.1.1 toString対応
 * TODO n.koseki 2013/09/02 V0.3.1.2 セーブポイント対応
 * n.koseki 2014/01/31 V0.3.2.1 ログアダプタ対応
 * n.koseki 2014/01/31 V0.3.2.1 プリペアキャッシュ外部化対応
 * n.koseki 2014/01/31 V0.3.2.1 SQL編集リゾルバ対応
 * n.koseki 2014/01/31 V0.3.2.1 未使用のコードを除去
 * TODO n.koseki 2014/01/31 V0.3.2.2 初期処理軽量化対応
 * TODO n.koseki 2014/01/31 V0.3.2.2 SQL実行リゾルバ対応
 * TODO n.koseki 2014/01/31 V0.3.2.2 クエリBeanマップ化対応
 * TODO n.koseki 2014/01/31 V0.3.2.3 パフォーマンスアナライザ対応
 * 
 * @version 0.3.1
 */

/*
 * [概要]
 * 
 * =====================================================================================
 * [履歴]
 * 2014-05-09 n.koseki 新規作成
 * 
 * =====================================================================================
 */
package org.dyndns.nuda.mapper;

import java.io.InputStream;
import java.lang.reflect.Array;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Proxy;
import java.lang.reflect.Type;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.dyndns.nuda.logger.CommonLogger;
import org.dyndns.nuda.logger.LoggerAdaptor;
import org.dyndns.nuda.mapper.annotation.JDBCQuery;
import org.dyndns.nuda.mapper.cache.CacheManager;
import org.dyndns.nuda.mapper.cache.EmbedCacheManager;
import org.dyndns.nuda.mapper.connection.SQLInterfaceConnection;
import org.dyndns.nuda.mapper.event.SQLInterfaceEvent;
import org.dyndns.nuda.mapper.event.SQLInterfaceEventProcessor;
import org.dyndns.nuda.mapper.event.implementation.DeleteEvent;
import org.dyndns.nuda.mapper.event.implementation.InsertEvent;
import org.dyndns.nuda.mapper.event.implementation.RollBackEvent;
import org.dyndns.nuda.mapper.event.implementation.UpdateEvent;
import org.dyndns.nuda.mapper.helper.CommandDataPair;
import org.dyndns.nuda.mapper.helper.SQLInterfaceCommandInvocationHandler;
import org.dyndns.nuda.mapper.helper.command.AcceptCommand;
import org.dyndns.nuda.mapper.helper.xml.SQLInterfaceCommandResultBean;
import org.dyndns.nuda.mapper.helper.xml.SQLInterfaceCommandXMLBean;
import org.dyndns.nuda.mapper.helper.xml.SQLXMLInterfaceCommandScope;
import org.dyndns.nuda.mapper.paramchecker.AutoParameterChecker;
import org.dyndns.nuda.mapper.paramchecker.ParameterCheckResult;
import org.dyndns.nuda.mapper.parameter.AutoSQLParameter;
import org.dyndns.nuda.mapper.parser.AutoSQLParser;
import org.dyndns.nuda.mapper.parser.SQLParser;
import org.dyndns.nuda.mapper.reserver.Reserver;
import org.dyndns.nuda.mapper.sql.editor.AutoSQLEditorResolver;
import org.dyndns.nuda.mapper.sql.editor.SQLEditorResolver;
import org.dyndns.nuda.tools.util.ReflectUtil;
import org.dyndns.nuda.tools.util.StringUtil;
import org.dyndns.nuda.tools.xml.Context;
import org.dyndns.nuda.tools.xml.DefaultXMLReader;

/**
 * JavaインタフェースクラスをSQLXMLと連動させるためのSQLInterfaceコンテナです
 * 
 * @author nkoseki
 * 
 */
public class JDBCXMLInvocationHandler implements InvocationHandler {
	
	private List<SQLInterfaceBean>	queryBeans			=
															new ArrayList<SQLInterfaceBean>();
	
	private SQLInterfaceConnection	con					= null;
	
	private boolean					useAutoCommit		= true;
	
	private boolean					manualTransaction	= false;
	
	public static final int			FLAGS				= Pattern.DOTALL
															| Pattern.MULTILINE;
	
	// -- ADD START 2014/01/31 V0.3.2.1 ログアダプタ対応
	private static LoggerAdaptor	logger				= null;
	
	static {
		logger = CommonLogger.getLoggerAdaptor();
		logger = logger.getLogger("JDBC-Container");
	}
	
	// -- ADD END 2014/01/31 V0.3.2.1 ログアダプタ対応
	
	// Getter/Setter
	
	// -- TODO START 2014/01/31 V0.3.1.2 初期処理軽量化対応
	
	public void setConnection(final SQLInterfaceConnection con) {
		
		logger
			.info("{}({}):{}", "setConnection", "SQLInterfaceConnection", con);
		this.con = con;
	}
	
	public boolean isUseAutoCommit() {
		logger.info("{}({}):{}", "isUseAutoCommit", "void", "");
		return this.useAutoCommit;
	}
	
	public void setUseAutoCommit(final boolean useAutoCommit) {
		logger.info("{}({}):{}", "setUseAutoCommit", "boolean", useAutoCommit);
		this.useAutoCommit = useAutoCommit;
	}
	
	public boolean isManualTransaction() {
		logger.info("{}({}):{}", "isManualTransaction", "void", "");
		return this.manualTransaction;
	}
	
	public void setManualTransaction(final boolean manualTransaction) {
		logger.info(
			"{}({}):{}",
			"setManualTransaction",
			"boolean",
			manualTransaction);
		this.manualTransaction = manualTransaction;
	}
	
	public Connection getConnection() {
		logger.info("{}({}):{}", "getConnection", "void", "");
		return this.con;
	}
	
	// -- TODO START 2013/09/02 V0.3.1.2 セーブポイント対応
	// -- TODO END 2013/09/02 V0.3.1.2 セーブポイント対応
	
	// Getter/Setter
	
	/**
	 * コネクションオブジェクト・ハンドラクラスを指定してJDBCハンドラを初期化します
	 * 
	 * @param con
	 *            コネクションオブジェクト
	 * @param handlerClass
	 *            ハンドラクラス
	 * @throws Exception
	 *             例外が起きた場合にスローされます
	 */
	public JDBCXMLInvocationHandler(final Connection con,
			final Class<?> handlerClass) throws Exception {
		logger.info(
			"{}({}, {}):{}, {}",
			"JDBCXMLInvocationHandler",
			"Connection",
			"Class<?>",
			con,
			handlerClass);
		this.initContainer(con, handlerClass, true, false);
	}
	
	/**
	 * コネクションオブジェクト・ハンドラクラスを指定してJDBCハンドラを初期化します
	 * 
	 * @param con
	 *            コネクションオブジェクト
	 * @param handlerClass
	 *            ハンドラクラス
	 * @throws Exception
	 *             例外が起きた場合にスローされます
	 */
	public JDBCXMLInvocationHandler(final Connection con,
			final Class<?> handlerClass, final boolean useAutoCommit)
			throws Exception {
		logger.info(
			"{}({}, {}, {}):{}, {}, {}",
			"JDBCXMLInvocationHandler",
			"Connection",
			"Class<?>",
			"boolean",
			con,
			handlerClass,
			useAutoCommit);
		this.initContainer(con, handlerClass, useAutoCommit, false);
	}
	
	/**
	 * コネクションオブジェクト・ハンドラクラスを指定してJDBCハンドラを初期化します
	 * 
	 * @param con
	 *            コネクションオブジェクト
	 * @param handlerClass
	 *            ハンドラクラス
	 * @throws Exception
	 *             例外が起きた場合にスローされます
	 */
	public JDBCXMLInvocationHandler(final Connection con,
			final Class<?> handlerClass, final boolean useAutoCommit,
			final boolean manualTransaction) throws Exception {
		logger.info(
			"{}({}, {}, {}, {}):{}, {}, {}, {}",
			"JDBCXMLInvocationHandler",
			"Connection",
			"Class<?>",
			"boolean",
			"boolean",
			con,
			handlerClass,
			useAutoCommit,
			manualTransaction);
		this.initContainer(con, handlerClass, useAutoCommit, manualTransaction);
		
		this.initSQLIFCtrlCommand();
	}
	
	public JDBCXMLInvocationHandler(final Connection con,
			final Class<?> handlerClass, final ClassLoader currentClassLoader,
			final boolean useAutoCommit, final boolean manualTransaction)
			throws Exception {
		logger.info(
			"{}({}, {}, {}, {}, {}):{}, {}, {}, {}, {}",
			"JDBCXMLInvocationHandler",
			"Connection",
			"Class<?>",
			"ClassLoader",
			"boolean",
			"boolean",
			con,
			handlerClass,
			currentClassLoader,
			useAutoCommit,
			manualTransaction);
		this.initContainer(
			con,
			handlerClass,
			currentClassLoader,
			useAutoCommit,
			manualTransaction);
		
		this.initSQLIFCtrlCommand();
	}
	
	/**
	 * コネクションオブジェクト・ハンドラクラス・クラスローダを指定してJDBCハンドラを初期化します
	 * 
	 * @param con
	 *            コネクションオブジェクト
	 * @param handlerClass
	 *            ハンドラクラス
	 * @param loader
	 *            sqlxmlを参照可能なクラスローダ
	 * @throws Exception
	 *             例外が起きた場合にスローされます
	 */
	public JDBCXMLInvocationHandler(final Connection con,
			final Class<?> handlerClass, final ClassLoader loader)
			throws Exception {
		logger.info(
			"{}({}, {}, {}):{}, {}, {}",
			"JDBCXMLInvocationHandler",
			"Connection",
			"Class<?>",
			"ClassLoader",
			con,
			handlerClass,
			loader);
		if (con == null) {
			logger.error("JDBC-Connection is NULL", new SQLMapperException(
				"JDBC-Connection is NULL"));
			throw new SQLMapperException("JDBC-Connection is NULL");
		}
		if (handlerClass == null) {
			logger.error("handler class is NULL", new SQLMapperException(
				"handler class is NULL"));
			throw new SQLMapperException("handler class is NULL");
		}
		if (!handlerClass.isInterface()) {
			String format = "unsupported class [{}]. class is not a interface";
			String message = StringUtil.format(format, handlerClass);
			SQLMapperException e = new SQLMapperException(message);
			logger.error(message, e);
			throw new SQLMapperException(message);
		}
		
		if (handlerClass.isAnnotationPresent(JDBCQuery.class)) {
			if (con instanceof SQLInterfaceConnection) {
				SQLInterfaceConnection sourceConnection =
					(SQLInterfaceConnection) con;
				this.con = sourceConnection;
			} else {
				
				this.con = new SQLInterfaceConnection(con);
			}
			this.manualTransaction = false;
			this.con.setUseManualTransaction(false);
			
			// SQLXMLの指定はもっとドラスティックに取得すべき
			// アノテーション指定だとクラス(インタフェース)との結合度が高くなるため
			// DIによる指定の方が良い？
			
			JDBCQuery query = handlerClass.getAnnotation(JDBCQuery.class);
			String xmlPath = query.sqlxml();
			
			QueryXMLReader reader = new QueryXMLReader();
			
			InputStream is = loader.getResourceAsStream(xmlPath);
			reader.read(is);
			
			List<QueryXMLBean> result = reader.getResult();
			
			is.close();
			
			for (QueryXMLBean xmlBean : result) {
				SQLInterfaceBean inBean =
					this.convertSQLFromXML(xmlBean.getType(), xmlBean.getSql());
				inBean.id = xmlBean.getId();
				inBean.type = xmlBean.getType();
				this.queryBeans.add(inBean);
			}
		} else {
			if (con instanceof SQLInterfaceConnection) {
				SQLInterfaceConnection sourceConnection =
					(SQLInterfaceConnection) con;
				this.con = sourceConnection;
			} else {
				
				this.con = new SQLInterfaceConnection(con);
			}
			this.manualTransaction = false;
			this.con.setUseManualTransaction(false);
			
			// クラス名からXMLファイルロケーションを自動生成
			String className =
				handlerClass.getCanonicalName().replace(".", "/");
			
			className = className + ".xml";
			
			String xmlPath = className;
			
			QueryXMLReader reader = new QueryXMLReader();
			
			InputStream is = loader.getResourceAsStream(xmlPath);
			reader.read(is);
			
			List<QueryXMLBean> result = reader.getResult();
			
			is.close();
			
			for (QueryXMLBean xmlBean : result) {
				SQLInterfaceBean inBean =
					this.convertSQLFromXML(xmlBean.getType(), xmlBean.getSql());
				inBean.id = xmlBean.getId();
				inBean.type = xmlBean.getType();
				this.queryBeans.add(inBean);
			}
		}
	}
	
	/**
	 * SQLインタフェースコンテナ初期化処理
	 * 
	 * @param con
	 *            JDBCコネクション
	 * @param handlerClass
	 *            SQLインタフェースクラス
	 * @param useAutoCommit
	 *            オートコミットモード
	 * @param manualTransaction
	 *            ユーザベーストラン制御区分
	 * @throws Exception
	 */
	private void initContainer(final Connection con,
			final Class<?> handlerClass, final boolean useAutoCommit,
			final boolean manualTransaction) throws Exception {
		logger.info(
			"{}({}, {}, {}, {}):{}, {}, {}, {}",
			"initContainer",
			"Connection",
			"Class<?>",
			"boolean",
			"boolean",
			con,
			handlerClass,
			useAutoCommit,
			manualTransaction);
		if (con == null) {
			logger.error("JDBC-Connection is NULL", new SQLMapperException(
				"JDBC-Connection is NULL"));
			throw new SQLMapperException("JDBC-Connection is NULL");
		}
		if (handlerClass == null) {
			logger.error("handler class is NULL", new SQLMapperException(
				"handler class is NULL"));
			throw new SQLMapperException("handler class is NULL");
		}
		if (!handlerClass.isInterface()) {
			String format = "unsupported class [{}]. class is not a interface";
			String message = StringUtil.format(format, handlerClass);
			SQLMapperException e = new SQLMapperException(message);
			logger.error(message, e);
			throw new SQLMapperException(message);
		}
		
		if (handlerClass.isAnnotationPresent(JDBCQuery.class)) {
			// 引数に指定されたコネクションオブジェクトが
			// ①SQLInterfaceConnectinoか
			// ②それ以外か
			// を判定する
			// JDBCXMLInvocationHandlerが内部で保持するのはすべて
			// SQLInterfaceConnection(SQLイベントの発火が可能なコネクションオブジェクト)となる
			
			if (con instanceof SQLInterfaceConnection) {
				SQLInterfaceConnection sourceConnection =
					(SQLInterfaceConnection) con;
				this.con = sourceConnection;
			} else {
				this.con = new SQLInterfaceConnection(con);
			}
			
			// ユーザベーストランザクション制御と
			// オートコミットモードの設定
			this.con.setUseManualTransaction(manualTransaction);
			this.manualTransaction = manualTransaction;
			this.useAutoCommit = useAutoCommit;
			
			// 引数に指定されたクラスから
			// SQLXMLのロケーションを取得し
			// XMLの解析を行う
			JDBCQuery query = handlerClass.getAnnotation(JDBCQuery.class);
			String xmlPath = query.sqlxml();
			
			// SQLXMLパースオブジェクト生成
			QueryXMLReader reader = new QueryXMLReader();
			
			InputStream is =
				Thread
					.currentThread()
					.getContextClassLoader()
					.getResourceAsStream(xmlPath);
			
			reader.read(is);
			
			List<QueryXMLBean> result = reader.getResult();
			
			for (QueryXMLBean xmlBean : result) {
				SQLInterfaceBean inBean =
					this.convertSQLFromXML(xmlBean.getType(), xmlBean.getSql());
				inBean.id = xmlBean.getId();
				inBean.type = xmlBean.getType();
				this.queryBeans.add(inBean);
			}
			
		} else {
			// String format =
			// "unsupported class [{}]. class is not supported annotation<org.dyndns.nuda.repserv.datastore.annotation.JDBCQuery>";
			// String message = StringUtil.format(format, handlerClass);
			//
			// logger.error(message, new SQLMapperException(message));
			// logger.info(message, new SQLException(message));
			//
			// throw new SQLMapperException(message);
			// 引数に指定されたコネクションオブジェクトが
			// ①SQLInterfaceConnectinoか
			// ②それ以外か
			// を判定する
			// JDBCXMLInvocationHandlerが内部で保持するのはすべて
			// SQLInterfaceConnection(SQLイベントの発火が可能なコネクションオブジェクト)となる
			
			if (con instanceof SQLInterfaceConnection) {
				SQLInterfaceConnection sourceConnection =
					(SQLInterfaceConnection) con;
				this.con = sourceConnection;
			} else {
				this.con = new SQLInterfaceConnection(con);
			}
			
			// ユーザベーストランザクション制御と
			// オートコミットモードの設定
			this.con.setUseManualTransaction(manualTransaction);
			this.manualTransaction = manualTransaction;
			this.useAutoCommit = useAutoCommit;
			
			// クラス名からXMLファイルロケーションを自動生成
			String className =
				handlerClass.getCanonicalName().replace(".", "/");
			
			className = className + ".xml";
			
			String xmlPath = className;
			
			// SQLXMLパースオブジェクト生成
			QueryXMLReader reader = new QueryXMLReader();
			
			InputStream is =
				Thread
					.currentThread()
					.getContextClassLoader()
					.getResourceAsStream(xmlPath);
			
			reader.read(is);
			
			List<QueryXMLBean> result = reader.getResult();
			
			for (QueryXMLBean xmlBean : result) {
				SQLInterfaceBean inBean =
					this.convertSQLFromXML(xmlBean.getType(), xmlBean.getSql());
				inBean.id = xmlBean.getId();
				inBean.type = xmlBean.getType();
				this.queryBeans.add(inBean);
			}
		}
	}
	
	/**
	 * SQLインタフェースコンテナ初期化処理
	 * 
	 * @param con
	 *            JDBCコネクション
	 * @param handlerClass
	 *            SQLインタフェースクラス
	 * @param currentClassLoader
	 *            SQLインタフェースクラスをロードしたクラスローダ
	 * @param useAutoCommit
	 *            オートコミットモード
	 * @param manualTransaction
	 *            ユーザベーストラン制御区分
	 * @throws Exception
	 */
	private void initContainer(final Connection con,
			final Class<?> handlerClass, final ClassLoader currentClassLoader,
			final boolean useAutoCommit, final boolean manualTransaction)
			throws Exception {
		logger.info(
			"{}({}, {}, {}, {}, {}):{}, {}, {}, {}, {}",
			"initContainer",
			"Connection",
			"Class<?>",
			"ClassLoader",
			"boolean",
			"boolean",
			con,
			handlerClass,
			currentClassLoader,
			useAutoCommit,
			manualTransaction);
		if (con == null) {
			logger.error("JDBC-Connection is NULL", new SQLMapperException(
				"JDBC-Connection is NULL"));
			throw new SQLMapperException("JDBC-Connection is NULL");
		}
		if (handlerClass == null) {
			logger.error("handler class is NULL", new SQLMapperException(
				"handler class is NULL"));
			throw new SQLMapperException("handler class is NULL");
		}
		if (!handlerClass.isInterface()) {
			String format = "unsupported class[{}]. class is not a interface";
			String message = StringUtil.format(format, handlerClass);
			SQLMapperException e = new SQLMapperException(message);
			logger.error(message, e);
			throw new SQLMapperException(message);
		}
		
		if (handlerClass.isAnnotationPresent(JDBCQuery.class)) {
			if (con instanceof SQLInterfaceConnection) {
				SQLInterfaceConnection sourceConnection =
					(SQLInterfaceConnection) con;
				this.con = sourceConnection;
			} else {
				this.con = new SQLInterfaceConnection(con);
				
			}
			this.con.setUseManualTransaction(manualTransaction);
			this.manualTransaction = manualTransaction;
			
			this.useAutoCommit = useAutoCommit;
			
			JDBCQuery query = handlerClass.getAnnotation(JDBCQuery.class);
			String xmlPath = query.sqlxml();
			
			QueryXMLReader reader = new QueryXMLReader();
			
			InputStream is = null;
			if (currentClassLoader == null) {
				is =
					Thread
						.currentThread()
						.getContextClassLoader()
						.getResourceAsStream(xmlPath);
			} else {
				is = currentClassLoader.getResourceAsStream(xmlPath);
			}
			
			reader.read(is);
			
			List<QueryXMLBean> result = reader.getResult();
			
			for (QueryXMLBean xmlBean : result) {
				SQLInterfaceBean inBean =
					this.convertSQLFromXML(xmlBean.getType(), xmlBean.getSql());
				inBean.id = xmlBean.getId();
				inBean.type = xmlBean.getType();
				this.queryBeans.add(inBean);
			}
			
		} else {
			if (con instanceof SQLInterfaceConnection) {
				SQLInterfaceConnection sourceConnection =
					(SQLInterfaceConnection) con;
				this.con = sourceConnection;
			} else {
				this.con = new SQLInterfaceConnection(con);
				
			}
			this.con.setUseManualTransaction(manualTransaction);
			this.manualTransaction = manualTransaction;
			
			this.useAutoCommit = useAutoCommit;
			
			// クラス名からXMLファイルロケーションを自動生成
			String className =
				handlerClass.getCanonicalName().replace(".", "/");
			
			className = className + ".xml";
			
			String xmlPath = className;
			
			QueryXMLReader reader = new QueryXMLReader();
			
			InputStream is = null;
			if (currentClassLoader == null) {
				is =
					Thread
						.currentThread()
						.getContextClassLoader()
						.getResourceAsStream(xmlPath);
			} else {
				is = currentClassLoader.getResourceAsStream(xmlPath);
			}
			
			reader.read(is);
			
			List<QueryXMLBean> result = reader.getResult();
			
			for (QueryXMLBean xmlBean : result) {
				SQLInterfaceBean inBean =
					this.convertSQLFromXML(xmlBean.getType(), xmlBean.getSql());
				inBean.id = xmlBean.getId();
				inBean.type = xmlBean.getType();
				this.queryBeans.add(inBean);
			}
		}
	}
	
	// -- START 2013/09/02 V0.3.1.1 SQLインタフェースコンテナ制御機能
	/**
	 * SQLインタフェースコマンド初期化処理
	 **/
	private void initSQLIFCtrlCommand() {
		logger.info("{}({}):{}", "initSQLIFCtrlCommand", "void", "");
		// 現在のクラスローダを取得
		ClassLoader l = Thread.currentThread().getContextClassLoader();
		
		SQLInterfaceCommandInvocationHandler h =
			new SQLInterfaceCommandInvocationHandler(this);
		
		List<Class<?>> commandList = new ArrayList<Class<?>>();
		
		// interface-command.xml読み込み処理
		Context<SQLInterfaceCommandResultBean> ctx =
			new Context<SQLInterfaceCommandResultBean>();
		
		ctx.setValue(new SQLInterfaceCommandResultBean());
		
		DefaultXMLReader<SQLInterfaceCommandResultBean> reader =
			new DefaultXMLReader<SQLInterfaceCommandResultBean>(
				SQLXMLInterfaceCommandScope.SQLInterfaceCommand,
				ctx);
		
		reader.read("interface-command.xml");
		
		SQLInterfaceCommandResultBean resultBean = reader.getResult();
		for (SQLInterfaceCommandXMLBean xmlBean : resultBean.beans) {
			try {
				Class<?> intf = Class.forName(xmlBean.type);
				commandList.add(intf);
			} catch (ClassNotFoundException e) {
				throw new SQLMapperException(e);
			}
		}
		
		commandList.add(AcceptCommand.class);
		
		// SQLインタフェースコマンド実行ハンドラを使って
		// プロキシを生成する
		Object proxy =
			Proxy.newProxyInstance(
				l,
				commandList.toArray(new Class<?>[commandList.size()]),
				h);
		
		this.ctrlCommandProxy = proxy;
		this.commandAcceptor = (AcceptCommand) proxy;
	}
	
	/**
	 * SQLインタフェースコマンドプロキシクラス
	 **/
	private Object					ctrlCommandProxy	= null;
	
	/**
	 * SQLインタフェースコマンドアクセプタクラス
	 **/
	private AcceptCommand			commandAcceptor		= null;
	
	// -- END 2013/09/02 V0.3.1.1 SQLインタフェースコンテナ制御機能
	
	private static final String		QST					= "?";
	private static final String		SPACE				= " ";
	private static final String		BLANK				= "";
	
	/*
	 * 2013/12/09 課題:SQLのパース処理を外部化する
	 */

	// SELECT用(選択系SQL入力シーケンスのマッチとプレースホルダ置換処理)
	private static final String		p1StrSub			=
															"(like|=|>|<|<=|>=)\\s+?(\\S+?\\s*?/\\*.+?\\*/)";
	private static final String		p2Str				=
															"(\\S+?\\s*/\\*\\s*?(.+?)\\s*?\\*/)";
	
	// INSERT用(更新系SQL入力シーケンスのマッチとプレースホルダ置換処理)
	private static final String		p1InsStr			=
															"((\\d+|'.*?')\\s*?(/\\*.+?\\*/))";
	private static final Pattern	P1INS				= Pattern.compile(
															p1InsStr,
															FLAGS);
	
	private static final String		p2InsStr			=
															"((\\S+?\\s*|'.+')\\s*?/\\*\\s*?(.+?)\\s*?\\*/)";
	private static final Pattern	P2INS				= Pattern.compile(
															p2InsStr,
															FLAGS);
	
	// 第一正規表現(id = 1/* id */ のようなシーケンスにマッチする)
	private static final Pattern	P1					= Pattern
															.compile(p1StrSub);
	
	// 第二正規表現(1/* id */ のようなシーケンスにマッチする)
	private static final Pattern	P2					= Pattern.compile(
															p2Str,
															FLAGS);
	
	/**
	 * クエリタイプ・SQL文字列を指定して整形済みSQL文字列を含むJavaBeanを生成します.
	 * S2DaoタイプのプレースホルダをJava純正のプレースホルダ('?')に変換し
	 * ?の出現順とエイリアスを同時に保持するJavaBeansを返します
	 * 
	 * @param queryType
	 *            クエリタイプ(SELECT/INSERT/UPDATE/CREATE/OTHER)
	 * @param queryStr
	 *            SQL文字列
	 * @return 整形済みSQL文字列を含むJavaBeanを生成します
	 */
	private SQLInterfaceBean convertSQLFromXML(final String queryType,
			final String queryStr) {
		logger.info(
			"{}({}, {}):{}, {}",
			"convertSQLFromXML",
			"String",
			"String",
			queryType,
			queryStr);
		SQLInterfaceBean bean = new SQLInterfaceBean();
		
		if ("SELECT".equals(queryType)) {
			// 判定対象文字列(SQL)
			String source = queryStr;
			
			Matcher m = P1.matcher(source);
			
			String result0 = "";
			
			// '?'の位置(インデックス)とプロパティ名を対応させるためのマップ
			Map<Integer, String> matchMap = new HashMap<Integer, String>();
			int matchIndex = 0;
			
			// 第一正規表現でループさせる
			while (m.find()) {
				int startRegion = m.regionStart();
				int endRegion = m.regionEnd();
				
				// マッチした部分のサブシーケンスを取得する
				Matcher subMatcher = m.region(startRegion, endRegion);
				
				// 正規表現パターンを第二正規表現に変更する
				subMatcher.usePattern(P2);
				
				// 第二正規表現でループさせる
				while (subMatcher.find()) {
					
					// 1/* id */ のようなシーケンスのうち'id'の部分を抽出する
					String matchResult = subMatcher.group(2);
					
					// マッチ部分のインデックスとマッチ結果をマップに保存
					// PreparedStatementで使用するため1オリジンとする
					matchMap.put(
						matchIndex + 1,
						matchResult.replace(SPACE, BLANK));
					
					// マッチ部分を'?'にリプレースする(Javaで扱えるSQL文へ修正)
					String result2 = subMatcher.replaceFirst(QST);
					result0 = result2;
					
					// リプレース結果を新たなシーケンスとして設定し直す
					subMatcher.reset(result2);
					matchIndex++;
				}
			}
			
			if (result0.isEmpty()) {
				result0 = queryStr;
			}
			
			bean.sqlStr = result0;
			bean.map = matchMap;
		} else if ("INSERT".equals(queryType)
			|| "UPDATE".equals(queryType)
			|| "DELETE".equals(queryType)) {
			// 判定対象文字列(SQL)
			String source = queryStr;
			
			Matcher m = P1INS.matcher(source);
			
			String result0 = "";
			
			// '?'の位置(インデックス)とプロパティ名を対応させるためのマップ
			Map<Integer, String> matchMap = new HashMap<Integer, String>();
			int matchIndex = 0;
			
			// 第一正規表現でループさせる
			while (m.find()) {
				int startRegion = m.regionStart();
				int endRegion = m.regionEnd();
				
				// マッチした部分のサブシーケンスを取得する
				Matcher subMatcher = m.region(startRegion, endRegion);
				
				// 正規表現パターンを第二正規表現に変更する
				subMatcher.usePattern(P2INS);
				
				// 第二正規表現でループさせる
				while (subMatcher.find()) {
					
					// 1/* id */ のようなシーケンスのうち'id'の部分を抽出する
					String matchResult = subMatcher.group(3);
					// matchResult = matchResult.trim();
					
					// マッチ部分のインデックスとマッチ結果をマップに保存
					// PreparedStatementで使用するため1オリジンとする
					matchMap.put(
						matchIndex + 1,
						matchResult.replace(SPACE, BLANK));
					
					// マッチ部分を'?'にリプレースする(Javaで扱えるSQL文へ修正)
					String result2 = subMatcher.replaceFirst(QST);
					result0 = result2;
					
					// リプレース結果を新たなシーケンスとして設定し直す
					subMatcher.reset(result2);
					matchIndex++;
				}
			}
			
			if (result0.isEmpty()) {
				result0 = queryStr;
			}
			
			bean.sqlStr = result0;
			bean.map = matchMap;
			
		} else {
			bean.sqlStr = queryStr;
			bean.map = null;
		}
		
		return bean;
		
	}
	
	/**
	 * 整形済みSQL文字列を含むJavaBeansを用いてStatementオブジェクトにパラメータを設定します
	 * 
	 * @param inBean
	 *            整形済みSQL文字列を含むJavaBeans
	 * @param pstmt
	 *            ステートメントオブジェクト
	 * @param parameter
	 *            ステートメントオブジェクトに設定するパラメータオブジェクト
	 * @throws Exception
	 *             例外が発生した場合にスローされます
	 */
	private void initStatement(final SQLInterfaceBean inBean,
			final PreparedStatement pstmt, final Object parameter)
			throws Exception {
		logger.info(
			"{}({}, {}, {}):{}, {}, {}",
			"initStatement",
			"SQLInterfaceBean",
			"PreparedStatment",
			"Object",
			inBean,
			pstmt,
			parameter);
		try {
			if (inBean.map != null) {
				for (Entry<Integer, String> entry : inBean.map.entrySet()) {
					int index = entry.getKey();
					String propName = entry.getValue();
					if (parameter != null) {
						Object param01 = parameter;
						String realMethodName =
							ReflectUtil.PREFIX.GETTER.camelCaseTo(propName);
						
						Method getterMethod = null;
						
						try {
							getterMethod =
								param01.getClass().getDeclaredMethod(
									realMethodName,
									new Class<?>[] {});
						} catch (Exception e) {
							throw new SQLMapperException(e);
						}
						
						Class<?> paramType = getterMethod.getReturnType();
						
						Object paramValue =
							getterMethod.invoke(param01, new Object[] {});
						
						// LoggerFactory.getLogger("A").info("{}", paramValue);
						
						// -- START 2014/06/05 V0.3.1.2 パラメータ設定抽象化
						
						AutoSQLParameter.getInstance().setParameter(
							pstmt,
							index,
							paramValue,
							paramType);
						
						//						if (paramType == int.class) {
						//							// intの場合
						//							pstmt.setInt(index, Integer.class.cast(paramValue));
						//							
						//						} else if (paramType == String.class) {
						//							// Stringの場合
						//							pstmt.setString(
						//								index,
						//								String.class.cast(paramValue));
						//							
						//						} else if (paramType == java.util.Date.class) {
						//							// java.util.Dateの場合
						//							java.util.Date d =
						//								java.util.Date.class.cast(paramValue);
						//							
						//							pstmt
						//								.setDate(index, new java.sql.Date(d.getTime()));
						//						} else if (paramType == java.sql.Date.class) {
						//							// java.util.Dateの場合
						//							java.sql.Date d =
						//								java.sql.Date.class.cast(paramValue);
						//							
						//							pstmt.setDate(index, d);
						//							
						//						} else if (paramType == long.class) {
						//							// longの場合
						//							pstmt.setLong(index, Long.class.cast(paramValue));
						//							
						//						} else if (paramType == double.class) {
						//							// doubleの場合
						//							pstmt.setDouble(
						//								index,
						//								Double.class.cast(paramValue));
						//							
						//						} else if (paramType == BigInteger.class) {
						//							// BigIntegerの場合
						//							// 処理しない
						//							throw new SQLMapperException(
						//								"unsupported type BigInteger");
						//							
						//						} else if (paramType == BigDecimal.class) {
						//							// BigDecimalの場合
						//							pstmt.setBigDecimal(
						//								index,
						//								BigDecimal.class.cast(paramValue));
						//							
						//						} else if (paramType == boolean.class) {
						//							// booleanの場合
						//							pstmt.setBoolean(
						//								index,
						//								Boolean.class.cast(paramValue));
						//							
						//						} else {
						//							// それ以外の場合(無視)
						//							
						//							// throw new SQLMapperException("unsupported type "
						//							// + paramType);
						//							
						//							logger.error("", new SQLMapperException(
						//								"unsupported type " + paramType));
						//						}
						
						// -- END 2014/06/05 V0.3.1.2 パラメータ設定抽象化
					} else {
						// パラメータがnullの場合
					}
				}
			} else {
				// プレースホルダバインドマップがnullの場合
			}
		} catch (Exception e) {
			logger.error("", e);
			// throw e;
		}
		
	}
	
	// -- TODO END 2014/01/31 V0.3.1.2 初期処理軽量化対応
	
	// -- DEL END 2014/01/31 V0.3.2.1 プリペアキャッシュ外部化対応
	/*
	 * プリペアドステートメントキャッシュのためのハッシュマップ 2013/12/09 課題:キャッシュ機構は外部化したほうが良い？
	 */
	// private static Map<String, SQLInterfacePreparedStatement> prepareCache =
	// new HashMap<String, SQLInterfacePreparedStatement>();
	// -- DEL END 2014/01/31 V0.3.2.1 プリペアキャッシュ外部化対応
	
	/*
	 * invokeメソッドの問題点 ①メソッドステップ数が長いため保守性に劣る ②複雑な条件分岐が散見されるため、可読性が低い
	 * →acceptメソッドパターンによりif分岐を除去すべきか？
	 * 
	 * ③イベント通信による処理の有用性が不明確
	 */

	// -- ADD START 2014/01/31 V0.3.1.1 toString対応
	/*
	 * メソッド名:toString
	 */
	private static final String	METHOD_NAME_TO_STRING	= "toString";
	// -- ADD END 2014/01/31 V0.3.1.1 toString対応
	
	// -- ADD START 2014/01/31 V0.3.2.1 プリペアキャッシュ外部化対応
	/*
	 * キャッシュマネージャ
	 */
	private CacheManager		cacheManager			= null;
	// -- END START 2014/01/31 V0.3.2.1 プリペアキャッシュ外部化対応
	
	// -- ADD START 2014/01/31 V0.3.2.1 SQL編集リゾルバ対応
	/*
	 * SQLステートメント編集リゾルバ
	 */
	private SQLEditorResolver	sqlEditorResolver		= AutoSQLEditorResolver
															.getInstance();
	
	// -- ADD START 2014/01/31 V0.3.2.1 SQL編集リゾルバ対応
	
	// -- ADD START 2014/04/25 V0.3.3.0 リザーバ対応
	private boolean				isReserved				= false;
	
	public boolean isReserved() {
		return this.isReserved;
	}
	
	public void setReserved(final boolean isReserved) {
		this.isReserved = isReserved;
	}
	
	private Reserver	currentReserver	= null;
	
	public void setReserver(final Reserver reserver) {
		this.currentReserver = reserver;
	}
	
	public Reserver getReserver() {
		return this.currentReserver;
	}
	
	// -- ADD END 2014/04/25 V0.3.3.0 リザーバ対応
	
	/**
	 * SQLインタフェースにおいてメソッドが実行された場合 このメソッドがコールされます
	 */
	@SuppressWarnings("unchecked")
	@Override
	public Object invoke(final Object proxy, final Method method,
			final Object[] args) throws Throwable {
		
		// -- DEL START 2014/01/31 V0.3.2.1 未使用のコードを除去
		// String mes = "";
		// if (args != null) {
		// for (Object arg : args) {
		// mes += arg + ",";
		// }
		// }
		// if (mes.length() > 0) {
		// mes = mes.substring(0, mes.length() - 1);
		// }
		// -- DEL END 2014/01/31 V0.3.2.1 未使用のコードを除去
		
		logger.info(
			"{}({}, {}, {}):{}, {}, {}",
			"invoke",
			"Object",
			"Method",
			"Object",
			"SQLInterface<X>",
			method.getName(),
			"argument<X>");
		
		// -- TODO START 2014/01/31 V0.3.2.3 パフォーマンスアナライザ対応
		long remainingTime = System.currentTimeMillis();
		// -- TODO END 2014/01/31 V0.3.2.3 パフォーマンスアナライザ対応
		
		// -- ADD START 2014/01/31 V0.3.2.1 プリペアキャッシュ外部化対応
		// キャッシュマネージャを用意する
		if (this.cacheManager == null) {
			this.cacheManager = new EmbedCacheManager();
		}
		// -- ADD END 2014/01/31 V0.3.2.1 プリペアキャッシュ外部化対応
		
		// -- ADD START 2014/01/31 V0.3.1.1 toString対応
		// toStringメソッドの場合
		String methodName = method.getName();
		Class<?>[] toStringMethodParamTypes = method.getParameterTypes();
		Class<?> toStringMethodReturnType = method.getReturnType();
		if (toStringMethodParamTypes.length == 0
			|| String.class.equals(toStringMethodReturnType)) {
			if (methodName.equals(METHOD_NAME_TO_STRING)) {
				String result = this.queryBeans.toString();
				
				return result;
			}
		}
		// -- ADD END 2014/01/31 V0.3.1.1 toString対応
		
		// CommandSigniture s = new CommandSigniture(method);
		
		// -- START 2013/09/02 V0.3.1.1 SQLインタフェースコンテナ制御機能
		if (this.commandAcceptor.accept(method)) {
			Class<?> proxyCls = this.ctrlCommandProxy.getClass();
			Method cmdMethod =
				proxyCls.getDeclaredMethod(
					method.getName(),
					method.getParameterTypes());
			return cmdMethod.invoke(this.ctrlCommandProxy, args);
		}
		// -- END 2013/09/02 V0.3.1.1 SQLインタフェースコンテナ制御機能
		
		SQLInterfaceBean inBean = null;
		
		// -- TODO START 2014/01/31 V0.3.2.2 クエリBeanマップ化対応
		
		SQLContainer sqlContainer = null;
		SQLParser parser = new AutoSQLParser();
		// メソッド名を元にXMLよりクエリを取得する
		for (SQLInterfaceBean aBean : this.queryBeans) {
			if (methodName.equals(aBean.id)) {
				inBean = aBean;
				
				sqlContainer = parser.parse(aBean.id, aBean.type, aBean.sqlStr);
				
				break;
			}
		}
		// -- TODO END 2014/01/31 V0.3.2.2 クエリBeanマップ化対応
		
		if (inBean == null) {
			String message0 =
				StringUtil.format(
					"原因:メソッド名:{} に対応するSQLインタフェースメソッドが存在しませんでした",
					methodName);
			String message =
				StringUtil.format("QueryInterfaceを生成できませんでした<{}>", message0);
			throw new SQLMapperException(message);
		}
		
		// -- START 2014/01/31 V0.3.2.1 引数チェック対応
		ParameterCheckResult paramCheckResult =
			AutoParameterChecker.getInstance().checkParameter(method, args);
		if (!paramCheckResult.isValid()) {
			String message0 = "";
			if (paramCheckResult.getErrorMessage() != null) {
				if (paramCheckResult.getCause() != null) {
					message0 =
						StringUtil.format(
							"原因:メソッド名{}:{} ({}):パラメータのチェックに失敗しました",
							methodName,
							paramCheckResult.getErrorMessage(),
							paramCheckResult.getCause().getMessage());
				} else {
					message0 =
						StringUtil.format(
							"原因:メソッド名{}:{}:パラメータのチェックに失敗しました",
							methodName,
							paramCheckResult.getErrorMessage());
				}
				
			} else {
				message0 =
					StringUtil.format(
						"原因:メソッド名:{}:パラメータのチェックに失敗しました",
						methodName);
			}
			String message =
				StringUtil.format("SQLインタフェースのパラメータが不正です<{}>", message0);
			throw new SQLMapperException(message);
		}
		// -- END 2014/01/31 V0.3.2.1 引数チェック対応
		
		// -- MOD START 2014/01/31 V0.3.2.1 SQL編集リゾルバ対応
		// SQLステートメント編集
		String conditionedSQL = "";
		if (this.sqlEditorResolver.accept(args, inBean.sqlStr)) {
			conditionedSQL =
				this.sqlEditorResolver.editSql(args, inBean.sqlStr);
		}
		
		// コンディション設定
		// SQL編集機能はAutoResolverパターンを使うように変更すべき？
		// if (args != null && args.length > 1) {
		// Object obj = args[1];
		// if (obj instanceof QueryCondition) {
		// Pattern p = Pattern.compile("#@condition#");
		// Matcher m = p.matcher(inBean.sqlStr);
		// if (m.find()) {
		// QueryCondition condition = (QueryCondition) obj;
		// String conditionStr = condition.renderCondition();
		// conditionedSQL =
		// inBean.sqlStr.replace("#@condition#", conditionStr);
		// }
		// }
		// }
		
		// if (conditionedSQL.isEmpty()) {
		// // カスタム条件が存在しない場合
		// conditionedSQL = inBean.sqlStr;
		// }
		// -- MOD END 2014/01/31 V0.3.2.1 SQL編集リゾルバ対応
		
		// -- MOD START 2014/01/31 V0.3.2.1 プリペアキャッシュ外部化対応
		// プリペアキャッシュここも外部化可能？
		PreparedStatement pstmt =
			this.cacheManager.getStatement(conditionedSQL, this.con, inBean);
		
		// if (prepareCache.containsKey(conditionedSQL)) {
		// pstmt = prepareCache.get(conditionedSQL);
		//
		// if (pstmt.isClosed()) {
		// pstmt = this.con.prepareStatement(conditionedSQL);
		// prepareCache.put(
		// conditionedSQL,
		// (SQLInterfacePreparedStatement) pstmt);
		// }
		// } else {
		// pstmt = this.con.prepareStatement(conditionedSQL);
		// ((SQLInterfacePreparedStatement) pstmt).setInterfaceBean(inBean);
		// prepareCache.put(
		// conditionedSQL,
		// (SQLInterfacePreparedStatement) pstmt);
		// }
		// -- MOD END 2014/01/31 V0.3.2.1 プリペアキャッシュ外部化対応
		
		pstmt.clearParameters();
		
		// -- TODO START 2014/01/31 V0.3.2.2 SQL実行リゾルバ対応
		// SELECT句の処理
		if ("SELECT".equals(inBean.type)) {
			if (inBean.map != null) {
				try {
					
					if (args != null && args.length != 0 && args[0] != null) {
						
						// args[0]のチェックが必要
						this.initStatement(inBean, pstmt, args[0]);
					}
				} catch (Exception e) {
				}
			}
			
			// SELECT句
			Object result0 = null;
			
			// 戻り値の型を判定する
			Class<?> cls = method.getReturnType();
			Class<?> returnType = null;
			
			if (cls.equals(List.class)) {
				// メソッド戻り値がListの場合(複数の結果を返す)
				ParameterizedType paramType =
					(ParameterizedType) method.getGenericReturnType();
				
				Type[] types = paramType.getActualTypeArguments();
				
				// Listの型パラメータを戻り値の型とする
				returnType = (Class<?>) types[0];
				
				if (returnType == null) {
					throw new SQLMapperException(
						"query SELECT unknown return type");
				}
				
				// コンパイル時に型消滅するため
				// Listのrow許容型を定義する
				@SuppressWarnings("rawtypes")
				List result = new ArrayList();
				
				ResultSet rs = pstmt.executeQuery();
				
				while (rs.next()) {
					// returnTypeはList<T> の型パラメータTと一致する
					Object obj = DaoHelper.convertResult(rs, returnType);
					
					// コンパイル時に型パラメータ情報は消えているため
					// Objectの実際の型がList<T>と一致していればOK
					result.add(obj);
				}
				
				try {
					rs.close();
					// -- START DEL 2013/12/07 V0.3.1.2 プリペアキャッシュのためにクローズ処理をオミット
					// pstmt.close();
					// -- END DEL 2013/12/07 V0.3.1.2 プリペアキャッシュのためにクローズ処理をオミット
				} catch (Exception e) {
					
				}
				
				result0 = result;
			} else {
				// 戻り値がvoidの場合は？？？
				
				// 単一の結果を返す
				ResultSet rs = pstmt.executeQuery();
				Object obj = null;
				
				Class<?> cls0 = null;
				
				// プリミティブからオブジェクトに型変更
				if (cls == int.class) {
					// intの場合
					cls0 = Integer.class;
				} else if (cls == long.class) {
					// longの場合
					cls0 = Long.class;
				} else if (cls == double.class) {
					// doubleの場合
					cls0 = Double.class;
				} else if (cls == boolean.class) {
					// booleanの場合
					cls0 = Boolean.class;
				} else {
					// それ以外の場合(無視)
					cls0 = cls;
				}
				
				if (rs.next()) {
					obj = DaoHelper.convertResult(rs, cls0);
				} else {
					// 結果が0行の場合は空のリストを返す
					// これはまずいのでは？？？
					List<?> result = Arrays.asList(cls0.newInstance());
					result.clear();
					obj = result;
				}
				
				try {
					rs.close();
				} catch (Exception e) {
				}
				
				result0 = obj;
			}
			
			return result0;
			// INSERT/UPDATE/DELETE句の処理
		} else if ("INSERT".equals(inBean.type)
			|| "UPDATE".equals(inBean.type)
			|| "DELETE".equals(inBean.type)) {
			
			pstmt.clearParameters();
			// INSERT句
			Class<?>[] paramTypes = method.getParameterTypes();
			List<Object> operatedList = new ArrayList<Object>();
			if (paramTypes.length > 0) {
				
				Class<?> paramType = paramTypes[0];
				if (List.class.equals(paramType)) {
					// 引数がリストの場合
					Object param = args[0];
					
					@SuppressWarnings("rawtypes")
					List paramList = (List) param;
					for (Object o : paramList) {
						if (inBean.map != null) {
							this.initStatement(inBean, pstmt, o);
							
							pstmt.addBatch();
							pstmt.clearParameters();
							operatedList.add(o);
						}
						
					}
					
				} else if (paramType.isArray()) {
					// 引数が配列の場合
					int length = Array.getLength(args[0]);
					for (int i = 0; i < length; i++) {
						Object o = Array.get(args[0], i);
						if (inBean.map != null) {
							this.initStatement(inBean, pstmt, o);
							pstmt.addBatch();
							pstmt.clearParameters();
							operatedList.add(o);
						}
					}
				} else {
					// 引数が単数オブジェクトの場合
					if (inBean.map != null) {
						this.initStatement(inBean, pstmt, args[0]);
						pstmt.addBatch();
						pstmt.clearParameters();
						operatedList.add(args[0]);
					}
				}
			} else {
				pstmt.addBatch();
			}
			
			try {
				if (!this.manualTransaction) {
					this.con.setAutoCommit(this.useAutoCommit);
				}
				
				int[] results = pstmt.executeBatch();
				
				if (!this.manualTransaction) {
					if (!this.useAutoCommit) {
						SQLInterfaceEvent event = null;
						remainingTime =
							System.currentTimeMillis() - remainingTime;
						if ("INSERT".equals(inBean.type)) {
							InsertEvent event0 = new InsertEvent();
							event0.setInsertedObjects(operatedList);
							event0.setInterfaceBean(inBean);
							event0.setRemainingTime(remainingTime);
							event = event0;
						} else if ("UPDATE".equals(inBean.type)) {
							UpdateEvent event0 = new UpdateEvent();
							event0.setUpdatedObjects(operatedList);
							event0.setInterfaceBean(inBean);
							event0.setRemainingTime(remainingTime);
							event = event0;
						} else if ("DELETE".equals(inBean.type)) {
							DeleteEvent event0 = new DeleteEvent();
							event0.setDeletedObjects(operatedList);
							event0.setInterfaceBean(inBean);
							event0.setRemainingTime(remainingTime);
							event = event0;
						}
						this.con.commitWith(event);
						
					}
				} else {
					CommandDataPair pair = new CommandDataPair();
					pair.setSqlInterfaceBean(inBean);
					for (Object obj : operatedList) {
						pair.addData(obj);
					}
					this.con.addCommand(pair);
				}
				
				Class<?> methodReturnType = method.getReturnType();
				if (methodReturnType.getComponentType() != null) {
					if (methodReturnType.getComponentType().equals(int.class)) {
						return results;
					}
				}
			} catch (SQLException e) {
				if (!this.manualTransaction) {
					if (!this.useAutoCommit) {
						
						this.con.rollback();
						
						RollBackEvent event = new RollBackEvent();
						event.setCause(e);
						event.setInterfaceBean(inBean);
						SQLInterfaceEventProcessor.newInstance().broadcast(
							event);
						// -- START DEL 2013/12/07 V0.3.1.2
						// 異常なオペレーション時に例外を発生させるためにオミット
						// pstmt.getWarnings().printStackTrace();
						// Class<?> methodReturnType = method.getReturnType();
						// if (methodReturnType.getComponentType() != null) {
						// if (methodReturnType.getComponentType().equals(
						// int.class)) {
						// return new int[0];
						// }
						// }
						// -- END DEL 2013/12/07 V0.3.1.2
						// 異常なオペレーション時に例外を発生させるためにオミット
					}
					// -- START ADD 2013/12/07 V0.3.1.2
					// 異常なオペレーション時に例外を発生させるために追加
					throw new SQLMapperException(e);
					// -- START ADD 2013/12/07 V0.3.1.2
					// 異常なオペレーション時に例外を発生させるために追加
				} else {
					throw new SQLMapperException(e);
				}
				
			}
			// CREATE/DROP文の処理
		} else if ("CREATE".equals(inBean.type)) {
			// CREATE TABLE/DROP TABLE
			try {
				pstmt.executeUpdate();
			} catch (SQLException e) {
				throw new SQLMapperException(e);
			}
			// その他truncate文などの処理
		} else {
			try {
				// truncate文など(パラメータを必要としないCRUD以外のSQL)
				if (!this.manualTransaction) {
					this.con.setAutoCommit(this.useAutoCommit);
				}
				
				pstmt.execute();
				
				if (!this.manualTransaction) {
					if (!this.useAutoCommit) {
						this.con.commit();
					}
				}
				
			} catch (SQLException e) {
				if (!this.manualTransaction) {
					if (!this.useAutoCommit) {
						this.con.rollback();
						RollBackEvent event = new RollBackEvent();
						event.setCause(e);
						SQLInterfaceEventProcessor.newInstance().broadcast(
							event);
					}
					throw new SQLMapperException(e);
				} else {
					throw new SQLMapperException(e);
				}
				
			}
		}
		// -- TODO END 2014/01/31 V0.3.2.2 SQL実行リゾルバ対応
		
		return null;
	}
	
}