/*
 * MatchImage.java
 *
 * Created on 2005/01/22, 14:00
 */

package orch;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.KeyEvent;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.util.LinkedList;
import sos.util.MinMaxDouble;
import sos.util.MinMaxInteger;
import sos.graphics.GraphicsAffineTransform;
import sos.graphics.SOSRectangle;

/**
 *
 * @author Scientific Open Source projects (Gaku Tanaka)
 * @version 1.0 (2005/02/15)
 */
public abstract class Matching extends javax.swing.JFrame {
	static protected final int[] dX = new int[]{0,1,0,-1};
	static protected final int[] dY = new int[]{1,0,-1,0};
	static private Color transBlue = new Color(0,0,255,15);
	
	protected volatile boolean pauseFlag;
	protected Rectangle searchBounds = new Rectangle();
	protected SOSRectangle matchedBounds = new SOSRectangle();
	protected SOSRectangle moveArea = new SOSRectangle();
	protected double horizontalOverlapRatio;
	protected double verticalOverlapRatio;
	protected double minOverlapRatio;
	
	private AffineTransform scaleTransform = new AffineTransform();
	private BufferedImage fixedImages;
	
	private ImagePiece[][] ipArray;
	private int nRow;
	private int nColumn;
	private Point startCell;
	
	private GraphicsAffineTransform transform;
	
	/** Creates new form MatchImage */
	public Matching() {
		moveArea.setColor(transBlue);
	}
	
	public void init(ImagePiece[][] array){
		initComponents();
		
		ipArray = array;
		nRow = ipArray.length;
		nColumn = ipArray[0].length;
		int count = 0;
		for (int y=0; y<nRow; y++){
			for (int x=0; x<nColumn; x++){
				if (ipArray[y][x]!=null){
					ipArray[y][x].setIndex(count);
					count++;
				}
			}
		}
		setupCanvas();
		
		pack();
		setVisible(true);
		
		pauseFlag = false;
	}
	
	private void setupCanvas(){
		int rowWidth = 0;
		int originalHeight = 0;
		MinMaxInteger mmiHeight = new MinMaxInteger();
		MinMaxInteger mmiWidth = new MinMaxInteger();
		mmiWidth.setInit(0);
		for (int y=0; y<nRow; y++){
			rowWidth = 0;
			mmiHeight.setInit(0);
			for (int x=0; x<nColumn; x++){
				if (ipArray[y][x]!=null){
					Dimension size = ipArray[y][x].getSize();
					mmiHeight.compareMax(size.height);
					if (x==0 || x==nColumn-1){
						rowWidth += (int)(size.width*1.5);
					}else{
						rowWidth += size.width;
					}
				}
			}
			if (y==0 || y==nRow-1){
				originalHeight += (int)(mmiHeight.getValue()*1.5);
			}else{
				originalHeight += mmiHeight.getValue();
			}
			mmiWidth.compareMax(rowWidth);
		}
		int originalWidth = mmiWidth.getValue();
		
		int canvasWidth = jCanvasMain.getWidth();
		int canvasHeight = (int)(1.0*originalHeight*canvasWidth/originalWidth);
		if (jCanvasMain.getHeight()<canvasHeight){
			canvasHeight = jCanvasMain.getHeight();
			canvasWidth = (int)(1.0*originalWidth*canvasHeight/originalHeight);
		}
		jCanvasMain.setSize(canvasWidth,canvasHeight);
		
		//calculate the mean of the cells which have image
		double meanX = (nColumn-1)/2.0;
		double meanY = (nRow-1)/2.0;
		MinMaxDouble mmd = new MinMaxDouble();
		mmd.setInit(Double.POSITIVE_INFINITY);
		for (int y=0; y<nRow; y++){
			for (int x=0; x<nColumn; x++){
				if (ipArray[y][x]!=null){
					double value = (meanX-x)*(meanX-x)+(meanY-y)*(meanY-y);
					mmd.compareMin(value, new Point(x,y));
				}
			}
		}
		startCell = (Point)mmd.getObject();
		
		int unitWidth = jCanvasMain.getWidth()/(nColumn+1); // +1 for the blank space
		int unitHeight = jCanvasMain.getHeight()/(nRow+1); // +1 for the blank space
		double startX = unitWidth*(startCell.x+0.5);
		double startY = unitHeight*(startCell.y+0.5);
		scaleTransform.setToTranslation(startX, startY);
		
		double ratioWidth = 1.0*jCanvasMain.getWidth()/originalWidth;
		double ratioHeight = 1.0*canvasHeight/originalHeight;
		scaleTransform.scale(ratioWidth,ratioHeight);
		
		// setup graphics objects
		fixedImages = (BufferedImage)createImage(canvasWidth, canvasHeight);
		jCanvasMain.setImage(fixedImages);
		jCanvasMain.getImageTransform().setFullSrc();
		
		transform = new GraphicsAffineTransform(scaleTransform);	
	}
	
	public void start(double horizontal, double vertical, double min){
		horizontalOverlapRatio = horizontal;
		verticalOverlapRatio = vertical;
		minOverlapRatio = min;
		
		LinkedList<Point> fifo = new LinkedList<Point>();
		fifo.add(startCell);
		ImagePiece ip1 = ipArray[startCell.y][startCell.x];
		ip1.setFixed(true);
		drawFixedImage(ip1);
		
		ImagePiece ip2 = null;
		ImagePiece ipSub = null;
		do{
			Point p = fifo.removeFirst();
			ip1 = ipArray[p.y][p.x];
			for (int i=0; i<dX.length; i++){
				int x2 = p.x+dX[i];
				int y2 = p.y+dY[i];
				if (!pauseFlag && 0<=x2 && x2<nColumn && 0<=y2 && y2<nRow){
					ip2 = ipArray[y2][x2];
					if (ip2!=null && !ip2.isFixed()){
						ipSub = getSubImagePiece(x2,y2,i);
						switch(i){
							case 0:
								setDirectionAndSearch(ip1,ip2,ipSub,false);
								break;
							case 1:
								setDirectionAndSearch(ip1,ip2,ipSub,true);
								break;
							case 2:
								setDirectionAndSearch(ip2,ip1,ipSub,false);
								break;
							case 3:
								setDirectionAndSearch(ip2,ip1,ipSub,true);
								break;
						}
						fifo.add(new Point(x2,y2));
					}
				}
			}
		}while(0<fifo.size());
	}
	
	private ImagePiece getSubImagePiece(int x, int y, int i){
		ImagePiece ip = null;
		
		int nx = x+dY[i];
		int ny = y+dX[i];
		if (0<=nx && nx<nColumn && 0<=ny && ny<nRow){
			ip = ipArray[ny][nx];
			if (ip!=null && ip.isFixed()){
				return ip;
			}
		}
		
		nx = x-dY[i];
		ny = y-dX[i];
		if (0<=nx && nx<nColumn && 0<=ny && ny<nRow){
			ip = ipArray[ny][nx];
			if (ip!=null && ip.isFixed()){
				return ip;
			}
		}
		
		return null;
	}
	
	private void setDirectionAndSearch(ImagePiece ip1, ImagePiece ip2, ImagePiece ipSub, boolean isHorizontal){
		if (ip1.isFixed()&ip2.isFixed()){
			return;
		}
		
		ImagePiece ipUnmove;
		ImagePiece ipMove;
		if (ip1.isFixed()){
			ipUnmove = ip1;
			ipMove = ip2;
		}else{
			ipUnmove = ip2;
			ipMove = ip1;
		}
		
		searchBounds = getSearchBounds(ipUnmove,ipMove,isHorizontal);
		//matchedBounds = null;
		drawRects(ipUnmove,ipMove,ipSub,null,null);
		
		if (ipSub!=null){
			Rectangle anotherSearchBounds = getSearchBounds(ipSub,ipMove,!isHorizontal);
			searchBounds = searchBounds.intersection(anotherSearchBounds);
			
			matchThreeImage(ipUnmove,ipMove,ipSub,isHorizontal);
		}else{
			matchTwoImage(ipUnmove,ipMove,isHorizontal);
		}
		ipMove.getBounds().setLocation(matchedBounds.getLocation());
		ipMove.setFixed(true);
		drawFixedImage(ipMove);
	}
	
	abstract protected Rectangle getSearchBounds(ImagePiece ipUnmove, ImagePiece ipMove, boolean isHorizontal);
	
	abstract protected void matchThreeImage(ImagePiece ipUnmove, ImagePiece ipMove, ImagePiece ipSub, boolean isHorizontal);
	
	abstract protected void matchTwoImage(ImagePiece ipUnmove, ImagePiece ipMove, boolean isHorizontal);
	
	private void updateFixedImage(ImagePiece ip){
		Graphics2D g = (Graphics2D)fixedImages.getGraphics();
		AffineTransform positionTransform = new AffineTransform(scaleTransform);
		Rectangle rect = ip.getBounds();
		positionTransform.translate(rect.x, rect.y);
		g.drawImage(ip.getImage(),positionTransform,jCanvasMain);
	}
	
	protected void drawRects(ImagePiece ipUnmove, ImagePiece ipMove, ImagePiece ipSub, SearchPoint sp1, SearchPoint sp2){
		
		jCanvasMain.clearGraphicObject();
		
		jCanvasMain.addGraphicObject(transform);
		
		jCanvasMain.addGraphicObject(ipUnmove);
		if (ipSub!=null){
			jCanvasMain.addGraphicObject(ipSub);
		}
		
		Rectangle rectMove = ipMove.getBounds();
		
		moveArea.setBounds(searchBounds);
		moveArea.width += rectMove.width;
		moveArea.height += rectMove.height;
		jCanvasMain.addGraphicObject(moveArea);
		
		if (matchedBounds!=null){
			matchedBounds.setSize(rectMove.width,rectMove.height);
			jCanvasMain.addGraphicObject(matchedBounds);
		}
		if (sp1!=null){
			jCanvasMain.addGraphicObject(sp1);
		}
		if (sp2!=null){
			jCanvasMain.addGraphicObject(sp2);
		}
		
		jCanvasMain.repaint();
	}
	
	private void drawFixedImage(ImagePiece ip){
		updateFixedImage(ip);
		jCanvasMain.repaint();
	}
	
	/** This method is called from within the constructor to
	 * initialize the form.
	 * WARNING: Do NOT modify this code. The content of this method is
	 * always regenerated by the Form Editor.
	 */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents() {
        jCanvasMain = new sos.canvas.JObjectCanvas();

        setDefaultCloseOperation(javax.swing.WindowConstants.EXIT_ON_CLOSE);
        setTitle("Matching");
        addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyPressed(java.awt.event.KeyEvent evt) {
                formKeyPressed(evt);
            }
        });

        jCanvasMain.setPreferredSize(new java.awt.Dimension(640, 480));
        getContentPane().add(jCanvasMain, java.awt.BorderLayout.CENTER);

        pack();
    }
    // </editor-fold>//GEN-END:initComponents
	
	private void formKeyPressed(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_formKeyPressed
		if (evt.getKeyCode()==KeyEvent.VK_ESCAPE){
			pauseFlag = true;
		}
	}//GEN-LAST:event_formKeyPressed
	
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private sos.canvas.JObjectCanvas jCanvasMain;
    // End of variables declaration//GEN-END:variables
	
}
