package jp.sourceforge.sos.gunmetry.image;

import java.awt.Image;
import java.awt.Window;
import java.awt.image.BufferedImage;
import java.awt.image.DataBufferByte;
import java.io.File;

import javax.swing.SwingUtilities;

import jp.sourceforge.sos.framework.IAbstraction;
import jp.sourceforge.sos.framework.IModel;
import jp.sourceforge.sos.gunmetry.common.Mediator;
import jp.sourceforge.sos.gunmetry.main.ViewFrameMain;
import jp.sourceforge.sos.gunmetry.marker.ModelMarker;
import jp.sourceforge.sos.lib.canvas.JObjectCanvas;
import jp.sourceforge.sos.lib.canvas.JTransformCanvas;
import jp.sourceforge.sos.lib.colorspace.ColorVector;
import jp.sourceforge.sos.lib.graphics.GraphicsImage;
import jp.sourceforge.sos.lib.graphics.GraphicsOffscreen;
import jp.sourceforge.sos.lib.graphics.IGraphics;
import jp.sourceforge.sos.lib.image.ImageInfo;
import jp.sourceforge.sos.lib.io.SOSImageIO;
import jp.sourceforge.sos.lib.morphology.data.DataDoubleArray;
import jp.sourceforge.sos.lib.morphology.data.Elements;
import jp.sourceforge.sos.lib.morphology.extra.Sobel;
import jp.sourceforge.sos.lib.util.MinMaxDouble;

public class ModelImage implements IModel {

	private SOSImageIO			imageFileIO		= new SOSImageIO();

	private File				fileOriginal	= null;

	private File				fileCluster		= null;

	private JTransformCanvas	canvas;

	private GraphicsImage		graphicsImage;

	private double[]			gradient;

	private ImageInfo			imiOriginal		= new ImageInfo();

	private ImageInfo			imiCluster;

	private ModelMarker			modelMarker;

	private GraphicsOffscreen	graphicsOffscreen;

	public ModelImage() {}

	public void setAbstraction(IAbstraction abstraction) {
		modelMarker = (ModelMarker)abstraction;
		canvas = (JTransformCanvas)modelMarker.getCanvas();
		graphicsImage = new GraphicsImage();
		graphicsOffscreen = (GraphicsOffscreen)canvas.getGraphicObject(1);
		graphicsOffscreen.addGraphicObject(0, graphicsImage);
	}

	void setOriginal() {
		if (fileOriginal!=null) {
			setImage(SOSImageIO.readImage(fileOriginal));
		}
	}

	void setGradient() {
		if (gradient!=null) {
			setImage(createGradientImage(gradient));
		}
	}

	void setCluster() {
		if (fileCluster!=null) {
			setImage(SOSImageIO.readImage(fileCluster));
		}
	}

	private void setImage(BufferedImage img) {
		if (img==null) {
			return;
		}
		graphicsImage.setImage(img);
		graphicsOffscreen.update();
		repaint();
	}

	boolean loadOriginal() {
		clearPreviousData();

		imageFileIO.resetChoosableFileFilters();
		imageFileIO.setFileFilter(ViewFrameMain.filterImg);
		fileOriginal = imageFileIO.getOpenFile("Select the original image...");
		if (fileOriginal==null) {
			return false;
		}

		BufferedImage img = SOSImageIO.readImage(fileOriginal);
		imiOriginal = new ImageInfo(img);
		modelMarker.setImageInfo(imiOriginal);
		createGradient();

		if (!isSameSize(img)) {
			canvas.setSize(img.getWidth(), img.getHeight());
			canvas.setSourceSize(img.getWidth(), img.getHeight());
			graphicsOffscreen.create(img.getWidth(), img.getHeight());
		}
		graphicsImage.setImage(img);
		graphicsOffscreen.update();
		canvas.getGraphicsTransform().setFullSrc();

		Window window = SwingUtilities.getWindowAncestor(canvas);
		window.pack();

		// modelMarker.setTitle("Gunmetry Ver2.0 ("+fileOriginal.getName()+")");
		return true;
	}

	private void createGradient() {
		double[][] LAB = new double[imiOriginal.getImageSize()][3];
		ColorVector.getLAB(imiOriginal.getPixels(), LAB);
		
		Sobel sobel = new Sobel();
		Elements elements = new Elements(imiOriginal.getNeighbor(), new DataDoubleArray(LAB));
		sobel.operate(elements);
		gradient = (double[]) elements.getResultData();
		modelMarker.setGradient(gradient);
	}

	private void repaint() {
		canvas.repaint();
	}

	private void clearPreviousData() {
		fileCluster = null;
		imiCluster = null;
		gradient = null;
		
		clearGraphicsOffScreen();
		modelMarker.init();

		Mediator.inform(Mediator.Subjects.READY_TO_ANALYSIS, Boolean.FALSE);
		Mediator.inform(Mediator.Subjects.POST_ANALYSIS, Boolean.FALSE);
	}

	private void clearGraphicsOffScreen() {
		for (int i=0; i<graphicsOffscreen.getObjectsSize(); i++){
			IGraphics obj = graphicsOffscreen.getGraphicObject(i);
			if (obj instanceof GraphicsImage){
				((GraphicsImage)obj).setImage(null);
			}
		}
		graphicsOffscreen.update();
	}

	private boolean isSameSize(BufferedImage img) {
		if (canvas.getSourceWidth()==img.getWidth()&&canvas.getSourceHeight()==img.getHeight()) {
			return true;
		}
		return false;
	}

	private BufferedImage createGradientImage(double[] gradient) {
		// make a BufferedImage as result
		BufferedImage img = new BufferedImage(imiOriginal.getWidth(), imiOriginal.getHeight(), BufferedImage.TYPE_BYTE_GRAY);
		DataBufferByte buffer = (DataBufferByte)img.getRaster().getDataBuffer();

		MinMaxDouble mmd = new MinMaxDouble();
		mmd.compareMax(gradient);
		double maxGradient = mmd.getValue();
		int value;
		for (int i = 0; i<gradient.length; i++) {
			value = (byte) (255*gradient[i]/maxGradient);
			buffer.setElem(i, value);
		}

		return img;
	}

	boolean loadCluster() {
		imageFileIO.resetChoosableFileFilters();
		imageFileIO.setFileFilter(ViewFrameMain.filterGifPng);
		fileCluster = imageFileIO.getOpenFile("Select a file for cluster image...");
		if (fileCluster==null) {
			return false;
		}

		BufferedImage img = SOSImageIO.readImage(fileCluster);
		if (!isSameSize(img)) {
			return false;
		}
		imiCluster = new ImageInfo(img);
		setMarker();

		return true;
	}

	void setMarker() {
		if (imiCluster!=null) {
			modelMarker.setColorPicker(imiCluster);
		}
	}

	JObjectCanvas getCanvas() {
		return canvas;
	}

	Image getImage() {
		return graphicsImage.getImage();
	}

}
