/*
 * 

 AbstractTranslator.java
 
 Copyright 2004 KUBO Hiroya (hiroya@sfc.keio.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */
package net.sf.sqs_xml.translator.logic;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URL;

import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamSource;
import javax.xml.transform.sax.TransformerHandler;
import javax.xml.transform.sax.SAXResult;
import javax.xml.transform.sax.SAXSource;
import javax.xml.transform.sax.SAXTransformerFactory;

import net.sf.sqs_xml.httpd.SQSHttpdFactory;
import net.sf.sqs_xml.translator.logic.Translator;

import org.xml.sax.XMLReader;
import org.xml.sax.helpers.XMLReaderFactory;
import org.apache.xml.serializer.Serializer;
import org.apache.xml.serializer.SerializerFactory;
import org.apache.xml.serializer.OutputPropertiesFactory;
import org.xml.sax.SAXException;
import org.xml.sax.InputSource;
//import org.apache.avalon.framework.logger.NullLogger;
import org.apache.avalon.framework.logger.ConsoleLogger;

public abstract class AbstractTranslator implements Translator{
    final static String SQS_PROPERTY_NAME = "sqs";
    final static java.util.ResourceBundle prop = java.util.ResourceBundle.getBundle(SQS_PROPERTY_NAME);	

	TransformerFactory tFactory = null;
	
	public static void main(String[] args, String message, String[] xsltFilenames, AbstractTranslator self, String baseuri)throws TranslatorException{
		if (args.length < 2) {
			System.err.println(message);
			return;
		}
		final String inFile = args[0];
		final String outFile = args[1];
		self.translate(inFile, new TranslatorResource(new ConsoleLogger(), xsltFilenames), outFile, baseuri);
	}
	
	
	public AbstractTranslator()throws TranslatorException{
        System.setProperty("javax.xml.transform.TransformerFactory", 
        	"org.apache.xalan.processor.TransformerFactoryImpl");
        System.setProperty("javax.xml.parsers.DocumentBuilderFactory",
        	"org.apache.xerces.jaxp.DocumentBuilderFactoryImpl");
        System.setProperty("javax.xml.parsers.SAXParserFactory",
        	"org.apache.xerces.jaxp.SAXParserFactoryImpl");		
		this.tFactory = TransformerFactory.newInstance();
		//this.tFactory.setAttribute("http://apache.org/xalan/features/incremental", Boolean.TRUE);
		//this.tFactory.setAttribute("http://apache.org/xalan/features/source_location", Boolean.TRUE);
		init();		
	}

	public void init()throws TranslatorException{
	    SQSHttpdFactory.getInitialHttpd();
	}
	
	public StreamSource createStreamSource(String href, String base){
		try{
			URL baseURL = new URL(base);
			URL u = new URL(baseURL, href);
			return new StreamSource(u.openStream(), u.toString());
		}catch(java.io.IOException ex){
			throw new RuntimeException(ex);  
		}
	}
	
	public void translate(String sourceFile, TranslatorResource resource, String targetFile, String baseURI) throws TranslatorException{
		BufferedInputStream sourceInputStream = null;
		BufferedOutputStream targetOutputStream = null;
		try{
			targetOutputStream = new BufferedOutputStream(new FileOutputStream(targetFile));
			sourceInputStream = new BufferedInputStream(new FileInputStream(sourceFile));
			translate(sourceInputStream, resource, targetOutputStream, baseURI);
		}catch(IOException ex){
			throw new TranslatorException(ex);
		}finally{
			try{
				sourceInputStream.close();
				targetOutputStream.flush();
				targetOutputStream.close();
			}catch(Exception ignore){
			}
		}
	}

	public void translate(InputStream inputStream, TranslatorResource resource, OutputStream outputStream, String baseURI)throws TranslatorException{
		if (! tFactory.getFeature(SAXSource.FEATURE) || ! tFactory.getFeature(SAXResult.FEATURE)){
			throw new TranslatorException("SAXSource.FEATURE and/or SAXResult.FEATURE are not supported.") ;
		}
		try{
			SAXTransformerFactory saxTFactory = ((SAXTransformerFactory) tFactory);
			java.util.List list = resource.getXsltFilename();
			TransformerHandler[] tHandler = createTransformerHandlers(saxTFactory, list, baseURI);
			XMLReader reader = XMLReaderFactory.createXMLReader();
			reader.setContentHandler(tHandler[0]);
			execute(inputStream, resource, outputStream,
					tHandler[0], tHandler[list.size()-1]);
			reader.parse(new InputSource(inputStream));
		}catch(SAXException ex){
			throw new TranslatorException(ex);
		}catch(IOException ex){
			throw new TranslatorException(ex);
		}catch(TransformerException ex){
			throw new TranslatorException(ex);
		}
	}

	/**
     * @param saxTFactory
     * @param list
     * @return
     * @throws TransformerConfigurationException
     */
    private TransformerHandler[] createTransformerHandlers(SAXTransformerFactory saxTFactory, java.util.List list, String baseURI) throws TransformerConfigurationException {
        TransformerHandler[] tHandler = new TransformerHandler[list.size()];
        for(int i=0; i < list.size(); i++){
        	String filename = (String)list.get(i);
        	tHandler[i] = saxTFactory.newTransformerHandler(createStreamSource(filename, baseURI));
        	tHandler[i].setSystemId(baseURI);
        	initParameters(filename, tHandler[i]);
        	if(0 < i){
        		tHandler[i-1].setResult(new SAXResult(tHandler[i]));					
        	}
        }
        return tHandler;
    }

    public void execute(InputStream inputStream, TranslatorResource resource,
            OutputStream outputStream, TransformerHandler starterHandler,
            TransformerHandler terminatorHandler) throws TranslatorException {
        try {
            Serializer serializer = SerializerFactory
                    .getSerializer(OutputPropertiesFactory
                            .getDefaultMethodProperties("xml"));
            serializer.setOutputStream(outputStream);
            terminatorHandler.setResult(new SAXResult(serializer
                    .asContentHandler()));
        } catch (IOException ex) {
            throw new TranslatorException(ex);
        }
    }
	
	public void initParameters(String filename, TransformerHandler tHandler){
		String[] params = getParameters(filename);
		Transformer transformer = tHandler.getTransformer();
		if(params != null){
			for(int i=0; i < params.length; i+=2){
				transformer.setParameter(params[i], params[i+1]);
			}
		}
	}
	public String[] getParameters(String filename){
	    return new String[]{};
	}
}
