/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew.command;

import java.sql.*;

import net.argius.logging.*;
import net.argius.stew.*;

/**
 * f[^x[Xe[ȕ\R}hB
 */
public final class Report extends Command {

    private static final Logger log = LoggerFactory.getLogger(Report.class);

    @Override
    public void execute(Connection conn, Parameter parameter) throws CommandException {
        if (!parameter.has(1)) {
            throw new UsageException(getUsage());
        }
        try {
            final String p1 = parameter.at(1);
            if (p1.equals("-")) {
                reportDBInfo(conn);
            } else {
                ResultSetReference ref = getResult(conn, parameter);
                try {
                    output(ref);
                    outputMessage("i.selected", ref.getRecordCount());
                } finally {
                    ref.getResultSet().close();
                }
            }
        } catch (SQLException ex) {
            throw new CommandException(ex);
        }
    }

    @Override
    public boolean isReadOnly() {
        return true;
    }

    ResultSetReference getResult(Connection conn, Parameter p) throws SQLException {
        if (!p.has(1)) {
            throw new UsageException(getUsage());
        }
        final String cmd = p.asString();
        final String tableName = p.at(1);
        final String option = p.at(2);
        try {
            DatabaseMetaData dbmeta = conn.getMetaData();
            if (option.equalsIgnoreCase("FULL")) {
                return getTableFullDescription(dbmeta, tableName, cmd);
            } else if (option.equalsIgnoreCase("PK")) {
                return getPrimaryKeyInfo(dbmeta, tableName, cmd);
            } else if (option.equalsIgnoreCase("INDEX")) {
                return getIndexInfo(dbmeta, tableName, cmd);
            }
            return getTableDescription(dbmeta, tableName, cmd);
        } catch (Throwable th) {
            if (th instanceof SQLException) {
                throw (SQLException)th;
            } else if (th instanceof RuntimeException) {
                throw (RuntimeException)th;
            }
            throw new CommandException(th);
        }
    }

    /**
     * e[u`ׂĕ\B
     * @param dbmeta DB^f[^
     * @param tableName e[u
     * @param cmd R}h
     * @return 
     * @throws Throwable
     */
    private ResultSetReference getTableFullDescription(DatabaseMetaData dbmeta,
                                                       String tableName,
                                                       String cmd) throws Throwable {
        if (log.isDebugEnabled()) {
            log.debug("report table-full-description of : " + tableName);
        }
        ResultSet rs = dbmeta.getColumns(null, null, convertPattern(tableName), null);
        try {
            return new ResultSetReference(rs, cmd);
        } catch (Throwable th) {
            rs.close();
            throw th;
        }
    }

    /**
     * e[u`\B
     * @param dbmeta DB^f[^
     * @param tableName e[u
     * @param cmd R}h
     * @throws Throwable 
     */
    private ResultSetReference getTableDescription(DatabaseMetaData dbmeta,
                                                   String tableName,
                                                   String cmd) throws Throwable {
        if (log.isDebugEnabled()) {
            log.debug("report table-description of : " + tableName);
        }
        ResultSet rs = dbmeta.getColumns(null, null, convertPattern(tableName), null);
        try {
            ResultSetReference ref = new ResultSetReference(rs, cmd);
            ColumnOrder order = ref.getOrder();
            order.addOrder(17, getColumnName("sequence"));
            order.addOrder(4, getColumnName("columnname"));
            order.addOrder(18, getColumnName("nullable"));
            order.addOrder(6, getColumnName("type"));
            order.addOrder(7, getColumnName("size"));
            order.addOrder(2, getColumnName("schema"));
            return ref;
        } catch (Throwable th) {
            rs.close();
            throw th;
        }
    }

    /**
     * vC}L[\B
     * @param dbmeta DB^f[^
     * @param tableName e[u
     * @param cmd R}h
     * @throws Throwable 
     */
    private ResultSetReference getPrimaryKeyInfo(DatabaseMetaData dbmeta,
                                                 String tableName,
                                                 String cmd) throws Throwable {
        if (log.isDebugEnabled()) {
            log.debug("report primary-key of : " + tableName);
        }
        ResultSet rs = dbmeta.getPrimaryKeys(null, null, convertPattern(tableName));
        try {
            ResultSetReference ref = new ResultSetReference(rs, cmd);
            ColumnOrder order = ref.getOrder();
            order.addOrder(1, getColumnName("catalog"));
            order.addOrder(2, getColumnName("schema"));
            order.addOrder(3, getColumnName("tablename"));
            order.addOrder(5, getColumnName("sequence"));
            order.addOrder(4, getColumnName("columnname"));
            order.addOrder(6, getColumnName("keyname"));
            return ref;
        } catch (Throwable th) {
            rs.close();
            throw th;
        }
    }

    /**
     * CfbNX\B
     * @param dbmeta DB^f[^
     * @param tableName e[u
     * @param cmd R}h
     * @throws Throwable 
     */
    private ResultSetReference getIndexInfo(DatabaseMetaData dbmeta, String tableName, String cmd) throws Throwable {
        if (log.isDebugEnabled()) {
            log.debug("report index of : " + tableName);
        }
        ResultSet rs = dbmeta.getIndexInfo(null, null, convertPattern(tableName), false, false);
        try {
            ResultSetReference ref = new ResultSetReference(rs, cmd);
            ColumnOrder order = ref.getOrder();
            order.addOrder(1, getColumnName("catalog"));
            order.addOrder(2, getColumnName("schema"));
            order.addOrder(3, getColumnName("tablename"));
            order.addOrder(8, getColumnName("sequence"));
            order.addOrder(9, getColumnName("columnname"));
            order.addOrder(6, getColumnName("keyname"));
            return ref;
        } catch (Throwable th) {
            rs.close();
            throw th;
        }
    }

    /**
     * f[^x[X\B
     * @param conn RlNV
     * @throws SQLException
     */
    private void reportDBInfo(Connection conn) throws SQLException {
        if (log.isDebugEnabled()) {
            log.debug("report dbinfo");
        }
        DatabaseMetaData meta = conn.getMetaData();
        final String userName = meta.getUserName();
        outputMessage("Report.dbinfo",
                      meta.getDatabaseProductName(),
                      meta.getDatabaseProductVersion(),
                      meta.getDriverName(),
                      meta.getDriverVersion(),
                      (userName == null) ? "" : userName,
                      meta.getURL());
    }

    private String getColumnName(String key) {
        return getMessage("Report.label." + key);
    }

}
