package jp.naist.se.stigmata.ui.swing;

/*
 * $Id: WellknownClassesSettingsPane.java 247 2007-10-10 08:45:20Z tama3 $
 */

import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.HashMap;
import java.util.Map;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.Box;
import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.border.TitledBorder;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import jp.naist.se.stigmata.ui.swing.actions.PopupShowAction;
import jp.naist.se.stigmata.utils.WellknownClassJudgeRule;
import jp.naist.se.stigmata.utils.WellknownClassManager;

/**
 * Well-known classes judge rules management pane.
 * 
 * @author Haruaki TAMADA
 * @version $Revision: 247 $ $Date: 2007-10-10 17:45:20 +0900 (Wed, 10 Oct 2007) $
 */
public class WellknownClassesSettingsPane extends JPanel{
    private static final long serialVersionUID = 329734546345634532L;

    private StigmataFrame stigmata;
    private WellknownClassManager manager;
    private JList list;
    private DefaultListModel listmodel;

    private Map<String, String> matchTypeMap = new HashMap<String, String>();
    private Map<String, String> partTypeMap = new HashMap<String, String>();

    public WellknownClassesSettingsPane(StigmataFrame stigmata){
        this.stigmata = stigmata;
        this.manager = stigmata.getEnvironment().getWellknownClassManager();

        initLayouts();
        initializeData();
    }

    public synchronized void setWellknownClasses(WellknownClassManager manager){
        manager.clear();
        for(int i = 0; i < listmodel.getSize(); i++){
            WellknownClassJudgeRule rule = (WellknownClassJudgeRule)listmodel.getElementAt(i);
            manager.add(rule);
        }
    }

    public synchronized WellknownClassManager createWellknownClassManager(){
        WellknownClassManager manager = new WellknownClassManager();
        setWellknownClasses(manager);
        return manager;
    }

    public void reset(){
        listmodel.clear();
        initializeData();
    }

    private synchronized boolean isWellknownClass(String className){
        WellknownClassManager manager = createWellknownClassManager();

        return manager.isWellKnownClass(className);
    }

    private String findType(JComboBox combo, Map<String, String> map){
        String item = (String)combo.getSelectedItem();
        for(Map.Entry<String, String> entry: map.entrySet()){
            if(item.equals(entry.getValue())){
                return entry.getKey();
            }
        }
        return null;
    }

    public void addRule(WellknownClassJudgeRule rule){
        if(rule != null){
            listmodel.addElement(rule);
            stigmata.setNeedToSaveSettings(true);
        }
    }

    public void editRule(int index){
        WellknownClassJudgeRule rule = (WellknownClassJudgeRule)listmodel.getElementAt(index);
        WellknownClassJudgeRule newrule = createOrUpdateRule(rule);
        if(newrule != null){
            listmodel.setElementAt(newrule, index);
            stigmata.setNeedToSaveSettings(true);
        }
    }


    private void initializeData(){
        for(WellknownClassJudgeRule rule : manager){
            listmodel.addElement(rule);
        }

        for(WellknownClassJudgeRule.MatchType type: WellknownClassJudgeRule.MatchType.values()){
            matchTypeMap.put(type.name(), Messages.getString("matchtype." + type.name()));
        }
        for(WellknownClassJudgeRule.MatchPartType type: WellknownClassJudgeRule.MatchPartType.values()){
            partTypeMap.put(type.name(), Messages.getString("matchparttype." + type.name()));
        }
    }

    private void initLayouts(){
        setLayout(new BorderLayout());
        JPanel center = new JPanel(new BorderLayout());
        listmodel = new DefaultListModel();

        list = new JList(listmodel);
        JScrollPane scroll = new JScrollPane(list);

        center.add(scroll, BorderLayout.CENTER);
        center.add(createSouthPane(), BorderLayout.SOUTH);

        add(center, BorderLayout.CENTER);
        add(createCheckPane(), BorderLayout.SOUTH);
    }

    private JComponent createCheckPane(){
        final JTextField text = new JTextField();
        final JButton checkButton = GUIUtility.createButton("checkwellknown");
        final JLabel label = new JLabel(GUIUtility.getIcon("wellknownclasschecker.default.icon"));
        checkButton.setEnabled(false);

        ActionListener listener = new ActionListener(){
            public void actionPerformed(ActionEvent e){
                String t = text.getText().trim();
                if(t.length() > 0){
                    String message = Messages.getString("wellknownclasschecker.wellknown.tooltip");
                    if(isWellknownClass(t)){
                        label.setIcon(GUIUtility.getIcon("wellknownclasschecker.wellknown.icon"));
                    }
                    else{
                        label.setIcon(GUIUtility.getIcon("wellknownclasschecker.notwellknown.icon"));
                        message = Messages.getString("wellknownclasschecker.notwellknown.tooltip");
                    }
                    label.setToolTipText(message);
                    String dm = String.format(
                        "<html><body><dl><dt>%s</dt><dd>%s</dd></body></html>", t, message
                    );
                    JOptionPane.showMessageDialog(
                        stigmata, dm, Messages.getString("wellknownclasschecker.dialog.title"),
                        JOptionPane.INFORMATION_MESSAGE
                    );
                }
            }
        };
        checkButton.addActionListener(listener);
        text.addActionListener(listener);
        text.getDocument().addDocumentListener(new DocumentListener(){
            public void changedUpdate(DocumentEvent arg0){
                checkButton.setEnabled(text.getText().trim().length() > 0);
            }

            public void insertUpdate(DocumentEvent arg0){
                checkButton.setEnabled(text.getText().trim().length() > 0);
            }

            public void removeUpdate(DocumentEvent arg0){
                checkButton.setEnabled(text.getText().trim().length() > 0);
            }
        });

        JComponent south = Box.createHorizontalBox();
        south.add(Box.createHorizontalGlue());
        south.add(text);
        south.add(Box.createHorizontalGlue());
        south.add(checkButton);
        south.add(Box.createHorizontalGlue());
        south.add(label);
        south.add(Box.createHorizontalGlue());

        south.setBorder(new TitledBorder(Messages.getString("wellknownclasschecker.border")));

        return south;
    }

    private JComponent createSouthPane(){
        JComponent southPanel = Box.createHorizontalBox();
        Action addAction = new AbstractAction(){
            private static final long serialVersionUID = -8749957850400877529L;

            public void actionPerformed(ActionEvent e){
                addRule(createOrUpdateRule(null));
            }
        };
        final Action removeAction = new AbstractAction(){
            private static final long serialVersionUID = 8776209200186477040L;

            public void actionPerformed(ActionEvent e){
                int[] indeces = list.getSelectedIndices();
                for(int i = indeces.length - 1; i >= 0; i--){
                    listmodel.removeElementAt(indeces[i]);
                }
                list.getSelectionModel().clearSelection();
                stigmata.setNeedToSaveSettings(true);
            }
        };
        final Action updateAction = new AbstractAction(){
            private static final long serialVersionUID = 852965501722574084L;

            public void actionPerformed(ActionEvent e){
                editRule(list.getSelectedIndex());
            }
        };
        JButton addButton = GUIUtility.createButton("addwellknown", addAction);
        JButton removeButton = GUIUtility.createButton("removewellknown", removeAction);
        JButton updateButton = GUIUtility.createButton("updatewellknown", updateAction);
        removeAction.setEnabled(false);
        updateAction.setEnabled(false);

        southPanel.add(Box.createHorizontalGlue());
        southPanel.add(addButton);
        southPanel.add(Box.createHorizontalGlue());
        southPanel.add(updateButton);
        southPanel.add(Box.createHorizontalGlue());
        southPanel.add(removeButton);
        southPanel.add(Box.createHorizontalGlue());

        list.addMouseListener(new MouseAdapter(){
            @Override
            public void mouseClicked(MouseEvent e){
                if(e.getClickCount() == 2){
                    editRule(list.getSelectedIndex());
                }
            }
        });
        JPopupMenu popup = new JPopupMenu();
        popup.add(GUIUtility.createJMenuItem("addwellknown", addAction));
        popup.add(GUIUtility.createJMenuItem("updatewellknown", updateAction));
        popup.add(GUIUtility.createJMenuItem("removewellknown", removeAction));

        list.addMouseListener(new PopupShowAction(popup));
        list.getSelectionModel().addListSelectionListener(new ListSelectionListener(){
            public void valueChanged(ListSelectionEvent arg0){
                int[] indeces = list.getSelectedIndices();
                removeAction.setEnabled(indeces != null && indeces.length > 0);
                updateAction.setEnabled(indeces != null && indeces.length == 1);
            }
        });

        return southPanel;
    }

    private WellknownClassJudgeRule createOrUpdateRule(WellknownClassJudgeRule rule){
        JTextField text = new JTextField();
        text.setOpaque(true);
        GUIUtility.decorateJComponent(text, "addwellknown.newrule.pattern");
        JCheckBox excludeCheck = new JCheckBox(Messages.getString("addwellknown.newrule.exclude.label"), false);
        GUIUtility.decorateJComponent(excludeCheck, "addwellknown.newrule.exclude");
        JComboBox matchTypeComboBox = new JComboBox();
        for(Map.Entry<String, String> entry: matchTypeMap.entrySet()){
            matchTypeComboBox.addItem(entry.getValue());
        }
        JComboBox partTypeComboBox = new JComboBox();
        for(Map.Entry<String, String> entry: partTypeMap.entrySet()){
            partTypeComboBox.addItem(entry.getValue());
        }
        if(rule != null){
            text.setText(rule.getPattern());
            excludeCheck.setSelected(rule.isExclude());
            matchTypeComboBox.setSelectedItem(matchTypeMap.get(rule.getMatchType().name()));
            partTypeComboBox.setSelectedItem(partTypeMap.get(rule.getMatchPartType().name()));
        }

        JPanel panel = new JPanel(new BorderLayout());
        panel.setOpaque(true);
        JPanel centerPanel = new JPanel(new FlowLayout());

        centerPanel.add(partTypeComboBox);
        centerPanel.add(matchTypeComboBox);

        panel.add(excludeCheck, BorderLayout.NORTH);
        panel.add(centerPanel, BorderLayout.CENTER);
        panel.add(text, BorderLayout.SOUTH);

        int value = JOptionPane.showConfirmDialog(
            stigmata, panel, Messages.getString("addwellknown.dialog.title"),
            JOptionPane.OK_CANCEL_OPTION
        );
        if(value == JOptionPane.OK_OPTION){
            String matchType = findType(matchTypeComboBox, matchTypeMap);
            String partType = findType(partTypeComboBox, partTypeMap);
            WellknownClassJudgeRule.MatchType match = null;
            WellknownClassJudgeRule.MatchPartType part = null;
            String pattern = text.getText();
            boolean excludeFlag = excludeCheck.isSelected();

            if(matchType != null && partType != null){
                match = WellknownClassJudgeRule.MatchType.valueOf(matchType);
                part = WellknownClassJudgeRule.MatchPartType.valueOf(partType);
            }

            if(match != null && partType != null && pattern != null && !pattern.equals("")){
                return new WellknownClassJudgeRule(pattern, match, part, excludeFlag);
            }
        }
        return null;
    }
}
