#version 330 core
out vec4 FragColor;
in vec2 TexCoords;
in vec3 WorldPos;
in vec3 Normal;

in vec4 oPos;

// material parameters
uniform sampler2D albedoMap;
uniform sampler2D normalMap;
uniform sampler2D metallicMap;
uniform sampler2D roughnessMap;
uniform sampler2D aoMap;

// lights
uniform vec3 lightPos0;
uniform vec3 lightPos1;
uniform vec3 lightPos2;

uniform vec3 lightColor;

uniform vec3 camPos;

const float PI = 3.14159265359;
// ----------------------------------------------------------------------------
// Easy trick to get tangent-normals to world-space to keep PBR code simplified.
// Don't worry if you don't get what's going on; you generally want to do normal 
// mapping the usual way for performance anways; I do plan make a note of this 
// technique somewhere later in the normal mapping tutorial.
vec3 getNormalFromMap()
{
    vec3 tangentNormal = texture(normalMap, TexCoords).xyz * 2.0 - 1.0;

    vec3 Q1  = dFdx(WorldPos);
    vec3 Q2  = dFdy(WorldPos);
    vec2 st1 = dFdx(TexCoords);
    vec2 st2 = dFdy(TexCoords);

    vec3 N   = normalize(Normal);
    vec3 T  = normalize(Q1*st2.t - Q2*st1.t);
    vec3 B  = -normalize(cross(N, T));
    mat3 TBN = mat3(T, B, N);

    return normalize(TBN * tangentNormal);
}
// ----------------------------------------------------------------------------
float DistributionGGX(vec3 N, vec3 H, float roughness)
{
    float a = roughness*roughness;
    float a2 = a*a;
    float NdotH = max(dot(N, H), 0.0);
    float NdotH2 = NdotH*NdotH;

    float nom   = a2;
    float denom = (NdotH2 * (a2 - 1.0) + 1.0);
    denom = PI * denom * denom;

    return nom / denom;
}
// ----------------------------------------------------------------------------
float GeometrySchlickGGX(float NdotV, float roughness)
{
    float r = (roughness + 1.0);
    float k = (r*r) / 8.0;

    float nom   = NdotV;
    float denom = NdotV * (1.0 - k) + k;

    return nom / denom;
}
// ----------------------------------------------------------------------------
float GeometrySmith(vec3 N, vec3 V, vec3 L, float roughness)
{
    float NdotV = max(dot(N, V), 0.0);
    float NdotL = max(dot(N, L), 0.0);
    float ggx2 = GeometrySchlickGGX(NdotV, roughness);
    float ggx1 = GeometrySchlickGGX(NdotL, roughness);

    return ggx1 * ggx2;
}
// ----------------------------------------------------------------------------
vec3 fresnelSchlick(float cosTheta, vec3 F0)
{
    return F0 + (1.0 - F0) * pow(1.0 - cosTheta, 5.0);
}
// ----------------------------------------------------------------------------






//////// insert 20dec17 begin /////////////////////////////////////
uniform int fogcolr=1; //  0=>noFog, 1=>white, 2=>brownish, 3=>purple, 4=>gray
uniform int foglevl=0; // 0=>noFog, 1=>light, 1=>Fog, 2=>heavy

const vec4 vFogColor=vec4(0.9,0.9,0.9,1.0); //normal white
const vec4 vDfogColor=vec4(0.7,0.7,0.7,1.0); //normal gray
const vec4 vSootColor=vec4(50.0/255,30.0/255,10.0/255,1.0); // brown
const vec4 vMystColor=vec4(0.7,0.5,0.7,1.0); // purple
//////// insert 20dec17 end /////////////////////////////////////


const float fStart=0.0;
const float lEnd=20.0; // light fog
const float fEnd=12.0; // normal fog
const float hEnd= 8.0; // heavy fog

float getFogFactor(float rng)
{

	float fResult = (lEnd-rng)/(lEnd-fStart);
	if( foglevl>1 ) fResult = (fEnd-rng)/(fEnd-fStart);
	if( foglevl>2 ) fResult = (hEnd-rng)/(hEnd-fStart);
	
	fResult = 1.0-clamp(fResult, 0.0, 1.0);
	
	return sqrt(fResult);
}







void main()
{		
    vec3 albedo     = pow(texture(albedoMap, TexCoords).rgb, vec3(2.2));
    float metallic  = texture(metallicMap, TexCoords).r;
    float roughness = texture(roughnessMap, TexCoords).r;
    float ao        = texture(aoMap, TexCoords).r;

    vec3 N = getNormalFromMap();
    vec3 V = normalize(camPos - WorldPos);

    // calculate reflectance at normal incidence; if dia-electric (like plastic) use F0 
    // of 0.04 and if it's a metal, use the albedo color as F0 (metallic workflow)    
    vec3 F0 = vec3(0.04); 
    F0 = mix(F0, albedo, metallic); //metallic=%albedo

    // reflectance equation
    vec3 Lo = vec3(0.0);
	 vec3 lpos;
    for(int i = 0; i < 3; ++i) 
    {
	 		if( i==0 ) lpos=lightPos0;
			if( i==1 ) lpos=lightPos1;
			if( i==2 ) lpos=lightPos2;

        // calculate per-light radiance
        vec3 L = normalize(lpos - WorldPos);
        vec3 H = normalize(V + L);
        float distance = length(lpos - WorldPos);
        float attenuation = 1.0 / (distance * distance);
        vec3 radiance = lightColor * attenuation;

        // Cook-Torrance BRDF
        float NDF = DistributionGGX(N, H, roughness);   
        float G   = GeometrySmith(N, V, L, roughness);      
        vec3 F    = fresnelSchlick(max(dot(H, V), 0.0), F0);
           
        vec3 nominator    = NDF * G * F; 
        float denominator = 4 * max(dot(N, V), 0.0) * max(dot(N, L), 0.0) + 0.001; 
        vec3 specular = nominator / denominator;

//fastrgv test: specular *= 2.0;
        
        // kS is equal to Fresnel
        vec3 kS = F;
        // for energy conservation, the diffuse and specular light can't
        // be above 1.0 (unless the surface emits light); to preserve this
        // relationship the diffuse component (kD) should equal 1.0 - kS.
        vec3 kD = vec3(1.0) - kS;
        // multiply kD by the inverse metalness such that only non-metals 
        // have diffuse lighting, or a linear blend if partly metal (pure metals
        // have no diffuse light).
        kD *= 1.0 - metallic;	  

        // scale light by NdotL
        float NdotL = max(dot(N, L), 0.0);        

        // add to outgoing radiance Lo
        Lo += (kD * albedo / PI + specular) * radiance * NdotL; 
    }   
    
    // ambient lighting (note that the next IBL tutorial will replace 
    // this ambient lighting with environment lighting).
    vec3 ambient = vec3(0.03) * albedo * ao;
    
    vec3 color = ambient + Lo;

    // HDR tonemapping
    color = color / (color + vec3(1.0));
    // gamma correct
    color = pow(color, vec3(1.0/2.2)); 

    FragColor = vec4(color, 1.0);


	if(foglevl>0) {
		float dist2obj = length(camPos-oPos.xyz);
		const float yc=-1.0;
		float dist = exp(-1.0*abs(oPos.y-yc)) * dist2obj;

		if( fogcolr==4 )
			FragColor = mix(FragColor,  vDfogColor, getFogFactor(dist)); //gray
		else if( fogcolr==3 )
			FragColor = mix(FragColor, vMystColor, getFogFactor(dist)); //purple
		else if( fogcolr==2 )
			FragColor = mix(FragColor, vSootColor, getFogFactor(dist)); //brown
		else if( fogcolr==1 )
			FragColor = mix(FragColor,  vFogColor, getFogFactor(dist)); //white

	}


}
