"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getGlobLists = exports.parseDefaultGlobsFile = exports.parseGlobsFile = exports.processGlobs = exports.formatGlob = exports.optimizeGlobLists = exports.optimizeGlobs = exports.wrapGlobs = exports.toAbsoluteGlobLists = exports.toPosixPath = exports.mergeGlobLists = exports.updateGlobLists = exports.makeGlobMatcher = void 0;
const fs_1 = require("fs");
const path_1 = __importDefault(require("path"));
const picomatch_1 = __importDefault(require("picomatch"));
const constants_1 = require("../constants");
const filesystem_1 = require("./filesystem");
function makeGlobMatcher(globs, picoOptions) {
    return (0, picomatch_1.default)(globs, { ...constants_1.DEFAULT_PICO_OPTIONS, ...picoOptions });
}
exports.makeGlobMatcher = makeGlobMatcher;
function updateGlobLists(globLists, updateFunc) {
    const { included, includedDirs, excluded } = globLists;
    return {
        ...globLists,
        included: updateFunc(included, 'included'),
        includedDirs: includedDirs ? updateFunc(includedDirs, 'includedDirs') : includedDirs,
        excluded: updateFunc(excluded, 'excluded'),
    };
}
exports.updateGlobLists = updateGlobLists;
function mergeGlobLists(globListsA, globListsB) {
    return {
        ...updateGlobLists(globListsA, (globs, key) => [...globs, ...(globListsB[key] || [])]),
        originalIncluded: [...globListsA.originalIncluded, ...globListsB.originalIncluded],
    };
}
exports.mergeGlobLists = mergeGlobLists;
function toPosixPath(pathStr) {
    return path_1.default.sep === '/' ? pathStr : pathStr.replace(/\\/g, '/');
}
exports.toPosixPath = toPosixPath;
function toAbsoluteGlobLists(globLists, absoluteNodeModulesPath) {
    const absolutePathWithPosixSeparator = toPosixPath(absoluteNodeModulesPath);
    return updateGlobLists(globLists, globs => globs.map(glob => absolutePathWithPosixSeparator + '/' + glob));
}
exports.toAbsoluteGlobLists = toAbsoluteGlobLists;
function wrapGlobs(globs, prefix) {
    return `${prefix || ''}@(${globs.map(glob => '(' + glob + ')').join('|')})`;
}
exports.wrapGlobs = wrapGlobs;
const GLOBSTAR_START_REGEX = /^(\/?\*\*\/)+/;
function optimizeGlobs(globs) {
    const globstarStartGlobs = [];
    const fixedPathGlobs = [];
    globs.forEach(glob => {
        if (glob.match(GLOBSTAR_START_REGEX)) {
            globstarStartGlobs.push(glob);
        }
        else {
            fixedPathGlobs.push(glob);
        }
    });
    const result = [];
    if (globstarStartGlobs.length) {
        const withoutGlobstar = globstarStartGlobs.map(glob => glob.replace(GLOBSTAR_START_REGEX, ''));
        result.push(wrapGlobs(withoutGlobstar, '**/'));
    }
    if (fixedPathGlobs.length) {
        result.push(wrapGlobs(fixedPathGlobs));
    }
    return result;
}
exports.optimizeGlobs = optimizeGlobs;
function optimizeGlobLists(globLists) {
    return updateGlobLists(globLists, optimizeGlobs);
}
exports.optimizeGlobLists = optimizeGlobLists;
const EXCLAMATION_START = /^\s*!/;
const DIR_GLOB_REGEX = /\/\**$/;
const EXCLUDED_GLOB_REGEX = /^!/;
const ESCAPED_NEGATIVE_GLOB_REGEX = /^\\!/;
const NOT_FIXED_START_GLOB_REGEX = /^([^/])/;
const FIXED_START_GLOB_REGEX = /^\//;
function formatGlob(glob) {
    return glob
        .trim()
        .replace(EXCLUDED_GLOB_REGEX, '')
        .replace(ESCAPED_NEGATIVE_GLOB_REGEX, '!')
        .replace(DIR_GLOB_REGEX, '/**')
        .replace(NOT_FIXED_START_GLOB_REGEX, '**/$1')
        .replace(FIXED_START_GLOB_REGEX, '');
}
exports.formatGlob = formatGlob;
function processGlobs(globs, excludedOnlyGlobs = []) {
    const globLists = {
        excluded: excludedOnlyGlobs.map(formatGlob),
        included: [],
        includedDirs: [],
        originalIncluded: [],
    };
    globs.forEach(glob => {
        const isExcluded = !!glob.match(EXCLAMATION_START);
        const formattedGlob = formatGlob(glob);
        if (!formattedGlob) {
            return;
        }
        else if (isExcluded) {
            globLists.excluded.push(formattedGlob);
        }
        else {
            if (formattedGlob.endsWith('/**')) {
                globLists.includedDirs.push(formattedGlob.replace(DIR_GLOB_REGEX, ''));
            }
            globLists.included.push(formattedGlob);
            globLists.originalIncluded.push(glob.trim());
        }
    });
    return globLists;
}
exports.processGlobs = processGlobs;
const COMMENT_OR_EMPTY_REGEX = /^\s*(#|$)/;
async function parseGlobsFile(filePath) {
    let fileContents;
    try {
        fileContents = (await fs_1.promises.readFile(filePath, { encoding: 'utf-8' })).toString() || '';
    }
    catch (error) {
        console.error(`Failed to read glob file (${filePath})`);
        throw error;
    }
    const fileGlobs = fileContents.split(/\r?\n/).filter(line => !line.match(COMMENT_OR_EMPTY_REGEX)) || [];
    return processGlobs(fileGlobs);
}
exports.parseGlobsFile = parseGlobsFile;
async function parseDefaultGlobsFile() {
    return parseGlobsFile(constants_1.DEFAULT_GLOBS_FILE_PATH);
}
exports.parseDefaultGlobsFile = parseDefaultGlobsFile;
async function getGlobLists({ argGlobs, useDefaultGlobs, userGlobsFilePath, }) {
    let globLists = processGlobs(argGlobs.included, argGlobs.excluded);
    if (useDefaultGlobs) {
        const defaultGlobLists = await parseDefaultGlobsFile();
        globLists = mergeGlobLists(globLists, defaultGlobLists);
    }
    if (userGlobsFilePath && (await (0, filesystem_1.fileExists)(userGlobsFilePath))) {
        const userFileGlobLists = await parseGlobsFile(userGlobsFilePath);
        globLists = mergeGlobLists(globLists, userFileGlobLists);
    }
    return globLists;
}
exports.getGlobLists = getGlobLists;
