"use strict";
/**
 * This script is injected into a plugins custom settings ui by the Homebridge UI
 * You should not include it in your own code, however you can use it for type information if desired.
 * It provides the interface to interact with the Homebridge UI service.
 */
let EventTargetConstructor = window.EventTarget;
/**
 * Pollyfill for older browsers that do not support EventTarget as a constructor.
 * https://developer.mozilla.org/en-US/docs/Web/API/EventTarget
 */
if (!Object.prototype.hasOwnProperty.call(window.EventTarget, 'caller')) {
    EventTargetConstructor = function () {
        this['listeners'] = {};
    };
    EventTargetConstructor.prototype['listeners'] = null;
    EventTargetConstructor.prototype.addEventListener = function (type, callback) {
        if (!(type in this['listeners'])) {
            this['listeners'][type] = [];
        }
        this['listeners'][type].push(callback);
    };
    EventTargetConstructor.prototype.removeEventListener = function (type, callback) {
        if (!(type in this['listeners'])) {
            return;
        }
        const stack = this['listeners'][type];
        for (let i = 0, l = stack.length; i < l; i++) {
            if (stack[i] === callback) {
                stack.splice(i, 1);
                return;
            }
        }
    };
    EventTargetConstructor.prototype.dispatchEvent = function (event) {
        if (!(event.type in this['listeners'])) {
            return true;
        }
        const stack = this['listeners'][event.type].slice();
        for (let i = 0, l = stack.length; i < l; i++) {
            stack[i].call(this, event);
        }
        return !event.defaultPrevented;
    };
}
class HomebridgePluginUi extends EventTargetConstructor {
    constructor() {
        super();
        this.origin = '';
        this.lastBodyHeight = 0;
        this.toast = new HomebridgeUiToastHelper();
        this.plugin = window['_homebridge'].plugin;
        this.serverEnv = window['_homebridge'].serverEnv;
        window.addEventListener('message', this._handleIncomingMessage.bind(this), false);
    }
    _handleIncomingMessage(e) {
        switch (e.data.action) {
            case 'ready': {
                this.origin = e.origin;
                document.body.style.display = 'block';
                this.dispatchEvent(new Event('ready'));
                this.fixScrollHeight();
                this._monitorFrameHeight();
                break;
            }
            case 'response': {
                this.dispatchEvent(new MessageEvent(e.data.requestId, {
                    data: e.data,
                }));
                break;
            }
            case 'stream': {
                this.dispatchEvent(new MessageEvent(e.data.event, {
                    data: e.data.data,
                }));
                break;
            }
            case 'body-class': {
                this._setBodyClass(e);
                break;
            }
            case 'inline-style': {
                this._setInlineStyle(e);
                break;
            }
            case 'link-element': {
                this._setLinkElement(e);
                break;
            }
            default:
                console.log(e.data);
        }
    }
    _postMessage(message) {
        window.parent.postMessage(message, this.origin || '*');
    }
    _setBodyClass(e) {
        document.body.classList.add(e.data.class);
    }
    _setInlineStyle(e) {
        const styleElement = document.createElement('style');
        styleElement.innerHTML = e.data.style;
        document.head.appendChild(styleElement);
    }
    _setLinkElement(e) {
        const linkElement = document.createElement('link');
        linkElement.setAttribute('href', e.data.href);
        linkElement.setAttribute('rel', e.data.rel);
        document.head.appendChild(linkElement);
    }
    _monitorFrameHeight() {
        if (window['ResizeObserver']) {
            // use ResizeObserver if available
            const resizeObserver = new window['ResizeObserver'](() => {
                this.fixScrollHeight();
            });
            resizeObserver.observe(document.body);
        }
        else {
            // fall back to polling
            setInterval(() => {
                if (document.body.scrollHeight !== this.lastBodyHeight) {
                    this.lastBodyHeight = document.body.scrollHeight;
                    this.fixScrollHeight();
                }
            }, 250);
        }
    }
    async _requestResponse(payload) {
        // generate a random request id so we can link the response
        const requestId = Math.random().toString(36).substring(2);
        payload.requestId = requestId;
        // post message to parent
        this._postMessage(payload);
        // wait for response
        return new Promise((resolve, reject) => {
            const responseHandler = (event) => {
                this.removeEventListener(requestId, responseHandler);
                if (event.data.success) {
                    resolve(event.data.data);
                }
                else {
                    reject(event.data.data);
                }
            };
            this.addEventListener(requestId, responseHandler);
        });
    }
    fixScrollHeight() {
        this._postMessage({ action: 'scrollHeight', scrollHeight: document.body.scrollHeight });
    }
    closeSettings() {
        this._postMessage({ action: 'close' });
    }
    showSpinner() {
        this._postMessage({ action: 'spinner.show' });
    }
    hideSpinner() {
        this._postMessage({ action: 'spinner.hide' });
    }
    showSchemaForm() {
        this._postMessage({ action: 'schema.show' });
    }
    hideSchemaForm() {
        this._postMessage({ action: 'schema.hide' });
    }
    createForm(schema, data, submitButton, cancelButton) {
        return new HomebridgeUiFormHelper(this, schema, data, submitButton, cancelButton);
    }
    endForm() {
        this._postMessage({ action: 'form.end' });
    }
    async getPluginConfig() {
        return await this._requestResponse({ action: 'config.get' });
    }
    async updatePluginConfig(pluginConfig) {
        return await this._requestResponse({ action: 'config.update', pluginConfig: pluginConfig });
    }
    async savePluginConfig() {
        return await this._requestResponse({ action: 'config.save' });
    }
    async getPluginConfigSchema() {
        return await this._requestResponse({ action: 'config.schema' });
    }
    async getCachedAccessories() {
        return await this._requestResponse({ action: 'cachedAccessories.get' });
    }
    async request(path, body) {
        return await this._requestResponse({ action: 'request', path: path, body: body });
    }
    async i18nCurrentLang() {
        return await this._requestResponse({ action: 'i18n.lang' });
    }
    async i18nGetTranslation() {
        return await this._requestResponse({ action: 'i18n.translations' });
    }
}
class HomebridgeUiToastHelper {
    _postMessage(type, message, title) {
        window.parent.postMessage({ action: 'toast.' + type, message: message, title: title }, '*');
    }
    success(message, title) {
        this._postMessage('success', message, title);
    }
    error(message, title) {
        this._postMessage('error', message, title);
    }
    warning(message, title) {
        this._postMessage('warning', message, title);
    }
    info(message, title) {
        this._postMessage('info', message, title);
    }
}
class HomebridgeUiFormHelper {
    constructor(parent, schema, data, submitButton, cancelButton) {
        this.parent = parent;
        this.formId = Math.random().toString(36).substring(2);
        this.parent._postMessage({ action: 'form.create', formId: this.formId, schema, data, submitButton, cancelButton });
        const handle = this._eventHandle.bind(this);
        this.parent.addEventListener(this.formId, handle);
        this.end = () => {
            this.parent.removeEventListener(this.formId, handle);
            this.parent._postMessage({ action: 'form.end', formId: this.formId, schema: schema, data: data });
        };
    }
    _eventHandle(event) {
        switch (event.data.formEvent) {
            case 'change': {
                // general change events
                if (this._changeHandle && typeof this._changeHandle === 'function') {
                    this._changeHandle(event.data.formData);
                }
                else {
                    console.info('Homebridge Custom Plugin UI: Missing form onChange handler.');
                }
                break;
            }
            case 'submit': {
                // submit form events
                if (this._submitHandle && typeof this._submitHandle === 'function') {
                    this._submitHandle(event.data.formData);
                }
                else {
                    console.info('Homebridge Custom Plugin UI: Missing form onSubmit handler.');
                }
                break;
            }
            case 'cancel': {
                // cancel form events
                if (this._cancelHandle && typeof this._cancelHandle === 'function') {
                    this._cancelHandle(event.data.formData);
                }
                else {
                    console.info('Homebridge Custom Plugin UI: Missing form onCancel handler.');
                }
                break;
            }
            default: {
                console.info('Unknown form event type:', event.data);
            }
        }
    }
    onChange(fn) {
        if (typeof fn !== 'function') {
            console.error('Homebridge Custom Plugin UI: Form onChange handler must be a function.');
            return;
        }
        this._changeHandle = fn;
    }
    onSubmit(fn) {
        if (typeof fn !== 'function') {
            console.error('Homebridge Custom Plugin UI: Form onSubmit handler must be a function.');
            return;
        }
        this._submitHandle = fn;
    }
    onCancel(fn) {
        if (typeof fn !== 'function') {
            console.error('Homebridge Custom Plugin UI: Form onCancel handler must be a function.');
            return;
        }
        this._cancelHandle = fn;
    }
}
window['homebridge'] = new HomebridgePluginUi();
//# sourceMappingURL=ui.js.map