"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AirQualitySensorCreator = void 0;
const z2mModels_1 = require("../z2mModels");
const hap_1 = require("../hap");
const helpers_1 = require("../helpers");
class AirQualitySensorCreator {
    createServicesFromExposes(accessory, exposes) {
        const endpointMap = (0, helpers_1.groupByEndpoint)(exposes.filter(e => (0, z2mModels_1.exposesHasProperty)(e) && (0, z2mModels_1.exposesIsPublished)(e) && !accessory.isPropertyExcluded(e.property) &&
            AirQualitySensorHandler.propertyFactories.find((f) => f.canUseExposesEntry(e)) !== undefined).map(e => e));
        endpointMap.forEach((value, key) => {
            if (!accessory.isServiceHandlerIdKnown(AirQualitySensorHandler.generateIdentifier(key))) {
                this.createService(key, value, accessory);
            }
        });
    }
    createService(endpoint, exposes, accessory) {
        try {
            const handler = new AirQualitySensorHandler(endpoint, exposes, accessory);
            accessory.registerServiceHandler(handler);
        }
        catch (error) {
            accessory.log.warn('Failed to setup Air Quality Sensor service ' +
                `for accessory ${accessory.displayName} for endpoint ${endpoint}: ${error}`);
        }
    }
}
exports.AirQualitySensorCreator = AirQualitySensorCreator;
class PassthroughAirQualityProperty {
    constructor(expose, service, characteristic) {
        this.expose = expose;
        this.service = service;
        this.characteristic = characteristic;
        this.latestAirQuality = hap_1.hap.Characteristic.AirQuality.UNKNOWN;
        const c = (0, helpers_1.getOrAddCharacteristic)(service, characteristic);
        (0, helpers_1.copyExposesRangeToCharacteristic)(expose, c);
    }
    updateState(state) {
        var _a;
        if (this.expose.property in state) {
            const sensorValue = state[this.expose.property];
            if (sensorValue !== null && sensorValue !== undefined) {
                this.service.updateCharacteristic(this.characteristic, sensorValue);
                this.latestAirQuality = (_a = this.convertToAirQuality(sensorValue)) !== null && _a !== void 0 ? _a : hap_1.hap.Characteristic.AirQuality.UNKNOWN;
            }
        }
    }
}
class VolatileOrganicCompoundsProperty extends PassthroughAirQualityProperty {
    constructor(expose, service) {
        super(expose, service, hap_1.hap.Characteristic.VOCDensity);
    }
    static canUseExposesEntry(entry) {
        return (0, z2mModels_1.exposesHasNumericProperty)(entry) && entry.name === VolatileOrganicCompoundsProperty.NAME;
    }
    convertToAirQuality(sensorValue) {
        if (sensorValue <= 333) {
            return hap_1.hap.Characteristic.AirQuality.EXCELLENT;
        }
        if (sensorValue <= 1000) {
            return hap_1.hap.Characteristic.AirQuality.GOOD;
        }
        if (sensorValue <= 3333) {
            return hap_1.hap.Characteristic.AirQuality.FAIR;
        }
        if (sensorValue <= 8332) {
            return hap_1.hap.Characteristic.AirQuality.INFERIOR;
        }
        return hap_1.hap.Characteristic.AirQuality.POOR;
    }
}
VolatileOrganicCompoundsProperty.NAME = 'voc';
class ParticulateMatter10Property extends PassthroughAirQualityProperty {
    constructor(expose, service) {
        super(expose, service, hap_1.hap.Characteristic.PM10Density);
    }
    static canUseExposesEntry(entry) {
        return (0, z2mModels_1.exposesHasNumericProperty)(entry) && entry.name === ParticulateMatter10Property.NAME;
    }
    convertToAirQuality(sensorValue) {
        if (sensorValue <= 25) {
            return hap_1.hap.Characteristic.AirQuality.EXCELLENT;
        }
        if (sensorValue <= 50) {
            return hap_1.hap.Characteristic.AirQuality.GOOD;
        }
        if (sensorValue <= 100) {
            return hap_1.hap.Characteristic.AirQuality.FAIR;
        }
        if (sensorValue <= 300) {
            return hap_1.hap.Characteristic.AirQuality.INFERIOR;
        }
        return hap_1.hap.Characteristic.AirQuality.POOR;
    }
}
ParticulateMatter10Property.NAME = 'pm10';
class ParticulateMatter2Dot5Property extends PassthroughAirQualityProperty {
    constructor(expose, service) {
        super(expose, service, hap_1.hap.Characteristic.PM2_5Density);
    }
    static canUseExposesEntry(entry) {
        return (0, z2mModels_1.exposesHasNumericProperty)(entry) && entry.name === ParticulateMatter2Dot5Property.NAME;
    }
    convertToAirQuality(sensorValue) {
        if (sensorValue <= 15) {
            return hap_1.hap.Characteristic.AirQuality.EXCELLENT;
        }
        if (sensorValue <= 35) {
            return hap_1.hap.Characteristic.AirQuality.GOOD;
        }
        if (sensorValue <= 55) {
            return hap_1.hap.Characteristic.AirQuality.FAIR;
        }
        if (sensorValue <= 75) {
            return hap_1.hap.Characteristic.AirQuality.INFERIOR;
        }
        return hap_1.hap.Characteristic.AirQuality.POOR;
    }
}
ParticulateMatter2Dot5Property.NAME = 'pm25';
class AirQualitySensorHandler {
    constructor(endpoint, exposes, accessory) {
        this.accessory = accessory;
        this.properties = [];
        this.identifier = AirQualitySensorHandler.generateIdentifier(endpoint);
        const serviceName = accessory.getDefaultServiceDisplayName(endpoint);
        accessory.log.debug(`Configuring Air Quality Sensor for ${serviceName}`);
        this.service = accessory.getOrAddService(new hap_1.hap.Service.AirQualitySensor(serviceName, endpoint));
        (0, helpers_1.getOrAddCharacteristic)(this.service, hap_1.hap.Characteristic.AirQuality);
        for (const e of exposes) {
            const factory = AirQualitySensorHandler.propertyFactories.find((f) => f.canUseExposesEntry(e));
            if (factory === undefined) {
                accessory.log.warn(`Air Quality Sensor does not know how to handle ${e.property} (on ${serviceName})`);
                continue;
            }
            this.properties.push(new factory(e, this.service));
        }
        if (this.properties.length === 0) {
            throw new Error(`Air Quality Sensor (${serviceName}) did not receive any suitable exposes entries.`);
        }
    }
    get getableKeys() {
        const keys = [];
        for (const property of this.properties) {
            if ((0, z2mModels_1.exposesCanBeGet)(property.expose)) {
                keys.push(property.expose.property);
            }
        }
        return keys;
    }
    updateState(state) {
        let airQuality = hap_1.hap.Characteristic.AirQuality.UNKNOWN;
        for (const p of this.properties) {
            p.updateState(state);
            airQuality = AirQualitySensorHandler.getWorstAirQuality(airQuality, p.latestAirQuality);
        }
        this.service.updateCharacteristic(hap_1.hap.Characteristic.AirQuality, airQuality);
    }
    static getWorstAirQuality(a, b) {
        return (a > b) ? a : b;
    }
    static generateIdentifier(endpoint) {
        let identifier = hap_1.hap.Service.AirQualitySensor.UUID;
        if (endpoint !== undefined) {
            identifier += '_' + endpoint.trim();
        }
        return identifier;
    }
}
AirQualitySensorHandler.propertyFactories = [
    VolatileOrganicCompoundsProperty,
    ParticulateMatter10Property,
    ParticulateMatter2Dot5Property,
];
//# sourceMappingURL=air_quality.js.map