"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SwitchCreator = void 0;
const z2mModels_1 = require("../z2mModels");
const hap_1 = require("../hap");
const helpers_1 = require("../helpers");
const monitor_1 = require("./monitor");
class SwitchCreator {
    createServicesFromExposes(accessory, exposes) {
        exposes.filter(e => e.type === z2mModels_1.ExposesKnownTypes.SWITCH && (0, z2mModels_1.exposesHasFeatures)(e)
            && !accessory.isServiceHandlerIdKnown(SwitchHandler.generateIdentifier(e.endpoint)))
            .forEach(e => this.createService(e, accessory));
    }
    createService(expose, accessory) {
        try {
            const handler = new SwitchHandler(expose, accessory);
            accessory.registerServiceHandler(handler);
        }
        catch (error) {
            accessory.log.warn(`Failed to setup switch for accessory ${accessory.displayName} from expose "${JSON.stringify(expose)}": ${error}`);
        }
    }
}
exports.SwitchCreator = SwitchCreator;
class SwitchHandler {
    constructor(expose, accessory) {
        this.accessory = accessory;
        const endpoint = expose.endpoint;
        this.identifier = SwitchHandler.generateIdentifier(endpoint);
        const potentialStateExpose = expose.features.find(e => (0, z2mModels_1.exposesHasBinaryProperty)(e) && !accessory.isPropertyExcluded(e.property)
            && e.name === 'state' && (0, z2mModels_1.exposesCanBeSet)(e) && (0, z2mModels_1.exposesIsPublished)(e));
        if (potentialStateExpose === undefined) {
            throw new Error('Required "state" property not found for Switch.');
        }
        this.stateExpose = potentialStateExpose;
        const serviceName = accessory.getDefaultServiceDisplayName(endpoint);
        accessory.log.debug(`Configuring Switch for ${serviceName}`);
        const service = accessory.getOrAddService(new hap_1.hap.Service.Switch(serviceName, endpoint));
        (0, helpers_1.getOrAddCharacteristic)(service, hap_1.hap.Characteristic.On).on('set', this.handleSetOn.bind(this));
        const onOffValues = new Map();
        onOffValues.set(this.stateExpose.value_on, true);
        onOffValues.set(this.stateExpose.value_off, false);
        this.monitor = new monitor_1.MappingCharacteristicMonitor(this.stateExpose.property, service, hap_1.hap.Characteristic.On, onOffValues);
    }
    get getableKeys() {
        const keys = [];
        if ((0, z2mModels_1.exposesCanBeGet)(this.stateExpose)) {
            keys.push(this.stateExpose.property);
        }
        return keys;
    }
    updateState(state) {
        this.monitor.callback(state);
    }
    handleSetOn(value, callback) {
        const data = {};
        data[this.stateExpose.property] = value ? this.stateExpose.value_on : this.stateExpose.value_off;
        this.accessory.queueDataForSetAction(data);
        callback(null);
    }
    static generateIdentifier(endpoint) {
        let identifier = hap_1.hap.Service.Switch.UUID;
        if (endpoint !== undefined) {
            identifier += '_' + endpoint.trim();
        }
        return identifier;
    }
}
//# sourceMappingURL=switch.js.map