// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef BASE_BASE_PATHS_H_
#define BASE_BASE_PATHS_H_

// This file declares path keys for the base module.  These can be used with
// the PathService to access various special directories and files.

#include "build/build_config.h"

#if defined(OS_WIN)
#include "base/base_paths_win.h"
#elif defined(OS_FUCHSIA)
#include "base/base_paths_fuchsia.h"
#elif defined(OS_APPLE)
#include "base/base_paths_mac.h"
#elif defined(OS_ANDROID)
#include "base/base_paths_android.h"
#endif

#if defined(OS_POSIX)
#include "base/base_paths_posix.h"
#endif

namespace base {

enum BasePathKey {
  PATH_START = 0,

  // The following refer to the current application.
  FILE_EXE,     // Path and filename of the current executable.
  FILE_MODULE,  // Path and filename of the module containing the code for
                // the PathService (which could differ from FILE_EXE if the
                // PathService were compiled into a shared object, for
                // example).
  DIR_EXE,      // Directory containing FILE_EXE.
  DIR_MODULE,   // Directory containing FILE_MODULE.
  DIR_ASSETS,   // Directory that contains application assets.

  // The following refer to system and system user directories.
  DIR_TEMP,          // Temporary directory for the system and/or user.
  DIR_HOME,          // User's root home directory. On Windows this will look
                     // like "C:\Users\<user>"  which isn't necessarily a great
                     // place to put files.
  DIR_USER_DESKTOP,  // The current user's Desktop.

  // The following refer to the applications current environment.
  DIR_CURRENT,  // Current directory.

  // The following are only for use in tests.
  // On some platforms, such as Android and Fuchsia, tests do not have access to
  // the build file system so the necessary files are bundled with the test
  // binary. On such platforms, these will return an appropriate path inside the
  // bundle.
  DIR_SRC_TEST_DATA_ROOT,  // The root of files in the source tree that are
                           // made available to tests. Useful for tests that use
                           // resources that exist in the source tree.
  DIR_SOURCE_ROOT = DIR_SRC_TEST_DATA_ROOT,  // Legacy name still widely used.
                                             // TODO(crbug.com/1264897): Replace
                                             // all instances and remove alias.
  DIR_GEN_TEST_DATA_ROOT,  // The root of files created by the build that are
                           // made available to tests. On platforms that do
                           // not bundle test files, this is usually the
                           // directory containing the test binary.
  DIR_TEST_DATA,           // Directory containing test data for //base tests.
                           // Only for use in base_unittests. Equivalent to
                           // DIR_SRC_TEST_DATA_ROOT + "/base/test/data".

  PATH_END
};

}  // namespace base

#endif  // BASE_BASE_PATHS_H_
