#include "main.h"

char *print_addr(u8 *addr) {
	static char mac_str[16][20];
	static int index = 0;

	index = (index + 1) % 16;

	snprintf(mac_str[index], sizeof(mac_str[index]) - 1, "%02x:%02x:%02x:%02x:%02x:%02x",
		 addr[0], addr[1], addr[2], addr[3], addr[4], addr[5]);
	mac_str[index][sizeof(mac_str[0]) - 1] = 0;

	return mac_str[index];
}

/**
 * monitor_tx_meshmerize_packet - transport a packet with meshmerize header on radiotap
 * @global: global structures
 * @skb: packet information and data
 * @pinfo: auxiliary information on the packet
 *
 * Returns 0 on success, < 0 otherwise.
 */
int monitor_tx_meshmerize_packet(struct global *global, struct sk_buff *skb, struct packet_info *pinfo) {
	int ret;

	if (build_ieee80211(skb, pinfo) < 0)
		return -1;

	if (build_radiotap(skb, pinfo) < 0)
		return -1;

	ret = write(global->mon_fd, skb->data, skb->len);
	if (ret < 0)
		fprintf(stderr, "transmit failed\n");

	return 0;
}

#define MAX_SEQNO        0xffff

int rate_rx_packet(struct global *global, struct sk_buff *skb, struct packet_info *pinfo) {
	struct station_info *sta_info = NULL, *sta_temp;
	int32_t seqno;
	u32 round;

	list_for_each_entry(sta_temp, &global->sta_info_list, list) {
		if (memcmp(sta_temp->mac, pinfo->mac_src, ETH_ALEN) != 0)
			continue;

		sta_info = sta_temp;
		break;
	}

	if (!sta_info) {
		char path[1024];
		char *init_string = "[\n";

		sta_info = malloc(sizeof(*sta_info));
		memset(sta_info, 0, sizeof(*sta_info));

		memcpy(sta_info->mac, pinfo->mac_src, ETH_ALEN);
		INIT_LIST_HEAD(&sta_info->list);

		snprintf(path, sizeof(path), "/tmp/%s/rate_analysis_%s.dump",
			 print_addr(global->wlan_addr),
			 print_addr(sta_info->mac));
		path[sizeof(path) - 1] = 0;

		sta_info->fp = fopen(path, "w");
		if (!sta_info->fp) {
			free(sta_info);
			return -1;
		}

		fwrite(init_string, strlen(init_string), 1, sta_info->fp);

		list_add(&sta_info->list, &global->sta_info_list);
	}

	if (skb->len < sizeof(uint32_t))
		return -1;

	seqno = ntohl(*((uint32_t *) skb->data));
	round = ntohl(*(((uint32_t *) skb->data) + 1));

	if (seqno > MAX_SEQNO)
		seqno = MAX_SEQNO;

	/* restart if mcs/bitrate has changed or seqno is lower */
	if (sta_info->mcs != pinfo->mcs ||
	    sta_info->bitrate != pinfo->bitrate ||
	    seqno < sta_info->seqno ||
	    sta_info->round != round) {
		char open_string[1024];

		if (sta_info->json_started) {
			char *close_string = "\" },\n";

			while (++sta_info->seqno < MAX_SENDS)
				fwrite("0", 1, 1, sta_info->fp);

			fwrite(close_string, strlen(close_string), 1, sta_info->fp);
		}

		sta_info->seqno = -1;
		sta_info->round = round;
		sta_info->mcs = pinfo->mcs;
		sta_info->bitrate = pinfo->bitrate;
		//printf("opening for bitrate %d, mcs %d\n", sta_info->bitrate, sta_info->mcs);

		if (sta_info->bitrate)
			snprintf(open_string, sizeof(open_string),
				 "{ \"round\": %d, \"legacy\": % 3.1f, \"bitmask\": \"", round,
				 ((float) sta_info->bitrate) / 2);
		else
			snprintf(open_string, sizeof(open_string), "{ \"round\": %d, \"mcs\": %d, \"bitmask\": \"",
				 round,
				 sta_info->mcs);

		open_string[sizeof(open_string) - 1] = 0;
		fwrite(open_string, strlen(open_string), 1, sta_info->fp);
		sta_info->json_started = 1;
	}

	while (sta_info->seqno + 1 < seqno) {
		fwrite("0", 1, 1, sta_info->fp);
		sta_info->seqno++;
	}

	fwrite("1", 1, 1, sta_info->fp);
	sta_info->seqno = seqno;

	fflush(sta_info->fp);

	return 0;
}

int monitor_rx_packet(struct global *global, struct sk_buff *skb) {
	struct packet_info pinfo;

	memset(&pinfo, 0, sizeof(pinfo));

	if (parse_radiotap(skb, &pinfo) < 0)
		return -1;

	if (parse_ieee80211(skb, &pinfo) < 0)
		return -1;

	rate_rx_packet(global, skb, &pinfo);

	return 0;
}

void ev_handler_mon_read(int fd, short ev __unused, void *arg) {
	struct global *global = arg;
	struct sk_buff skb;
	static u8 buf[2048];
	ssize_t len;

	memset(buf, 0, sizeof(buf));
	skb.head = &buf[0];
	skb.data = skb.head;
	len = read(fd, skb.data, sizeof(buf));
	if (len < 0) {
		log_printf(global, LOG_INTERFACE, "failed to read from monitor\n");
		exit(-1);
	}
	skb.len = (size_t) len;

	monitor_rx_packet(global, &skb);
}

static u8 legacy_rates[] = {2, 4, 11, 22, 12, 18, 24, 36, 48, 72, 96, 108, -1};

void increment_rate(struct global *global) {
	if (global->disable_legacy) {
		global->bitrate_legacy = (int) sizeof(legacy_rates) - 1;
	} else if (global->bitrate_legacy < (int) sizeof(legacy_rates) - 1) {
		global->bitrate_legacy++;
	}
	if (global->bitrate_legacy >= (int) sizeof(legacy_rates) - 1) {
		global->bitrate_mcs++;
	}
	if (global->bitrate_mcs > 15) {
		/* restart process */
		global->bitrate_legacy = global->disable_legacy ? (int) sizeof(legacy_rates) - 1 : 0;
		global->bitrate_mcs = global->disable_legacy ? 0 : -1;
		global->output_round++;
	}

}

void ev_handler_output_timer(int fd __unused, short ev __unused, void *arg) {
	struct global *global = arg;
	struct packet_info pinfo;
	u8 buf[2048];
	int i;
	struct sk_buff skb;


	//usleep(rand() % (SEND_INTERVAL * 1000000));
	increment_rate(global);

	printf("sending %d packets on bitrate %f / MCS %d\n", MAX_SENDS,
	       legacy_rates[global->bitrate_legacy] / 2.0, global->bitrate_mcs);

	for (i = 0; i < MAX_SENDS; i++) {
		memset(&pinfo, 0, sizeof(pinfo));
		memset(buf, 0, sizeof(buf));
		skb.head = &buf[0];
		skb.data = skb.head + 128;

		skb.len = 1400;
		*((u32 *) skb.data) = htonl(global->output_seqno);
		*((u32 *) (skb.data + 4)) = htonl(global->output_round);
		global->output_seqno++;

		memcpy(&pinfo.mac_dst, "\xff\xff\xff\xff\xff\xff", ETH_ALEN);
		memcpy(&pinfo.mac_src, global->wlan_addr, ETH_ALEN);
		memcpy(&pinfo.ether_dst, "\xff\xff\xff\xff\xff\xff", ETH_ALEN);
		memcpy(&pinfo.ether_src, global->wlan_addr, ETH_ALEN);

		if (global->bitrate_legacy < (int) sizeof(legacy_rates) - 1)
			pinfo.bitrate = legacy_rates[global->bitrate_legacy];
		else
			pinfo.mcs = (u8) global->bitrate_mcs;

		monitor_tx_meshmerize_packet(global, &skb, &pinfo);
	}
	global->output_seqno = 0;

}

void ev_handler_sigterm(int fd __unused, short ev __unused, void *arg) {
	struct station_info *sta_info;
	struct global *global = arg;

	printf("sigterm\n");

	list_for_each_entry(sta_info, &global->sta_info_list, list) {
		if (sta_info->json_started) {
			char *close_string = "\" }\n]\n";

			while (++sta_info->seqno < MAX_SENDS)
				fwrite("0", 1, 1, sta_info->fp);

			fwrite(close_string, strlen(close_string), 1, sta_info->fp);
		} else {
			char *close_string = "]\n";
			fwrite(close_string, strlen(close_string), 1, sta_info->fp);
		}
		fclose(sta_info->fp);

		/* TODO: free'ing the stuff would be cleaner, but we exit afterwards anyway */

		break;
	}

	event_base_loopbreak(global->ev_base);
}

static void usage(const char *argv0) {
	fprintf(stderr, "Usage: %s [-p phy][-i ibss] [-r send/recv/duplex] [-m]\n", argv0);
	fprintf(stderr, "\tOptions:"
		"\n\t\t -p \tphy interface to use;  Default: Phy0"
		"\n\t\t -i \tibss interface to use; Default: wlan0"
		"\n\t\t -r \tRole of the node;      Default: duplex"
		"\n\t\t -m \tSend only in mcs rates");
	exit(EXIT_FAILURE);
}

int main(int argc, char *argv[]) {
	struct global global;
	struct event *ev_mon;
	struct event *ev_sigterm, *ev_sighup, *ev_sigint;
	char cmd[1024];
	char *phy = "phy0";
	char *wlan = "wlan0";
	char *role = "duplex";
	struct stat st = {0};
	char out_path[1024];
	int opt;
	struct timeval output_timer_interval = {DEFAULT_SEND_INTERVAL, 0};
	memset(&global, 0, sizeof(global));

	while ((opt = getopt(argc, argv, "p:i:r:t:m::")) != -1) {
		switch (opt) {
			case 'p':
				phy = optarg;
				break;
			case 'i':
				wlan = optarg;
				break;
			case 'r':
				role = optarg;
				break;
			case 't':
				output_timer_interval.tv_sec = atoi(optarg);
				break;
			case 'm':
				global.disable_legacy = true;
				break;
			default:
				usage(argv[0]);
		}
	}
	global.bitrate_legacy = -1;
	global.bitrate_mcs = -1;

	if (get_device_mac(wlan, global.wlan_addr) < 0) {
		fprintf(stderr, "Couldn't acquire wlan address\n");
		return -1;
	}

	INIT_LIST_HEAD(&global.sta_info_list);

	snprintf(cmd, sizeof(cmd), "iw phy %s interface add %s_mon type monitor", phy, phy);
	cmd[sizeof(cmd) - 1] = 0;
	system(cmd);
	snprintf(cmd, sizeof(cmd), "ifconfig %s_mon up", phy);
	cmd[sizeof(cmd) - 1] = 0;
	system(cmd);
	snprintf(cmd, sizeof(cmd), "%s_mon", phy);
	cmd[sizeof(cmd) - 1] = 0;

	global.mon_fd = rawsock_create(cmd);

	if (global.mon_fd < 0) {
		fprintf(stderr, "Can't create monitor interface - are you root?\n");
		return -1;
	}

	global.ev_base = event_base_new();
	if (!global.ev_base) {
		fprintf(stderr, "Couldn't set up event base\n");
		return -1;
	}

	sprintf(out_path, "/tmp/%s", print_addr(global.wlan_addr));
	if (stat(out_path, &st) == -1) {
		mkdir(out_path, 0755);
	}

	ev_sigterm = event_new(global.ev_base, SIGTERM, EV_SIGNAL | EV_PERSIST, ev_handler_sigterm, &global);
	event_add(ev_sigterm, NULL);
	ev_sighup = event_new(global.ev_base, SIGHUP, EV_SIGNAL | EV_PERSIST, ev_handler_sigterm, &global);
	event_add(ev_sighup, NULL);
	ev_sigint = event_new(global.ev_base, SIGINT, EV_SIGNAL | EV_PERSIST, ev_handler_sigterm, &global);
	event_add(ev_sigint, NULL);

	if (!strcmp(role, "recv") || !strcmp(role, "duplex")) {
		ev_mon = event_new(global.ev_base, global.mon_fd, EV_READ | EV_PERSIST, ev_handler_mon_read, &global);
		event_add(ev_mon, NULL);
	}

	if (!strcmp(role, "send") || !strcmp(role, "duplex")) {
		global.output_timer = event_new(global.ev_base, -1, EV_PERSIST, ev_handler_output_timer, &global);
		event_add(global.output_timer, &output_timer_interval);
	}

	event_base_loop(global.ev_base, 0);


	return 0;
}
