---
title: Meso
---

# From 30.000 feet

Starting with version 0.26.0 the mesoscopic model is available in the
[sumo](../sumo.md) distribution ([MESO](../Simulation/Meso.md) has been
developed alongside SUMO since 2002 but was not publicly available).
[MESO](../Simulation/Meso.md) refers to a mesoscopic simulation model which
uses the same input data as the main [sumo](../sumo.md) model. It
computes vehicle movements with queues and runs up to 100 times faster
than the microscopic model of [sumo](../sumo.md). Additionally, due
to using a coarser model for intersections and lane-changing it is more
tolerant of network modelling errors than [sumo](../sumo.md).

- **Purpose:** Simulates a defined scenario
- **System:** portable (Linux/Windows is tested); runs on command line
- **Input (mandatory):**
  - A) a road network as generated via [netconvert](../netconvert.md) or [netgenerate](../netgenerate.md), see [Building Networks](../index.md#network_building)
  - B) a set of routes (as generated by [duarouter](../duarouter.md), [jtrrouter](../jtrrouter.md), [dfrouter](../dfrouter.md), or [activitygen](../activitygen.md), see also [Definition of Vehicles, Vehicle Types, and Routes](../Definition_of_Vehicles,_Vehicle_Types,_and_Routes.md))

- **Input (optional):** Additional definitions of traffic lights, variable speed signs, output detectors etc.
- **Output:** SUMO allows to generate a wide set of outputs; visualization is done using [sumo-gui](../sumo-gui.md)
- **Programming Language:** C++

# Usage Description

The mesoscopic model uses the same input files and generates (almost)
the same outputs as [sumo](../sumo.md). To use
[MESO](../Simulation/Meso.md) the option **--mesosim** {{DT_BOOL}} must be set. Additional options are
described at [sumo#mesoscopic](../sumo.md#mesoscopic)

# Model Description

The simulation model is based on the work of [Eissfeldt, Vehicle-based modelling of traffic](http://kups.ub.uni-koeln.de/1274/). The original
model was focused on motorway traffic. The current implementation
contains significant extensions to support the simulation of
heterogeneous urban traffic as well.

## Used vType parameters
Only a few vType parameters affect the mesoscopic simulation. They are listed below

- vClass
- length
- minGap
- maxSpeed
- speedFactor, speedDev (see [Further Congestion Effects](#further_congestion_effects))
- [impatience](#impatience)
- accel,decel (only for computing junction passing time when the [microscopic junction model](#junction_model) is active)

## Longitudinal Model

Vehicles are placed in traffic queues with a maximum length specified by
option **--meso-edgelength**. To accomplish this each edge is split into 1 or more queues of
equal length. Generally a queue represents all lanes of an edge but at
intersections with dedicated turning lanes, multiple queues are used.
Vehicles generally leave the queues in the order in which they entered
it. The core idea of the model is to compute the time at which a vehicle
exits the queue according to the following criteria:

1.  the traffic state in the current and subsequent queue.
2.  the minimum travel time based on vehicle speed, and speed limit
3.  the state of the intersection for the last queue of an edge
    according to the configuration of the
    [\#Junction_Model](#junction_model)

The first point is accomplished by classifying each queue as either
*jammed* or *free* according to a traffic density threshold (configured
with option **--meso-jam-threshold**). The minimum headway between vehicles is then computed
differently for each of the four distinct possible cases. The behavior
for each case is configured using the options

- **meso-tauff**: minimum headway when traveling from free segment to
  free segment
- **meso-taufj**: minimum headway when traveling from free segment to
  jammed segment
- **meso-taujf**: minimum headway when traveling from jammed segment
  to free segment
- **meso-taujj**: headway for 'spaces' to travel backwards through the
  jam. When a segment is completely occupied the actual headway is
  VEHICLE_NUMBER \* taujj.

The headways are scaled according to the inverse of the lane number
(with three lanes, the headways are divided by 3).

!!! note
    The values tauff, taufj and taujf denote net-time gaps (vehicle front bumper to leader back-bumper). The actual gross time gap is computed from this by taking into account the vehicle length.

The value taujj configures the speed with which the jam front moves
upstream when a multi-segment jam clears.

Backward-moving jams are possible due to the headway effects when
meso-tauff is lower than meso-taujf.

### meso-jam-threshold

For each queue (also called segment in the GUI) an occupancy threshold
value determines whether that queue is *jammed* or *free*. The following
numerical values are supported for option **--meso-jam-treshold** {{DT_FLOAT}}:

- value = -1: Threshold is computed so that vehicles driving at the
  speed limit do not jam. This is the default behavior which is
  recommend for networks with varying speed limits since the free-flow
  occupancy depends on speed.
- value < 0: Threshold is computed as above but the free-flow
  occupancy is computed for (speedLimit \* -value). As this value
  approaches 0 (from below) jamming is reduced. Values below -1 cause
  increased jamming
- value \> 0: The value is used directly as the threshold fraction for
  all queues (i.e. 0.4 means that above 40% occupancy the queue is
  jammed)
- value = 1: Jamming is disabled

### Further Congestion Effects

Another mechanism which creates a negative correlation between density
and speed comes through vehicle (preferred)-speed distributions. At
higher traffic densities the probability of being slowed down because a
slow vehicle is ahead in the queue rises. This effect size can be
controlled by [changing the spread of the speed distribution.](../Definition_of_Vehicles,_Vehicle_Types,_and_Routes.md#speed_distributions)
On multi-lane edges, [overtaking can be enabled](#lateral_model) to reduce this slow-down effect.

### Jam-Resolution

The option **--meso-recheck** {{DT_TIME}} can be used to delay traffic flow into a fully occupied segment. Whenever a vehicle cannot move into the next segment because it is full, the given value acts as a time delay before checking again whether the segment has capacity to receive another vehicle. By default this delay is set to 0. 

## Lateral Model

Lateral movement is not modelled explicitly. Vehicles may overtake each
other (swap their position) if the option **--meso-overtaking** is enabled. This is a
randomized process depending on vehicle speeds and density.

## Junction Model

There are 3 basic options for modelling junction control

1. **--meso-junction-control false** (the default): No junction control takes place. This should only be
    used for motorway scenarios or in combination with the penalty options below

2. **--meso-junction-control true**: junctions are modeled as in the [simplified microsim model](../Simulation/Intersections.md#internal_links) (**--no-internal-links true**).

3. **--meso-junction-control.limited true**: No junction control takes place as long as the target queue is in
    the unjammed state (*free*). When trying to enter a jammed queue,
    junction control as in case 2 is used.

Additional options can be used to control intersection delays:

### TLS-Penalty

When setting **--meso-tls-penalty** {{DT_FLOAT}}: to a value \> 0 (default is 0), a travel time penalty is applied
according to the average delay time (based on red phase duration).
This takes precedence over any of the above **--meso-junction-control** settings for tls controlled
intersections. FLOAT is used as a scaling factor that roughly
corresponds to coordination (1.0 corresponds to uncoordinated traffic
lights, whereas values approaching 0 model near-perfect coordination).
(The limit case of no-penalty can be achieved by setting **--meso-junction-control false**)

For an option value of *p*, The time penalty is computed by scaling the
expected waiting time for random arrival within the cycle

```
travelTimePenalty = p * (redTime * redTime + redTime) / (2 * cycleTime)
```

### TLS-Flow-Penalty
When setting **--meso-tls-flow-penalty** {{DT_FLOAT}}: to a value \> 0 (default is 0), a headway penalty is applied
 which serves to reduce the maximum flow across a tls-controlled intersection (In contrast to actual junction control, they flow is spread evently across the phase cycle rather than being concentrated during the green phases).
When the flow penalty is set to a value of 1 the minimum headway time is increased to model the maximum capacity
according to the proportion of green time to cycle time.
Higher penalty values can be used to reduce the flow even further while lower values increase the maximum flow.
The latter is useful if the green split is not known exactly (because the traffic light program is guessed heuristically).
```
greenFraction = MIN2(1.0, (cycleTime - redDuration) / cycleTime) / penalty))
headway = defaultHeadway / greenFraction
```

Note, that the maximum flow cannot exceed the value at permenant green light regardless of penalty value.

### Penalty at uncontrolled intersections

**--meso-minor-penalty** {{DT_TIME}} can be used to apply a fixed time penalty when passing an unprioritzed
link (this is disabled when --meso-junction-control.limited is set and
junction control is active for that link)

### Impatience

Vehicles that reach an [impatience](../Definition_of_Vehicles,_Vehicle_Types,_and_Routes.md#impatience) value of 1 (maximum) can pass an intersection regardless of foe traffic with higher priority. The time to reach maximum impatience can be configured with option **--time-to-impatience** {{DT_TIME}}. 

## Configuration by edge type

The model parameters described above can be customized for each edge type id by loading the following configuration from an additional file:
```
<additional>
    <type id="highway.motorway" ...>
        <meso tauff="1.13" taufj="1.13" taujf="1.73" taujj="1.4" jamThreshold="-1"
             multiQueue="true" junctionControl="false" tlsPenalty="0" tlsFlowPenalty="0"
             minorPenalty="0" overtaking="false"/>
    </type>
</additional>
```
All attributes are optional and default to the value of the option with the corresponding option name. (i.e. tauff ~ **meso-tauff**, tlsPenalty ~ **meso-tls-penalty**, ...).

## Outputs

Many of the output options of SUMO are supported but the resulting files
differ somewhat:

- No lane specific output is possible.
  - [<laneData\>-output](../Simulation/Output/Lane-_or_Edge-based_Traffic_Measures.md#lane-based_network_states)
    will be treated like `<edgeData>`-output.
  - [netstate-dump](../Simulation/Output/RawDump.md) will
    report vehicles as child elements of `<edge>` instead of `<lane>`.
- Induction loops write attributes that are similar to meanData output

The following outputs are not supported:

- [Lane area detectors (E2)](../Simulation/Output/Lanearea_Detectors_(E2).md)
- [Multi-Entry-Exit detectors (E3)](../Simulation/Output/Multi-Entry-Exit_Detectors_(E3).md)

## Limitations compared to SUMO

The following SUMO features are not supported:

- [Actuated traffic lights](Traffic_Lights.md#actuated_traffic_lights)
- Electric model
- Wireless model
- Opposite-direction driving
- Sublane-model
- Boarding durations for passengers

# MESO-GUI

The mesoscopic simulation can be observed with
[sumo-gui](../sumo-gui.md) when running with option **--mesosim**. The
difference between micro- and meso-simulation is not visible from
isolated vehicles. Once there are multiple vehicles waiting in a queue
it becomes more apparent due to differences in spacing between the
vehicles. And from the fact that vehicles appear to *jump* in 100m
increments (which is the default segment length). When right-clicking on
vehicles or edges and selecting 'Show parameters' the set of values is
different between the microscopic and mesoscopic model. For larger
simulations the speed difference will also be noticeable. Due to being
based on edges rather than lanes, some of the visualization options are
different when running the mesoscopic model as well.
