"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createMessageGenerator = exports.secureMessageGeneratorS2 = exports.secureMessageGeneratorS0 = exports.simpleMessageGenerator = exports.waitForNodeUpdate = void 0;
const cc_1 = require("@zwave-js/cc");
const SecurityCC_1 = require("@zwave-js/cc/SecurityCC");
const core_1 = require("@zwave-js/core");
const deferred_promise_1 = require("alcalzone-shared/deferred-promise");
const SendDataShared_1 = require("../serialapi/transport/SendDataShared");
const StateMachineShared_1 = require("./StateMachineShared");
async function waitForNodeUpdate(driver, msg, timeoutMs) {
    try {
        return await driver.waitForMessage((received) => {
            return msg.isExpectedNodeUpdate(received);
        }, timeoutMs);
    }
    catch (e) {
        throw new core_1.ZWaveError(`Timed out while waiting for a response from the node`, core_1.ZWaveErrorCodes.Controller_NodeTimeout);
    }
}
exports.waitForNodeUpdate = waitForNodeUpdate;
/** A simple message generator that simply sends a message, waits for the ACK (and the response if one is expected) */
const simpleMessageGenerator = async function* (driver, msg, onMessageSent, additionalCommandTimeoutMs = 0) {
    // Pass this message to the send thread and wait for it to be sent
    let result;
    let commandTimeMs;
    // At this point we can't have received a premature update
    msg.prematureNodeUpdate = undefined;
    try {
        // The yield can throw and must be handled here
        result = yield msg;
        // Figure out how long the message took to be handled
        msg.markAsCompleted();
        commandTimeMs = Math.ceil(msg.rtt / 1e6);
        onMessageSent(msg, result);
    }
    catch (e) {
        msg.markAsCompleted();
        throw e;
    }
    // If the message was sent to a node and came back with a NOK callback,
    // we want to inspect the callback, for example to look at TX statistics
    // or update the node status.
    //
    // We now need to throw because the callback was passed through so we could inspect it.
    if ((0, SendDataShared_1.isTransmitReport)(result) && !result.isOK()) {
        // Throw the message in order to short-circuit all possible generators
        throw result;
    }
    // If the sent message expects an update from the node, wait for it
    if (msg.expectsNodeUpdate()) {
        // We might have received the update prematurely. In that case, return it.
        if (msg.prematureNodeUpdate)
            return msg.prematureNodeUpdate;
        // CommandTime is measured by the application
        // ReportTime timeout SHOULD be set to CommandTime + 1 second.
        const timeout = commandTimeMs +
            driver.options.timeouts.report +
            additionalCommandTimeoutMs;
        return waitForNodeUpdate(driver, msg, timeout);
    }
    return result;
};
exports.simpleMessageGenerator = simpleMessageGenerator;
/** A simple (internal) generator that simply sends a command, and optionally returns the response command */
async function* sendCommandGenerator(driver, command, onMessageSent, options) {
    const msg = driver.createSendDataMessage(command, options);
    const resp = yield* (0, exports.simpleMessageGenerator)(driver, msg, onMessageSent);
    if (resp && (0, cc_1.isCommandClassContainer)(resp)) {
        driver.unwrapCommands(resp);
        return resp.command;
    }
}
/** A message generator for security encapsulated messages (S0) */
const secureMessageGeneratorS0 = async function* (driver, msg, onMessageSent) {
    if (!(0, SendDataShared_1.isSendData)(msg)) {
        throw new core_1.ZWaveError("Cannot use the S0 message generator for a command that's not a SendData message!", core_1.ZWaveErrorCodes.Argument_Invalid);
    }
    else if (typeof msg.command.nodeId !== "number") {
        throw new core_1.ZWaveError("Cannot use the S0 message generator for multicast commands!", core_1.ZWaveErrorCodes.Argument_Invalid);
    }
    else if (!(msg.command instanceof SecurityCC_1.SecurityCCCommandEncapsulation)) {
        throw new core_1.ZWaveError("The S0 message generator can only be used for Security S0 command encapsulation!", core_1.ZWaveErrorCodes.Argument_Invalid);
    }
    // Step 1: Acquire a nonce
    const secMan = driver.securityManager;
    const nodeId = msg.command.nodeId;
    let additionalTimeoutMs;
    // Try to get a free nonce before requesting a new one
    let nonce = secMan.getFreeNonce(nodeId);
    if (!nonce) {
        // No free nonce, request a new one
        const cc = new SecurityCC_1.SecurityCCNonceGet(driver, {
            nodeId: nodeId,
            endpoint: msg.command.endpointIndex,
        });
        const nonceResp = yield* sendCommandGenerator(driver, cc, (msg, result) => {
            additionalTimeoutMs = Math.ceil(msg.rtt / 1e6);
            onMessageSent(msg, result);
        }, {
            // Only try getting a nonce once
            maxSendAttempts: 1,
        });
        if (!nonceResp) {
            throw new core_1.ZWaveError("No nonce received from the node, cannot send secure command!", core_1.ZWaveErrorCodes.SecurityCC_NoNonce);
        }
        nonce = nonceResp.nonce;
    }
    msg.command.nonce = nonce;
    // Now send the actual secure command
    return yield* (0, exports.simpleMessageGenerator)(driver, msg, onMessageSent, additionalTimeoutMs);
};
exports.secureMessageGeneratorS0 = secureMessageGeneratorS0;
/** A message generator for security encapsulated messages (S2) */
const secureMessageGeneratorS2 = async function* (driver, msg, onMessageSent) {
    if (!(0, SendDataShared_1.isSendData)(msg)) {
        throw new core_1.ZWaveError("Cannot use the S2 message generator for a command that's not a SendData message!", core_1.ZWaveErrorCodes.Argument_Invalid);
    }
    else if (typeof msg.command.nodeId !== "number") {
        throw new core_1.ZWaveError("Cannot use the S2 message generator for multicast commands!", // (yet)
        core_1.ZWaveErrorCodes.Argument_Invalid);
    }
    else if (!(msg.command instanceof cc_1.Security2CCMessageEncapsulation)) {
        throw new core_1.ZWaveError("The S2 message generator can only be used for Security S2 command encapsulation!", core_1.ZWaveErrorCodes.Argument_Invalid);
    }
    const secMan = driver.securityManager2;
    const nodeId = msg.command.nodeId;
    const spanState = secMan.getSPANState(nodeId);
    let additionalTimeoutMs;
    if (spanState.type === core_1.SPANState.None ||
        spanState.type === core_1.SPANState.LocalEI) {
        // Request a new nonce
        // No free nonce, request a new one
        const cc = new cc_1.Security2CCNonceGet(driver, {
            nodeId: nodeId,
            endpoint: msg.command.endpointIndex,
        });
        const nonceResp = yield* sendCommandGenerator(driver, cc, (msg, result) => {
            additionalTimeoutMs = Math.ceil(msg.rtt / 1e6);
            onMessageSent(msg, result);
        }, {
            // Only try getting a nonce once
            maxSendAttempts: 1,
        });
        if (!nonceResp) {
            throw new core_1.ZWaveError("No nonce received from the node, cannot send secure command!", core_1.ZWaveErrorCodes.Security2CC_NoSPAN);
        }
        // Storing the nonce is not necessary, this will be done automatically when the nonce is received
    }
    // Now send the actual secure command
    let response = yield* (0, exports.simpleMessageGenerator)(driver, msg, onMessageSent, additionalTimeoutMs);
    if ((0, cc_1.isCommandClassContainer)(response) &&
        response.command instanceof cc_1.Security2CCNonceReport) {
        const command = response.command;
        if (command.SOS && command.receiverEI) {
            // The node couldn't decrypt the last command we sent it. Invalidate
            // the shared SPAN, since it did the same
            secMan.storeRemoteEI(nodeId, command.receiverEI);
        }
        driver.controllerLog.logNode(nodeId, {
            message: `failed to decode the message, retrying with SPAN extension...`,
            direction: "none",
        });
        // Prepare the message for re-transmission
        msg.callbackId = undefined;
        msg.command.unsetSequenceNumber();
        // Send the message again
        response = yield* (0, exports.simpleMessageGenerator)(driver, msg, onMessageSent, additionalTimeoutMs);
        if ((0, cc_1.isCommandClassContainer)(response) &&
            response.command instanceof cc_1.Security2CCNonceReport) {
            // No dice
            driver.controllerLog.logNode(nodeId, {
                message: `failed to decode the message after re-transmission with SPAN extension, dropping the message.`,
                direction: "none",
                level: "warn",
            });
            throw new core_1.ZWaveError("The node failed to decode the message.", core_1.ZWaveErrorCodes.Security2CC_CannotDecode);
        }
    }
    return response;
};
exports.secureMessageGeneratorS2 = secureMessageGeneratorS2;
function createMessageGenerator(driver, msg, onMessageSent) {
    const resultPromise = (0, deferred_promise_1.createDeferredPromise)();
    const generator = {
        parent: undefined,
        current: undefined,
        self: undefined,
        start: () => {
            const resetGenerator = () => {
                generator.current = undefined;
                generator.self = undefined;
            };
            async function* gen() {
                // Determine which message generator implementation should be used
                let implementation = exports.simpleMessageGenerator;
                if ((0, SendDataShared_1.isSendData)(msg)) {
                    if (msg.command instanceof cc_1.Security2CCMessageEncapsulation) {
                        implementation = exports.secureMessageGeneratorS2;
                    }
                    else if (msg.command instanceof SecurityCC_1.SecurityCCCommandEncapsulation) {
                        implementation = exports.secureMessageGeneratorS0;
                    }
                }
                // Step through the generator so we can easily cancel it and don't
                // accidentally forget to unset this.current at the end
                const gen = implementation(driver, msg, onMessageSent);
                let sendResult;
                let result;
                while (true) {
                    // This call passes the previous send result (if it exists already) to the generator and saves the
                    // generated or returned message in `value`. When `done` is true, `value` contains the returned result of the message generator
                    try {
                        const { value, done } = await gen.next(sendResult);
                        if (done) {
                            result = value;
                            break;
                        }
                        // Pass the generated message to the transaction machine and remember the result for the next iteration
                        generator.current = value;
                        sendResult = yield generator.current;
                    }
                    catch (e) {
                        if (e instanceof Error) {
                            // There was an actual error, reject the transaction
                            resultPromise.reject(e);
                        }
                        else if ((0, SendDataShared_1.isTransmitReport)(e) && !e.isOK()) {
                            // The generator was prematurely ended by throwing a NOK transmit report.
                            // If this cannot be handled (e.g. by moving the messages to the wakeup queue), we need
                            // to treat this as an error
                            if (driver.handleMissingNodeACK(generator.parent)) {
                                resetGenerator();
                                return;
                            }
                            else {
                                resultPromise.reject((0, StateMachineShared_1.sendDataErrorToZWaveError)("callback NOK", generator.parent, e));
                            }
                        }
                        else {
                            // The generator was prematurely ended by throwing a Message
                            resultPromise.resolve(e);
                        }
                        break;
                    }
                }
                resultPromise.resolve(result);
                resetGenerator();
                return;
            }
            generator.self = gen();
            return generator.self;
        },
    };
    return { resultPromise, generator };
}
exports.createMessageGenerator = createMessageGenerator;
//# sourceMappingURL=MessageGenerators.js.map