module TapKit

	class Relationship
		attr_accessor :entity, :name, :own, :to_many, :join_semantic, :mandatory,\
			:delete_rule, :propagate, :destination
		attr_accessor :joins, :destination_entity, :source_attributes, :destination_attributes

		INNER_JOIN       = :inner_join
		FULL_OUTER_JOIN  = :full_outer_join
		LEFT_OUTER_JOIN  = :left_outer_join
		RIGHT_OUTER_JOIN = :right_outer_join

		DELETE_RULE_NULLIFY = ClassDescription::DELETE_RULE_NULLIFY
		DELETE_RULE_CASCADE = ClassDescription::DELETE_RULE_CASCADE
		DELETE_RULE_DENY    = ClassDescription::DELETE_RULE_DENY

		def initialize( list = {}, entity = nil )
			@entity        = entity
			@name          = list['name']
			@destination   = list['destination']
			@to_many       = list['to_many']       || false
			@mandatory     = list['mandatory']     || false
			@delete_rule   = list['delete_rule']   || 'nullify'
			@own           = list['own']           || false
			@propagate     = list['propagate']     || false
			@join_semantic = list['join_semantic'] || 'inner'

			@joins                  = []
			@source_attributes      = []
			@destination_attributes = []

			case @delete_rule
			when 'nullify' then @delete_rule = DELETE_RULE_NULLIFY
			when 'cascade' then @delete_rule = DELETE_RULE_CASCADE
			when 'deny'    then @delete_rule = DELETE_RULE_DENY
			end

			case @join_semantic
			when 'inner'       then @join_semantic = INNER_JOIN
			when 'full_outer'  then @join_semantic = FULL_OUTER_JOIN
			when 'left_outer'  then @join_semantic = LEFT_OUTER_JOIN
			when 'right_outer' then @join_semantic = RIGHT_OUTER_JOIN
			end

			if entity then
				@destination_entity = entity.model.model_group.entity @destination
			end

			list['joins'] ||= []
			list['joins'].each do |join|
				source      = @entity.attribute join['source']
				destination = @destination_entity.attribute join['destination']
				@source_attributes << source
				@destination_attributes << destination
				add Join.new(source, destination)
			end

			if (list.empty? == false) and entity then
				validate_required_attributes
			end
		end

		def beautify_name
			if @name then
				@name = @name.downcase
			end
		end

		def add( join )
			if @joins.size > 2 then
				raise "Joins have already added."
			end

			@joins << join
		end

		def remove( join )
			@joins.delete join
		end

		def to_many?
			@to_many
		end

		def mandatory?
			@mandatory
		end

		def own?
			@own
		end

		def propagate?
			@propagate
		end

		def destination_attribute( source_attribute )
			@joins.each do |join|
				if join.source == source_attribute then
					return join.destination
				end
			end
			nil
		end

		def inverse_relationship
			@destination_entity.relationships.each do |rel|
				if rel.destination_entity == @entity then
					rel.joins.each do |join|
						if @destination_attributes.include?(join.source) and \
						   @source_attributes.include?(join.destination) then
							return rel
						end
					end
				end
			end
			nil
		end

		def validate_value( value )
			if FaultHandler === value then
				return value
			end

			errors = []
			_validate_type(errors, value)
			_validate_mandatory(errors, value)

			unless errors.empty? then
				raise ValidationError.aggregate(errors)
			end

			value
		end

		private

		def _property
			"The '#@name' relationship of '#{@entity.name}'"
		end

		def _validate_type( errors, value )
			if value.nil? then
				return
			end

			msg = "#{_property} is wrong type #{value.class}"
			if to_many? then
				unless (Array === value) or (FaultingDelayEvaluationArray === value) then
					errors << ValidationError.new(msg, @name)
				end
			else
				unless DatabaseObject === value then
					errors << ValidationError.new(msg, @name)
				end
			end
		end

		def _validate_mandatory( errors, value )
			if mandatory? then
				if value.nil? then
					errors << ValidationError.new("#{_property} can't allow null", @name)
				elsif to_many? then
					if value.size == 0 then
						errors << ValidationError.new( \
							"#{_property} can't allow no relationships", @name)
					end
				end
			end
		end

		public

		def validate_required_attributes
			msg = "Relationship requires attributes: 'name', 'destination', " +
				"'joins' (this requires 'source' and 'destination')"

			if @name.nil? or @destination.nil? or @joins.empty? then
				key = @name || :relationship
				error = ValidationError.new(msg, key)
				raise error
			end
		end

		def to_s
			@name
		end

		def to_h
			case @delete_rule
			when DELETE_RULE_NULLIFY then delete_rule = 'nullify'
			when DELETE_RULE_CASCADE then delete_rule = 'cascade'
			when DELETE_RULE_DENY    then delete_rule = 'deny'
			end

			case @join_semantic
			when INNER_JOIN       then join_semantic = 'inner'
			when FULL_OUTER_JOIN  then join_semantic = 'full_outer'
			when LEFT_OUTER_JOIN  then join_semantic = 'left_outer'
			when RIGHT_OUTER_JOIN then join_semantic = 'right_outer'
			end

			joins = []
			@joins.each do |join|
				joins << join.to_h
			end

			{
				'name'          => @name,
				'destination'   => @destination_entity.name,
				'to_many'       => @to_many,
				'mandatory'     => @mandatory,
				'delete_rule'   => delete_rule,
				'join_semantic' => join_semantic,
				'joins'         => joins
			}
		end
	end
end
