module TapKit

	# == Top level objects
	# ModelGroup, ObjectStoreCoordinator, ObserverCenter, SharedEditingContext
	class Application
		attr_reader :model_group, :fetch_timestamp_lag, :observer_center, :notification_center
		attr_reader :object_store, :shared_editing_context, :databases
		attr_accessor :log_options

		def initialize( paths = [] )
			@observer_center = ObserverCenter.new
			@notification_center = NotificationCenter.new
			@model_group = ModelGroup.new
			@object_store = ObjectStoreCoordinator.new self
			@shared_editing_context = SharedEditingContext.new @object_store
			@databases = {}
			@log_options = {:sql => false, :out => $stdout}

			if String === paths then
				paths = [paths]
			end

			paths.each do |path|
				model = Model.new path
				_ready_access model
			end

			unless paths.empty? then
				_generate_relationships
			end

			@notification_center.add(DatabaseContext, :handle_notification,
				ObjectStoreCoordinator::COS_NEEDED_NOTIFICATION)
		end

		private

		def _ready_access( model )
			@model_group.add model
			db = Database.new(self, model)
			add_database(model, db)
		end

		def _generate_relationships
			models.each do |model|
				list = model.model_list
				list['entities'].each do |entity_list|
					entity_name = entity_list['name']
					entity = @model_group.entity entity_name
					if re_lists = entity_list['relationships'] then
						re_lists.each do |re_list|
							relation = Relationship.new(re_list, entity)
							entity.add_relationship relation
						end
					end
				end
			end
		end

		public

		def add_database( model, db )
			@databases[model] = db
		end

		def database( model )
			@databases[model]
		end

		def adapter( model )
			if db = @databases[model] then
				db.adapter
			end
		end

		def add_model( model )
			if @model_group.models.empty? then
				_ready_access model
			else
				@model_group.add model
			end
		end

		alias << add_model

		def add_models( models )
			models.each do |model|
				add_model model
			end
		end

		def models
			@model_group.models
		end

		def model( name )
			@model_group.model name
		end

		def entities
			@model_group.entities
		end

		def entity( name )
			@model_group.entity name
		end

		def create_editing_context
			ec = EditingContext.new @object_store
			ec.shared_editing_context = @shared_editing_context
			ec
		end

		alias ec create_editing_context
	end
end

