# * do SQL statements with and without transaction
# * do SQL statements (select, insert, update, delete)

$LOAD_PATH.unshift '../../../lib'
require 'tapkit'
require 'test/unit'
require 'test/unit/ui/console/testrunner'

MODEL = 'testadapter.yaml'

class TestMySQLAdapter < Test::Unit::TestCase
	include TapKit

	def setup
		app = Application.new
		model = Model.new MODEL
		@adapter = MySQLAdapter.new(model, app)
	end

	def test_using_classes
		assert(MySQLExpression, MySQLAdapter.expression_class)
		assert_kind_of(MySQLExpressionFactory, @adapter.expression_factory)
	end

	def test_internal_type
		tests = {
			'tinyint'=>Integer, 'tinyint unsigned'=>Integer,
			'smallint'=>Integer, 'smallint unsigned'=>Integer,
			'mediumint'=>Integer, 'mediumint unsigned'=>Integer,
			'int'=>Integer, 'int unsigned'=>Integer, 
			'bigint'=>Integer, 'bigint unsigned'=>Integer, 
			'float'=>Float, 'double'=>Float, 'double precision'=>Float,
			'real'=>Float, 'decimal'=>String, 'numeric'=>Float,
			'datetime'=>Timestamp, 'year'=>Date,
			'char'=>String, 'varchar'=>String, 'tinyblob'=>String,
			'tinytext'=>String, 'blob'=>String, 'text'=>String,
			'mediumblob'=>String, 'mediumtext'=>String, 'longblob'=>String,
			'longtext'=>String
		}

		tests.each do |type, expected|
			assert_equal(expected, MySQLAdapter.internal_type(type))
		end
	end
end


class TestMySQLAccessing < Test::Unit::TestCase
	include TapKit

	def setup
		@app = Application.new [MODEL]
		@adapter = @app.adapter(@app.model(MODEL))
		@context = MySQLContext.new @adapter
		@channel = @context.create_channel
		@ec = @app.ec
	end

	def test_commit_without_transaction
		_commit(false)
	end

	def test_commit_with_transaction
		_commit(true)
	end

	def _commit( transaction, pk = 100 )
		@adapter.connection['transaction'] = transaction
		# @app.log_options[:sql] = true

		str = "test_#{Time.now}"
		num = rand(10000)
		date = Date.today
		expr = MySQLExpression.new @app.entity('TestAdapter')

		# insert
		expr.statement = "insert into TEST_ADAPTER (TEST_ID, TEST_STRING, TEST_NUMBER, TEST_DATE) values (#{pk}, '#{str}', #{num}, '#{date}')"
		@context.begin_transaction
		@channel.evaluate(expr)
		@context.commit_transaction

		# update
		num += 1
		expr.statement = "update TEST_ADAPTER set TEST_NUMBER = #{num} where TEST_ID = #{pk}"
		@context.begin_transaction
		@channel.evaluate expr
		@context.commit_transaction

		# select
		expr.statement = "select TEST_ID, TEST_STRING, TEST_NUMBER, TEST_DATE from TEST_ADAPTER where TEST_ID = #{pk}"
		@context.begin_transaction
		state = @channel.evaluate expr
		@context.commit_transaction
		row = state.fetch_all.first
		assert_equal(pk, row[0])
		assert_equal(str, row[1])
		assert_equal(num, row[2])
		assert_equal(date.to_s, row[3].to_s)

		# delete
		expr.statement = "delete from TEST_ADAPTER where TEST_ID = #{pk}"
		@context.begin_transaction
		@channel.evaluate expr
		@context.commit_transaction
	end

	def test_rollback
		@adapter.connection['transaction'] = true
		# @app.log_options[:sql] = true

		pk = 200
		str = "test_#{Time.now}"
		num = rand(10000)
		date = Date.today
		expr = MySQLExpression.new @app.entity('TestAdapter')

		# send SQL statements
		expr.statement = "insert into TEST_ADAPTER (TEST_ID, TEST_STRING, TEST_NUMBER, TEST_DATE) values (#{pk}, '#{str}', #{num}, '#{date}')"
		@context.begin_transaction
		@channel.evaluate(expr)

		# check
		expr.statement = "select TEST_ID, TEST_STRING, TEST_NUMBER, TEST_DATE from TEST_ADAPTER where TEST_ID = #{pk}"
		state = @channel.evaluate expr
		row = state.fetch_all.first
		assert_equal(pk, row[0])
		assert_equal(str, row[1])
		assert_equal(num, row[2])
		assert_equal(date.to_s, row[3].to_s)

		# rollback
		@context.rollback_transaction

		# check again
		state = @channel.evaluate expr
		assert state.fetch_all.empty?
	end

end


if __FILE__ == $0 then
	suite = Test::Unit::TestSuite.new 'TapKit TestSuite'
	suite << TestMySQLAdapter.suite
	suite << TestMySQLAccessing.suite
	runner = Test::Unit::UI::Console::TestRunner.new suite
	runner.start
end
