/* 
 * Copyright 2007 Tatooine Project <http://tatooine.sourceforge.jp/> 
 *  
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *  
 *     http://www.apache.org/licenses/LICENSE-2.0
 *  
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.tatooine.gtx;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import jp.sf.tatooine.base.utils.NameUtils;
import jp.sf.tatooine.gtx.filter.Capitalize;
import jp.sf.tatooine.gtx.filter.Center;
import jp.sf.tatooine.gtx.filter.Chomp;
import jp.sf.tatooine.gtx.filter.Chop;
import jp.sf.tatooine.gtx.filter.Classify;
import jp.sf.tatooine.gtx.filter.Left;
import jp.sf.tatooine.gtx.filter.Lowercase;
import jp.sf.tatooine.gtx.filter.Lpad;
import jp.sf.tatooine.gtx.filter.Mid;
import jp.sf.tatooine.gtx.filter.Right;
import jp.sf.tatooine.gtx.filter.Rpad;
import jp.sf.tatooine.gtx.filter.Substring;
import jp.sf.tatooine.gtx.filter.Swapcase;
import jp.sf.tatooine.gtx.filter.Trim;
import jp.sf.tatooine.gtx.filter.Uncapitalize;
import jp.sf.tatooine.gtx.filter.Underscore;
import jp.sf.tatooine.gtx.filter.Uppercase;

import org.apache.commons.jexl.Expression;
import org.apache.commons.jexl.ExpressionFactory;
import org.apache.commons.jexl.JexlContext;
import org.apache.commons.jexl.JexlHelper;
import org.apache.commons.lang.StringUtils;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import com.sun.org.apache.xerces.internal.parsers.DOMParser;

/**
 * GtxParser.
 *
 * @author  Tooru Noda
 * @version 1.0 2007/04/11
 * @since   JDK5.0 Tiger
 */
public class GtxParser {
	
	/** 名前空間URI. */
	public static final String GTX_NS_URI = "http://tatooine.sourceforge.jp/ns/gtx";
	
	/** EL書式:${el}. */
	private static final Pattern PTN_EL = Pattern.compile("\\$\\{(.*?)\\}");
	
	/** 変数名規約. */
	private static final String REGEXP_IDENTIFIER = "[a-zA-Z_$][a-zA-Z0-9_\\-#$]*";
	
	/** 関数規約. */
	private static final String REGEXP_FUNC = "(" + REGEXP_IDENTIFIER + ")\\s*(\\((.*)\\))?";
	
	/** 関数書式：func_name(args). */
	private static final Pattern PTN_FUNC = Pattern.compile(REGEXP_FUNC);
	
	/** 関数名位置 */
	private static final int POS_IDENTIFIER = 1;
	
	/** パラメータリスト位置 */
	private static final int POS_PARAM_LIST = 3;
	
	/** コンテキスト変数. */
	private GtxContext _context = new GtxContext();
	
	/** フィルタマップ */
	private HashMap<String, Filter> _filterMap = new HashMap<String, Filter>();
	
	/** テンプレートXML. */
	private Document _templateXml = null;
	
	/**
	 * <code>GtxParser</code>を構築する.
	 */
	public GtxParser() {
		initFilterMap();
	}
	
	/* 定義済みフィルタのロード */
	private void initFilterMap() {
		addFilter(new Capitalize());
		addFilter(new Center());
		addFilter(new Chomp());
		addFilter(new Chop());
		addFilter(new Classify());
		addFilter(new Left());
		addFilter(new Lowercase());
		addFilter(new Lpad());
		addFilter(new Mid());
		addFilter(new Right());
		addFilter(new Rpad());
		addFilter(new Substring());
		addFilter(new Swapcase());
		addFilter(new Trim());
		addFilter(new Uncapitalize());
		addFilter(new Underscore());
		addFilter(new Uppercase());
	}
	
	/**
	 * カスタムフィルタをパーサに登録する.
	 * 
	 * @param filter	カスタムフィルタ
	 */
	public final void addFilter(Filter filter) {
		_filterMap.put(NameUtils.underscore(filter.getClass().getSimpleName()), filter);
	}
	
	/**
	 * コンテキストを設定する.
	 * 
	 * @param context	コンテキスト
	 */
	public final void setContext(GtxContext context) {
		_context = context;
	}
	
	/**
	 * コンテキストを取得する.
	 * 
	 * @return	コンテキスト
	 */
	public final GtxContext getContext() {
		return _context;
	}
	
	/**
	 * XMLファイルに記述されたテンプレート命令を処理する.
	 * 
	 * @param stream			入力ストリーム
	 * @throws IOException	入出力エラー発生時に送出する
	 * @throws SAXException	XML構文エラー発生時に送出する
	 * @throws GtxSyntaxException	gtx構文エラー発生時に送出する
	 */
	public final void parse(InputStream stream) 
			throws IOException, SAXException, GtxSyntaxException {
		
		parse(stream, new GtxContext());
	}
	
	/**
	 * XMLファイルに記述されたテンプレート命令を処理する.
	 * 
	 * @param stream			入力ストリーム
	 * @throws IOException	入出力エラー発生時に送出する
	 * @throws SAXException	XML構文エラー発生時に送出する
	 * @throws GtxSyntaxException	gtx構文エラー発生時に送出する
	 */
	public final void parse(InputStream stream, GtxContext context) 
			throws IOException, SAXException, GtxSyntaxException {
		
		DOMParser parser = new DOMParser();
		parser.parse(new InputSource(stream));
		_templateXml = parser.getDocument();
		Node rootNode = _templateXml.getDocumentElement();
		if (_context != null) {
			context.setParent(_context);
		}
		parseNode(context, rootNode);
		
		// TODO: xmlns:gtx="～"を削除したい。名前空間の削除は可能？よくわからん。要調査。
	}
	
	/**
	 * テンプレート実行結果をストリームに出力する.
	 * 
	 * @param out	出力先ストリーム
	 */
	public void print(OutputStream out) {
		
		try {
			TransformerFactory transFactory = TransformerFactory.newInstance();
			Transformer transformer = transFactory.newTransformer();
			StreamResult sr = new StreamResult(out);
			
			DOMSource source = new DOMSource(_templateXml);
			transformer.transform(source, sr);
		}
		catch (TransformerException e) {
			throw new GtxException(e);
		}
	}
	
	/**
	 * XMLノードツリーを再帰的に走査する。
	 * 
	 * @param context	コンテキストパラメータ
	 * @param currentNode	走査対象ノード
	 * @throws GtxSyntaxException	gtx構文エラー発生時に送出する
	 */
	private void parseNode(GtxContext context, Node currentNode) 
			throws GtxSyntaxException {
		
		if (currentNode.getNodeType() == Node.TEXT_NODE) {
			
			/* EL解析 */
			String value = evaluateEL(context, currentNode.getNodeValue());
			currentNode.setNodeValue(value);
		}
		else if (currentNode.getNodeType() == Node.PROCESSING_INSTRUCTION_NODE) {
			System.out.print(currentNode.getNodeValue());
			
			// TODO: gtx-macro解析
		}
		else if (currentNode.getNodeType() == Node.ELEMENT_NODE) {
			parseElementNode(context, currentNode);
		}
	}
	
	/**
	 * 要素ノードを走査する.
	 * 
	 * @param context	コンテキスト
	 * @param currentNode	現在ノード
	 * @throws GtxSyntaxException	構文エラーの時送出される
	 */
	private void parseElementNode(GtxContext context, Node currentNode) 
			throws GtxSyntaxException {
		
		GtxContext nsContext = new GtxContext();
		nsContext.setParent(context);
		NamedNodeMap attrs = currentNode.getAttributes();
		
		Node attr = null;
		
		// TODO: 長くなってきたら分割しよう。
		GTX_ATTR_TEMPLATING: {
			
			/* ignore */
			attr = attrs.getNamedItemNS(GTX_NS_URI, GtxConsts.IGNORE);
			if (attr != null) {
				
				/* EL解析 */
				attr.setNodeValue(evaluateEL(nsContext, attr.getNodeValue()));
				
				if (Boolean.valueOf(attr.getNodeValue())) {
					removeAllGtxAttr(attrs);
					break GTX_ATTR_TEMPLATING;
				}
				else {
					attrs.removeNamedItemNS(GTX_NS_URI, GtxConsts.IGNORE);
				}
			}
			/* if */
			attr = attrs.getNamedItemNS(GTX_NS_URI, GtxConsts.IF);
			if (attr != null) {
				
				/* EL解析 */
				attr.setNodeValue(evaluateEL(nsContext, attr.getNodeValue()));
				
				if (Boolean.valueOf(attr.getNodeValue())) {
					attrs.removeNamedItemNS(GTX_NS_URI, GtxConsts.IF);
				}
				else {
					Node parentNode = currentNode.getParentNode();
					parentNode.removeChild(currentNode);
					break GTX_ATTR_TEMPLATING;
				}
			}
			/* unless */
			attr = attrs.getNamedItemNS(GTX_NS_URI, GtxConsts.UNLESS);
			if (attr != null) {
				
				/* EL解析 */
				attr.setNodeValue(evaluateEL(nsContext, attr.getNodeValue()));
				
				if (Boolean.valueOf(attr.getNodeValue())) {
					Node parentNode = currentNode.getParentNode();
					parentNode.removeChild(currentNode);
					break GTX_ATTR_TEMPLATING;
				}
				else {
					attrs.removeNamedItemNS(GTX_NS_URI, GtxConsts.UNLESS);
				}
			}
			/* inner-text */
			attr = attrs.getNamedItemNS(GTX_NS_URI, GtxConsts.INNER_TEXT);
			if (attr != null) {
				
				/* EL解析 */
				attr.setNodeValue(evaluateEL(nsContext, attr.getNodeValue()));
				
				String innerText = attr.getNodeValue();
				if (innerText != null) {
					NodeList list = currentNode.getChildNodes();
					for (int i = 0; i < list.getLength(); i++) {
						currentNode.removeChild(list.item(i));
					}
					currentNode.appendChild(_templateXml.createTextNode(innerText));
				}
				attrs.removeNamedItemNS(GTX_NS_URI, GtxConsts.INNER_TEXT);
			}
			/* null */
			attr = attrs.getNamedItemNS(GTX_NS_URI, GtxConsts.NULL);
			if (attr != null) {
				
				/* EL解析 */
				attr.setNodeValue(evaluateEL(nsContext, attr.getNodeValue()));
				attrs.removeNamedItemNS(GTX_NS_URI, GtxConsts.NULL);
			}
			/* for */
			attr = attrs.getNamedItemNS(GTX_NS_URI, GtxConsts.FOR);
			if (attr != null) {
				
				boolean inside = false;
				
				/* インデックス変数名取得 */
				Node indexNode = attrs.getNamedItemNS(GTX_NS_URI, GtxConsts.INDEX);
				String index = null;
				if (indexNode != null) {
					indexNode.setNodeValue(evaluateEL(nsContext, indexNode.getNodeValue()));
					index = indexNode.getNodeValue();
					if (index != null) {
						index = index.trim();
						if (!index.matches(REGEXP_IDENTIFIER)) {
							throw new GtxSyntaxException(index);
						}
					}
					attrs.removeNamedItemNS(GTX_NS_URI, GtxConsts.INDEX);
				}
				/* inside命令取得 */
				Node insideNode = attrs.getNamedItemNS(GTX_NS_URI, GtxConsts.INSIDE);
				if (insideNode != null) {
					
					attrs.removeNamedItemNS(GTX_NS_URI, GtxConsts.INSIDE);
					
					/* EL解析 */
					insideNode.setNodeValue(evaluateEL(nsContext, insideNode.getNodeValue()));
					
					inside = Boolean.valueOf(insideNode.getNodeValue());
				}
				/* 走査数取得 */
				attr.setNodeValue(evaluateEL(nsContext, attr.getNodeValue()));
				LoopRange range = LoopRange.parse(attr.getNodeValue());
				
				/* step命令取得 */
				Node stepNode = attrs.getNamedItemNS(GTX_NS_URI, GtxConsts.STEP);
				if (stepNode != null) {
					
					attrs.removeNamedItemNS(GTX_NS_URI, GtxConsts.STEP);
					
					/* EL解析 */
					stepNode.setNodeValue(evaluateEL(nsContext, stepNode.getNodeValue()));
					
					range.setStep(Integer.parseInt(stepNode.getNodeValue()));
				}
				/* 挿入場所取得 */
				Node refNode = currentNode.getNextSibling();
				Node parentNode = currentNode.getParentNode();
				
				attrs.removeNamedItemNS(GTX_NS_URI, GtxConsts.FOR);
				
				/* inside命令あり */
				if (inside) {
					
					/* テンプレート作成 */
					Node templateNode = currentNode.cloneNode(true);
					
					/* 子ノードを削除 */
					NodeList childNodes = currentNode.getChildNodes();
					int removeCnt = childNodes.getLength();
					for (int i = 0; i < removeCnt; i++) {
						currentNode.removeChild(childNodes.item(0));
					}
					/* 指定数分テンプレートを挿入 */
					NodeList newChildNodes = templateNode.getChildNodes();
					for (int i = range.getBegin(); i < range.getEnd(); i = range.getNext(i)) {
						
						if (index != null) {
							nsContext.put(index, i);
						}
						for (int j = 0; j < newChildNodes.getLength(); j++) {
							Node newChildNode = newChildNodes.item(j).cloneNode(true);
							currentNode.appendChild(newChildNode);
							parseNode(nsContext, newChildNode);
						}
					}
				}
				/* inside命令なし */
				else {
					
					/* テンプレート作成 */
					Node templateNode = currentNode.cloneNode(true);
					parentNode.removeChild(currentNode);
					
					/* 指定数分テンプレートを挿入 */
					for (int i = range.getBegin(); i < range.getEnd(); i = range.getNext(i)) {
					
						if (index != null) {
							nsContext.put(index, i);
						}
						Node spacer = refNode.getPreviousSibling();
						Node newNode = templateNode.cloneNode(true);
						parentNode.insertBefore(newNode, refNode);
						
						/* 直前に空白ノードがあれば挿入（整形のため） */
						if (spacer != null && spacer.getNodeType() == Node.TEXT_NODE) {
							if (spacer.getNodeValue().trim().length() == 0) {
								parentNode.insertBefore(spacer.cloneNode(false), refNode);
							}
						}
						parseNode(nsContext, newNode);
					}
				}
				// TODO: DOMの走査が想定通りにいかないのでとりあえず重複コードにした。要調査。
				
				/* 残りの属性値のELを評価 */
				for (int i = 0; i < attrs.getLength(); i++) {
					
					String value = attrs.item(i).getNodeValue();
					
					/* EL解析 */
					attrs.item(i).setNodeValue(evaluateEL(nsContext, value));
				}
				return; /* 子ノードを既に走査しているため */
			}
			/* 残りの属性値のELを評価 */
			for (int i = 0; i < attrs.getLength(); i++) {
				
				String value = attrs.item(i).getNodeValue();
				
				/* EL解析 */
				attrs.item(i).setNodeValue(evaluateEL(nsContext, value));
			}
			
		} /* End of GTX_ATTR_TEMPLATING */
		
		/* 子ノードを走査する */
		NodeList childNodes = currentNode.getChildNodes();
		for (int i = 0; i < childNodes.getLength(); i++) {
			
			parseNode(nsContext, childNodes.item(i));
		}
	}
	
	/* EL式を評価する */
	String evaluateEL(GtxContext context, String text) throws GtxSyntaxException {
		
		StringBuffer buf = new StringBuffer();
		try {
			Matcher m = PTN_EL.matcher(text);
			while (m.find()) {
				String el = m.group(1);
				String[] array = el.split(":");
				String result = null;
				if (array.length != 0) {
					Expression e = ExpressionFactory.createExpression(array[0].trim());
					JexlContext jexlContext = JexlHelper.createContext();
					jexlContext.setVars(context.flatten());
					Object o = e.evaluate(jexlContext);
					if (o == null) {
						result = "null";
					}
					else {
						result = o.toString();
						for (int i = 1; i < array.length; i++) {
							String filter = array[i].trim();
							if (StringUtils.isEmpty(filter)) {
								throw new GtxSyntaxException(text);
							}
							result = evaluateFilter(result, filter);
						}
						Map map = jexlContext.getVars();
						for (Object entry : map.entrySet()) {
							Map.Entry tmp = (Map.Entry) entry;
							context.put((String) tmp.getKey(), tmp.getValue());
						}
					}
				}
				m.appendReplacement(buf, result);
			}
			m.appendTail(buf);
		}
		catch (Exception e) {
			throw new GtxSyntaxException(e);
		}
		return buf.toString();
	}
	
	/* Filterを評価する。 */
	String evaluateFilter(String value, String filter) throws GtxSyntaxException {
		
		// <method>::= <identifier> [ "(" [ <parameter_list> ] ")" ]
		// <identifier> ::= [a-zA-Z_$][a-zA-Z0-9_\-#$]*
		// <parameter_list> ::= <parameter> { "," <parameter> }
		// <parameter> ::= .+
		
		try {
			Matcher m = PTN_FUNC.matcher(filter);
			if (!m.matches()) {
				throw new GtxSyntaxException(filter);
			}
			String id = m.group(POS_IDENTIFIER);
			String[] params = new String[0];
			if (m.group(POS_PARAM_LIST) != null) {
				params = m.group(POS_PARAM_LIST).split(",");
				for (int i = 0; i < params.length; i++) {
					params[i] = params[i].trim();
				}
			}
			if (!_filterMap.containsKey(id)) {
				throw new GtxSyntaxException("Unknown Filter name. : " + id);
			}
			Filter filterObj = _filterMap.get(id);
			return filterObj.invoke(value, params);
		}
		catch (Exception e) {
			throw new GtxSyntaxException(e);
		}
	}

	/* gtx属性をすべて削除する */
	private void removeAllGtxAttr(NamedNodeMap attrs) {
		
		for (String name : GtxConsts.ATTR_NAMES) {
			if (attrs.getNamedItemNS(GTX_NS_URI, name) != null) {
				attrs.removeNamedItemNS(GTX_NS_URI, name);
			}
		}
	}
}
