#ifndef _TMQUATERNION_HPP
#define _TMQUATERNION_HPP

/** 
	@file quaternion.hpp
	@brief Defines tempest::quaternion<T,Sz>.
	@author ototoi / Toru Matsuoka
	@date 2004/05/14 
*/
#include<cstddef>
#include<cmath>
#include<stdexcept>

#include <complex>

#include<ostream>
#include<sstream>

#include"vector.hpp"

namespace tempest{	

/**
	@class quaternion
	@brief Quaternion class template.
	@todo	
	@code
	tempest::quaterion<double> q;	
	@endcode
	@bug	

*/	
template<class T>
class quaternion:public vector_base< quaternion<T>, T, 4>{
public:
	//-----------------------------------------------
    //type defines
	typedef		T										value_type;
	typedef		T&										reference;							  
	typedef		const T&								const_reference;
	typedef		quaternion<T>							this_type;
	
	typedef		T										scalar_type;
	typedef		vector<T,3>								vector_type;
	
	typedef		std::size_t								size_type;
	typedef		std::ptrdiff_t							difference_type;
	
	typedef		T*										pointer;
	typedef		const T*								const_pointer;
	
	typedef		pointer									iterator;
	typedef		const_pointer							const_iterator;
	
#ifdef _TEMPEST_USE_META_
	typedef		typename param_type<T>::type			param_type;
#else
	typedef		T										param_type;
#endif
	
public:
	static const size_type c_size = 4;//container size
	
	//-----------------------------------------------
    //functions for iterator
	iterator		begin()			{return element;}
	iterator		end()			{return element+c_size;}
	const_iterator	begin()	const	{return element;}
	const_iterator	end()	const	{return element+c_size;}
	
	//-----------------------------------------------
    //constructors and destructor
	quaternion(){}
	
	quaternion(param_type a,param_type b,param_type c,param_type d):m0(a),m1(b),m2(c),m3(d){}
	
	quaternion(const quaternion<T>& rhs):m0(rhs.m0),m1(rhs.m1),m2(rhs.m2),m3(rhs.m3){}
	
	template<class X>
    explicit quaternion(const quaternion<X> &rhs):m0(rhs[0]),m1(rhs[1]),m2(rhs[2]),m3(rhs[3]){}
	
	template<class X>
	explicit quaternion(const X* rhs):m0(rhs[0]),m1(rhs[1]),m2(rhs[2]),m3(rhs[3]){}//X const (&rhs)[c_size]
	
	template<class X,class Y>
	quaternion(X s, const vector<Y,3> & v):m0(s),m1(v[0]),m2(v[1]),m3(v[2]){}//scalar , vector
	
	template<class X,class Y>
	quaternion(const vector<X,3> & v, Y theta){//pivot , theta
		T s;
		
		theta /= 2;
		s = std::sin(theta);
		m0 = std::cos(theta);
		m1 = s*v[0];
		m2 = s*v[1];
		m3 = s*v[2];
	}
	
	explicit quaternion(
		std::complex<T> const & z0,
		std::complex<T> const & z1 = std::complex<T>()
	):
		m0(z0.real()),
		m1(z0.imag()),
		m2(z1.real()),
		m3(z1.imag()){} // nothing to do!
	
	
	
	template<class Self,class X>
	explicit quaternion(const vector_base<Self,X,4> & rhs):
		m0(static_cast<T>(static_cast<const Self &>(rhs)[0])),
		m1(static_cast<T>(static_cast<const Self &>(rhs)[1])),
		m2(static_cast<T>(static_cast<const Self &>(rhs)[2])),
		m3(static_cast<T>(static_cast<const Self &>(rhs)[3])){}
	
	//-----------------------------------------------
	//inserters
	this_type& operator= (const this_type &rhs){
		m0 = rhs.m0 ;
    	m1 = rhs.m1 ;
    	m2 = rhs.m2 ;
		m3 = rhs.m3 ;
    	return *this;
	}

    template<class X>
    this_type& operator= (const quaternion<X> &rhs){
		m0 = rhs[0];
		m1 = rhs[1];
		m2 = rhs[2];
		m3 = rhs[3];
    	return *this;
	}
	
	template<class Self,class X>
	this_type& operator= (const vector_base<Self,X,4> & rhs){
		m0 = static_cast<T>(static_cast<const Self &>(rhs)[0]);
    	m1 = static_cast<T>(static_cast<const Self &>(rhs)[1]);
    	m2 = static_cast<T>(static_cast<const Self &>(rhs)[2]);
		m3 = static_cast<T>(static_cast<const Self &>(rhs)[3]);
		return *this;
	}
	
	template<class IT>
	void assign( IT start, IT end ){
		assert(std::difference(start,end)<=c_size);//debug
		
		std::copy(start,end,begin());				
	}
	
	void assign( size_type num, param_type val ){
		std::fill_n(begin(),(num<c_size)?num:c_size,val);		
	}
	
	//-----------------------------------------------
	//capacity
	size_type size ()     const { return c_size; }
    size_type max_size () const { return c_size; }
    bool      empty ()    const { return false;	 }
	
	//-----------------------------------------------
	//operators
	
#define DECLARE_OP_EQUAL(OP)							\
	template<class X>									\
	this_type& operator OP (const quaternion<X> & rhs){	\
		m0 OP rhs[0];									\
		m1 OP rhs[1];									\
		m2 OP rhs[2];									\
		m3 OP rhs[3];									\
		return *this;									\
	}													\
	template<class X>									\
	this_type& operator OP (param_type rhs){			\
		a OP at;										\
		return(*this);									\
	}													\
	template<class X>									\
	this_type& operator OP (const std::complex<X> &rhs){\
		X at = static_cast<X>(m0);						\
		X bt = static_cast<X>(m1);						\
		at OP (rhs.real());								\
		bt OP (rhs.imag());								\
														\
		m0 = static_cast<T>(at);						\
		m1 = static_cast<T>(bt);						\
		return *this;									\
	}
	
//----use
	DECLARE_OP_EQUAL(+=)
	DECLARE_OP_EQUAL(-=)
//
	
#undef DECLARE_OP_EQUAL
	
	template<class X>
	this_type& operator*= (std::complex<X> const & rhs){
		X    ar = rhs.real();
		X    br = rhs.imag();

		T    at = static_cast<T>(+m0*ar-m1*br);
		T    bt = static_cast<T>(+m0*br+m1*ar);
		T    ct = static_cast<T>(+m2*ar+m3*br);
		T    dt = static_cast<T>(-m2*br+m3*ar);

		a = at;
		b = bt;
		c = ct;
		d = dt;

		return(*this);
	}
	
	this_type& operator*= (const this_type & rhs){
		T tmp0 = m0;
		T tmp1 = m1;
		T tmp2 = m2;
		
		m0 = tmp0*rhs[0] - tmp1*rhs[1] - tmp2*rhs[2] - m3*rhs[3];
		m1 = tmp0*rhs[1] + tmp1*rhs[0] + tmp2*rhs[3] - m3*rhs[2];
		m2 = tmp0*rhs[2] - tmp1*rhs[3] + tmp2*rhs[0] + m3*rhs[1];
		m3 = tmp0*rhs[3] + tmp1*rhs[2] - tmp2*rhs[1] + m3*rhs[0];		
		return *this;
	}
		
	template<class X>
	this_type& operator*= (const quaternion<X> & rhs){
		
		T m0 = static_cast<T>(m0*rhs[0] - m1*rhs[1] - mp2*rhs[2] - m3*rhs[3]);
		T m1 = static_cast<T>(m0*rhs[1] + m1*rhs[0] + mp2*rhs[3] - m3*rhs[2]);
		T m2 = static_cast<T>(m0*rhs[2] - m1*rhs[3] + mp2*rhs[0] + m3*rhs[1]);
		T m3 = static_cast<T>(m0*rhs[3] + m1*rhs[2] - mp2*rhs[1] + m3*rhs[0]);

		m0 = at;
		m1 = bt;
		m2 = ct;
		m3 = dt;
		return *this;
	}
	
	template<class X>
	this_type& operator*= (const X & rhs){		
		m0 = static_cast<T>(m0 * rhs);
		m1 = static_cast<T>(m1 * rhs);
		m2 = static_cast<T>(m2 * rhs);
		m3 = static_cast<T>(m3 * rhs);
		return *this;
	}
	
	template<class X>
	this_type& operator/= (const X & rhs){		
		m0 = static_cast<T>(m0 / rhs);
		m1 = static_cast<T>(m1 / rhs);
		m2 = static_cast<T>(m2 / rhs);
		m3 = static_cast<T>(m3 / rhs);
		return *this;
	}
	
	this_type& operator*= (param_type rhs){		
		m0 *= rhs;
		m1 *= rhs;
		m2 *= rhs;
		m3 *= rhs;
		return *this;
	}
	
	this_type& operator/= (param_type rhs){		
		m0 /= rhs;
		m1 /= rhs;
		m2 /= rhs;
		m3 /= rhs;
		return *this;
	}
	
	
	//-----------------------------------------------
	T & operator[](size_type i){
    	return element[i];
	}
	
	param_type operator[](size_type i) const {
    	return element[i];
	}
	
	reference at(size_type i){
		if(c_size<=i){throw std::out_of_range("tempest::quaternion");}
		return element[i];
	}
	const_reference at(size_type i) const {
		if(c_size<=i){throw std::out_of_range("tempest::quaternion");}
		return element[i];
	}
	
	//-----------------------------------------------
	//utilities
	
	T sqr_length()const{
		return m0*m0 + m1*m1 + m2*m2 + m3*m3;
	}
	T length()const{
		T temp = sqr_length();
		return std::sqrt(temp);
	}
	T dot(const quaternion<T>& rhs)const{
		return m0*rhs[0] + m1*rhs[1] + m2*rhs[2] + m3*rhs[3];		
	}
	
	quaternion<T>& normalize(){
		T length = sqr_length(); 		//||q||^2
    	if (length == T()) return *this;

		length = RSQRT(length);	// 1 / ||q||
		//length = rsqrt(length);
    	m0 *= length;
		m1 *= length;
		m2 *= length;
		m3 *= length;
		
		return *this;
	}
private:
	
	union{
		struct{
			T m0,m1,m2,m3;
		};
		T element[4];
	};

};

/**
	@relates quaternion
 */
//@{
	
//-----------------------------------------------
//binary operators
template<class T>
inline quaternion<T> operator- (const quaternion<T> & rhs){
	return quaternion<T>(rhs) *= -1;
}
	
template<class T> 
inline quaternion<T> operator+ (const quaternion<T> &lhs, const quaternion<T> &rhs){
	return quaternion<T>(lhs) += rhs;
}
template<class T> 
inline quaternion<T> operator- (const quaternion<T> &lhs, const quaternion<T> &rhs){
	return quaternion<T>(lhs) -= rhs;
}


template<class T, class X> 
inline quaternion<T> operator* (const X lhs,const quaternion<T> & rhs){ 
	return quaternion<T>(rhs) *= lhs ; 
}

template<class T, class X> 
inline quaternion<T> operator* (const quaternion<T> &lhs,const X rhs){ 
	return quaternion<T>(lhs) *= rhs ;
}	
	
template<class T, class X> 
inline quaternion<T> operator/ (const quaternion<T> &lhs,const X rhs){ 
	return quaternion<T>(lhs) /= rhs ;
}
	
template<class T>
inline  quaternion<T> operator* (const quaternion<T>& lhs,const quaternion<T>& rhs){
	return quaternion<T>(
		lhs[0]*rhs[0] - lhs[1]*rhs[1] - lhs[2]*rhs[2] - lhs[3]*rhs[3],
		lhs[0]*rhs[1] + lhs[1]*rhs[0] + lhs[2]*rhs[3] - lhs[3]*rhs[2],
		lhs[0]*rhs[2] - lhs[1]*rhs[3] + lhs[2]*rhs[0] + lhs[3]*rhs[1],
		lhs[0]*rhs[3] + lhs[1]*rhs[2] - lhs[2]*rhs[1] + lhs[3]*rhs[0]
	);
}
	

	
//-----------------------------------------------	
// utility functions

template<class T,std::size_t Sz>
inline quaternion<T> normalize(const quaternion<T> &rhs){
    return quaternion<T>(rhs).normalize();
}

template<class T,std::size_t Sz>
inline T length(const quaternion<T> &rhs){
	return rhs.length();
}

template<class T,std::size_t Sz>
inline T sqr_length(const quaternion<T> &rhs){
	return rhs.sqr_length();
}
template<class T>
inline T dot(const quaternion<T>& lhs,const quaternion<T>& rhs){
	return lhs.dot(rhs);		
}

template<class T> 
inline  quaternion<T> operator~(const quaternion<T>& rhs){
	T l = rhs.sqr_length();//lq|*|q|
	if (l == T()) return rhs;
	l = T(1)/l;
	
	return quaternion<T>(rhs[0]*l,-rhs[1]*l,-rhs[2]*l,-rhs[3]*l); 
}
	
template<class T,class X>
inline quaternion<T> lerp(const quaternion<T> &lhs,const quaternion<T> &rhs, X t){
	return ((1-t)*lhs + t*rhs).normalize();
}
	
template<class T,class X>
inline quaternion<T> slerp(const quaternion<T> &lhs,const quaternion<T> &rhs, X t){
	
	X theta = std::acos( dot(lhs,rhs) );//
	T s = T(1)/std::sin(theta);
	
	return (std::sin((1-t)*theta)*lhs + std::sin(t*theta)*rhs)*s;		
}

template<typename T, typename _CharT, class _Traits>
std::basic_ostream<_CharT, _Traits>& operator<<(std::basic_ostream<_CharT, _Traits>& os, const quaternion<T>& rhs){
	
	std::basic_ostringstream<_CharT, _Traits> s;
	s.flags(os.flags());
	s.imbue(os.getloc());
	s.precision(os.precision());
	s << "(" << rhs[0] << "," << rhs[1] << "," << rhs[2] << ")";
	
	return os << s.str();
}
	
//@}
	
	
//-----------------------------------------------
	template<>    class quaternion<float>;
	template<>    class quaternion<double>;
	template<>    class quaternion<long double>;

}//

#endif

