/* eel-gdk-pixbuf-extensions.c: Routines to augment what's in gdk-pixbuf.

   Copyright (C) 2000 Eazel, Inc.

   The Gnome Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The Gnome Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the Gnome Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Authors: Darin Adler <darin@eazel.com>
            Ramiro Estrugo <ramiro@eazel.com>
*/

/* ********************************************** gdk-pixbuf-extension.c *** *
 * gdk-pixbufγĥؿ
 *
 * Copyright (C) 2001 Yasuyuki SUGAYA <sugaya@suri.it.okayama-u.ac.jp
 * Okayama University
 *                                  Time-stamp: <2001-11-06 18:44:22 sugaya>
 * ************************************************************************* */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <png.h>
#include "gdk-pixbuf-extension.h"
#include "ps.h"
#include "wmf.h"

/* ************************************************************************* */
void
free_buffer (guchar	*pixels,
	     gpointer	data) {
  free (pixels);
}

/* GdkPixbufǡpng¸ؿ ********************************** */
gboolean
gdk_pixbuf_save_to_file (const GdkPixbuf	*pixbuf,
			 const char		*file_name) {
  FILE *handle;
  char *buffer;
  gboolean has_alpha;
  int width, height, depth, rowstride;
  guchar *pixels;
  png_structp png_ptr;
  png_infop info_ptr;
  png_text text[2];
  int i;

  g_return_val_if_fail (pixbuf != NULL, FALSE);
  g_return_val_if_fail (file_name != NULL, FALSE);
  g_return_val_if_fail (file_name[0] != '\0', FALSE);

  handle = fopen (file_name, "wb");
  if (handle == NULL) {
    return FALSE;
  }

  png_ptr = png_create_write_struct (PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
  if (png_ptr == NULL) {
    fclose (handle);
    return FALSE;
  }

  info_ptr = png_create_info_struct (png_ptr);
  if (info_ptr == NULL) {
    png_destroy_write_struct (&png_ptr, (png_infopp)NULL);
    fclose (handle);
    return FALSE;
  }

  if (setjmp (png_ptr->jmpbuf)) {
    png_destroy_write_struct (&png_ptr, &info_ptr);
    fclose (handle);
    return FALSE;
  }

  png_init_io (png_ptr, handle);

  has_alpha = gdk_pixbuf_get_has_alpha (pixbuf);
  width = gdk_pixbuf_get_width (pixbuf);
  height = gdk_pixbuf_get_height (pixbuf);
  depth = gdk_pixbuf_get_bits_per_sample (pixbuf);
  pixels = gdk_pixbuf_get_pixels (pixbuf);
  rowstride = gdk_pixbuf_get_rowstride (pixbuf);

  png_set_IHDR (png_ptr, info_ptr, width, height,
		depth, PNG_COLOR_TYPE_RGB_ALPHA,
		PNG_INTERLACE_NONE,
		PNG_COMPRESSION_TYPE_DEFAULT,
		PNG_FILTER_TYPE_DEFAULT);

  /* Some text to go with the png image */
  text[0].key = "Title";
  text[0].text = (char *) file_name;
  text[0].compression = PNG_TEXT_COMPRESSION_NONE;
  text[1].key = "Software";
  text[1].text = "Eel Thumbnail";
  text[1].compression = PNG_TEXT_COMPRESSION_NONE;
  png_set_text (png_ptr, info_ptr, text, 2);

  /* Write header data */
  png_write_info (png_ptr, info_ptr);

  /* if there is no alpha in the data, allocate buffer to expand into */
  if (has_alpha) {
    buffer = NULL;
  } else {
    buffer = g_malloc(4 * width);
  }
	
  /* pump the raster data into libpng, one scan line at a time */	
  for (i = 0; i < height; i++) {
    if (has_alpha) {
      png_bytep row_pointer = pixels;
      png_write_row (png_ptr, row_pointer);
    } else {
      /* expand RGB to RGBA using an opaque alpha value */
      int x;
      char *buffer_ptr = buffer;
      char *source_ptr = pixels;
      for (x = 0; x < width; x++) {
	*buffer_ptr++ = *source_ptr++;
	*buffer_ptr++ = *source_ptr++;
	*buffer_ptr++ = *source_ptr++;
	*buffer_ptr++ = 255;
      }
      png_write_row (png_ptr, (png_bytep) buffer);		
    }
    pixels += rowstride;
  }
	
  png_write_end (png_ptr, info_ptr);
  png_destroy_write_struct (&png_ptr, &info_ptr);
	
  g_free (buffer);
		
  fclose (handle);
  return TRUE;
}

/* ץ졼ĥǡǡǡ ********** */
GdkPixbuf*
gdk_pixbuf_remove_alpha (const GdkPixbuf	*src) {
  GdkPixbuf	*dst;
  unsigned char	*data, *src_ptr, *dst_ptr;
  int		size, n;

  size = gdk_pixbuf_get_width (src) * gdk_pixbuf_get_height(src) * 3;
  src_ptr = gdk_pixbuf_get_pixels (src);
  dst_ptr = data = (unsigned char *)
    malloc (sizeof (unsigned char) * size);
			
  for (n = size/3; n > 0; n--) {
    *dst_ptr++ = *src_ptr++;
    *dst_ptr++ = *src_ptr++;
    *dst_ptr++ = *src_ptr++;
    src_ptr++;
  }

  dst = gdk_pixbuf_new_from_data (data, GDK_COLORSPACE_RGB, FALSE, 8,
				  gdk_pixbuf_get_width (src),
				  gdk_pixbuf_get_height(src),
				  gdk_pixbuf_get_width (src) * 3,
				  free_buffer, NULL);
  return dst;
}

/* ************************************************************************* */
void
gdk_pixbuf_calc_map_tables (GdkPixbufModifier		*mod) {
  int			i;
  double		g, b, c, ii, v;

  g_return_if_fail (mod  != NULL);

  /* If we are using the defaults, there is no need to get a map */
  if ((mod->mod.gamma == 256) && (mod->mod.brightness == 256) &&
      (mod->mod.contrast == 256) &&
      (mod->rmod.gamma == 256) && (mod->rmod.brightness == 256) &&
      (mod->rmod.contrast == 256) &&
      (mod->gmod.gamma == 256) && (mod->gmod.brightness == 256) &&
      (mod->gmod.contrast == 256) &&
      (mod->bmod.gamma == 256) && (mod->bmod.brightness == 256) &&
      (mod->bmod.contrast == 256)) {
    if (mod->map) {
      free (mod->map);
      mod->map = NULL;
    }
    return;
  }
  if (!mod->map) {
    mod->map = (GdkPixbufModifierMap *) malloc (sizeof (GdkPixbufModifierMap));
    if (!mod->map) return;
  }
  g = ((double) mod->mod.gamma) / 256;
  b = ((double) mod->mod.brightness) / 256;
  c = ((double) mod->mod.contrast) / 256;

  if (g < 0.01) g = 0.01;

  for (i = 0; i < 256; i++) {
    ii = ((double) i) / 256;
    v = ((ii - 0.5) * c) + 0.5 + (b - 1);
    if (v > 0) {
      v = pow (((ii - 0.5) * c) + 0.5 + (b - 1), 1 / g) * 256;
    } else {
      v = 0;
    }
    if (v > 255) {
      v = 255;
    } else if (v < 0) {
      v = 0;
    }
    mod->map->rmap[i] = (unsigned char) v;
    mod->map->gmap[i] = (unsigned char) v;
    mod->map->bmap[i] = (unsigned char) v;
  }
  g = ((double) mod->rmod.gamma) / 256;
  b = ((double) mod->rmod.brightness) / 256;
  c = ((double) mod->rmod.contrast) / 256;

  if (g < 0.01) g = 0.01;

  for (i = 0; i < 256; i++) {
    ii = ((double) mod->map->rmap[i]) / 256;
    v = ((ii - 0.5) * c) + 0.5 + (b - 1);
    if (v > 0) {
      v = pow (((ii - 0.5) * c) + 0.5 + (b - 1), 1 / g) * 256;
    } else {
      v = 0;
    }
    if (v > 255) {
      v = 255;
    } else if (v < 0) {
      v = 0;
    }
    mod->map->rmap[i] = (unsigned char) v;
  }
  g = ((double) mod->gmod.gamma) / 256;
  b = ((double) mod->gmod.brightness) / 256;
  c = ((double) mod->gmod.contrast) / 256;

  if (g < 0.01) g = 0.01;

  for (i = 0; i < 256; i++) {
    ii = ((double) mod->map->gmap[i]) / 256;
    v = ((ii - 0.5) * c) + 0.5 + (b - 1);
    if (v > 0) {
      v = pow (((ii - 0.5) * c) + 0.5 + (b - 1), 1 / g) * 256;
    } else {
      v = 0;
    }
    if (v > 255) {
      v = 255;
    } else if (v < 0) {
      v = 0;
    }
    mod->map->gmap[i] = (unsigned char) v;
  }
  g = ((double) mod->bmod.gamma) / 256;
  b = ((double) mod->bmod.brightness) / 256;
  c = ((double) mod->bmod.contrast) / 256;

  if (g < 0.01) g = 0.01;
  for (i = 0; i < 256; i++) {
    ii = ((double) mod->map->bmap[i]) / 256;
    v = ((ii - 0.5) * c) + 0.5 + (b - 1);
    if (v > 0) {
      v = pow (((ii - 0.5) * c) + 0.5 + (b - 1), 1 / g) * 256;
    } else {
      v = 0;
    }
    if (v > 255) {
      v = 255;
    } else if (v < 0) {
      v = 0;
    }
    mod->map->bmap[i] = (unsigned char) v;
  }
}

/* ************************************************************************* */
void
gdk_pixbuf_set_image_modifier (GdkPixbufColorModifier	*cmod,
			       GdkPixbufModifier	*mod) {
  g_return_if_fail (cmod != NULL);
  g_return_if_fail (mod  != NULL);

  mod->mod.gamma	= cmod->gamma;
  mod->mod.brightness	= cmod->brightness;
  mod->mod.contrast	= cmod->contrast;

  gdk_pixbuf_calc_map_tables (mod);
}

/* ************************************************************************* */
void
gdk_pixbuf_set_image_red_modifier (GdkPixbufColorModifier	*cmod,
				   GdkPixbufModifier		*mod) {
  g_return_if_fail (cmod != NULL);
  g_return_if_fail (mod  != NULL);

  mod->rmod.gamma	= cmod->gamma;
  mod->rmod.brightness	= cmod->brightness;
  mod->rmod.contrast	= cmod->contrast;

  gdk_pixbuf_calc_map_tables (mod);
}

/* ************************************************************************* */
void
gdk_pixbuf_set_image_green_modifier (GdkPixbufColorModifier	*cmod,
				     GdkPixbufModifier		*mod) {
  g_return_if_fail (cmod != NULL);
  g_return_if_fail (mod  != NULL);

  mod->gmod.gamma	= cmod->gamma;
  mod->gmod.brightness	= cmod->brightness;
  mod->gmod.contrast	= cmod->contrast;

  gdk_pixbuf_calc_map_tables (mod);
}

/* ************************************************************************* */
void
gdk_pixbuf_set_image_blue_modifier (GdkPixbufColorModifier	*cmod,
				    GdkPixbufModifier		*mod) {
  g_return_if_fail (cmod != NULL);
  g_return_if_fail (mod  != NULL);

  mod->bmod.gamma	= cmod->gamma;
  mod->bmod.brightness	= cmod->brightness;
  mod->bmod.contrast	= cmod->contrast;

  gdk_pixbuf_calc_map_tables (mod);
}

/* ************************************************************************* */
void
gdk_pixbuf_get_image_modifier (GdkPixbufColorModifier	*src_mod,
			       GdkPixbufColorModifier	*dst_mod) {
  g_return_if_fail (src_mod != NULL);
  g_return_if_fail (dst_mod != NULL);

  dst_mod->gamma	= src_mod->gamma;
  dst_mod->brightness	= src_mod->brightness;
  dst_mod->contrast	= src_mod->contrast;
}

/* ************************************************************************* */
void
gdk_pixbuf_get_image_red_curve (GdkPixbufModifierMap	*src_map,
				unsigned char 		*mod) {
  int			i;

  g_return_if_fail (mod != NULL);

  if (src_map) {
    for (i = 0; i < 256; i++) mod[i] = src_map->rmap[i];
  } else {
    for (i = 0; i < 256; i++) mod[i] = i;
  }
}

/* ************************************************************************* */
void
gdk_pixbuf_get_image_green_curve (GdkPixbufModifierMap	*src_map,
				  unsigned char		*mod) {
  int			i;

  g_return_if_fail (mod != NULL);

  if (src_map) {
    for (i = 0; i < 256; i++) mod[i] = src_map->gmap[i];
  } else {
    for (i = 0; i < 256; i++) mod[i] = i;
  }
}

/* ************************************************************************* */
void
gdk_pixbuf_get_image_blue_curve (GdkPixbufModifierMap	*src_map,
				 unsigned char 		*mod) {
  int			i;

  g_return_if_fail (mod != NULL);

  if (src_map) {
    for (i = 0; i < 256; i++) mod[i] = src_map->bmap[i];
  } else {
    for (i = 0; i < 256; i++) mod[i] = i;
  }
}

/* ************************************************************************* */
void
gdk_pixbuf_apply_modifiers_to_rgb (GdkPixbuf			*pbuf,
				   GdkPixbufModifier		*mod,
				   int				clear_flg) {
  int			x, y;
  int			has_alpha;
  unsigned char		*ptr;
  
  g_return_if_fail (pbuf != NULL);
  g_return_if_fail (mod  != NULL);
  
  ptr = gdk_pixbuf_get_pixels (pbuf);
  has_alpha = gdk_pixbuf_get_has_alpha (pbuf);
  
  if (mod->map) {
    for (y = 0; y < gdk_pixbuf_get_height (pbuf); y++) {
      for (x = 0; x < gdk_pixbuf_get_width (pbuf); x++) {
	*ptr = mod->map->rmap[*ptr];
	ptr++;
	*ptr = mod->map->gmap[*ptr];
	ptr++;
	*ptr = mod->map->bmap[*ptr];
	ptr++;
	if (has_alpha) ptr++;
      }
    }
  }
  if (clear_flg) {
    mod->mod.gamma	= 256;
    mod->mod.brightness	= 256;
    mod->mod.contrast	= 256;
    mod->rmod.gamma	= 256;
    mod->rmod.brightness= 256;
    mod->rmod.contrast	= 256;
    mod->gmod.gamma	= 256;
    mod->gmod.brightness= 256;
    mod->gmod.contrast	= 256;
    mod->bmod.gamma	= 256;
    mod->bmod.brightness= 256;
    mod->bmod.contrast	= 256;

    gdk_pixbuf_calc_map_tables (mod);
  }
}

/* ************************************************************************* */
GdkPixbuf*
gdk_pixbuf_align_rowstride (GdkPixbuf	*src) {
  unsigned char	*src_data, *dst_data;
  int		row, col, w, h, p, plane;
  int		src_rowstride, dst_rowstride;

  w     = gdk_pixbuf_get_width (src);
  h     = gdk_pixbuf_get_height(src);
  plane = 3 + gdk_pixbuf_get_has_alpha (src);
  src_rowstride = gdk_pixbuf_get_rowstride (src);
  dst_rowstride = w * plane;

  src_data = gdk_pixbuf_get_pixels (src);
  dst_data = (unsigned char *)
    malloc (sizeof (unsigned char) * (w * h * plane));

  for (row = 0; row < h; row++) {
    for (col = 0; col < w; col++) {
      for (p = 0; p < plane; p++) {
	dst_data[dst_rowstride * row + plane * col + p]
	  = src_data[src_rowstride * row + plane * col + p];
      }
    }
  }
  if (plane == 3) {
    return gdk_pixbuf_new_from_data (dst_data, GDK_COLORSPACE_RGB, FALSE, 8,
				     w, h, dst_rowstride, free_buffer, NULL);
  } else {
    return gdk_pixbuf_new_from_data (dst_data, GDK_COLORSPACE_RGB, TRUE, 8,
				     w, h, dst_rowstride, free_buffer, NULL);
  }
}

/* ************************************************************************* */
GdkPixbuf*
gdk_pixbuf_new_from_file_extension (char	*filename) {
  GdkPixbuf	*pbuf;
  char		*ext;

  /* ĥҤγ */
  ext = (char *) strrchr (filename, '.');
  if (!ext) return NULL;

  if (strcmp (ext, ".eps") == 0) {
    GdkPixbuf	**tmp;
    int		n, pages;

    tmp = load_ps_image (filename, &pages);
    pbuf = gdk_pixbuf_copy (tmp[0]);
    for (n = 0; n < pages; n++) gdk_pixbuf_unref (tmp[n]);
    free (tmp);

    return pbuf;
  } else if (strcmp (ext, ".wmf") == 0) {
    return load_wmf_image (filename);
  } else {
    return gdk_pixbuf_new_from_file (filename);
  }
}

/* *************************************** End of gdk-pixbuf-extension.c *** */
