# vim: tabstop=4 shiftwidth=4 softtabstop=4
# -*- coding: utf-8 -*-
#

import re
from linkpair.device import Device
from linkpair.port import Port
from linkpair.collect.agent.base_agent import BaseAgent


class OVSAgent(BaseAgent):

    '''
    Open vSwitch Collector Agent
    '''

    def run(self):
        self.get_configuration()

    def get_bridge_metadata(self, br):
        br_meta = {}
        br_meta["dpid"] = ""
        br_meta["controller"] = ""
        br_meta["manager"] = ""

        res_mgr = self._runner.exec_cmd("ovs-vsctl get-manager")
        for mgr in res_mgr:
            br_meta["manager"] = self._u.str_join(
                br_meta["manager"], mgr.strip(), ", ")

        res_ctl = self._runner.exec_cmd("ovs-vsctl get-controller " + br)
        for ctl in res_ctl:
            br_meta["controller"] = self._u.str_join(
                br_meta["controller"], ctl.strip(), ",")

        res_ofsw = self._runner.exec_cmd("ovs-ofctl show " + br)
        for ofsw_spec in res_ofsw:
            ofsw_spec = ofsw_spec.strip()
            if self._u.d_push(re.search(r': dpid:(.*)', ofsw_spec)) is not None:
                br_meta["dpid"] = self._u.d_pop().group(1).strip()
            elif self._u.d_push(re.search(r'n_tables:(.*), n_buffers:(.*)', ofsw_spec)) is not None:
                match = self._u.d_pop()
                br_meta["tables"] = match.group(1).strip()
                br_meta["buffers"] = match.group(2).strip()
            elif self._u.d_push(re.search(r'capabilities:(.*)', ofsw_spec)) is not None:
                br_meta["capabilities"] = self._u.d_pop().group(1).strip()
            elif self._u.d_push(re.search(r'actions:(.*)', ofsw_spec)) is not None:
                br_meta["actions"] = self._u.d_pop().group(1).strip()

        return br_meta

    def get_port_metadata(self, port_name, br_name=""):
        port_meta = {}
        exist_port = self._cu.get_port(port_name)
        if exist_port != self._cu.PORT_NOT_FOUND:
            port_meta = exist_port.metadata

        res_ovsport = self._runner.exec_cmd("ovs-vsctl show")
        for i in range(0, len(res_ovsport)):
            port_start = res_ovsport[i].strip()
            if self._u.d_push(re.search(r'Port "?%s"?' % port_name, port_start)) is not None:
                for j in range(i + 1, len(res_ovsport)):
                    port_spec = res_ovsport[j].strip()
                    if self._u.d_push(re.search(r'tag: (.*)', port_spec)) is not None:
                        port_meta["tag"] = self._u.d_pop().group(1).strip()
                    elif self._u.d_push(re.search(r'type: (.*)', port_spec)) is not None:
                        port_meta["type"] = self._u.d_pop().group(1).strip()
                    elif self._u.d_push(re.search(r'options: {(.*)}', port_spec)) is not None:
                        port_options = self._u.d_pop().group(
                            1).strip().split(",")
                        for port_option in port_options:
                            (opt_name, opt_value) = port_option.split("=")
                            opt_name = opt_name.strip()
                            opt_value = opt_value.replace("\"", " ")
                            opt_value = opt_value.strip()
                            port_meta[opt_name] = opt_value
                            if opt_name == "remote_ip":
                                res_iproute = self._runner.exec_cmd(
                                    "ip route get " + opt_value)
                                for i in range(0, len(res_iproute)):
                                    remote_route = res_iproute[i].strip()
                                    if self._u.d_push(re.search(r'via (\S+) dev (\S+)  src (\S+)', remote_route)) is not None:
                                        match = self._u.d_pop()
                                        port_meta["remote_routing_if_ipaddr"] = match.group(
                                            1).strip()
                                        port_meta[
                                            "remote_routing_if_dev"] = match.group(2).strip()
                                        port_meta["remote_routing_src_ipaddr"] = match.group(
                                            3).strip()
                                    elif self._u.d_push(re.search(r'cache  mtu (\d+) advmss (\d+) hoplimit (\d+)', remote_route)) is not None:
                                        match = self._u.d_pop()
                                        port_meta["remote_peer_mtu"] = match.group(
                                            1).strip()
                                        port_meta[
                                            "remote_peer_mss"] = match.group(2).strip()
                                        port_meta[
                                            "remote_peer_hoplimit"] = match.group(3).strip()

                    elif self._u.d_push(re.search(r'Port "*', port_spec)) is not None:
                        break

        if br_name != "":
            res_ofctl = self._runner.exec_cmd("ovs-ofctl show " + br_name)
            for i in range(0, len(res_ofctl)):
                port_start = res_ofctl[i].strip()
                if self._u.d_push(re.search(r'(\d+)\(%s\): addr:(.*)' % port_name, port_start)) is not None:
                    match = self._u.d_pop()
                    port_meta["port_no"] = match.group(1).strip()
                    port_meta["mac_addr"] = match.group(2).strip()
                    break

            res_ofctl = self._runner.exec_cmd(
                "ovs-ofctl dump-ports " + br_name)
            if "port_no" in port_meta:
                found_port = False
                for i in range(0, len(res_ofctl)):
                    port_start = res_ofctl[i].strip()
                    if self._u.d_push(
                        re.search(r'port\s*%s:\s*rx pkts=(\d+), bytes=(\d+), drop=(\d+), errs=(\d+), frame=(\d+), over=(\d+), crc=(\d+)'
                                  % port_meta["port_no"], port_start)) is not None:
                        match = self._u.d_pop()
                        port_meta["rx_pkts"] = match.group(1).strip()
                        port_meta["rx_bytes"] = match.group(2).strip()
                        port_meta["rx_drop"] = match.group(3).strip()
                        port_meta["rx_errs"] = match.group(4).strip()
                        port_meta["rx_frame"] = match.group(5).strip()
                        port_meta["rx_over"] = match.group(6).strip()
                        port_meta["rx_crc"] = match.group(7).strip()
                        found_port = True
                    elif self._u.d_push(
                        re.search(
                            r'\s*tx pkts=(\d+), bytes=(\d+), drop=(\d+), errs=(\d+), coll=(\d+)',
                            port_start)) is not None and found_port:
                        match = self._u.d_pop()
                        port_meta["tx_pkts"] = match.group(1).strip()
                        port_meta["tx_bytes"] = match.group(2).strip()
                        port_meta["tx_drop"] = match.group(3).strip()
                        port_meta["tx_errs"] = match.group(4).strip()
                        port_meta["tx_coll"] = match.group(5).strip()
                        break

        return port_meta

    def get_configuration(self):
        patch_peers = {}
        veth_peers = {}

        result = self._runner.exec_cmd("ovs-vsctl list-br")
        for br_src in result:
            br_src = br_src.rstrip()
            br_src_meta = self.get_bridge_metadata(br_src)
            port_names = self._runner.exec_cmd(
                "ovs-vsctl list-ports " + br_src)
            for port_name in port_names:
                port_name = port_name.rstrip()
                port_meta = self.get_port_metadata(port_name, br_src)

                self._port_to_br[port_name] = br_src
                # patch port
                if self._cu.get_port_type(port_meta) == "patch":
                    patch_src = port_name
                    patch_dst = self._cu.get_port_peer(port_meta)
                    patch_peers[patch_src + ":" + patch_dst] = self.PEER_FOUND

                    if patch_dst + ":" + patch_src in patch_peers:
                        continue

                    patch_src_meta = port_meta
                    result3 = self._runner.exec_cmd(
                        "ovs-vsctl port-to-br " + patch_dst)
                    if result3 is not None and len(result3) > 0:
                        br_dst = result3[0].rstrip()
                        br_dst_meta = self.get_bridge_metadata(br_dst)
                        patch_dst_meta = self.get_port_metadata(
                            patch_dst, br_dst)
                        self._cu.add_linkpair(
                            Device(br_src, Device.BR_TYPE, br_src_meta),
                            Device(br_dst, Device.BR_TYPE, br_dst_meta),
                            Port(patch_src, Port.DEFAULT_TYPE, patch_src_meta),
                            Port(patch_dst, Port.DEFAULT_TYPE, patch_dst_meta),
                            self._formatter.PATCH_FORMAT)
                    else:
                        patch_dst_meta = self.get_port_metadata(
                            patch_dst, br_dst)
                        self._cu.add_linkpair(
                            Device(br_src, Device.BR_TYPE, br_src_meta),
                            Device("NOT CONNECTED", Device.NOT_CONNECTED_TYPE),
                            Port(patch_src, Port.DEFAULT_TYPE, patch_src_meta),
                            Port(patch_dst, Port.DEFAULT_TYPE, patch_dst_meta),
                            self._formatter.PATCH_FORMAT)

                # veth port
                elif self._cu.get_port_veth_peer(port_name) != self._cu.PORT_PEER_UNSPECIFIED:
                    peer_src = port_name
                    peer_dst = self._cu.get_port_veth_peer(port_name)
                    veth_peers[peer_src + ":" + peer_dst] = self.PEER_FOUND

                    if peer_dst + ":" + peer_src in veth_peers:
                        continue

                    peer_src_meta = port_meta
                    result3 = self._runner.exec_cmd(
                        "ovs-vsctl port-to-br " + peer_dst)
                    if result3 is not None and len(result3) > 0:
                        br_dst = result3[0].rstrip()
                        br_dst_meta = self.get_bridge_metadata(br_dst)
                        peer_dst_meta = self.get_port_metadata(
                            peer_dst, br_dst)
                        self._cu.add_linkpair(
                            Device(br_src, Device.BR_TYPE, br_src_meta),
                            Device(br_dst, Device.BR_TYPE, br_dst_meta),
                            Port(peer_src, Port.DEFAULT_TYPE, peer_src_meta),
                            Port(peer_dst, Port.DEFAULT_TYPE, peer_dst_meta),
                            self._formatter.VETH_FORMAT)
                    else:
                        peer_dst_meta = self.get_port_metadata(
                            peer_dst, br_dst)
                        self._cu.add_linkpair(
                            Device(br_src, Device.BR_TYPE, br_src_meta),
                            Device("NOT CONNECTED", Device.NOT_CONNECTED_TYPE),
                            Port(peer_src, Port.DEFAULT_TYPE, peer_src_meta),
                            Port(peer_dst, Port.DEFAULT_TYPE, peer_dst_meta),
                            self._formatter.VETH_FORMAT)

                # none patch/veth port
                else:
                    # NOT Internal Bridge Port.
                    if self._cu.get_port_type(port_meta) != "internal":

                        # physical interface
                        if self._u.d_push(re.match(r'^eth\d+$', port_name)) \
                            or self._u.d_push(re.match(r'^em\d+$', port_name)) \
                                or self._u.d_push(re.match(r'^igb\d+$', port_name)) \
                                or self._u.d_push(re.match(r'^bond\d+$', port_name)):
                            self._cu.add_linkpair(
                                Device(br_src, Device.BR_TYPE, br_src_meta),
                                Device("Physical NW", Device.PHYNET_TYPE),
                                Port(port_name, Port.DEFAULT_TYPE, port_meta),
                                Port(""))

                        # tunnel port
                        elif self._u.d_push(re.match(r'(vxlan\d+)', port_name)) \
                            or self._cu.get_port_type(port_meta) == "gre" \
                                or self._cu.get_port_type(port_meta) == "stt":
                            self._cu.add_linkpair(
                                Device(br_src, Device.BR_TYPE, br_src_meta),
                                Device("OS Routing", Device.OS_ROUTE_TYPE),
                                Port(port_name, Port.DEFAULT_TYPE, port_meta),
                                Port(""),
                                self._formatter.OS_ROUTING_FORMAT)

                        # Other OVSPort
                        else:
                            # VLAN interface
                            if self._cu.get_port_vlan_master_if(port_name) != self._cu.PORT_VLAN_UNSPECIFIED:
                                master_if = self._cu.get_port_vlan_master_if(
                                    port_name)
                                if master_if in self._port_to_br:
                                    br_dst = self._port_to_br[master_if]
                                    br_dst_obj = self._cu.get_device(br_dst)
                                    port_src = port_name
                                    port_src_meta = port_meta
                                    port_dst = master_if
                                    #port_dst_obj = self._cu.get_port(master_if)
                                    port_dst_meta = self.get_port_metadata(
                                        port_dst, br_dst)
                                    self._cu.add_linkpair(
                                        Device(
                                            br_src, Device.BR_TYPE, br_src_meta),
                                        br_dst_obj,
                                        Port(
                                            port_src, Port.DEFAULT_TYPE, port_src_meta),
                                        # port_dst_obj,
                                        Port(
                                            port_dst, Port.DEFAULT_TYPE, port_dst_meta),
                                        self._formatter.VLAN_DIST_FORMAT)

                                elif self._u.d_push(re.match(r'^eth\d+$', master_if)) \
                                    or self._u.d_push(re.match(r'^em\d+$', master_if)) \
                                        or self._u.d_push(re.match(r'^igb\d+$', master_if)) \
                                        or self._u.d_push(re.match(r'^bond\d+$', master_if)):
                                    self._cu.add_linkpair(
                                        Device(
                                            br_src, Device.BR_TYPE, br_src_meta),
                                        Device(
                                            "Physical NW", Device.PHYNET_TYPE),
                                        Port(
                                            port_name, Port.DEFAULT_TYPE, port_meta),
                                        Port(""))
                            else:
                                self._cu.add_linkpair(
                                    Device(
                                        br_src, Device.BR_TYPE, br_src_meta),
                                    Device(
                                        "NOT CONNECTED", Device.NOT_CONNECTED_TYPE),
                                    Port(
                                        port_name, Port.DEFAULT_TYPE, port_meta),
                                    Port(""))

                    # internal
                    elif self._cu.get_port_type(port_meta) == "internal":
                        if port_name in self._iface_to_nss:
                            self._cu.add_linkpair(
                                Device(br_src, Device.BR_TYPE, br_src_meta),
                                Device(self._iface_to_nss[
                                       port_name], Device.NAMESPACE_TYPE),
                                Port(port_name, Port.DEFAULT_TYPE, port_meta),
                                Port(""),
                                self._formatter.NAMESPACE_FORMAT)
                        else:
                            self._cu.add_linkpair(
                                Device(br_src, Device.BR_TYPE, br_src_meta),
                                Device("INTERNAL", Device.OS_ROUTE_TYPE),
                                Port(port_name, Port.DEFAULT_TYPE, port_meta),
                                Port(""))
                    # else:
                    # Other OVSPort
                    #        self._port_to_br[port_name] = br_src
                    else:
                        continue
