# vim: tabstop=4 shiftwidth=4 softtabstop=4
# -*- coding: utf-8 -*-
#

import sys
import os
import re
from socket import gethostname
from linkpair.device import Device
from linkpair.port import Port
from linkpair.linkpair import LinkPair
from linkpair.formatter.grapheasy import GraphEasyFormatter
from linkpair.utils.common import CommonUtils
from linkpair.collect.agent.commandrunner import CommandRunner


class CollectUtils(object):

    '''LinkpPair collector utility

    This class
    '''

    PEER_FOUND = 1
    DEVICE_NOT_FOUND = -1
    PORT_NOT_FOUND = -1
    LINKPAIR_NOT_FOUND = -1
    PORT_TYPE_UNSPECIFIED = "UNSPECIFIED"
    PORT_PEER_UNSPECIFIED = "UNSPECIFIED"
    PORT_VLAN_UNSPECIFIED = "UNSPECIFIED"
    METADATA_NOT_FOUND = {}

    def __init__(self, cd, util, formatter=GraphEasyFormatter()):
        self._devices = cd.devices
        self._ports = cd.ports
        self._linkpairs = cd.linkpairs
        self._port_to_br = cd.port_to_br
        self._iface_to_nss = cd.iface_to_nss
        self._veth_peers = cd.veth_peers
        self._u = util
        self._formatter = formatter

    def add_linkpair(self, dev1, dev2, port1, port2, format=""):
        if dev1.label in self._devices:
            self._devices[dev1.label].metadata.update(dev1.metadata)
        else:
            self.add_device(dev1)
        if dev2.label in self._devices:
            self._devices[dev2.label].metadata.update(dev2.metadata)
        else:
            self.add_device(dev2)
        if port1.label in self._ports:
            self._ports[port1.label].metadata.update(port1.metadata)
        else:
            self.add_port(port1)
        if port2.label in self._ports:
            self._ports[port2.label].metadata.update(port2.metadata)
        else:
            self.add_port(port2)
        if format == "":
            format = self._formatter.DEFAULT_FORMAT

        if self.exist_linkpair_by_ports(port1.label, port2.label) \
                and port2.label != "INTERNAL":
            exist_linkpair = self.get_linkpair_by_ports(
                port1.label, port2.label)
            regist_dev1 = exist_linkpair.device1
            regist_dev2 = exist_linkpair.device2
            regist_port1 = exist_linkpair.port1
            regist_port2 = exist_linkpair.port2

            if exist_linkpair.device2.label == "NOT CONNECTED":
                if dev1.label == exist_linkpair.device1.label:
                    regist_dev1 = self._devices[dev1.label]
                    regist_dev2 = self._devices[dev2.label]
                    regist_port1 = exist_linkpair.port1
                    regist_port2 = exist_linkpair.port2
                elif dev2.label == exist_linkpair.device1.label:
                    regist_dev1 = self._devices[dev2.label]
                    regist_dev2 = self._devices[dev1.label]
                    regist_port1 = exist_linkpair.port2
                    regist_port2 = exist_linkpair.port1

                self.delete_linkpair(
                    exist_linkpair.device1.label,
                    exist_linkpair.device2.label,
                    exist_linkpair.port1.label,
                    exist_linkpair.port2.label)
                self._linkpairs.append(
                    LinkPair(
                        regist_dev1,
                        regist_dev2,
                        regist_port1,
                        regist_port2,
                        format))

        else:
            self._linkpairs.append(
                LinkPair(
                    self._devices[dev1.label],
                    self._devices[dev2.label],
                    self._ports[port1.label],
                    self._ports[port2.label],
                    format))
#        if self._dbu.enable_db:
#            insert_record(fmt, src, src_style, label, dst, dst_style)

    def add_device(self, device):
        if device.label in self._devices:
            self._devices[device.label].metadata.update(device.metadata)
        else:
            self._devices[device.label] = device

    def get_device(self, device_name):
        if device_name in self._devices:
            return self._devices[device_name]
        else:
            return self.DEVICE_NOT_FOUND

    def get_linkpair(self, dev1_name, dev2_name, port1_name, port2_name):
        for linkpair in self._linkpairs:
            if linkpair.device1.label == dev1_name \
                and linkpair.device2.label == dev2_name \
                and linkpair.port1.label == port1_name \
                    and linkpair.port2.label == port2_name:
                    return linkpair
            elif linkpair.device2.label == dev1_name \
                and linkpair.device1.label == dev2_name \
                and linkpair.port2.label == port1_name \
                    and linkpair.port1.label == port2_name:
                    return linkpair
        return self.LINKPAIR_NOT_FOUND

    def delete_linkpair(self, dev1_name, dev2_name, port1_name, port2_name):
        for i in range(0, len(self._linkpairs)):
            if self._linkpairs[i].device1.label == dev1_name \
                and self._linkpairs[i].device2.label == dev2_name \
                and self._linkpairs[i].port1.label == port1_name \
                    and self._linkpairs[i].port2.label == port2_name:
                    del self._linkpairs[i]
                    return True
            elif self._linkpairs[i].device2.label == dev1_name \
                and self._linkpairs[i].device1.label == dev2_name \
                and self._linkpairs[i].port2.label == port1_name \
                    and self._linkpairs[i].port1.label == port2_name:
                    del self._linkpairs[i]
                    return True
        return False

    def get_linkpair_by_ports(self, port1_name, port2_name):
        for linkpair in self._linkpairs:
            if linkpair.port1.label == port1_name \
                    and linkpair.port2.label == port2_name:
                    return linkpair
            elif linkpair.port2.label == port1_name \
                    and linkpair.port1.label == port2_name:
                    return linkpair
        return self.LINKPAIR_NOT_FOUND

    def exist_linkpair_by_ports(self, port1_name, port2_name):
        for linkpair in self._linkpairs:
            if linkpair.port1.label == port1_name \
                    and linkpair.port2.label == port2_name:
                    return True
            elif linkpair.port2.label == port1_name \
                    and linkpair.port1.label == port2_name:
                    return True
        return False

    def exist_port(self, device_name):
        if device_name in self._devices:
            return True
        else:
            return False

    def add_port(self, port):
        if port.label in self._ports:
            self._ports[port.label].metadata.update(port.metadata)
        else:
            self._ports[port.label] = port

    def get_port(self, port_name):
        if port_name in self._ports:
            return self._ports[port_name]
        else:
            return self.PORT_NOT_FOUND

    def exist_port(self, port_name):
        if port_name in self._ports:
            return True
        else:
            return False

    def get_linkpairs(self):
        return self._linkpairs

    def drop_linkpairs(self):
        self._linkpairs = []

    def get_veth_peers(self):
        return self._veth_peers

    def drop_veth_peer(self):
        self._veth_peers = []

    def regist_to_port2br(self, device, bridge):
        if device in self._port_to_br:
            if bridge == "":
                return self._port_to_br[device]
            else:
                return bridge
        else:
            self._port_to_br[device] = bridge
            return bridge

    def get_port_type(self, port_meta):
        if "type" in port_meta:
            return port_meta["type"]
        else:
            return self.PORT_TYPE_UNSPECIFIED

    def get_port_peer(self, port_meta):
        if self.get_port_type(port_meta) == "patch":
            return port_meta["peer"].replace("\"", "")
        else:
            return self.PORT_TYPE_UNSPECIFIED

#     def get_port_peer(self, port_name):
#         port = self.get_port(port_name)
#         if port != self.PORT_NOT_FOUND:
#             port_meta = port.metadata
#             if self.get_port_type(port_meta) == "patch":
#                 return port_meta["peer"].replace("\"", "")
#             else:
#                 return self.PORT_TYPE_UNSPECIFIED
#         else:
#             return self.PORT_TYPE_UNSPECIFIED

#     def get_port_vlan_master_if(self, port_meta):
#         if "vlan_master_if" in port_meta:
#             return port_meta["vlan_master_if"]
#         else:
#             return self.PORT_VLAN_UNSPECIFIED

    def get_port_vlan_master_if(self, port_name):
        port = self.get_port(port_name)
        if port != self.PORT_NOT_FOUND:
            port_meta = port.metadata
            if "vlan_master_if" in port_meta:
                return port_meta["vlan_master_if"]
            else:
                return self.PORT_VLAN_UNSPECIFIED
        else:
            return self.PORT_VLAN_UNSPECIFIED

    def get_port_veth_peer(self, port_name):
        port = self.get_port(port_name)
        if port != self.PORT_NOT_FOUND:
            port_meta = port.metadata
            if "veth_peer" in port_meta:
                return port_meta["veth_peer"].replace("\"", "")
            else:
                return self.PORT_PEER_UNSPECIFIED
        else:
            return self.PORT_PEER_UNSPECIFIED
