# vim: tabstop=4 shiftwidth=4 softtabstop=4
# -*- coding: utf-8 -*-
#

from linkpair.linkpair import LinkPair
from linkpair.device import Device
from linkpair.port import Port
from baseformatter import BaseFormatter


class GraphvizTBFormatter(BaseFormatter):
    DEFAULT_STYLE = 'fillcolor="#ddddff", peripheries=2'
    VM_STYLE = 'color="#000000", fillcolor="#dddddd"'
    BR_STYLE = 'fillcolor="#ddddff", peripheries=2'
    OS_ROUTE_STYLE = 'fillcolor="#ffffff", style="filled,dotted"'
    PHYNET_STYLE = 'fillcolor="#ffffff", style="filled,dotted"'
    NAMESPACE_STYLE = 'fillcolor="#ddffdd", style="filled,dotted"'
    PATCH_STYLE = 'fillcolor="#ddddff", peripheries=2'
    NOT_CONNECTED_STYLE = 'fillcolor="#ffdddd", style="filled,dotted"'

    DEFAULT_FORMAT_DESC = '"%s" -> "%s" [ color="#000000", fontcolor="#663333", label = "%s     %s" ]'
    OS_ROUTING_FORMAT_DESC = '"%s" -> "%s" [ color="#000000", fontcolor="#663333", label = "%s     %s", style="dotted" ]'
    NAMESPACE_FORMAT_DESC = '"%s" -> "%s" [ color="#000000", fontcolor="#663333", label = "%s     %s" ]'
    PATCH_FORMAT_DESC = '"%s" -> "%s" [ dir=both, arrowhead=box, arrowtail=box, color="#000000:#000000", fontcolor="#333366", label = "%s : %s" ]'
    VETH_FORMAT_DESC = '"%s" -> "%s" [ dir=both, arrowhead=obox, arrowtail=obox, color="#000000", fontcolor="#336633", label="%s \| %s" ]'
    VLAN_DIST_FORMAT_DESC = '"%s" -> "%s" [ dir=forward, arrowhead=crow, arrowtail=none, color="#660000", fontcolor="#336633", label="%s\(%s\)", style="dotted" ]'

    # ADDNODE_STYLE = '"%s" [ fillcolor="#ffffff", style="filled,dotted" ]'
    ADDNODE_STYLE = '"%s" [ %s ]'
    ADDEDGE_START = ""
    ADDEDGE_END = ""

    RANK_STYLE = '{rank = same; "%s"; "%s"}'
    GROUP_STYLE = '"%s" [ group = %s ]'

    formatter_name = "graphviz-tb"

# [ br0 ] { fill: #ddddff; border: double; bordercolor: blue; } -- { label: "eth5   "; }

    HEADER_FORMAT = '''
digraph GRAPH_0 {

edge [ arrowhead=none, arrowtail=none, fontsize=10, ];
ratio = "auto"
graph [ rankdir=TB ];
node [
    fontsize=12,
    fillcolor=white,
    style=filled,
    shape=box ];

"Level_1" -> "Level_2" -> "Level_3" -> "Level_4" -> "Level_5" [color ="transparent"];
"Level_1" [color = "transparent", label="", fixedsize = true, width = 0.1, height = 0.1, fontsize=2];
"Level_2" [color = "transparent", label="", fixedsize = true, width = 0.1, height = 0.1, fontsize=2];
"Level_3" [color = "transparent", label="", fixedsize = true, width = 0.1, height = 0.1, fontsize=2];
"Level_4" [color = "transparent", label="", fixedsize = true, width = 0.1, height = 0.1, fontsize=2];
"Level_5" [color = "transparent", label="", fixedsize = true, width = 0.1, height = 0.1, fontsize=2];

'''

    FOOTER_FORMAT = '''
}
'''

    def format_linkpairs(self):
        formatted_lps = []
        formatted_lps.append(self.get_header())
        for device in self.get_all_devices(self._linkpairs):
            node = self.ADDNODE_STYLE % (
                device, self.get_device_style(device.device_type))
            formatted_lps.append(node)

        formatted_lps.append(self.ADDEDGE_START)
        for lp in self._linkpairs:
            formatted_lps.append(self.format_linkpair(lp))

        formatted_lps[len(formatted_lps) - 1] = formatted_lps[len(
            formatted_lps) - 1].rstrip(",")
        formatted_lps.append(self.ADDEDGE_END)

        ''' graphviz rank output'''
        for lp in self._linkpairs:
            rank = self.rank_linkpair(lp)
            if rank != "":
                formatted_lps.append(rank)
        formatted_lps = formatted_lps + self.rank_nodes(self._linkpairs)

        formatted_lps.append(self.get_footer())
        return formatted_lps

    def rank_linkpair(self, linkpair):
        rank_linkpair = ""
        if linkpair.format == self.PATCH_FORMAT:
            rank_linkpair += self.GROUP_STYLE % (
                linkpair.device1, self._groupid)
            rank_linkpair += "\n" + self.GROUP_STYLE % (
                linkpair.device2, self._groupid)
            rank_linkpair += "\n" + self.RANK_STYLE % (
                linkpair.device1, linkpair.device2)
            self._groupid += 1
        return rank_linkpair

    @classmethod
    def rank_nodes(cls, linkpairs):
        rank_linkpair = []
        os_routing = False
        physical_nw = False
        internal_nw = False

        for lp in linkpairs:
            if str(lp.device1) == "OS Routing" or str(lp.device2) == "OS Routing":
                os_routing = True
            elif str(lp.device1) == "Physical NW" or str(lp.device2) == "Physical NW":
                physical_nw = True
            elif str(lp.device1) == "INTERNAL" or str(lp.device2) == "INTERNAL":
                internal_nw = True

            if lp.device1.device_type == Device.VM_TYPE:
                rank_linkpair.append(
                    '{rank = min; "Level_1"; \"' + str(lp.device1) + '\"}')
            if lp.device2.device_type == Device.VM_TYPE:
                rank_linkpair.append(
                    '{rank = min; "Level_1"; \"' + str(lp.device2) + '\"}')

        if os_routing:
            rank_linkpair.append('{rank = max; "Level_5"; "OS Routing"}')
        if internal_nw:
            rank_linkpair.append('{rank = max; "Level_5"; "INTERNAL"}')
        if physical_nw:
            rank_linkpair.append('{rank = max; "Level_5"; "Physical NW"}')

        return rank_linkpair

    @classmethod
    def format_linkpair(cls, linkpair):
        formatted_lp = ""
        device1_label = linkpair.device1
        device2_label = linkpair.device2
        port1_label = linkpair.port1
        port2_label = linkpair.port2

        if "remote_ip" in linkpair.port1.metadata:
            port1_label = port1_label + " remote=\\\"" + \
                linkpair.port1.metadata["remote_ip"] + "\\\""
        if "tag" in linkpair.port1.metadata:
            port1_label = port1_label + " vlan=" + \
                linkpair.port1.metadata["tag"]
        if "vlan_vid" in linkpair.port1.metadata:
            port1_label = port1_label + " vid=" + \
                linkpair.port1.metadata["vlan_vid"]
        if "vxlan_vnid" in linkpair.port1.metadata:
            port1_label = port1_label + " vnid=" + \
                linkpair.port1.metadata["vxlan_vnid"]
        if "remote_ip" in linkpair.port2.metadata:
            port2_label = port2_label + " remote=\\\"" + \
                linkpair.port2.metadata["remote_ip"] + "\\\""
        if "tag" in linkpair.port2.metadata:
            port2_label = port2_label + " vlan=" + \
                linkpair.port2.metadata["tag"]
        if "vlan_vid" in linkpair.port2.metadata:
            port2_label = port2_label + " vid=" + \
                linkpair.port2.metadata["vlan_vid"]
        if "vxlan_vnid" in linkpair.port2.metadata:
            port2_label = port2_label + " vnid=" + \
                linkpair.port2.metadata["vxlan_vnid"]

        formatted_lp = cls.get_format_desc(linkpair.format) % (
            device1_label,
            device2_label,
            port1_label,
            port2_label)
        return formatted_lp
