/*
    mdi
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "fileio.h"
#include "misc_mdi.h"


static void misc_mdi_class_init (MiscMdiClass *klass);
static void misc_mdi_init       (MiscMdi      *mdi);


static GtkNotebookClass *parent_class = NULL;


enum
{
  ADD_WINDOW_SIGNAL,
  CHANGE_WINDOW_SIGNAL,
  CLOSE_WINDOW_SIGNAL,
  REMOVE_WINDOW_SIGNAL,
  SWITCH_WINDOW_SIGNAL,
  LAST_SIGNAL
};


struct _MiscMdiTab
{
  gpointer page_data;
  gboolean created;     /* ja:TRUE:新規作成,FALSE:既存ファイル */
  gboolean edited;      /* ja:TRUE:編集済み,FALSE:未編集 */
  gchar *file;
  gint same;            /* -2:名前のみ,-1:フルパス,0...:フルパス+数字 */
  GtkWidget *image, *label;
  MiscMdi *mdi;
};


static gint misc_mdi_signals[LAST_SIGNAL] = {0};


/******************************************************************************
*                                                                             *
******************************************************************************/
static void
misc_mdi_label_markup (MiscMdiTab *tab)
{
  gchar *label, *str;

  if (tab->same < -1)
    {
      str = g_path_get_basename (tab->file);
      label = g_filename_to_utf8 (str, -1, NULL, NULL, NULL);
      g_free (str);
    }
  else
    {
      label = g_filename_to_utf8 (tab->file, -1, NULL, NULL, NULL);
    }
  if (tab->edited)
    {
      str = g_strconcat ("<b><i>", label, "</i></b>", NULL);
      g_free (label);
      label = str;
    }
  if (tab->same >= 0)
    {
      str = g_strdup_printf ("%s:%d", label, tab->same);
      g_free (label);
      label = str;
    }
  gtk_label_set_markup (GTK_LABEL (tab->label), label);
  g_free (label);
}


static void
file_add_edit (MiscMdi    *mdi,
               MiscMdiTab *tab0)
{
  gint i;
  GList *glist;
  MiscMdiTab *tab1;

  tab0->same = -2;
  /* ja:ファイル名を調べる */
  glist = gtk_container_children (GTK_CONTAINER (mdi));
  /* ja:同名同パスを調べる */
  for (i = g_list_length (glist) - 1; i >= 0; i--)
    {
      tab1 = g_object_get_data (G_OBJECT (gtk_notebook_get_tab_label
            (GTK_NOTEBOOK (mdi), g_list_nth_data (glist, i))), "user_data");
      if (tab1 && tab0 != tab1 && g_strfilecmp (tab0->file, tab1->file) == 0)
        {
          if (tab1->same < 0)
            {
              tab1->same = 0;
              misc_mdi_label_markup (tab1);
            }
          if (tab0->same <= tab1->same)
            tab0->same = tab1->same + 1;
        }
    }
  if (tab0->same < 0)
    {
      gchar *name0;

      /* ja:同名同パスはないとき */
      /* ja:同名異パスを調べる */
      name0 = g_path_get_basename (tab0->file);
      for (i = g_list_length (glist) - 1; i >= 0; i--)
        {
          tab1 = g_object_get_data (G_OBJECT (gtk_notebook_get_tab_label
            (GTK_NOTEBOOK (mdi), g_list_nth_data (glist, i))), "user_data");
          if (tab1 && tab0 != tab1)
            {
              gchar *name1;

              name1 = g_path_get_basename (tab1->file);
              if (g_strfilecmp (name0, name1) == 0)
                {
                  if (tab1->same < -1)
                    {
                      tab1->same = -1;
                      misc_mdi_label_markup (tab1);
                    }
                  g_free (name1);
                  break;
                }
              g_free (name1);
            }
        }
      g_free (name0);
      if (i >= 0)
        tab0->same = -1;
    }
  g_list_free (glist);
}


static void
file_remove_edit (MiscMdi     *mdi,
                  MiscMdiTab  *tab0,
                  const gchar *file)
{
  gint i, count0 = 0, count1 = 0;
  gchar *name0, *name1;
  GList *glist;
  MiscMdiTab *tab1;

  /* ja:ファイル名を調べる */
  glist = gtk_container_children (GTK_CONTAINER (mdi));
  name0 = g_path_get_basename (tab0->file);
  for (i = g_list_length (glist) - 1; i >= 0; i--)
    {
      tab1 = g_object_get_data (G_OBJECT (gtk_notebook_get_tab_label
            (GTK_NOTEBOOK (mdi), g_list_nth_data (glist, i))), "user_data");
      if (tab1 && tab0 != tab1)
        {
          /* ja:同名同パスを調べる */
          if (g_strfilecmp (tab0->file, tab1->file) == 0)
            count0++;
          /* ja:同名異パスを調べる */
          name1 = g_path_get_basename (tab1->file);
          if (g_strfilecmp (name0, name1) == 0)
            count1++;
          g_free (name1);
        }
    }
  if (count1 == 1)/* ja:同名のファイルが1つだけあるとき */
    {
      gchar *name2;

      name2 = file ? g_path_get_basename (file) : NULL;
      if (!name2 || g_strfilecmp (name0, name2) != 0)
        /* 名前が変わっているとき */
        for (i = g_list_length (glist) - 1; i >= 0; i--)
          {
            /* ja:同名のファイルを通常表示に変更する */
            tab1 = g_object_get_data (G_OBJECT (gtk_notebook_get_tab_label
                            (GTK_NOTEBOOK (mdi), g_list_nth_data (glist, i))),
                                                                "user_data");
            if (tab1 && tab0 != tab1)
              {
                /* ja:同名を調べる */
                name1 = g_path_get_basename (tab1->file);
                if (g_strfilecmp (name0, name1) == 0)
                  {
                    tab1->same = -2;
                    misc_mdi_label_markup (tab1);
                  }
                g_free (name1);
              }
          }
      g_free (name2);
    }
  else if (count0 == 1)/* ja:同名同パスが1つだけある(異パスは複数)とき */
    for (i = g_list_length (glist) - 1; i >= 0; i--)
      {
        /* ja:同名同パスのファイルをフルパスに変更する */
        tab1 = g_object_get_data (G_OBJECT (gtk_notebook_get_tab_label
            (GTK_NOTEBOOK (mdi), g_list_nth_data (glist, i))), "user_data");
        /* ja:同名同パスを調べる */
        if (tab1 && tab0 != tab1 && g_strfilecmp (tab0->file, tab1->file) == 0)
          {
            tab1->same = -1;
            misc_mdi_label_markup (tab1);
          }
      }
  g_free (name0);
  g_list_free (glist);
}


/******************************************************************************
*                                                                             *
******************************************************************************/
GtkType
misc_mdi_get_type (void)
{
  static GType misc_mdi_type = 0;

  if (!misc_mdi_type)
    {
      static const GTypeInfo mdi_info =
      {
        sizeof (MiscMdiClass),
        NULL,               /* base_init */
        NULL,               /* base_finalize */
        (GClassInitFunc)misc_mdi_class_init,
        NULL,               /* class_finalize */
        NULL,               /* class_data */
        sizeof (MiscMdi),
        0,              /* n_preallocs */
        (GInstanceInitFunc)misc_mdi_init,
      };

      misc_mdi_type = g_type_register_static (GTK_TYPE_NOTEBOOK, "MiscMdi",
                                                                &mdi_info, 0);
    }

  return misc_mdi_type;
}


static void
misc_mdi_switch_page (GtkNotebook     *notebook,
                      GtkNotebookPage *page,
                      guint            page_num)
{
  GtkWidget *child;

  parent_class->switch_page (notebook, page, page_num);
  child = gtk_notebook_get_nth_page (notebook, page_num);
  if (child)
    {
      GtkWidget *tab_label;

      tab_label = gtk_notebook_get_tab_label (notebook, child);
      if (tab_label)
        {
          MiscMdiTab *tab;

          tab = g_object_get_data (G_OBJECT (tab_label), "user_data");
          if (tab)
            g_signal_emit (MISC_MDI (notebook),
                                        misc_mdi_signals[SWITCH_WINDOW_SIGNAL],
                                        0,
                                        page_num,
                                        tab->page_data);
        }
    }
}


static void
misc_mdi_class_init (MiscMdiClass *klass)
{
  GObjectClass *gobject_class;
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkNotebookClass *notebook_class;

  gobject_class = (GObjectClass*) klass;
  object_class = (GtkObjectClass*) klass;
  widget_class = (GtkWidgetClass*) klass;
  notebook_class = (GtkNotebookClass*) klass;

  parent_class = g_type_class_peek_parent (klass);

  notebook_class->switch_page = misc_mdi_switch_page;

  misc_mdi_signals[ADD_WINDOW_SIGNAL]
                = g_signal_new ("add_window",
                                G_TYPE_FROM_CLASS (gobject_class),
                                G_SIGNAL_RUN_LAST,
                                G_STRUCT_OFFSET (MiscMdiClass, add_window),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__UINT_POINTER,
                                G_TYPE_NONE, 2,
                                G_TYPE_UINT,
                                G_TYPE_POINTER);
  misc_mdi_signals[CHANGE_WINDOW_SIGNAL]
                = g_signal_new ("change_window",
                                G_TYPE_FROM_CLASS (gobject_class),
                                G_SIGNAL_RUN_LAST,
                                G_STRUCT_OFFSET (MiscMdiClass, change_window),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__UINT_POINTER,
                                G_TYPE_NONE, 2,
                                G_TYPE_UINT,
                                G_TYPE_POINTER);
  misc_mdi_signals[CLOSE_WINDOW_SIGNAL]
                = g_signal_new ("close_window",
                                G_TYPE_FROM_CLASS (gobject_class),
                                G_SIGNAL_RUN_LAST,
                                G_STRUCT_OFFSET (MiscMdiClass, close_window),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__UINT_POINTER,
                                G_TYPE_NONE, 2,
                                G_TYPE_UINT,
                                G_TYPE_POINTER);
  misc_mdi_signals[REMOVE_WINDOW_SIGNAL]
                = g_signal_new ("remove_window",
                                G_TYPE_FROM_CLASS (gobject_class),
                                G_SIGNAL_RUN_LAST,
                                G_STRUCT_OFFSET (MiscMdiClass, remove_window),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__UINT_POINTER,
                                G_TYPE_NONE, 2,
                                G_TYPE_UINT,
                                G_TYPE_POINTER);
  misc_mdi_signals[SWITCH_WINDOW_SIGNAL]
                = g_signal_new ("switch_window",
                                G_TYPE_FROM_CLASS (gobject_class),
                                G_SIGNAL_RUN_LAST,
                                G_STRUCT_OFFSET (MiscMdiClass, switch_window),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__UINT_POINTER,
                                G_TYPE_NONE, 2,
                                G_TYPE_UINT,
                                G_TYPE_POINTER);
}


static void
misc_mdi_init (MiscMdi *mdi)
{
  mdi->tooltips = gtk_tooltips_new ();
}


/******************************************************************************
*                                                                             *
* ja:更新関数群                                                               *
*                                                                             *
******************************************************************************/
/*  ja:新規作成
    RET,ウィジェット                                                        */
GtkWidget *
misc_mdi_new (void)
{
  return GTK_WIDGET (g_object_new (MISC_TYPE_MDI, NULL));
}


/* ja:ボタンが押されたとき */
static void
misc_mdi_clicked (GtkWidget *widget,
                  GtkWidget *tab_label)
{
  MiscMdiTab *tab;

  tab = g_object_get_data (G_OBJECT (tab_label), "user_data");
  if (tab)
    {
      gint i;
      GList *glist;

      glist = gtk_container_children (GTK_CONTAINER (tab->mdi));
      for (i = g_list_length (glist) - 1; i >= 0; i--)
        {
          GtkWidget *child;

          child = g_list_nth_data (glist, i);
          if (gtk_notebook_get_tab_label (GTK_NOTEBOOK (tab->mdi), child)
                                                                == tab_label)
            {
              g_signal_emit (tab->mdi,
                     misc_mdi_signals[CLOSE_WINDOW_SIGNAL],
                     0,
                     gtk_notebook_page_num (GTK_NOTEBOOK (tab->mdi), child),
                     tab->page_data);
              break;
            }
        }
      g_list_free (glist);
    }
}


/* ja:子ウィジェットが破棄されるとき */
static void
misc_mdi_destroy (GtkWidget *widget,
                  MiscMdi   *mdi)
{
  GtkWidget *tab_label;

  tab_label = gtk_notebook_get_tab_label (GTK_NOTEBOOK (mdi), widget);
  if (tab_label)
    {
      MiscMdiTab *tab;

      tab = g_object_get_data (G_OBJECT (tab_label), "user_data");
      if (tab)
        {
          g_signal_emit (mdi,
                         misc_mdi_signals[REMOVE_WINDOW_SIGNAL],
                         0,
                         gtk_notebook_page_num (GTK_NOTEBOOK (mdi), widget),
                         tab->page_data);
          file_remove_edit (MISC_MDI (mdi), tab, NULL);
          g_free (tab->file);
          g_free (tab);
          g_object_set_data (G_OBJECT (tab_label), "user_data", NULL);
        }
    }
}


/*  ja:ページのタブを作る
          mdi,ウィジェット
        child,内容
         file,ファイル名
      created,TRUE:新規作成,FALSE:既存ファイル
    page_data,ページデータ
          RET,ウィジェット                                                  */
static GtkWidget *
misc_mdi_create_tab_label (MiscMdi     *mdi,
                           GtkWidget   *child,
                           const gchar *file,
                           gboolean     created,
                           gpointer     page_data)
{
  GdkPixbuf *pixbuf;
  GtkWidget *button, *frame, *hbox, *event;
  MiscMdiTab *tab;
  static const gchar *xpm[] = {
"8 7 2 1",
"   c None",
".  c #000000",
"..    ..",
" ..  .. ",
"  ....  ",
"   ..   ",
"  ....  ",
" ..  .. ",
"..    .."};

  if (!mdi)
    return NULL;
  tab = g_malloc0 (sizeof (MiscMdiTab));
  tab->page_data = page_data;
  tab->created = created;
  tab->file = file ? created ? g_strdup (file) : fileio_get_full_path (file)
                                                            : g_strdup ("");
  tab->same = -2;
  tab->mdi = mdi;
  file_add_edit (mdi, tab);
  /* ja:ボタン */
  button = gtk_button_new ();
  frame = gtk_frame_new (NULL);
  gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
  pixbuf = gdk_pixbuf_new_from_xpm_data (xpm);
  gtk_container_add (GTK_CONTAINER (frame),
                                        gtk_image_new_from_pixbuf (pixbuf));
  g_object_unref (pixbuf);
  gtk_container_add (GTK_CONTAINER (button), frame);
  gtk_widget_show_all (button);
  /* ja:ラベル */
  tab->label = gtk_label_new ("");
  misc_mdi_label_markup (tab);
  gtk_widget_show (tab->label);
  /* ja:アイコン */
  pixbuf = fileio_extract_icon (tab->file);
  if (pixbuf)
    {
      tab->image = gtk_image_new_from_pixbuf (pixbuf);
      g_object_unref (pixbuf);
      gtk_widget_show (tab->image);
    }
  /* ja:表示 */
  hbox = gtk_hbox_new (FALSE, 0);
  if (tab->image)
    gtk_box_pack_start (GTK_BOX (hbox), tab->image, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox), tab->label, TRUE, TRUE, 0);
  gtk_widget_show (hbox);
  /* ja:ツールチップス */
  event = gtk_event_box_new ();
  gtk_container_add (GTK_CONTAINER (event), hbox);
  gtk_tooltips_set_tip (mdi->tooltips, event, tab->file, tab->file);
  g_object_set_data (G_OBJECT (event), "user_data", tab);

  g_signal_connect (G_OBJECT (button), "clicked",
                                        G_CALLBACK (misc_mdi_clicked), event);
  g_signal_connect (G_OBJECT (child), "destroy",
                                        G_CALLBACK (misc_mdi_destroy), mdi);
  return event;
}


/*  ja:ページを挿入する
          mdi,ウィジェット
        child,内容
         menu,メニュー
     position,位置(0...)
         file,ファイル名
      created,TRUE:新規作成,FALSE:既存ファイル
    page_data,ページデータ
       RET,ページ(0...,-1:エラー)                                           */
gint
misc_mdi_insert_page_menu (MiscMdi     *mdi,
                           GtkWidget   *child,
                           GtkWidget   *menu,
                           gint         position,
                           const gchar *file,
                           gboolean     created,
                           gpointer     page_data)
{
  gint page_num = -1;
  GtkWidget *tab_label;

  tab_label = misc_mdi_create_tab_label (mdi, child, file, created, page_data);
  if (tab_label)
    {
      page_num = gtk_notebook_insert_page_menu (GTK_NOTEBOOK (mdi), child,
                                                    tab_label, menu, position);
      g_signal_emit (mdi,
                     misc_mdi_signals[ADD_WINDOW_SIGNAL],
                     0,
                     page_num,
                     page_data);
    }
  return page_num;
}


#if !GLIB_CHECK_VERSION(2,2,0)
/*  ja:ページ数を取得する
    mdi,ウィジェット
    RET,ページ数                                                            */
gint
misc_mdi_get_n_pages (MiscMdi *mdi)
{
  gint n_pages = 0;

  if (mdi)
    {
      GList *glist;

      glist = gtk_container_children (GTK_CONTAINER (mdi));
      if (glist)
        {
          n_pages = g_list_length (glist);
          g_list_free (glist);
        }
    }
  return n_pages;
}
#endif /* not GLIB_CHECK_VERSION(2,2,0) */


/*  ja:ページをデータから取得する
          mdi,ウィジェット
    page_data,ページデータ
          RET,ページ(0...),-1:なし                                          */
gint
misc_mdi_get_page_from_data (MiscMdi  *mdi,
                             gpointer  page_data)
{
  if (mdi)
    {
      gint i;

      for (i = misc_mdi_get_n_pages (MISC_MDI (mdi)) - 1; i >= 0; i--)
        if (misc_mdi_get_data (MISC_MDI (mdi), i) == page_data)
          return i;
    }
  return -1;
}


/*  ja:ページの内部を取得する
         mdi,ウィジェット
    page_num,ページ(0...),-1:カレント
         RET,ページ内部データ                                               */
static MiscMdiTab *
misc_mdi_get_tab (MiscMdi *mdi,
                  gint     page_num)
{
  if (mdi)
    {
      GtkWidget *child;

      child = gtk_notebook_get_nth_page (GTK_NOTEBOOK (mdi), page_num >= 0
            ? page_num : gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi)));
      if (child)
        {
          GtkWidget *tab_label;

          tab_label = gtk_notebook_get_tab_label (GTK_NOTEBOOK (mdi), child);
          if (tab_label)
            return g_object_get_data (G_OBJECT (tab_label), "user_data");
        }
    }
  return NULL;
}


/*  ja:ページデータを取得する
         mdi,ウィジェット
    page_num,ページ(0...),-1:カレント
         RET,ページデータ                                                   */
gpointer
misc_mdi_get_data (MiscMdi *mdi,
                   gint     page_num)
{
  MiscMdiTab *tab;

  tab = misc_mdi_get_tab (mdi, page_num);
  return tab ? tab->page_data: NULL;
}


/*  ja:ページデータを設定する
           mdi,ウィジェット
      page_num,ページ(0...),-1:カレント
     page_data,ページデータ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
misc_mdi_set_data (MiscMdi *mdi,
                   gint     page_num,
                   gpointer page_data)
{
  MiscMdiTab *tab;

  tab = misc_mdi_get_tab (mdi, page_num);
  if (tab)
    {
      tab->page_data = page_data;
      return TRUE;
    }
  return FALSE;
}


/*  ja:ページのファイル名を取得する
         mdi,ウィジェット
    page_num,ページ(0...),-1:カレント
         RET,ファイル名,NULL:エラー                                         */
const gchar *
misc_mdi_get_file (MiscMdi *mdi,
                   gint     page_num)
{
  MiscMdiTab *tab;

  tab = misc_mdi_get_tab (mdi, page_num);
  return tab ? tab->file : NULL;
}


/*  ja:ページのファイル名を設定する
         mdi,ウィジェット
    page_num,ページ(0...),-1:カレント
        file,ファイル名
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
misc_mdi_set_file (MiscMdi     *mdi,
                   gint         page_num,
                   const gchar *file)
{
  if (mdi && file)
    {
      GtkWidget *child;

      child = gtk_notebook_get_nth_page (GTK_NOTEBOOK (mdi), page_num >= 0
            ? page_num : gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi)));
      if (child)
        {
          GtkWidget *tab_label;

          tab_label = gtk_notebook_get_tab_label (GTK_NOTEBOOK (mdi), child);
          if (tab_label)
            {
              MiscMdiTab *tab;

              tab = g_object_get_data (G_OBJECT (tab_label), "user_data");
              if (tab)
                {
                  GdkPixbuf *pixbuf;
                  GtkWidget *hbox;

                  /* ja:ラベル */
                  file_remove_edit (mdi, tab, file);
                  g_free (tab->file);
                  tab->file = tab->created ? g_strdup (file)
                                           : fileio_get_full_path (file);
                  tab->same = -2;
                  file_add_edit (mdi, tab);
                  misc_mdi_label_markup (tab);
                  /* ja:ツールチップス */
                  gtk_tooltips_set_tip (mdi->tooltips, tab_label,
                                                        tab->file, tab->file);
                  /* ja:アイコン */
                  hbox = gtk_bin_get_child (GTK_BIN (tab_label));
                  if (tab->image)
                    gtk_container_remove (GTK_CONTAINER (hbox), tab->image);
                  tab->image = NULL;
                  pixbuf = fileio_extract_icon (tab->file);
                  if (pixbuf)
                    {
                      tab->image = gtk_image_new_from_pixbuf (pixbuf);
                      g_object_unref (pixbuf);
                      gtk_widget_show (tab->image);
                    }
                  if (tab->image)
                    gtk_box_pack_start (GTK_BOX (hbox),
                                                tab->image, FALSE, FALSE, 0);
                  g_signal_emit (mdi,
                                 misc_mdi_signals[CHANGE_WINDOW_SIGNAL],
                                 0,
                                 page_num,
                                 tab->page_data);
                  return TRUE;
                }
            }
        }
    }
  return FALSE;
}


/*  ja:ページの作成状態を取得する
         mdi,ウィジェット
    page_num,ページ(0...),-1:カレント
         RET,TRUE:新規作成,FALSE:既存ファイル                               */
gboolean
misc_mdi_get_created (MiscMdi *mdi,
                      gint     page_num)
{
  MiscMdiTab *tab;

  tab = misc_mdi_get_tab (mdi, page_num);
  return tab ? tab->created : FALSE;
}


/*  ja:ページの作成状態を設定する
          mdi,ウィジェット
     page_num,ページ(0...),-1:カレント
    is_create,TRUE:新規作成,FALSE:既存ファイル
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
misc_mdi_set_created (MiscMdi  *mdi,
                      gint      page_num,
                      gboolean  is_create)
{
  MiscMdiTab *tab;

  tab = misc_mdi_get_tab (mdi, page_num);
  if (tab)
    {
      if (is_create ^ tab->created)
        tab->created = is_create;
      return TRUE;
    }
  return FALSE;
}


/*  ja:ページの編集状態を取得する
         mdi,ウィジェット
    page_num,ページ(0...),-1:カレント
         RET,TRUE:編集済み,FALSE:未編集                                     */
gboolean
misc_mdi_get_edited (MiscMdi *mdi,
                     gint     page_num)
{
  MiscMdiTab *tab;

  tab = misc_mdi_get_tab (mdi, page_num);
  return tab ? tab->edited : FALSE;
}


/*  ja:ページの編集状態を設定する
         mdi,ウィジェット
    page_num,ページ(0...),-1:カレント
     is_edit,TRUE:編集済み,FALSE:未編集
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
misc_mdi_set_edited (MiscMdi  *mdi,
                     gint      page_num,
                     gboolean  is_edit)
{
  MiscMdiTab *tab;

  tab = misc_mdi_get_tab (mdi, page_num);
  if (tab)
    {
      if (is_edit ^ tab->edited)
        {
          tab->edited = is_edit;
          misc_mdi_label_markup (tab);
        }
      return TRUE;
    }
  return FALSE;
}
