/*
    Text maid
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "charset.h"
#include "command.h"
#include "edit.h"
#include "find.h"
#include "general.h"
#include "sigfile.h"
#include "toolbar.h"
#include "misc/charuty.h"
#include "misc/misc.h"
#include "misc/misc_mdi.h"


/******************************************************************************
*                                                                             *
* ja:一般関数群                                                               *
*                                                                             *
******************************************************************************/
/*  ja:キャレットを消す
    tmaid,ウインドウ情報                                                    */
static void
hide_caret (TmaidWindow *tmaid)
{
  GdkRectangle rc;

  gtk_timeout_remove (timer_id);
  timer_id = 0;
  if (caret)
    {
      caret = FALSE;
      rc.x = edit_get_align_pos (tmaid,
                                 tmaid->cursor.x, tmaid->cursor.y, FALSE);
      rc.y = tmaid->cursor.y;
      rc.x = (rc.x - tmaid->top.x) * tmaid->font_width;
      rc.y = (rc.y - tmaid->top.y) * tmaid->font_height;
      rc.width = tmaid->font_width;
      rc.height = tmaid->font_height;
      gtk_widget_draw (tmaid->drawing, &rc);
    }
}


/*  ja:キャレットの描画
     tmaid,ウインドウ情報
    cursor,古いキャレット位置                                               */
void
draw_caret (TmaidWindow *tmaid,
            GdkPoint    *cursor)
{
  gboolean visible;
  gchar *text;
  gint x, y;
  guint context_id;
  GdkRectangle rc;
#ifndef G_OS_WIN32
  GtkWidget *widget;
#endif /* not G_OS_WIN32 */

  if (cursor && timer_id != 0
            && (tmaid->cursor.x != cursor->x || tmaid->cursor.y != cursor->y))
    {
      /* ja:タイマ再設定 */
      gtk_timeout_remove (timer_id);
      timer_id = 0;
      /* ja:古いキャレットを消去 */
      rc.x = edit_get_align_pos (tmaid, cursor->x, cursor->y, FALSE);
      rc.y = cursor->y;
      rc.x = (rc.x - tmaid->top.x) * tmaid->font_width;
      rc.y = (rc.y - tmaid->top.y) * tmaid->font_height;
      rc.width = tmaid->font_width;
      rc.height = tmaid->font_height;
      gtk_widget_draw (tmaid->drawing, &rc);
    }
  /* ja:新しいキャレットを描画 */
  if (timer_id == 0)
    caret=TRUE;
  x = edit_get_align_pos (tmaid, tmaid->cursor.x, tmaid->cursor.y, FALSE);
  y = tmaid->cursor.y;
  rc.x = (x - tmaid->top.x) * tmaid->font_width;
  rc.y = (y - tmaid->top.y) * tmaid->font_height;
  rc.width = tmaid->font_width;
  rc.height = tmaid->font_height;
  gtk_widget_draw (tmaid->drawing, &rc);
  if (timer_id == 0)
    timer_id = gtk_timeout_add (500, signal_timeout, NULL);
  text = g_strdup_printf (_("L%d,C%d"), y + 1, x + 1);
  gtk_label_set_text (GTK_LABEL (label_cursor), text);
  g_free (text);
  text = tmaid->select.x >= 0
                    ? g_strdup_printf (_("Selected(L%d,C%d)-(L%d,C%d)"),
                        tmaid->select.y + 1, tmaid->select.x + 1, y + 1, x + 1)
                    : g_strdup ("");
  context_id = gtk_statusbar_get_context_id (GTK_STATUSBAR (status), "Status");
  gtk_statusbar_pop (GTK_STATUSBAR (status), context_id);
  gtk_statusbar_push (GTK_STATUSBAR (status), context_id, text);
  g_free (text);
#ifndef G_OS_WIN32
  gdk_window_get_origin (tmaid->drawing->window, &x, &y);
  widget = gtk_widget_get_parent (tmaid->preedit);
  visible = GTK_WIDGET_VISIBLE (widget);
  if (!visible)
    gtk_window_resize (GTK_WINDOW (widget), 1, 1);
  gtk_widget_hide (widget);
  gtk_window_move (GTK_WINDOW (widget), rc.x + x, rc.y + y);
  if (visible)
    gtk_widget_show (widget);
#endif /* not G_OS_WIN32 */
  rc.width = 0;
  gtk_im_context_set_cursor_location (tmaid->im_context, &rc);
}


/*  ja:メニューを設定する
    tmaid,ウインドウ情報                                                 */
void
set_menu_bar (TmaidWindow *tmaid)
{
  if (tmaid)
    {
      gboolean created;

      created = misc_mdi_get_created (MISC_MDI (mdi),
                        misc_mdi_get_page_from_data (MISC_MDI (mdi), tmaid));
      gtk_widget_set_sensitive (gtk_menu_get_attach_widget (GTK_MENU
            (gtk_item_factory_get_widget (ifactory_menu, "<main>/Edit"))),
                                                                        TRUE);
      gtk_widget_set_sensitive (gtk_menu_get_attach_widget (GTK_MENU
            (gtk_item_factory_get_widget (ifactory_menu, "<main>/Search"))),
                                                                        TRUE);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                                "<main>/File/Close"), TRUE);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                                "<main>/File/Save"), TRUE);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                                "<main>/File/Save As..."), TRUE);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                                "<main>/File/Reload..."), !created);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                                "<main>/File/Print..."), TRUE);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                                "<main>/File/Properties..."), TRUE);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                                "<main>/Edit/Undo"), tmaid->undo != NULL);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                                "<main>/Edit/Redo"), tmaid->redo != NULL);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                                "<main>/Edit/Cut"), tmaid->select.x >= 0);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                                "<main>/Edit/Copy"), tmaid->select.x >= 0);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                                "<main>/Edit/Delete"), tmaid->select.x >= 0);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                                "<main>/Search/Next"), find_num > 0);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                                "<main>/Search/Previous"), find_num > 0);
      gtk_widget_set_sensitive (toolbar_items[ 2].item, TRUE);
      gtk_widget_set_sensitive (toolbar_items[ 4].item, TRUE);
      gtk_widget_set_sensitive (toolbar_items[ 6].item, TRUE);
      gtk_widget_set_sensitive (toolbar_items[ 8].item, tmaid->select.x >= 0);
      gtk_widget_set_sensitive (toolbar_items[ 9].item, tmaid->select.x >= 0);
      gtk_widget_set_sensitive (toolbar_items[10].item, TRUE);
      gtk_widget_set_sensitive (toolbar_items[11].item, tmaid->select.x >= 0);
      gtk_widget_set_sensitive (toolbar_items[13].item, tmaid->undo != NULL);
      gtk_widget_set_sensitive (toolbar_items[14].item, tmaid->redo != NULL);
      gtk_widget_set_sensitive (toolbar_items[16].item, TRUE);
      gtk_widget_set_sensitive (toolbar_items[17].item, TRUE);
    }
  else
    {
      gtk_widget_set_sensitive (gtk_menu_get_attach_widget (GTK_MENU
            (gtk_item_factory_get_widget (ifactory_menu, "<main>/Edit"))),
                                                                        FALSE);
      gtk_widget_set_sensitive (gtk_menu_get_attach_widget (GTK_MENU
            (gtk_item_factory_get_widget (ifactory_menu, "<main>/Search"))),
                                                                        FALSE);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                                "<main>/File/Close"), FALSE);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                                "<main>/File/Save"), FALSE);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                                "<main>/File/Save As..."), FALSE);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                                "<main>/File/Reload..."), FALSE);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                                "<main>/File/Print..."), FALSE);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                                "<main>/File/Properties..."), FALSE);
      gtk_widget_set_sensitive (toolbar_items[ 2].item, FALSE);
      gtk_widget_set_sensitive (toolbar_items[ 4].item, FALSE);
      gtk_widget_set_sensitive (toolbar_items[ 6].item, FALSE);
      gtk_widget_set_sensitive (toolbar_items[ 8].item, FALSE);
      gtk_widget_set_sensitive (toolbar_items[ 9].item, FALSE);
      gtk_widget_set_sensitive (toolbar_items[10].item, FALSE);
      gtk_widget_set_sensitive (toolbar_items[11].item, FALSE);
      gtk_widget_set_sensitive (toolbar_items[13].item, FALSE);
      gtk_widget_set_sensitive (toolbar_items[14].item, FALSE);
      gtk_widget_set_sensitive (toolbar_items[16].item, FALSE);
      gtk_widget_set_sensitive (toolbar_items[17].item, FALSE);
    }
}


/*  ja:指定範囲を再描画する
    tmaid,ウインドウ情報
    start,開始
      end,終了                                                              */
void
clear_sel (TmaidWindow *tmaid,
           GdkPoint    *start,
           GdkPoint    *end)
{
  gint st, ed;
  GdkRectangle rc;

  if (start->y == end->y)
    {
      /* ja:同じ行 */
      st = edit_get_align_pos (tmaid, MIN (start->x, end->x), start->y, FALSE);
      ed = edit_get_align_pos (tmaid, MAX (start->x, end->x), start->y, TRUE);
      rc.x = (st - tmaid->top.x) * tmaid->font_width;
      rc.y = (start->y - tmaid->top.y) * tmaid->font_height;
      rc.width = (ed - st + 1) * tmaid->font_width;
      rc.height = tmaid->font_height;
    }
  else
    {
      /* ja:違う行 */
      rc.x = 0;
      rc.y = (MIN (start->y, end->y) - tmaid->top.y) * tmaid->font_height;
      rc.width = tmaid->drawing->allocation.width;
      rc.height = (ABS (start->y - end->y) + 1) * tmaid->font_height;
    }
  gtk_widget_draw (tmaid->drawing, &rc);
}


/*  ja:移動する
    tmaid,ウインドウ情報
    top,古い左上の座標                                                      */
void
move_edit_window (TmaidWindow *tmaid,
                  GdkPoint    *top)
{
  if (tmaid->top.x != top->x || tmaid->top.y != top->y)
    misc_scroll_window (tmaid->drawing,
                                (top->x - tmaid->top.x) * tmaid->font_width,
                                (top->y - tmaid->top.y) * tmaid->font_height,
                                0, 0,
                                tmaid->drawing->allocation.width,
                                tmaid->drawing->allocation.height);
}


/*  ja:キャレット位置にデータを挿入/上書きする
     tmaid,ウインドウ情報
      text,データを格納するポインタ
    length,データのバイト数
     caret,TRUE:移動する,FALSE:移動しない
    select,TRUE:選択する,FALSE:選択しない
       RET,逆の操作を行うDOING構造体,NULL:エラー                            */
TmaidHistory *
edit_operation (TmaidWindow    *tmaid,
                const gchar    *text,
                const gint      length,
                const gboolean  caret,
                const gboolean  select)
{
  gint i, j, sx, sy, max, delete, put;
  GdkPoint cursor_old, cursor_new, top;
  GdkRectangle rc;
  LineBuffer *p, *q;
  TmaidHistory *d;

  hide_caret (tmaid);
  tmaid->cursor.x = edit_get_align_pos (tmaid,
                                    tmaid->cursor.x, tmaid->cursor.y, FALSE);
  top = tmaid->top;
  sx = MAX (tmaid->drawing->allocation.width / tmaid->font_width, 1);
  sy = MAX (tmaid->drawing->allocation.height / tmaid->font_height, 1);
  d = g_malloc (sizeof (TmaidHistory));
  if (tmaid->select.x < 0)
    {
      d->text = NULL;
      d->length = 0;
      d->caret = FALSE;
    }
  else
    {
      /* ja:削除 */
      d->length = edit_get_sel_bytes (tmaid, &tmaid->cursor, &tmaid->select);
      d->text = g_malloc (d->length * sizeof (gchar));
      edit_cpy_sel_mem (tmaid, &tmaid->cursor, &tmaid->select, d->text);
      delete = edit_del_sel_mem (tmaid, &tmaid->cursor, &tmaid->select);
      tmaid->max -= delete;/* ja:行数減少 */
      /* ja:選択範囲解除とキャレット移動 */
      if (tmaid->select.y < tmaid->cursor.y
                                        || (tmaid->select.y == tmaid->cursor.y
                                        && tmaid->select.x < tmaid->cursor.x))
        {
          tmaid->cursor = tmaid->select;
          d->caret = TRUE;
        }
      else
        {
          tmaid->cursor = tmaid->cursor;
          d->caret = FALSE;
        }
      tmaid->select.x = -1;
      /* ja:削除した行の削除した桁より右を初期化(右マージンのために1つ多い) */
      rc.x = (tmaid->cursor.x - tmaid->top.x - 1) * tmaid->font_width;
      rc.y = (tmaid->cursor.y - tmaid->top.y) * tmaid->font_height;
      rc.width = tmaid->drawing->allocation.width - rc.x;
      rc.height = tmaid->font_height;
      gtk_widget_draw (tmaid->drawing, &rc);
      /* ja:減った行数分だけスクロール */
      misc_scroll_window (tmaid->drawing, 0, -delete * tmaid->font_height,
        0,
        (tmaid->cursor.y - tmaid->top.y + 1) * tmaid->font_height,
        tmaid->drawing->allocation.width,
        MAX (tmaid->drawing->allocation.height
            - (tmaid->cursor.y - tmaid->top.y + 1) * tmaid->font_height, 0));
        /* ja:削除を終えた位置から右マージンの影響のある行数だけ初期化 */
      for (i = 0,
        p = edit_get_line_buf (&tmaid->start, &tmaid->off, tmaid->cursor.y);
                                                        p; i++, p = p->next)
        if (!p->margin)
          break;
      if (i>0)
        {
          rc.x = 0;
          rc.y = (tmaid->cursor.y - tmaid->top.y + 1) * tmaid->font_height;
          rc.width = tmaid->drawing->allocation.width;
          rc.height = i * tmaid->font_height;
          gtk_widget_draw (tmaid->drawing, &rc);
        }
    }
  if (length <= 0 || !text)
    {
      d->cursor = tmaid->cursor;
      d->select.x = -1;
    }
  else
    {
      /* ja:挿入 */
      tmaid->cursor.x = edit_get_align_pos (tmaid,
                                    tmaid->cursor.x, tmaid->cursor.y, FALSE);
      put = edit_put_mem (tmaid, &tmaid->cursor, &cursor_new, text, length);
      tmaid->max += put;/* ja:行数増加 */
      cursor_old = tmaid->cursor;
      if (caret)
        {
          /* ja:キャレット移動 */
          d->cursor = cursor_new;
          d->select = cursor_old;
          tmaid->cursor = cursor_new;
        }
      else
        {
          /* ja:キャレット移動なし */
          d->cursor = cursor_old;
          d->select = cursor_new;
        }
      if (select)
        {
          /* ja:選択する */
          tmaid->select = d->select;
          gtk_selection_owner_set (tmaid->drawing, GDK_SELECTION_PRIMARY,
                                                            GDK_CURRENT_TIME);
        }
      /* ja:挿入した行の挿入した桁より右を初期化 */
      rc.x = (cursor_old.x - tmaid->top.x) * tmaid->font_width;
      rc.y = (cursor_old.y - tmaid->top.y) * tmaid->font_height;
      rc.width = tmaid->drawing->allocation.width;
      rc.height = tmaid->font_height;
      gtk_widget_draw (tmaid->drawing, &rc);
      /* ja:増えた行数分だけスクロール */
      misc_scroll_window (tmaid->drawing, 0, put * tmaid->font_height,
            0,
            (cursor_old.y - tmaid->top.y + 1) * tmaid->font_height,
            tmaid->drawing->allocation.width,
            MAX (tmaid->drawing->allocation.height
                - (cursor_old.y - tmaid->top.y + 1) * tmaid->font_height, 0));
      /* ja:挿入を終えた位置から右マージンの影響のある行数だけ初期化 */
      p = q = edit_get_line_buf (&tmaid->start, &tmaid->off, cursor_new.y);
      for (i = cursor_new.y, j = 0;
                                p->prev && p->prev->margin && cursor_old.y < i;
                                                        i--, j++, p = p->prev);
      while (q->next && q->margin)
        j++, q = q->next;
      if (j > 0)
        {
          rc.x = 0;
          rc.y = (i - tmaid->top.y + 1) * tmaid->font_height;
          rc.width = tmaid->drawing->allocation.width;
          rc.height = j * tmaid->font_height;
          gtk_widget_draw (tmaid->drawing, &rc);
        }
    }
  if (tmaid->cursor.x < tmaid->top.x)
    tmaid->top.x = tmaid->cursor.x;
  else if (tmaid->cursor.x - sx + 1 > tmaid->top.x)
    tmaid->top.x = tmaid->cursor.x - sx + 1;
  if (tmaid->cursor.y < tmaid->top.y)
    tmaid->top.y = tmaid->cursor.y;
  else if (tmaid->cursor.y - sy + 1 > tmaid->top.y)
    tmaid->top.y = tmaid->cursor.y - sy + 1;
  if (tmaid->top.y > tmaid->max - sy)
    tmaid->top.y = MAX (tmaid->max - sy, 0);
  max = edit_get_width_max (tmaid);
  if (tmaid->top.x > max - sx + 1)
    tmaid->top.x = MAX (max - sx + 1, 0);
  move_edit_window (tmaid, &top);
  misc_set_scroll_bar (tmaid->hscroll,
                            G_CALLBACK (signal_value_changed_hscroll), tmaid,
                                            0, max + 1, sx, tmaid->top.x);
  misc_set_scroll_bar (tmaid->vscroll,
                            G_CALLBACK (signal_value_changed_vscroll), tmaid,
                                            0, tmaid->max, sy, tmaid->top.y);
  draw_caret (tmaid, NULL);
  return d;
}


/*  ja:リストから削除する
      d,リストの先頭
    RET,削除したリストの数                                                  */
gint
delete_list (TmaidHistory **d)
{
  gint count = 0;
  TmaidHistory *d0, *d1;

  for (d0 = *d; d0; d0 = d1)
    {
      g_free (d0->text);
      d1 = d0->next;
      g_free (d0);
      count++;
    }
  *d = NULL;
  return count;
}


/*  ja:右マージンで折り返す
    tmaid,ウインドウ情報                                                 */
void
modify_margin (TmaidWindow *tmaid)
{
    gint data_pos, screen_pos;
    LineBuffer *p, *q;

  /* ja:ラインバッファの先頭へ移動 */
  while (tmaid->start->prev)
    tmaid->start = tmaid->start->prev;
  tmaid->off = 0;
  p = tmaid->start;
  if (tmaid->ft.limit)
    /* ja:右マージンによる改行あり */
    while (p)
      {
        data_pos = screen_pos = 0;
        while (data_pos<p->length)
          if (p->text[data_pos] == '\t')
            {
              screen_pos = (screen_pos / tmaid->ft.tab + 1) * tmaid->ft.tab;
              if (tmaid->ft.margin < screen_pos)
                break;
              data_pos++;
            }
          else if (data_pos + charset_length (p->text[data_pos]) <= p->length)
            {
              screen_pos += charset_width (tmaid->layout, 
                                        p->text + data_pos,
                                        tmaid->font_width, tmaid->font_buf);
              if (tmaid->ft.margin < screen_pos)
                break;
              data_pos += charset_length (p->text[data_pos]);
            }
          else
            {
              screen_pos++;
              if (tmaid->ft.margin < screen_pos)
                break;
              data_pos++;
            }
        if (tmaid->ft.margin < screen_pos)
          {
            /* ja:右マージンを超えているとき、擬似改行する */
            q = g_malloc (sizeof (LineBuffer));
            q->length = p->length-data_pos;
            q->margin = p->margin;
            q->text = g_malloc (q->length * sizeof (gchar));
            q->prev = p;
            q->next = p->next;
            p->next = q;
            if (q->next)
              q->next->prev=q;
            g_memmove (q->text, p->text + data_pos,
                                                q->length * sizeof (gchar));
            p->length = data_pos;
            p->margin = TRUE;
            p->text = g_realloc (p->text, p->length * sizeof (gchar));
            p = q;
            tmaid->max++;
          }
        else if (screen_pos < tmaid->ft.margin && p->margin && p->next)
          {
            /* ja:右マージンを下回り擬似改行のとき、次の行とあわせる */
            data_pos = p->length;
            q = p->next;
            if (q->next)
              q->next->prev = p;
            p->next = q->next;
            p->length += q->length;
            p->margin = q->margin;
            p->text = g_realloc (p->text, p->length * sizeof (gchar));
            g_memmove (p->text + data_pos, q->text,
                                                q->length * sizeof (gchar));
            g_free (q->text);
            g_free (q);
            tmaid->max--;
          }
        else
          {
            p=p->next;
          }
      }
  else
    /* ja:右マージンによる改行なし */
    while (p)
      if (p->margin)
        {
          /* ja:擬似改行のとき、次の行とあわせる */
          data_pos = p->length;
          q = p->next;
          if (q->next)
            q->next->prev = p;
          p->next = q->next;
          p->length += q->length;
          p->margin = q->margin;
          p->text = g_realloc (p->text, p->length * sizeof (gchar));
          g_memmove (p->text + data_pos, q->text, q->length * sizeof (gchar));
          g_free (q->text);
          g_free (q);
          tmaid->max--;
        }
      else
        {
          p = p->next;
        }
}


/*  ja:ファイルの保存を問い合わせる
    tmaid,TXTウインドウ情報
      RET,TRUE:ファイルは閉じる,FALSE:ファイルを閉じない                    */
gboolean
prompt_close (TmaidWindow *tmaid)
{
  gint page;

  page = misc_mdi_get_page_from_data (MISC_MDI (mdi), tmaid);
  if (misc_mdi_get_edited (MISC_MDI (mdi), page))
    {
      gchar *text, *utf8str;

      utf8str = g_filename_to_utf8
            (misc_mdi_get_file (MISC_MDI (mdi), page), -1, NULL, NULL, NULL);
      text = g_strdup_printf (_("File %s was edited.\nSave?"), utf8str);
      g_free (utf8str);
      switch (misc_message_box ("Text maid", text, 0,
                                    _("_Yes"), _("_No"), _("Cancel"), NULL))
        {
          case 0:
            {
              gint page_num;

              page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
              if (page != page_num)
                gtk_notebook_set_current_page (GTK_NOTEBOOK (mdi), page);
              else
                page_num = -1;
              command_save (NULL, 0, NULL);
              if (page_num >= 0)
                gtk_notebook_set_current_page (GTK_NOTEBOOK (mdi), page_num);
              if (!misc_mdi_get_edited (MISC_MDI (mdi), page))
                break;
            }
          case 2:
          case -1:
            g_free (text);
            return FALSE;
        }
      g_free (text);
    }
  return TRUE;
}


/*  ja:キャラクターセットを置換する
    charlist,置換するキャラクターセット
       ft_id,ファイルタイプのID(-1:追加なし)                                */
void
charlist_renewal_all (const gchar *charlist,
                      const gint   ft_id)
{
  gchar *charset;
  gint i;

  for (i = 0; i < ftnum; i++)
    {
      charset = ftype[i].ft_id == ft_id
                                ? charuty_append (ftype[i].charset, charlist)
                                : charuty_rename (ftype[i].charset, charlist);
      g_free (ftype[i].charset);
      ftype[i].charset = charset;
    }
  for (i = misc_mdi_get_n_pages (MISC_MDI (mdi)) - 1; i >= 0; i--)
    {
      TmaidWindow *tmaid;

      tmaid = misc_mdi_get_data (MISC_MDI (mdi), i);
      charset = charuty_rename (tmaid->ft.charset, charlist);
      g_free (tmaid->ft.charset);
      tmaid->ft.charset = charset;
    }
}


/******************************************************************************
*                                                                             *
* ja:高レベル関数群                                                           *
*                                                                             *
******************************************************************************/
/*  ja:編集履歴操作
    tmaid,ウインドウ情報
     kind,TRUE:redo,FALSE:undo                                              */
void
history_operation (TmaidWindow    *tmaid,
                   const gboolean  kind)
{
  GdkPoint cursor, select;
  TmaidHistory *d0, *d1;

  if (kind)
    {
      d0 = tmaid->redo;
      tmaid->redo = d0->next;
    }
  else
    {
      d0 = tmaid->undo;
      tmaid->undo = d0->next;
    }
  cursor = tmaid->cursor;
  select = tmaid->select;
  tmaid->cursor = d0->cursor;
  tmaid->select = d0->select;
  if (select.x >= 0)
    clear_sel (tmaid, &select, &cursor);
  d1 = edit_operation (tmaid, d0->text, d0->length, d0->caret, TRUE);
  if (kind)
    {
      d1->next = tmaid->undo;
      tmaid->undo = d1;
    }
  else
    {
      d1->next = tmaid->redo;
      tmaid->redo = d1;
    }
  g_free (d0->text);
  g_free (d0);
  set_menu_bar (tmaid);
}


/*  ja:右マージンで改行する
    tmaid,ウインドウ情報                                                    */
void
margin_operation (TmaidWindow *tmaid)
{
  gboolean limit;
  LineBuffer *p;

  delete_list (&tmaid->undo);
  delete_list (&tmaid->redo);
  limit = tmaid->ft.limit;
  tmaid->ft.limit = TRUE;
  modify_margin (tmaid);
  for (p = tmaid->start; p->prev; p = p->prev);
  while (p)
    {
      p->margin = FALSE;
      p = p->next;
    }
  tmaid->ft.limit = limit;
  misc_mdi_set_edited (MISC_MDI (mdi),
                    misc_mdi_get_page_from_data (MISC_MDI (mdi), tmaid), TRUE);
  tmaid->cursor.x = tmaid->cursor.y = tmaid->top.x = tmaid->top.y = 0;
  tmaid->select.x = -1;
  misc_set_scroll_bar (tmaid->hscroll,
            G_CALLBACK (signal_value_changed_hscroll), tmaid,
            0, edit_get_width_max (tmaid) + 1,
            MAX (tmaid->drawing->allocation.width / tmaid->font_width, 1),
                                                                tmaid->top.x);
  misc_set_scroll_bar (tmaid->vscroll,
            G_CALLBACK (signal_value_changed_vscroll), tmaid,
            0, tmaid->max,
            MAX (tmaid->drawing->allocation.height / tmaid->font_height, 1),
                                                                tmaid->top.y);
  set_menu_bar (tmaid);
  gtk_widget_draw (tmaid->drawing, NULL);
}


/*  ja:タブをスペースに変換する
    tmaid,ウインドウ情報                                                    */
void
tab_operation (TmaidWindow *tmaid)
{
  gint data_pos, screen_pos;
  LineBuffer *p;

  delete_list (&tmaid->undo);
  delete_list (&tmaid->redo);
  for (p = tmaid->start; p->prev; p = p->prev);
  while (p)
    {
      data_pos = screen_pos = 0;
      while (data_pos < p->length)
        if (p->text[data_pos] == '\t')
          {
            gint space;

            space = (screen_pos / tmaid->ft.tab + 1) * tmaid->ft.tab
                                                                - screen_pos;
            if (space > 1)
              {
                p->text = g_realloc (p->text, (p->length + space - 1)
                                                            * sizeof (gchar));
                g_memmove (p->text + data_pos + space,
                                p->text + data_pos + 1,
                                (p->length - data_pos - 1) * sizeof (gchar));
                p->length += space - 1;
              }
            g_memset (p->text + data_pos, ' ', space * sizeof (gchar));
            screen_pos += space;
            data_pos += space;
          }
        else if (data_pos + charset_length (p->text[data_pos]) <= p->length)
          {
            screen_pos += charset_width (tmaid->layout, p->text + data_pos,
                                        tmaid->font_width, tmaid->font_buf);
            data_pos+=charset_length(p->text[data_pos]);
          }
        else
          {
            screen_pos++;
            data_pos++;
          }
      p = p->next;
    }
  misc_mdi_set_edited (MISC_MDI (mdi),
                    misc_mdi_get_page_from_data (MISC_MDI (mdi), tmaid), TRUE);
  tmaid->select.x = -1;
  set_menu_bar (tmaid);
  gtk_widget_draw (tmaid->drawing, NULL);
}
