/*
    Text maid
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <gdk/gdkkeysyms.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include "abort.h"
#include "charset.h"
#include "edit.h"
#include "print.h"
#ifndef G_OS_WIN32
# include "misc/fileio.h"
#endif /* not G_OS_WIN32 */
#include "misc/misc.h"
#ifdef G_OS_WIN32
# include "misc/misc_mdi.h"
# include <gdk/gdkwin32.h>
#endif /* G_OS_WIN32 */


PrintConfig printer;                        /* ja:印刷の設定 */


/******************************************************************************
*                                                                             *
* ja:印刷関数群                                                               *
*                                                                             *
******************************************************************************/
#ifdef G_OS_WIN32
BOOL CALLBACK
PrintProc (HDC hdc,
           int nError)
{
  return userbreak;
}
#endif /* G_OS_WIN32 */


/*  印刷
     tmaid,ウインドウ情報
    prncfg,印刷情報
       RET,TRUE:正常終了,FALSE:エラー                                       */
gboolean
print_out (TmaidWindow *tmaid,
           PrintConfig *prncfg)
{
  guchar *c;
  gint i, x, y, sx, sy, data_pos, end_pos, line, width, channels, rowstride;
  LineBuffer *p;
  GdkGC *gc;
  GdkPixmap *pixmap;
  GdkPixbuf *pixbuf;
  GdkPoint start,end;
  GtkWidget *dialog;
#ifdef G_OS_WIN32
  gchar *name, *utf8str;
  HBITMAP hBitmap;
  HDC hDC;
  DOCINFO di;
#else /* not G_OS_WIN32 */
  FILE *fp;
#endif /* not G_OS_WIN32 */

  sx = prncfg->column * prncfg->font_width;
#ifdef G_OS_WIN32
  sy = sx * prncfg->height / prncfg->width;
#else /* not G_OS_WIN32 */
  sy = sx * (prncfg->height - prncfg->left - prncfg->right)
                            / (prncfg->width - prncfg->top - prncfg->bottom);
#endif /* not G_OS_WIN32 */
  if (tmaid->select.x < 0 || !prncfg->select)
    {
      /* ja:選択範囲なし、すべて印刷 */
      start.x = start.y = 0;
      end.x = edit_get_width (tmaid,tmaid->max - 1);
      end.y = tmaid->max - 1;
    }
  else if (tmaid->select.y < tmaid->cursor.y
                                        || (tmaid->select.y == tmaid->cursor.y
                                        && tmaid->select.x < tmaid->cursor.x))
    {
      start = tmaid->select;
      end = tmaid->cursor;
    }
  else
    {
      start = tmaid->cursor;
      end = tmaid->select;
    }
#ifdef G_OS_WIN32
  hDC = CreateCompatibleDC (prncfg->pd.hDC);
  if (!hDC)
    {
      misc_message_box ("Text maid", "CreateCompatibleDC", 0, _("OK"), NULL);
      DeleteDC (prncfg->pd.hDC);
      return FALSE;
    }
  if (SetAbortProc (prncfg->pd.hDC, PrintProc) <= 0)
    {
      misc_message_box ("Text maid", "SetAbortProc", 0, _("OK"), NULL);
      DeleteDC (hDC);
      DeleteDC (prncfg->pd.hDC);
      return FALSE;
    }
  g_memset (&di, 0, sizeof (DOCINFO));
  di.cbSize = sizeof (DOCINFO);
  utf8str = g_filename_to_utf8 (misc_mdi_get_file (MISC_MDI (mdi),
                        misc_mdi_get_page_from_data (MISC_MDI (mdi), tmaid)),
                                                        -1, NULL, NULL, NULL);
# ifdef UNICODE
  name = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
# else /* not UNICODE */
  name = g_locale_from_utf8 (utf8str, -1, NULL, NULL, NULL);
# endif /* not UNICODE */
  g_free (utf8str);
  di.lpszDocName = name ? name : "Text maid";
  if (StartDoc (prncfg->pd.hDC, &di) <= 0)
    {
      misc_message_box ("Text maid", "StartDoc", 0, _("OK"), NULL);
      DeleteDC (hDC);
      DeleteDC (prncfg->pd.hDC);
      return FALSE;
    }
  g_free (name);
#else /* not G_OS_WIN32 */
# ifdef G_OS_UNIX
  if (prncfg->mode)
    {
      /* ja:プリンタ */
      fp = popen (prncfg->printer, "w");
      if (!fp)
        {
          misc_message_box ("Text maid", _("Can not open printer"), 0,
                                                                _("OK"), NULL);
          return FALSE;
        }
    }
  else
    {
# endif /* G_OS_UNIX */
      /* ja:ファイル */
      fp = fopen (prncfg->file, "wt");
      if (!fp)
        {
          misc_message_box ("Text maid", _("Can not open file"), 0,
                                                                _("OK"), NULL);
          return FALSE;
        }
# ifdef G_OS_UNIX
    }
# endif /* G_OS_UNIX */
  fputs ("%!PS-Adobe-3.0\n", fp);
#endif /* not G_OS_WIN32 */
  pixmap = gdk_pixmap_new (window->window, sx, sy, -1);
  pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, sx, sy);
  gc = gdk_gc_new (pixmap);
  gdk_gc_set_line_attributes (gc, 1,
                                GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER);
  for (i = 10; i < 12; i++)
    gdk_color_alloc (gdk_colormap_get_system (), (tmaid->ft.syscol
                                        ? system_color : tmaid->ft.color) + i);
  /* ja:ダイアログボックス */
  userbreak = TRUE;
  dialog = abort_dialog (_("Printing"));
  gtk_grab_add (dialog);
#if defined G_OS_UNIX
  for (i = 0; i < prncfg->mode ? 1 : prncfg->copies; i++)
#elif defined G_OS_WIN32
  for (i = 0; i < prncfg->copies; i++)
#endif
    {
      p = edit_get_line_buf (&tmaid->start, &tmaid->off, start.y);
      end_pos = edit_get_data_pos (tmaid,end.x, end.y, FALSE);
      line = start.y;
      data_pos = edit_get_data_pos (tmaid, start.x, start.y, FALSE);
      while (line <= end.y)
        {
          /* ja:ページスタート */
          /* ja:背景 */
          gdk_gc_set_foreground (gc, (tmaid->ft.syscol
                                    ? system_color : tmaid->ft.color) + 11);
          gdk_draw_rectangle (pixmap, gc, TRUE, 0, 0, sx, sy);
          gdk_gc_set_foreground (gc, (tmaid->ft.syscol
                                    ? system_color : tmaid->ft.color) + 10);
          x = y = 0;
          while (y <= sy - tmaid->font_height && line <= end.y)
            {
              while (gtk_events_pending ())
                gtk_main_iteration ();
              while (x < sx)
                {
                  if (data_pos >= (line == end.y ? end_pos : p->length))
                    {
                      /* ja:行の終わり */
                      data_pos = 0;
                      line++;
                      p = p->next;
                      if (!p || !p->margin)
                        break;
                    }
                  if (p->text[data_pos] == '\t')
                    width = (x / tmaid->font_width / tmaid->ft.tab + 1)
                                    * tmaid->ft.tab * tmaid->font_width - x;
                  else if (data_pos + charset_length (p->text[data_pos])
                                                                <= p->length)
                    width = tmaid->font_width * charset_width
                                        (tmaid->layout, p->text + data_pos,
                                        tmaid->font_width, tmaid->font_buf);
                  else
                    width = tmaid->font_width;
                  if (x > 0 && sx < x + width)
                    break;
                  if (p->text[data_pos] == '\t')
                    {
                      /* ja:タブ */
                      data_pos++;
                    }
                  else if (data_pos + charset_length (p->text[data_pos])
                                                                <= p->length)
                    {
                      /* ja:半角/全角 */
                      pango_layout_set_text (tmaid->layout,
                                        p->text + data_pos,
                                        charset_length (p->text[data_pos]));
                      gdk_draw_layout (pixmap, gc, x, y, tmaid->layout);
                      data_pos += charset_length (p->text[data_pos]);
                    }
                  else
                    {
                      /* ja:強制半角 */
                      pango_layout_set_text (tmaid->layout,
                                                        p->text + data_pos, 1);
                      gdk_draw_layout (pixmap, gc, x, y, tmaid->layout);
                      data_pos++;
                    }
                  x += width;
                }
              x = 0;
              y += tmaid->font_height;
            }
#ifdef G_OS_WIN32
          hBitmap = SelectObject (hDC, (HBITMAP)GDK_WINDOW_HWND (pixmap));
          if (StartPage (prncfg->pd.hDC) <= 0)
            {
              misc_message_box ("Text maid", "StartPage", 0, _("OK"), NULL);
              hBitmap = SelectObject (hDC, hBitmap);
              goto loop;
            }
          if (!StretchBlt (prncfg->pd.hDC, 0, 0, prncfg->width, prncfg->height,
                                                hDC, 0, 0, sx, sy, SRCCOPY))
            {
              misc_message_box ("Text maid", "StretchBlt", 0, _("OK"), NULL);
              EndPage (prncfg->pd.hDC);
              hBitmap = SelectObject (hDC, hBitmap);
              goto loop;
            }
          if (EndPage (prncfg->pd.hDC) <= 0)
            {
              misc_message_box ("Text maid", "EndPage", 0, _("OK"), NULL);
              hBitmap = SelectObject (hDC, hBitmap);
              goto loop;
            }
          hBitmap = SelectObject (hDC, hBitmap);
#else /* not G_OS_WIN32 */
          fprintf (fp,
                "gsave\n"
                "%d %d translate\n"
                "%d %d scale\n"
                "%d %d 8 [%d 0 0 %d 0 0]\n"
                "{currentfile %d string readhexstring pop}\n"
                "image\n",
                prncfg->left, prncfg->bottom,
                prncfg->width - prncfg->left - prncfg->right,
                prncfg->height - prncfg->top - prncfg->bottom,
                sx, sy, sx, sy, sx);
          gdk_pixbuf_get_from_drawable (pixbuf, pixmap,
                            gdk_colormap_get_system (), 0, 0, 0, 0, sx, sy);
          channels = gdk_pixbuf_get_n_channels (pixbuf);
          rowstride = gdk_pixbuf_get_rowstride (pixbuf);
          c = gdk_pixbuf_get_pixels (pixbuf);
          i = 0;
          switch (channels)
            {
              case 1:
                for (y = sy - 1; y >= 0; y--)
                  {
                    for (x = 0; x < sx; x++)
                      fprintf (fp, "%02x", c[x * channels + y * rowstride]);
                    fputc ('\n', fp);
                  }
                break;
              case 3:
              case 4:
                for (y = sy - 1;y >= 0; y--)
                  {
                    for (x = 0; x < sx; x++)
                      fprintf (fp, "%02x",
                                    (c[x * channels + y * rowstride] * 77
                                    +c[x * channels + y * rowstride + 1] * 150
                                    +c[x * channels + y * rowstride + 2] * 29)
                                                                        / 256);
                    fputc('\n',fp);
                  }
                break;
              default:
                for (y = 0; y < sy; y++)
                  {
                    for (x = 0; x < sx; x++)
                      fputs ("ff", fp);
                    fputc ('\n', fp);
                  }
            }
          fputs("grestore\nshowpage\n",fp);
#endif /* not G_OS_WIN32 */
          if (!userbreak)
            goto loop;
        }
    }
  loop:
  /* ja:終了処理 */
  gtk_grab_remove (dialog);
  gtk_widget_destroy (dialog);
  g_object_unref (gc);
  gdk_pixbuf_unref (pixbuf);
  gdk_pixmap_unref (pixmap);
#ifdef G_OS_WIN32
  if (EndDoc (prncfg->pd.hDC) <= 0)
    {
      misc_message_box ("Text maid", "EndDoc", 0, _("OK"), NULL);
      DeleteDC (hDC);
      DeleteDC (prncfg->pd.hDC);
      return FALSE;
    }
  if (!DeleteDC (prncfg->pd.hDC))
    {
      misc_message_box ("Text maid", "DeleteDC", 0, _("OK"), NULL);
      DeleteDC (hDC);
      return FALSE;
    }
  if (!DeleteDC (hDC))
    {
      misc_message_box ("Text maid", "DeleteDC", 0, _("OK"), NULL);
      return FALSE;
    }
#else /* not G_OS_WIN32 */
# ifdef G_OS_UNIX
  if (prncfg->mode) /* ja:プリンタ */
    pclose (fp);
  else              /* ja:ファイル */
# endif /* G_OS_UNIX */
    fclose (fp);
#endif /* not G_OS_WIN32 */
  return TRUE;
}


/******************************************************************************
*                                                                             *
* ja:印刷ダイアログ関数群                                                     *
*                                                                             *
******************************************************************************/
static gboolean print_clicked_ok;
static GtkWidget *dialog, *button1, *label1, *spin0;
#ifndef G_OS_WIN32
static GtkWidget *button0, *check, *entry1, *label3, *label4;
static GtkWidget *spin2, *spin3, *spin4, *spin5, *spin6, *spin7;
# ifdef G_OS_UNIX
static GtkWidget *entry0, *label2, *radio0, *spin1;
# endif /* G_OS_UNIX */
static struct
{
  gchar *name;
  const gint width, height;
} paper[] = { {"A3", 840, 1188}, {"A4", 594, 840}, {"A5", 420, 594},
              {"B4", 730, 1030}, {"B5", 515, 730}, {NULL, 0, 0}};
#endif /* not G_OS_WIN32 */


/* ja:OKボタンの表示/非表示を判定 */
static void
print_dialog_ok_sensitive (GtkWidget   *widget,
                           PrintConfig *prncfg)
{
  gint column, row;
#ifndef G_OS_WIN32
  gint width, height, left, top, right, bottom;
# ifdef G_OS_UNIX
  gboolean mode;
  gint copies;
# endif /* G_OS_UNIX */
#endif /* not G_OS_WIN32 */

  column = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin0));
#ifdef G_OS_WIN32
  row = column * prncfg->font_width * prncfg->height
                                    / (prncfg->width * prncfg->font_height);
  gtk_widget_set_sensitive (button1, column >= 1 && row >= 1);
#else /* not G_OS_WIN32 */
# ifdef G_OS_UNIX
  mode = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio0));
  copies = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin1));
# endif /* G_OS_UNIX */
  width = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin2));
  height = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin3));
  top = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin4));
  left = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin5));
  right = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin6));
  bottom = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin7));
  width -= left + right;
  height -= top + bottom;
  row = left >= 0 && top >= 0 && right >= 0 && bottom >= 0
                                                    && width > 0 && height > 0
                                    ? column * prncfg->font_width * height
                                        / (width * prncfg->font_height) : 0;
# ifdef G_OS_UNIX
  gtk_widget_set_sensitive (button1, column > 0 && row > 0
                                                        && (mode || copies > 0)
        && g_strlen (gtk_entry_get_text (GTK_ENTRY (mode ? entry0 : entry1)))
                                                                        > 0);
# else /* not G_OS_UNIX */
  gtk_widget_set_sensitive (button1, column > 0 && row > 0
                    && g_strlen (gtk_entry_get_text (GTK_ENTRY (entry1))) > 0);
# endif /* not G_OS_UNIX */
#endif /* not G_OS_WIN32 */
}


/* ja:スピンボタンが変化した */
static void
print_dialog_changed_spin (GtkWidget   *widget,
                           PrintConfig *prncfg)
{
  gchar *text;
  gint column, row;
#ifndef G_OS_WIN32
  gint width, height, left, top, right, bottom;
#endif /* not G_OS_WIN32 */

  column = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin0));
#ifdef G_OS_WIN32
  row = column * prncfg->font_width * prncfg->height
                                    / (prncfg->width * prncfg->font_height);
#else /* not G_OS_WIN32 */
  width = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin2));
  height = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin3));
  top = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin4));
  left = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin5));
  right = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin6));
  bottom = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin7));
  width -= left + right;
  height -= top + bottom;
  row = left >= 0 && top >= 0 && right >= 0 && bottom >= 0
                                                    && width > 0 && height > 0
                                    ? column * prncfg->font_width * height
                                        / (width * prncfg->font_height) : 0;
#endif /* not G_OS_WIN32 */
  text = g_strdup_printf (_("%d Lines"), MAX (row, 0));
  gtk_label_set_text (GTK_LABEL (label1), text);
  g_free (text);
  print_dialog_ok_sensitive (widget, prncfg);
}


#ifndef G_OS_WIN32
/* ja:コンボボックス */
static void
print_dialog_changed_combo (GtkWidget *widget,
                            gpointer   user_data)
{
# if GTK_CHECK_VERSION(2,4,0)
  gchar *name;
# else /* not GTK_CHECK_VERSION(2,4,0) */
  const gchar *name;
# endif /* not GTK_CHECK_VERSION(2,4,0) */
  gint i;

# if GTK_CHECK_VERSION(2,6,0)
  name = gtk_combo_box_get_active_text (GTK_COMBO_BOX (widget));
# elif GTK_CHECK_VERSION(2,4,0)
  GtkTreeIter iter;

  if (gtk_combo_box_get_active_iter (GTK_COMBO_BOX (widget), &iter))
    gtk_tree_model_get (gtk_combo_box_get_model (GTK_COMBO_BOX (widget)),
                                                        &iter, 0, &name, -1);
  else
    name = NULL;
# else /* not GTK_CHECK_VERSION(2,4,0) */
  name = gtk_entry_get_text (GTK_ENTRY (widget));
# endif /* not GTK_CHECK_VERSION(2,4,0) */
  for (i = 0; paper[i].name; i++)
    if (g_strcmp (paper[i].name, name) == 0)
      break;
# if GTK_CHECK_VERSION(2,4,0)
  g_free (name);
# endif /* GTK_CHECK_VERSION(2,4,0) */
  if (paper[i].name)
    {
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin2), paper[i].width);
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin3), paper[i].height);
      gtk_widget_set_sensitive (label3, FALSE);
      gtk_widget_set_sensitive (label4, FALSE);
      gtk_widget_set_sensitive (spin2, FALSE);
      gtk_widget_set_sensitive (spin3, FALSE);
    }
  else
    {
      /* ja:ユーザー設定 */
      gtk_widget_set_sensitive (label3, TRUE);
      gtk_widget_set_sensitive (label4, TRUE);
      gtk_widget_set_sensitive (spin2, TRUE);
      gtk_widget_set_sensitive (spin3, TRUE);
    }
}
#endif /* not G_OS_WIN32 */


/* ja:ラジオボタン */
#ifdef G_OS_UNIX
static void
print_dialog_toggled (GtkWidget   *widget,
                      PrintConfig *prncfg)
{
  gboolean result;

  result = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio0));
  gtk_widget_set_sensitive (entry0, result);
  gtk_widget_set_sensitive (entry1, !result);
  gtk_widget_set_sensitive (button0, !result);
# ifdef G_OS_UNIX
  gtk_widget_set_sensitive (label2, !result);
  gtk_widget_set_sensitive (spin1, !result);
# endif /* not G_OS_UNIX */
  print_dialog_ok_sensitive (widget, prncfg);
}
#endif /* G_OS_UNIX */


/* ja:OKボタンが押された */
static void
print_dialog_clicked_ok (GtkWidget   *widget,
                         PrintConfig *prncfg)
{
#ifndef G_OS_WIN32
# ifdef G_OS_UNIX
  prncfg->mode = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio0));
  if (prncfg->mode)
    {
      g_free (prncfg->printer);
      prncfg->printer = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry0)));
      prncfg->copies
                = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin1));
    }
  else
    {
# endif /* G_OS_UNIX */
      g_free (prncfg->file);
      prncfg->file = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry1)));
# ifdef G_OS_UNIX
    }
# endif /* G_OS_UNIX */
  prncfg->select = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check));
  prncfg->width = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin2));
  prncfg->height = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin3));
  prncfg->top = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin4));
  prncfg->left = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin5));
  prncfg->right = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin6));
  prncfg->bottom = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin7));
#endif /* not G_OS_WIN32 */
  prncfg->column = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin0));
  gtk_widget_destroy (dialog);
  print_clicked_ok = TRUE;
}


#ifndef G_OS_WIN32
# if ! GTK_CHECK_VERSION(2,4,0)
/* ja:ファイルの参照でOKボタンが押された */
static void
print_open_ok (GtkWidget *widget,
               GtkWidget *dialog_open)
{
  gchar **file;

  file = g_object_get_data (G_OBJECT (dialog_open), "user_data");
  *file = g_strdup (gtk_file_selection_get_filename
                                        (GTK_FILE_SELECTION (dialog_open)));
  gtk_widget_destroy (dialog_open);
}
# endif /* not GTK_CHECK_VERSION(2,4,0) */


/* ja:参照ボタンが押された */
static void
print_dialog_clicked_file (GtkWidget   *widget,
                           PrintConfig *prncfg)
{
  gchar *file = NULL;
  GtkWidget *dialog_open;

# if GTK_CHECK_VERSION(2,4,0)
  dialog_open = gtk_file_chooser_dialog_new (_("Save As"),
                        GTK_WINDOW (window), GTK_FILE_CHOOSER_ACTION_SAVE,
                        GTK_STOCK_SAVE_AS, GTK_RESPONSE_ACCEPT,
                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                        NULL);
  file = g_filename_from_utf8 (gtk_entry_get_text (GTK_ENTRY (entry1)),
                                                        -1, NULL, NULL, NULL);
  if (fileio_isfile (file))
    {
      gtk_file_chooser_set_filename (GTK_FILE_CHOOSER (dialog_open), file);
    }
  else
    {
      gchar *dir, *path;

      dir = g_path_get_dirname (file);
      path = fileio_get_full_path (dir);
      g_free (dir);
      if (fileio_isfile (path))
        gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog_open),
                                                                        path);
      g_free (path);
    }
  g_free (file);
  file = gtk_dialog_run (GTK_DIALOG (dialog_open)) == GTK_RESPONSE_ACCEPT
    ? gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog_open)) : NULL;
  gtk_widget_destroy (dialog_open);
# else /* not GTK_CHECK_VERSION(2,4,0) */
  dialog_open = gtk_file_selection_new (_("Save As"));
  g_signal_connect (G_OBJECT (dialog_open), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (dialog_open)->ok_button),
                        "clicked", G_CALLBACK (print_open_ok), dialog_open);
  g_signal_connect_swapped
        (G_OBJECT (GTK_FILE_SELECTION (dialog_open)->cancel_button),
        "clicked", G_CALLBACK (gtk_widget_destroy), G_OBJECT (dialog_open));
  gtk_file_selection_set_filename (GTK_FILE_SELECTION (dialog_open),
                                    gtk_entry_get_text (GTK_ENTRY (entry1)));
  g_object_set_data (G_OBJECT (dialog_open), "user_data", &file);
  gtk_widget_show (dialog_open);
  gtk_grab_add (dialog_open);
  gtk_main ();
# endif /* not GTK_CHECK_VERSION(2,4,0) */
  if (file)
    {
      gtk_entry_set_text (GTK_ENTRY (entry1), file);
      g_free (file);
    }
}
#endif /* not G_OS_WIN32 */


/* ja:エントリー/スピンボタンでリターンが押された */
static void
print_dialog_activate (GtkWidget   *widget,
                       PrintConfig *prncfg)
{
#ifdef G_OS_WIN32
  if (gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin0)) >= 1)
    print_dialog_clicked_ok (widget, prncfg);
#else /* not G_OS_WIN32 */
  gint left, top, right, bottom;
# ifdef G_OS_UNIX
  gboolean mode;
  gint copies;

  mode = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio0));
  copies = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin1));
# endif /* G_OS_UNIX */
  top = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin4));
  left = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin5));
  right = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin6));
  bottom = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin7));
  if (
# ifdef G_OS_UNIX
        g_strlen (gtk_entry_get_text (GTK_ENTRY (mode ? entry0 : entry1))) > 0
        && (mode || copies > 0)
# else /* not G_OS_UNIX */
        g_strlen (gtk_entry_get_text (GTK_ENTRY (entry1))) > 0
# endif /* not G_OS_UNIX */
        && gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin0)) >= 1
        && left >= 0 && top >= 0 && right >= 0 && bottom >= 0
        && gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin1))
                                                                > left + right
        && gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spin2))
                                                                > top + bottom)
    print_dialog_clicked_ok (widget, prncfg);
#endif /* not G_OS_WIN32 */
}


/* ja:ESCが押された */
static gboolean
print_dialog_key_press (GtkWidget   *widget,
                        GdkEventKey *event,
                        gpointer     user_data)
{
  if (event->keyval == GDK_Escape)
    gtk_widget_destroy (widget);
    return FALSE;
}


/*  ja:印刷ダイアログ表示
    prncfg,印刷情報
       RET,TRUE:OK,FALSE:Cancel                                             */
gboolean
print_dialog (PrintConfig *prncfg)
{
  GtkWidget *button2, *label0, *hbox0, *hbox1, *vbox;
#ifndef G_OS_WIN32
  gint i;
# if ! GTK_CHECK_VERSION(2,4,0)
  GList *glist = NULL;
# endif /* not GTK_CHECK_VERSION(2,4,0) */
  GtkWidget *combo, *frame0, *frame1, *label5, *label6, *label7, *label8;
# ifdef G_OS_UNIX
  GtkWidget *radio1;
# else /* not G_OS_UNIX */
  GtkWidget *label9;
# endif /* not G_OS_UNIX */
#endif /* not G_OS_WIN32 */

#ifdef G_OS_WIN32
  /* ja:プリンター */
  g_memset (&prncfg->pd, 0, sizeof (PRINTDLG));
  prncfg->pd.lStructSize = sizeof (PRINTDLG);
  prncfg->pd.hwndOwner = GDK_WINDOW_HWND (window->window);
  prncfg->pd.Flags = PD_HIDEPRINTTOFILE | PD_RETURNDC;
  if (prncfg->select)
    prncfg->pd.Flags |= PD_SELECTION;
  else
    prncfg->pd.Flags |= PD_NOSELECTION;
  prncfg->pd.nCopies = prncfg->copies;
  if (!PrintDlg (&prncfg->pd))
    return FALSE;
  prncfg->width = GetDeviceCaps (prncfg->pd.hDC,HORZRES);
  prncfg->height = GetDeviceCaps (prncfg->pd.hDC,VERTRES) * 24 / 25;
  if (prncfg->select)
    prncfg->select = (prncfg->pd.Flags & PD_SELECTION) != 0;
  prncfg->copies = prncfg->pd.nCopies;
#endif /* G_OS_WIN32 */

  /* ja:メインウインドウ */
  dialog = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (dialog), _("Print"));
  g_signal_connect_after (G_OBJECT (dialog), "key-press-event",
                                    G_CALLBACK (print_dialog_key_press), NULL);
  g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  /* ja:ボタン */
  button1 = gtk_button_new_with_label (_("Print"));
  button2 = gtk_button_new_with_label (_("Cancel"));
  g_signal_connect (G_OBJECT (button1), "clicked",
                                G_CALLBACK (print_dialog_clicked_ok), prncfg);
  g_signal_connect_swapped (G_OBJECT (button2), "clicked",
                        G_CALLBACK (gtk_widget_destroy), G_OBJECT (dialog));
  GTK_WIDGET_SET_FLAGS (button1, GTK_CAN_DEFAULT);
  GTK_WIDGET_SET_FLAGS (button2, GTK_CAN_DEFAULT);
  /* ja:スピンボタン */
  spin0 = gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new
                                (prncfg->column, 1, G_MAXINT, 1, 5, 0)), 0, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin0), TRUE);
  g_signal_connect (G_OBJECT (spin0), "activate",
                                G_CALLBACK (print_dialog_activate), prncfg);
  g_signal_connect (G_OBJECT (spin0), "changed",
                            G_CALLBACK (print_dialog_changed_spin), prncfg);
  /* ja:ラベル */
  label0 = gtk_label_new_with_mnemonic (_("_Columns"));
  label1 = gtk_label_new ("");
  gtk_label_set_mnemonic_widget (GTK_LABEL (label0), spin0);
#ifndef G_OS_WIN32
  /* ja:ボタン */
  button0 = gtk_button_new_with_label (_("Browse..."));
  g_signal_connect (G_OBJECT (button0), "clicked",
                            G_CALLBACK (print_dialog_clicked_file), dialog);
  GTK_WIDGET_SET_FLAGS (button0, GTK_CAN_DEFAULT);
  /* ja:チェックボタン */
  check = gtk_check_button_new_with_mnemonic (_("Print _Selection"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), prncfg->select);
  g_signal_connect (G_OBJECT (check), "toggled",
                            G_CALLBACK (print_dialog_ok_sensitive), prncfg);
  gtk_widget_set_sensitive (check, prncfg->select);
  /* ja:エントリー */
  entry1 = gtk_entry_new ();
  if (prncfg->file)
    gtk_entry_set_text (GTK_ENTRY (entry1), prncfg->file);
  g_signal_connect (G_OBJECT (entry1), "activate",
                            G_CALLBACK (print_dialog_clicked_ok), prncfg);
  g_signal_connect (G_OBJECT (entry1), "changed",
                            G_CALLBACK (print_dialog_ok_sensitive), prncfg);
  /* ja:スピンボタン */
  spin2 = gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new
                                (prncfg->width, 1, G_MAXINT, 1, 5, 0)), 0, 0);
  spin3 = gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new
                                (prncfg->height, 1, G_MAXINT, 1, 5, 0)), 0, 0);
  spin4 = gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new
                                (prncfg->top, 0, G_MAXINT, 1, 5, 0)), 0, 0);
  spin5 = gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new
                                (prncfg->left, 0, G_MAXINT, 1, 5, 0)), 0, 0);
  spin6 = gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new
                                (prncfg->right, 0, G_MAXINT, 1, 5, 0)), 0, 0);
  spin7 = gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new
                                (prncfg->bottom, 0, G_MAXINT, 1, 5, 0)), 0, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin2), TRUE);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin3), TRUE);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin4), TRUE);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin5), TRUE);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin6), TRUE);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin7), TRUE);
  g_signal_connect (G_OBJECT (spin2), "activate",
                                G_CALLBACK (print_dialog_activate), prncfg);
  g_signal_connect (G_OBJECT (spin3), "activate",
                                G_CALLBACK (print_dialog_activate), prncfg);
  g_signal_connect (G_OBJECT (spin4), "activate",
                                G_CALLBACK (print_dialog_activate), prncfg);
  g_signal_connect (G_OBJECT (spin5), "activate",
                                G_CALLBACK (print_dialog_activate), prncfg);
  g_signal_connect (G_OBJECT (spin6), "activate",
                                G_CALLBACK (print_dialog_activate), prncfg);
  g_signal_connect (G_OBJECT (spin7), "activate",
                                G_CALLBACK (print_dialog_activate), prncfg);
  g_signal_connect (G_OBJECT (spin2), "changed",
                            G_CALLBACK (print_dialog_changed_spin), prncfg);
  g_signal_connect (G_OBJECT (spin3), "changed",
                            G_CALLBACK (print_dialog_changed_spin), prncfg);
  g_signal_connect (G_OBJECT (spin4), "changed",
                            G_CALLBACK (print_dialog_changed_spin), prncfg);
  g_signal_connect (G_OBJECT (spin5), "changed",
                            G_CALLBACK (print_dialog_changed_spin), prncfg);
  g_signal_connect (G_OBJECT (spin6), "changed",
                            G_CALLBACK (print_dialog_changed_spin), prncfg);
  g_signal_connect (G_OBJECT (spin7), "changed",
                            G_CALLBACK (print_dialog_changed_spin), prncfg);
  /* ja:ラベル */
  label3 = gtk_label_new_with_mnemonic (_("_Width"));
  label4 = gtk_label_new_with_mnemonic (_("_Height"));
  label5 = gtk_label_new_with_mnemonic (_("_Top"));
  label6 = gtk_label_new_with_mnemonic (_("_Left"));
  label7 = gtk_label_new_with_mnemonic (_("_Right"));
  label8 = gtk_label_new_with_mnemonic (_("_Bottom"));
  gtk_label_set_mnemonic_widget (GTK_LABEL (label3), spin2);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label4), spin3);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label5), spin4);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label6), spin5);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label7), spin6);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label8), spin7);
# ifdef G_OS_UNIX
  /* ja:エントリー */
  entry0 = gtk_entry_new ();
  if (prncfg->printer)
    gtk_entry_set_text (GTK_ENTRY (entry0), prncfg->printer);
  g_signal_connect (G_OBJECT (entry0), "activate",
                            G_CALLBACK (print_dialog_clicked_ok), prncfg);
  g_signal_connect (G_OBJECT (entry0), "changed",
                            G_CALLBACK (print_dialog_ok_sensitive), prncfg);
  /* ja:スピンボタン */
  spin1 = gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new
                                (prncfg->copies, 1, G_MAXINT, 1, 5, 0)), 0, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin1), TRUE);
  g_signal_connect (G_OBJECT(spin1), "activate",
                                G_CALLBACK (print_dialog_activate), prncfg);
  g_signal_connect (G_OBJECT (spin1), "changed",
                            G_CALLBACK (print_dialog_changed_spin), prncfg);
  /* ja:ラベル */
  label2 = gtk_label_new_with_mnemonic (_("C_opies"));
  gtk_label_set_mnemonic_widget (GTK_LABEL (label2), spin1);
  /* ja:ラジオボタン */
  radio0 = gtk_radio_button_new_with_mnemonic (NULL, _("_Printer"));
  radio1 = gtk_radio_button_new_with_mnemonic_from_widget
                                    (GTK_RADIO_BUTTON (radio0), _("_File"));
  g_signal_connect (G_OBJECT (radio0), "toggled",
                                    G_CALLBACK (print_dialog_toggled), prncfg);
  g_signal_connect (G_OBJECT (radio1), "toggled",
                                    G_CALLBACK (print_dialog_toggled), prncfg);
  if (!prncfg->mode)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio1), TRUE);
# else /* not G_OS_UNIX */
  /* ja:ラベル */
  label9 = gtk_label_new_with_mnemonic (_("_File"));
  gtk_label_set_mnemonic_widget (GTK_LABEL (label9), entry1);
# endif /* not G_OS_UNIX */
  /* ja:コンボボックス */
# if GTK_CHECK_VERSION(2,4,0)
  combo = gtk_combo_box_new_text ();
  for (i = 0; paper[i].name; i++)
    gtk_combo_box_append_text (GTK_COMBO_BOX (combo), paper[i].name);
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo), _("User Define"));
  g_signal_connect (G_OBJECT (combo), "changed",
                                G_CALLBACK (print_dialog_changed_combo), NULL);
# else /* not GTK_CHECK_VERSION(2,4,0) */
  for (i = 0; paper[i].name; i++)
    glist = g_list_append (glist, paper[i].name);
  glist = g_list_append (glist, _("User Define"));
  combo = gtk_combo_new ();
  gtk_combo_set_popdown_strings (GTK_COMBO (combo), glist);
  gtk_combo_set_value_in_list (GTK_COMBO (combo), TRUE, FALSE);
  gtk_editable_set_editable (GTK_EDITABLE (GTK_COMBO (combo)->entry), FALSE);
  g_signal_connect (G_OBJECT (GTK_COMBO (combo)->entry), "changed",
                                G_CALLBACK (print_dialog_changed_combo), NULL);
  g_list_free (glist);
# endif /* not GTK_CHECK_VERSION(2,4,0) */
  for (i = 0; paper[i].name; i++)
    if (paper[i].width == prncfg->width && paper[i].height == prncfg->height)
      break;
# if GTK_CHECK_VERSION(2,4,0)
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo), i);
# else /* not GTK_CHECK_VERSION(2,4,0) */
  if (paper[i].name)
    gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combo)->entry), paper[i].name);
  else
    gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combo)->entry),
                                                            _("User Define"));
# endif /* not GTK_CHECK_VERSION(2,4,0) */

  /* ja:フレームとボックス */
  frame0 = gtk_frame_new (_("Paper"));
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
  gtk_box_pack_start (GTK_BOX (vbox), combo, TRUE, TRUE, 0);
  hbox0 = gtk_hbox_new (FALSE, SPACING);
  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), label3, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), spin2, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), hbox1, TRUE, TRUE, 0);
  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), label4, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), spin3, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), hbox1, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (frame0), vbox);

  frame1 = gtk_frame_new (_("Margin"));
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
  hbox0 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), label5, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), spin4, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
  hbox0 = gtk_hbox_new (FALSE, SPACING);
  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), label6, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), spin5, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), hbox1, TRUE, TRUE, 0);
  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), label7, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), spin6, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), hbox1, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
  hbox0 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), label8, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), spin7, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (frame1), vbox);
#endif /* not G_OS_WIN32 */

  /* ja:フレームとボックス */
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
  hbox0 = gtk_hbox_new (FALSE, SPACING);
  hbox1 = gtk_hbox_new (FALSE,0);
  gtk_box_pack_start (GTK_BOX (hbox1), spin0, TRUE, TRUE, 0);
  gtk_box_pack_end (GTK_BOX (hbox1), label0, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), hbox1, TRUE, TRUE, 0);
  gtk_box_pack_end (GTK_BOX (hbox0), label1, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);

#ifndef G_OS_WIN32
  gtk_box_pack_start (GTK_BOX (vbox), check, FALSE, FALSE, 0);

# ifdef G_OS_UNIX
  hbox0 = gtk_hbox_new (FALSE, SPACING);
  gtk_box_pack_start (GTK_BOX (hbox0), radio0, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), entry0, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
# endif /* G_OS_UNIX */

  hbox0 = gtk_hbox_new (FALSE, SPACING);
# ifdef G_OS_UNIX
  gtk_box_pack_start (GTK_BOX (hbox0), radio1, FALSE, FALSE, 0);
# else /* not G_OS_UNIX */
  gtk_box_pack_start (GTK_BOX (hbox0), label9, FALSE, FALSE, 0);
# endif /* not G_OS_UNIX */
  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), entry1, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), button0, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), hbox1, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);

# ifdef G_OS_UNIX
  hbox0 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), label2, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), spin1, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
# endif /* G_OS_UNIX */

  gtk_box_pack_start (GTK_BOX (vbox), frame0, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), frame1, FALSE, FALSE, 0);
#endif /* not G_OS_WIN32 */

  hbox0 = gtk_hbox_new (FALSE, SPACING);
  gtk_box_pack_end (GTK_BOX (hbox0), button2, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox0), button1, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);

  gtk_container_add (GTK_CONTAINER (dialog), vbox);

  /* ja:表示 */
#ifdef G_OS_UNIX
  gtk_widget_set_sensitive (entry0, prncfg->mode);
  gtk_widget_set_sensitive (entry1, !prncfg->mode);
  gtk_widget_set_sensitive (button0, !prncfg->mode);
#endif /* G_OS_UNIX */
  gtk_widget_grab_focus (button1);
  print_dialog_changed_spin (NULL, prncfg);
  print_clicked_ok = FALSE;

  gtk_window_set_policy (GTK_WINDOW (dialog), FALSE, TRUE, FALSE);
  gtk_grab_add (dialog);
  gtk_widget_show_all (dialog);
  gtk_main ();
#ifdef G_OS_WIN32
  if (!print_clicked_ok)
    DeleteDC (prncfg->pd.hDC);
#endif /* G_OS_WIN32 */
  return print_clicked_ok;
}
