/*
    Text maid
    copyright (c) 1998-2008 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "charset.h"
#include "edit.h"
#include "ftype.h"
#include "general.h"
#include "prop.h"
#include "sigfile.h"
#include "misc/misc.h"
#include "orz/orzcharlist.h"
#include "orz/orzcharset.h"


/******************************************************************************
*                                                                             *
* ja:プロパティ関数群                                                         *
*                                                                             *
******************************************************************************/
/*  ja:プロパティをコピーする
    ft_dst,コピー先
    ft_src,コピー元                                                         */
void
property_copy (FileType *ft_dst,
               FileType *ft_src)
{
  gboolean success[12];

  ft_dst->ft_id = ft_src->ft_id;
  if (ft_dst->crlf < 3 && ft_src->crlf < 3)
    ft_dst->crlf = ft_src->crlf;
  ft_dst->margin = ft_src->margin;
  ft_dst->tab = ft_src->tab;
  ft_dst->writemode = ft_src->writemode;
  ft_dst->autoindent = ft_src->autoindent;
  ft_dst->code = ft_src->code;
  ft_dst->eof = ft_src->eof;
  ft_dst->hexcase = ft_src->hexcase;
  ft_dst->html = ft_src->html;
  ft_dst->esc = ft_src->esc;
  ft_dst->limit = ft_src->limit;
  ft_dst->overwrite = ft_src->overwrite;
  ft_dst->space = ft_src->space;
  ft_dst->syscol = ft_src->syscol;
  ft_dst->tabconv = ft_src->tabconv;
  ft_dst->gline = ft_src->gline;
  ft_dst->mline = ft_src->mline;
  ft_dst->nline = ft_src->nline;
  ft_dst->uline = ft_src->uline;
  ft_dst->vline = ft_src->vline;
  g_free (ft_dst->font_name);
  ft_dst->font_name = ft_src->font_name ? g_strdup (ft_src->font_name) : NULL;
  g_memmove (ft_dst->color, ft_src->color, sizeof (GdkColor) * 12);
  gdk_colormap_alloc_colors (gdk_colormap_get_system (),
                                    ft_dst->color, 12, FALSE, TRUE, success);
}


/*  ja:プロパティを変更する
    tmaid,ウインドウ情報
       ft,ファイルタイプ                                                    */
void
property_change (TmaidWindow *tmaid,
                 FileType    *ft)
{
  gboolean modify;
  gint max, sx, sy;

  if (tmaid->ft.limit != ft->limit || tmaid->ft.tab != ft->tab
                        || (tmaid->ft.code && tmaid->ft.margin != ft->margin))
    {
      /* ja:折り返しの設定変更、タブ数変更 */
      delete_list (&tmaid->undo);
      delete_list (&tmaid->redo);
    }
  modify = tmaid->ft.limit != ft->limit || (tmaid->ft.limit
                && (tmaid->ft.tab != ft->tab || tmaid->ft.margin != ft->margin
                            || g_strcmp (tmaid->ft.font_name, ft->font_name)));
  property_copy (&tmaid->ft, ft);
  tmaid->select.x = -1;
  /* ja:フォント */
  g_object_unref (tmaid->layout);
  if (tmaid->ft.font_name)
    {
      PangoFontDescription *desc;

      desc = pango_font_description_from_string (tmaid->ft.font_name);
      gtk_widget_modify_font (tmaid->drawing, desc);
      pango_font_description_free (desc);
    }
  else
    {
      gtk_widget_modify_font (tmaid->drawing, system_font);
    }
  tmaid->layout = gtk_widget_create_pango_layout (tmaid->drawing, NULL);
  charset_size (tmaid->layout, &tmaid->font_width, &tmaid->font_height,
                                                            tmaid->font_buf);
  if (modify)
    {
      /* ja:折り返しの設定変更、
                            折り返しが有効でタブ数や右マージンやフォント変更 */
      tmaid->cursor.x = tmaid->cursor.y = tmaid->top.x = tmaid->top.y = 0;
      modify_margin (tmaid);
    }
  /* ja:スクロールバー */
  sx = MAX (tmaid->drawing->allocation.width / tmaid->font_width, 1);
  sy = MAX (tmaid->drawing->allocation.height / tmaid->font_height, 1);
  if (tmaid->top.y > tmaid->max - sy)
    tmaid->top.y = MAX (tmaid->max - sy, 0);
  max = edit_get_width_max (tmaid);
  if (tmaid->top.x > max - sx + 1)
    tmaid->top.x = MAX (max - sx + 1, 0);
  misc_set_scroll_bar (tmaid->hscroll,
                            G_CALLBACK (signal_value_changed_hscroll), tmaid,
                                            0, max + 1, sx, tmaid->top.x);
  misc_set_scroll_bar (tmaid->vscroll,
                            G_CALLBACK (signal_value_changed_vscroll), tmaid,
                                            0, tmaid->max, sy, tmaid->top.y);
  /* ja:再描画 */
  gtk_widget_queue_draw (tmaid->drawing);
  draw_caret (tmaid, NULL);
}


/******************************************************************************
*                                                                             *
* ja:プロパティダイアログ関数群                                               *
*                                                                             *
******************************************************************************/
typedef struct _PropertyDialog
{
  gchar *charset_list;
#if ! GTK_CHECK_VERSION(2,4,0)
  gchar *font_name;
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  gint ft_id;
  GdkColor gdk_color[12];
  GtkWidget *dialog;
  GtkWidget *button_color;
#if GTK_CHECK_VERSION(2,4,0)
  GtkWidget *button_font;
#endif /* GTK_CHECK_VERSION(2,4,0) */
  GtkWidget *charset;
  GtkWidget *check_gline;     /* ja:TRUE:グリッド線表示 */
  GtkWidget *check_space;     /* ja:TRUE:スペースを灰色の四角で表示 */
  GtkWidget *check_nline;     /* ja:TRUE:改行コードを表示する */
  GtkWidget *check_mline;     /* ja:TRUE:右マージンを表示 */
  GtkWidget *check_limit;     /* ja:TRUE:右マージンで折り返す */
  GtkWidget *check_code;      /* ja:TRUE:タブのコードを表示する */
  GtkWidget *check_uline;     /* ja:TRUE:タブ下線表示 */
  GtkWidget *check_vline;     /* ja:TRUE:タブ縦線表示 */
  GtkWidget *check_autoindent;/* ja:TRUE:自動字下げ */
  GtkWidget *check_tabconv;   /* ja:TRUE:タブをスペースに変換する */
  GtkWidget *check_eof;       /* ja:TRUE:1Ahをファイルの末尾に書き加える */
  GtkWidget *check_overwrite; /* ja:TRUE:ファイルを上書きするときに確認する */
  GtkWidget *check_syscol;    /* ja:TRUE:システム色を使う */
  GtkWidget *check_html;      /* ja:TRUE:HTML数値文字参照を変換する */
  GtkWidget *check_esc;       /* ja:TRUE:エスケープシーケンスを変換する */
  GtkWidget *check_associate; /* ja:TRUE:関連づける */
  GtkWidget *combo;
  GtkWidget *entry_text;
  GtkWidget *entry_ext;
  GtkWidget *label;
  GtkWidget *radio_abort;
  GtkWidget *radio_question;
  GtkWidget *radio_decimal;
  GtkWidget *radio_hex;
  GtkWidget *radio_escape;
  GtkWidget *radio_upper;
  GtkWidget *radio_lower;
  GtkWidget *spin_margin;
  GtkWidget *spin_tab;
  GtkWidget *tview;
} PropertyDialog;


/* ja:OKボタンの表示/非表示を判定 */
static void
property_dialog_ok_sensitive (GtkWidget      *widget,
                              PropertyDialog *propdlg)
{
  gtk_dialog_set_response_sensitive
                                (GTK_DIALOG (propdlg->dialog), GTK_RESPONSE_OK,
    (!propdlg->entry_text
        || g_strlen (gtk_entry_get_text (GTK_ENTRY (propdlg->entry_text))) > 0)
    && (!propdlg->entry_ext
        || g_strlen (gtk_entry_get_text (GTK_ENTRY (propdlg->entry_ext))) > 0)
    && (!propdlg->charset
        || orz_charset_is_valid (ORZ_CHARSET (propdlg->charset))));
}


/* ja:システム色 */
static void
property_dialog_toggled_check (GtkWidget      *widget,
                               PropertyDialog *propdlg)
{
  gtk_tree_selection_unselect_all
                (gtk_tree_view_get_selection (GTK_TREE_VIEW (propdlg->tview)));
  gtk_widget_set_sensitive (propdlg->tview,
    !gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (propdlg->check_syscol)));
  gtk_widget_set_sensitive (propdlg->button_color, FALSE);
  property_dialog_ok_sensitive (widget, propdlg);
}


/* ja:ラジオボタン */
static void
property_dialog_toggled_radio (GtkWidget      *widget,
                               PropertyDialog *propdlg)
{
  gboolean result;

  result = gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg->radio_hex))
        || gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg->radio_escape));
  gtk_widget_set_sensitive (propdlg->radio_upper, result);
  gtk_widget_set_sensitive (propdlg->radio_lower, result);
  property_dialog_ok_sensitive (widget, propdlg);
}


/* ja:リストボックスの選択 */
static void
property_dialog_changed (GtkTreeSelection *select,
                         PropertyDialog   *propdlg)
{
  gint i, row;

  row = -1;
  for (i = 0; i < 12; i++)
    {
      GtkTreeIter iter;

      if (gtk_tree_model_iter_nth_child (gtk_tree_view_get_model
                            (GTK_TREE_VIEW (propdlg->tview)), &iter, NULL, i)
                        && gtk_tree_selection_iter_is_selected (select, &iter))
        {
          row = i;
          break;
        }
    }
  gtk_widget_set_sensitive (propdlg->button_color, row >= 0);
}


/* ja:色の設定ボタンが押された */
static void
property_dialog_clicked_color (GtkWidget      *widget,
                               PropertyDialog *propdlg)
{
  gint i;
  GtkListStore *store;
  GtkTreeIter iter;
  GtkWidget *dialog;

  store = GTK_LIST_STORE (gtk_tree_view_get_model
                                            (GTK_TREE_VIEW (propdlg->tview)));
  dialog = gtk_color_selection_dialog_new (_("Color Configuration"));
  gtk_window_set_transient_for (GTK_WINDOW (dialog),
                                                GTK_WINDOW (propdlg->dialog));
  for (i = 0; i < 12; i++)
    if (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store), &iter, NULL, i)
            && gtk_tree_selection_iter_is_selected
                (gtk_tree_view_get_selection (GTK_TREE_VIEW (propdlg->tview)),
                                                                        &iter))
      {
        gtk_color_selection_set_current_color (GTK_COLOR_SELECTION
                            (GTK_COLOR_SELECTION_DIALOG (dialog)->colorsel),
                                                    propdlg->gdk_color + i);
        break;
      }
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    {
      gchar *xpm[18];
      GdkColor color;

      gtk_color_selection_get_current_color (GTK_COLOR_SELECTION
                    (GTK_COLOR_SELECTION_DIALOG (dialog)->colorsel), &color);
      xpm[0] = "24 16 1 1";
      for (i = 2; i < 18; i++)
        xpm[i] = "XXXXXXXXXXXXXXXXXXXXXXXX";
      for (i = 0; i < 12; i++)
        if (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                                &iter, NULL, i)
            && gtk_tree_selection_iter_is_selected
                (gtk_tree_view_get_selection (GTK_TREE_VIEW (propdlg->tview)),
                                                                        &iter))
          {
            propdlg->gdk_color[i].red = color.red;
            propdlg->gdk_color[i].green = color.green;
            propdlg->gdk_color[i].blue = color.blue;
            xpm[1] = g_strdup_printf ("X\tc #%02X%02X%02X",
                                            propdlg->gdk_color[i].red / 256,
                                            propdlg->gdk_color[i].green / 256,
                                            propdlg->gdk_color[i].blue / 256);
            gtk_list_store_set (store, &iter,
                    0, gdk_pixbuf_new_from_xpm_data ((const char **)xpm), -1);
            g_free (xpm[1]);
          }
      property_dialog_ok_sensitive (widget, propdlg);
    }
  gtk_widget_destroy (dialog);
}


/* ja:文字符号化方式の設定ボタンが押された */
static void
property_dialog_clicked_charlist (GtkWidget      *widget,
                                  PropertyDialog *propdlg)
{
  gint i;
  GtkWidget *dialog;

  dialog = orz_charlist_new ();
  gtk_window_set_transient_for (GTK_WINDOW (dialog),
                                                GTK_WINDOW (propdlg->dialog));
  orz_charlist_set_charlist (ORZ_CHARLIST (dialog), propdlg->charset_list);
  for (i = 0; i < ftnum; i++)
    orz_charlist_add_candidate (ORZ_CHARLIST (dialog), ftype[i].charset);
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    {
      g_free (propdlg->charset_list);
      propdlg->charset_list
                        = orz_charlist_get_charlist (ORZ_CHARLIST (dialog));
      property_dialog_ok_sensitive (widget, propdlg);
    }
  gtk_widget_destroy (dialog);
}


#if ! GTK_CHECK_VERSION(2,4,0)
/* ja:フォントボタンが押された */
static void
property_dialog_clicked_font (GtkWidget      *widget,
                              PropertyDialog *propdlg)
{
  GtkWidget *dialog;

  dialog = gtk_font_selection_dialog_new (_("Font"));
  gtk_window_set_transient_for (GTK_WINDOW (dialog),
                                                GTK_WINDOW (propdlg->dialog));
  if (propdlg->font_name)
    gtk_font_selection_dialog_set_font_name
        (GTK_FONT_SELECTION_DIALOG (dialog), propdlg->font_name);
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    {
      gchar *name;

      name = gtk_font_selection_dialog_get_font_name
                                        (GTK_FONT_SELECTION_DIALOG (dialog));
      if (name && name[0] != '\0')
        {
          g_free (propdlg->font_name);
          propdlg->font_name = name;
          property_dialog_ok_sensitive (widget, propdlg);
        }
      else
        {
          g_free (name);
        }
    }
  gtk_widget_destroy (dialog);
}
#endif /* not GTK_CHECK_VERSION(2,4,0) */


/* ja:ファイルタイプボタンが押された */
static void
property_dialog_clicked_ftype (GtkWidget      *widget,
                               PropertyDialog *propdlg)
{
  gint pos;

  pos = ftype_dialog (GTK_WINDOW (propdlg->dialog), TRUE);
  if (0 <= pos && pos < ftnum)
    {
      gchar *xpm[18];
      gint i;
      GtkListStore *store;
      GtkTreeIter iter;

      store = GTK_LIST_STORE (gtk_tree_view_get_model
                                            (GTK_TREE_VIEW (propdlg->tview)));
      propdlg->ft_id = ftype[pos].ft_id;
      gtk_label_set_text (GTK_LABEL (propdlg->label), ftype[pos].text);
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg->check_gline),
                                                        ftype[pos].gline);
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg->check_space),
                                                        ftype[pos].space);
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg->check_nline),
                                                        ftype[pos].nline);
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg->check_mline),
                                                        ftype[pos].mline);
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg->check_limit),
                                                        ftype[pos].limit);
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg->check_code),
                                                        ftype[pos].code);
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg->check_uline),
                                                        ftype[pos].uline);
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg->check_vline),
                                                        ftype[pos].vline);
      gtk_toggle_button_set_active
                                (GTK_TOGGLE_BUTTON (propdlg->check_autoindent),
                                                        ftype[pos].autoindent);
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg->check_tabconv),
                                                        ftype[pos].tabconv);
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg->check_eof),
                                                        ftype[pos].eof);
      gtk_toggle_button_set_active
                                (GTK_TOGGLE_BUTTON (propdlg->check_overwrite),
                                                        ftype[pos].overwrite);
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg->check_syscol),
                                                        ftype[pos].syscol);
      switch (ftype[pos].writemode)
        {
          case  1: gtk_toggle_button_set_active
                    (GTK_TOGGLE_BUTTON (propdlg->radio_question), TRUE); break;
          case  2: gtk_toggle_button_set_active
                    (GTK_TOGGLE_BUTTON (propdlg->radio_decimal), TRUE); break;
          case  3: gtk_toggle_button_set_active
                    (GTK_TOGGLE_BUTTON (propdlg->radio_hex), TRUE); break;
          case  4: gtk_toggle_button_set_active
                    (GTK_TOGGLE_BUTTON (propdlg->radio_escape), TRUE); break;
          default: gtk_toggle_button_set_active
                    (GTK_TOGGLE_BUTTON (propdlg->radio_abort), TRUE);
        }
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ftype[pos].hexcase
                        ? propdlg->radio_upper : propdlg->radio_lower), TRUE);
      if (ftype[pos].crlf < 3)
#if GTK_CHECK_VERSION(2,4,0)
        gtk_combo_box_set_active (GTK_COMBO_BOX (propdlg->combo),
                                                            ftype[pos].crlf);
#else /* not GTK_CHECK_VERSION(2,4,0) */
        gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (propdlg->combo)->entry),
                    gtk_label_get_text (GTK_LABEL (gtk_bin_get_child (GTK_BIN
                    (g_list_nth_data (gtk_container_get_children
                        (GTK_CONTAINER (GTK_COMBO (propdlg->combo)->list)),
                                                        ftype[pos].crlf))))));
#endif /* not GTK_CHECK_VERSION(2,4,0) */
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (propdlg->spin_margin),
                                                            ftype[pos].margin);
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (propdlg->spin_tab),
                                                            ftype[pos].tab);
      g_memmove (propdlg->gdk_color, ftype[pos].color, sizeof (GdkColor) * 12);
      xpm[0] = "24 16 1 1";
      for (i = 2; i < 18; i++)
        xpm[i] = "XXXXXXXXXXXXXXXXXXXXXXXX";
      for (i = 0; i < 12; i++)
        if (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                            &iter, NULL, i))
          {
            xpm[1] = g_strdup_printf ("X\tc #%02X%02X%02X",
                                            propdlg->gdk_color[i].red / 256,
                                            propdlg->gdk_color[i].green / 256,
                                            propdlg->gdk_color[i].blue / 256);
            gtk_list_store_set (store, &iter,
                    0, gdk_pixbuf_new_from_xpm_data ((const char **)xpm), -1);
            g_free (xpm[1]);
          }
#if GTK_CHECK_VERSION(2,4,0)
      gtk_font_button_set_font_name (GTK_FONT_BUTTON (propdlg->button_font),
                                                        ftype[pos].font_name);
#else /* not GTK_CHECK_VERSION(2,4,0) */
      g_free (propdlg->font_name);
      propdlg->font_name = g_strdup (ftype[pos].font_name);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
      gtk_dialog_set_response_sensitive (GTK_DIALOG (propdlg->dialog),
                                                        GTK_RESPONSE_OK, TRUE);
    }
  else if (pos >= ftnum)
    {
      propdlg->ft_id = -1;
      gtk_label_set_text (GTK_LABEL (propdlg->label),
                                                _("Independent File Type"));
      property_dialog_ok_sensitive (widget, propdlg);
    }
}


/* ja:スピンボタン/コンボボックスでリターンが押された */
static void
property_dialog_activate (GtkWidget      *widget,
                          PropertyDialog *propdlg)
{
  if ((!propdlg->entry_text
        || g_strlen (gtk_entry_get_text (GTK_ENTRY (propdlg->entry_text))) > 0)
    && (!propdlg->entry_ext
        || g_strlen (gtk_entry_get_text (GTK_ENTRY (propdlg->entry_ext))) > 0)
    && (!propdlg->charset
        || orz_charset_is_valid (ORZ_CHARSET (propdlg->charset))))
    gtk_dialog_response (GTK_DIALOG (propdlg->dialog), GTK_RESPONSE_OK);
}


/*  ja:プロパティダイアログ表示
    parent,親ウインドウ
        ft,ファイルタイプ
       RET,TRUE:OK,FALSE:Cancel                                             */
gboolean
property_dialog (GtkWindow *parent,
                 FileType  *ft)
{
  gboolean result;
  gchar *title, *xpm[18];
  gint i;
#if ! GTK_CHECK_VERSION(2,4,0)
  GList *glist;
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  GtkStockItem stock_item;
  GtkCellRenderer *renderer;
  GtkListStore *store;
  GtkTreeIter iter;
  GtkTreeSelection *select;
  GtkTreeViewColumn *column;
  GtkWidget *button_associated, *button_auto, *scroll, *label;
  GtkWidget *hbox, *tbox, *vbox;
  GtkWidget *frame0, *frame1, *frame2, *frame3, *frame4, *frame5, *frame6;
#if ! GTK_CHECK_VERSION(2,4,0)
  GtkWidget *button_font;
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  gchar *color_name[12] = {
            N_("Foreground"), N_("Background"), N_("Control Code"),
            N_("Full Width Space"), N_("Return Code"), N_("Tab Under Line"),
            N_("Right Margin"), N_("Grid"), N_("Selected Foreground"),
            N_("Selected Background"), N_("Foreground(Print)"),
            N_("Background(Print)")};
  PropertyDialog propdlg;

  /* ja:メインウインドウ */
  if (!ft->text || !ft->ext)
    {
      gtk_stock_lookup (GTK_STOCK_PROPERTIES, &stock_item);
      title = misc_mnemonic_to_text (stock_item.label);
    }
  else
    {
      title = NULL;
    }
  propdlg.dialog = gtk_dialog_new_with_buttons
                            (title ? title : _("Details of File Type"),
                            parent,
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
#ifdef G_OS_WIN32
                            GTK_STOCK_OK, GTK_RESPONSE_OK,
                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
#else /* not G_OS_WIN32 */
                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                            GTK_STOCK_OK, GTK_RESPONSE_OK,
#endif /* not G_OS_WIN32 */
                            NULL);
  g_free (title);
  g_signal_connect (G_OBJECT (propdlg.dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
  /* ja:ボタン */
  propdlg.button_color = gtk_button_new_with_label (_("Palette"));
#if GTK_CHECK_VERSION(2,4,0)
  propdlg.button_font = gtk_font_button_new_with_font (ft->font_name);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  button_font = gtk_button_new_with_mnemonic (_("_Font"));
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  g_signal_connect (G_OBJECT (propdlg.button_color), "clicked",
                        G_CALLBACK (property_dialog_clicked_color), &propdlg);
#if GTK_CHECK_VERSION(2,4,0)
  g_signal_connect (G_OBJECT (propdlg.button_font), "font-set",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  g_signal_connect (G_OBJECT (button_font), "clicked",
                        G_CALLBACK (property_dialog_clicked_font), &propdlg);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  if (ft->text && ft->ext)
    {
      button_auto = gtk_button_new_with_label (_("Auto Negotiation"));
      g_signal_connect (G_OBJECT (button_auto), "clicked",
                    G_CALLBACK (property_dialog_clicked_charlist), &propdlg);
    }
  else
    {
      button_auto = NULL;
    }
  if (!ft->text && !ft->ext)
    {
      button_associated
                = gtk_button_new_with_mnemonic (_("_Associated File Type"));
      g_signal_connect (G_OBJECT (button_associated), "clicked",
                        G_CALLBACK (property_dialog_clicked_ftype), &propdlg);
    }
  else
    {
      button_associated = NULL;
    }
  /* ja:チェックボタン */
  propdlg.check_gline = gtk_check_button_new_with_label (_("Grid"));
  propdlg.check_space = gtk_check_button_new_with_label
                                                    (_("Full Width Space"));
  propdlg.check_nline = gtk_check_button_new_with_label (_("Return Code"));
  propdlg.check_mline = gtk_check_button_new_with_label
                                                    (_("Show Right Margin"));
  propdlg.check_limit = gtk_check_button_new_with_label (_("Turn"));
  propdlg.check_code = gtk_check_button_new_with_label (_("Control Mark"));
  propdlg.check_uline = gtk_check_button_new_with_label (_("Under Line"));
  propdlg.check_vline = gtk_check_button_new_with_label (_("Vertical Line"));
  propdlg.check_autoindent = gtk_check_button_new_with_label
                                                            (_("Auto Indent"));
  propdlg.check_tabconv = gtk_check_button_new_with_label
                                                (_("Convert Tab to Space"));
  propdlg.check_eof = gtk_check_button_new_with_label (_("Append ^Z at EOF"));
  propdlg.check_overwrite = gtk_check_button_new_with_label
                                            (_("Over Write with Agreement"));
  propdlg.check_syscol = gtk_check_button_new_with_label
                                                    (_("Use System Color"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg.check_gline),
                                                            ft->gline);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg.check_space),
                                                            ft->space);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg.check_nline),
                                                            ft->nline);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg.check_mline),
                                                            ft->mline);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg.check_limit),
                                                            ft->limit);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg.check_code),
                                                            ft->code);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg.check_uline),
                                                            ft->uline);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg.check_vline),
                                                            ft->vline);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg.check_autoindent),
                                                            ft->autoindent);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg.check_tabconv),
                                                            ft->tabconv);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg.check_eof),
                                                            ft->eof);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg.check_overwrite),
                                                            ft->overwrite);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (propdlg.check_syscol),
                                                            ft->syscol);
  g_signal_connect (G_OBJECT (propdlg.check_gline), "toggled",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
  g_signal_connect (G_OBJECT (propdlg.check_space), "toggled",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
  g_signal_connect (G_OBJECT (propdlg.check_nline), "toggled",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
  g_signal_connect (G_OBJECT (propdlg.check_mline), "toggled",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
  g_signal_connect (G_OBJECT (propdlg.check_limit), "toggled",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
  g_signal_connect (G_OBJECT (propdlg.check_code), "toggled",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
  g_signal_connect (G_OBJECT (propdlg.check_uline), "toggled",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
  g_signal_connect (G_OBJECT (propdlg.check_vline), "toggled",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
  g_signal_connect (G_OBJECT (propdlg.check_autoindent), "toggled",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
  g_signal_connect (G_OBJECT (propdlg.check_tabconv), "toggled",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
  g_signal_connect (G_OBJECT (propdlg.check_eof), "toggled",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
  g_signal_connect (G_OBJECT (propdlg.check_overwrite), "toggled",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
  g_signal_connect (G_OBJECT (propdlg.check_syscol), "toggled",
                        G_CALLBACK (property_dialog_toggled_check), &propdlg);
  if (ft->text && ft->ext)
    {
      propdlg.check_html = gtk_check_button_new_with_label
                                    (_("HTML Numeric character reference"));
      propdlg.check_esc = gtk_check_button_new_with_label
                                    (_("Escape sequence"));
      propdlg.check_associate = gtk_check_button_new_with_mnemonic
                                    (_("_Associate File Type"));
      gtk_toggle_button_set_active
                (GTK_TOGGLE_BUTTON (propdlg.check_html), ft->html);
      gtk_toggle_button_set_active
                (GTK_TOGGLE_BUTTON (propdlg.check_esc), ft->esc);
      gtk_toggle_button_set_active
                (GTK_TOGGLE_BUTTON (propdlg.check_associate), ft->associate);
      g_signal_connect (G_OBJECT (propdlg.check_html), "toggled",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
      g_signal_connect (G_OBJECT (propdlg.check_esc), "toggled",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
      g_signal_connect (G_OBJECT (propdlg.check_associate), "toggled",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
    }
  else
    {
      propdlg.check_html = propdlg.check_esc = propdlg.check_associate = NULL;
    }
  /* ja:コンボボックス */
#if GTK_CHECK_VERSION(2,4,0)
  propdlg.combo = gtk_combo_box_new_text ();
  gtk_combo_box_append_text (GTK_COMBO_BOX (propdlg.combo), "CR+LF");
  gtk_combo_box_append_text (GTK_COMBO_BOX (propdlg.combo), "CR");
  gtk_combo_box_append_text (GTK_COMBO_BOX (propdlg.combo), "LF");
  if (ft->text && ft->ext)
    gtk_combo_box_append_text (GTK_COMBO_BOX (propdlg.combo), _("Auto"));
  gtk_combo_box_set_active (GTK_COMBO_BOX (propdlg.combo), ft->crlf);
  g_signal_connect (G_OBJECT (propdlg.combo), "changed",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  glist = g_list_append (NULL, "CR+LF");
  glist = g_list_append (glist, "CR");
  glist = g_list_append (glist, "LF");
  if (ft->text && ft->ext)
    glist = g_list_append (glist, _("Auto"));
  propdlg.combo = gtk_combo_new ();
  gtk_combo_set_popdown_strings (GTK_COMBO (propdlg.combo), glist);
  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (propdlg.combo)->entry),
                                            g_list_nth_data (glist, ft->crlf));
  gtk_editable_set_editable (GTK_EDITABLE (GTK_COMBO (propdlg.combo)->entry),
                                                                        FALSE);
  g_signal_connect (G_OBJECT (GTK_COMBO (propdlg.combo)->entry), "activate",
                        G_CALLBACK (property_dialog_activate), &propdlg);
  g_signal_connect (G_OBJECT (GTK_COMBO (propdlg.combo)->entry), "changed",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
  g_list_free (glist);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  /* ja:文字符号化方式 */
  if (!ft->text && !ft->ext)
    {
      propdlg.charset = orz_charset_new ();
      for (i = 0; i < ftnum; i++)
        orz_charset_add_candidate (ORZ_CHARSET (propdlg.charset),
                                                            ftype[i].charset);
      orz_charset_set_charset (ORZ_CHARSET (propdlg.charset), ft->charset);
      g_signal_connect (G_OBJECT (propdlg.charset), "activate",
                        G_CALLBACK (property_dialog_activate), &propdlg);
      g_signal_connect (G_OBJECT (propdlg.charset), "edited",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
    }
  else
    {
      propdlg.charset = NULL;
    }
  /* ja:エントリー/ラベル */
  propdlg.label = NULL;
  if (ft->text && ft->ext)
    {
      propdlg.entry_text = gtk_entry_new ();
      propdlg.entry_ext = gtk_entry_new ();
      gtk_entry_set_text (GTK_ENTRY (propdlg.entry_text), ft->text);
      gtk_entry_set_text (GTK_ENTRY (propdlg.entry_ext), ft->ext);
      g_signal_connect (G_OBJECT (propdlg.entry_text), "changed",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
      g_signal_connect (G_OBJECT (propdlg.entry_ext), "changed",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
      g_signal_connect (G_OBJECT (propdlg.entry_text), "activate",
                        G_CALLBACK (property_dialog_activate), &propdlg);
      g_signal_connect (G_OBJECT (propdlg.entry_ext), "activate",
                        G_CALLBACK (property_dialog_activate), &propdlg);
    }
  else
    {
      if (ft->ft_id >= 0)
        {
          /* ja:ファイルタイプあり */
          for (i = 0; i < ftnum; i++)
            if (ft->ft_id == ftype[i].ft_id)
              {
                propdlg.label = gtk_label_new (ftype[i].text);
                break;
              }
          if (!propdlg.label)
            propdlg.label = gtk_label_new (_("Unknow File Type"));
        }
      else
        {
          /* ja:ファイルタイプなし */
          propdlg.label = gtk_label_new (_("Independent File Type"));
        }
      gtk_misc_set_alignment (GTK_MISC (propdlg.label), 0, 0.5);
      propdlg.entry_text = propdlg.entry_ext = NULL;
    }
  /* ja:ラジオボタン */
  propdlg.radio_abort = gtk_radio_button_new_with_label (NULL, _("Abort"));
  propdlg.radio_question = gtk_radio_button_new_with_label_from_widget
                (GTK_RADIO_BUTTON (propdlg.radio_abort), _("Question Mark"));
  propdlg.radio_decimal = gtk_radio_button_new_with_label_from_widget
                            (GTK_RADIO_BUTTON (propdlg.radio_question),
                            _("HTML Numeric character reference (Decimal)"));
  propdlg.radio_hex = gtk_radio_button_new_with_label_from_widget
                        (GTK_RADIO_BUTTON (propdlg.radio_decimal),
                        _("HTML Numeric character reference (Hexadecimal)"));
  propdlg.radio_escape = gtk_radio_button_new_with_label_from_widget
                (GTK_RADIO_BUTTON (propdlg.radio_hex), _("Escape sequence"));
  propdlg.radio_upper = gtk_radio_button_new_with_label
                                                    (NULL, _("Upper case"));
  propdlg.radio_lower = gtk_radio_button_new_with_label_from_widget
                    (GTK_RADIO_BUTTON (propdlg.radio_upper), _("Lower case"));
  switch (ft->writemode)
    {
      case  1: gtk_toggle_button_set_active
                    (GTK_TOGGLE_BUTTON (propdlg.radio_question), TRUE); break;
      case  2: gtk_toggle_button_set_active
                    (GTK_TOGGLE_BUTTON (propdlg.radio_decimal), TRUE); break;
      case  3: gtk_toggle_button_set_active
                    (GTK_TOGGLE_BUTTON (propdlg.radio_hex), TRUE); break;
      case  4: gtk_toggle_button_set_active
                    (GTK_TOGGLE_BUTTON (propdlg.radio_escape), TRUE); break;
      default: gtk_toggle_button_set_active
                    (GTK_TOGGLE_BUTTON (propdlg.radio_abort), TRUE);
    }
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ft->hexcase
                        ? propdlg.radio_upper : propdlg.radio_lower), TRUE);
  g_signal_connect (G_OBJECT (propdlg.radio_abort), "toggled",
                        G_CALLBACK (property_dialog_toggled_radio), &propdlg);
  g_signal_connect (G_OBJECT (propdlg.radio_question), "toggled",
                        G_CALLBACK (property_dialog_toggled_radio), &propdlg);
  g_signal_connect (G_OBJECT (propdlg.radio_decimal), "toggled",
                        G_CALLBACK (property_dialog_toggled_radio), &propdlg);
  g_signal_connect (G_OBJECT (propdlg.radio_hex), "toggled",
                        G_CALLBACK (property_dialog_toggled_radio), &propdlg);
  g_signal_connect (G_OBJECT (propdlg.radio_escape), "toggled",
                        G_CALLBACK (property_dialog_toggled_radio), &propdlg);
  g_signal_connect (G_OBJECT (propdlg.radio_upper), "toggled",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
  g_signal_connect (G_OBJECT (propdlg.radio_lower), "toggled",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
  /* ja:リストボックス */
  store = gtk_list_store_new (2, GDK_TYPE_PIXBUF, G_TYPE_STRING);
  propdlg.tview = gtk_tree_view_new_with_model (GTK_TREE_MODEL (store));
  gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (propdlg.tview), FALSE);
  gtk_tree_view_set_enable_search (GTK_TREE_VIEW (propdlg.tview), FALSE);
  renderer = gtk_cell_renderer_pixbuf_new ();
  column = gtk_tree_view_column_new_with_attributes
                                        (NULL, renderer, "pixbuf", 0, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (propdlg.tview), column);
  renderer = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes
                                        (NULL, renderer, "text", 1, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (propdlg.tview), column);
  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (propdlg.tview));
  gtk_tree_selection_set_mode (select, GTK_SELECTION_MULTIPLE);
  g_signal_connect (G_OBJECT (select), "changed",
                            G_CALLBACK (property_dialog_changed), &propdlg);
  xpm[0] = "24 16 1 1";
  for (i = 2; i < 18; i++)
    xpm[i] = "XXXXXXXXXXXXXXXXXXXXXXXX";
  for (i = 0; i < 12; i++)
    {
      xpm[1] = g_strdup_printf ("X\tc #%02X%02X%02X",
                                                    ft->color[i].red / 256,
                                                    ft->color[i].green / 256,
                                                    ft->color[i].blue / 256);
      gtk_list_store_append (store, &iter);
      gtk_list_store_set (store, &iter,
                        0, gdk_pixbuf_new_from_xpm_data ((const char **)xpm),
                        1, _(color_name[i]), -1);
      g_free (xpm[1]);
    }
  g_object_unref (store);
  /* ja:スクロールウインドウ */
  scroll = gtk_scrolled_window_new (NULL, NULL);
  gtk_container_add (GTK_CONTAINER (scroll), propdlg.tview);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
                                    GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  /* ja:スピンボタン */
  propdlg.spin_margin = gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new
                                    (ft->margin, 1, G_MAXINT, 1, 5, 0)), 0, 0);
  propdlg.spin_tab = gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new
                                    (ft->tab, 1, G_MAXINT, 1, 5, 0)), 0, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (propdlg.spin_margin), TRUE);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (propdlg.spin_tab), TRUE);
  g_signal_connect (G_OBJECT (propdlg.spin_margin), "activate",
                        G_CALLBACK (property_dialog_activate), &propdlg);
  g_signal_connect (G_OBJECT (propdlg.spin_tab), "activate",
                        G_CALLBACK (property_dialog_activate), &propdlg);
  g_signal_connect (G_OBJECT (propdlg.spin_margin), "changed",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
  g_signal_connect (G_OBJECT (propdlg.spin_tab), "changed",
                        G_CALLBACK (property_dialog_ok_sensitive), &propdlg);
  /* ja:フレームとボックス */
  frame0 = gtk_frame_new ("");
  label = gtk_frame_get_label_widget (GTK_FRAME (frame0));
  gtk_label_set_text_with_mnemonic (GTK_LABEL (label), _("_Display"));
  gtk_label_set_mnemonic_widget (GTK_LABEL (label), propdlg.check_gline);
  hbox = gtk_hbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), SPACING);
  gtk_box_pack_start (GTK_BOX (hbox), propdlg.check_gline, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), propdlg.check_space, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), propdlg.check_nline, TRUE, TRUE, 0);
  gtk_container_add (GTK_CONTAINER (frame0), hbox);

  frame1 = gtk_frame_new ("");
  label = gtk_frame_get_label_widget (GTK_FRAME (frame1));
  gtk_label_set_text_with_mnemonic (GTK_LABEL (label), _("_Right Margin"));
  gtk_label_set_mnemonic_widget (GTK_LABEL (label), propdlg.spin_margin);
  hbox = gtk_hbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), SPACING);
  gtk_box_pack_start (GTK_BOX (hbox), propdlg.spin_margin, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), propdlg.check_mline, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), propdlg.check_limit, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (frame1), hbox);

  frame2 = gtk_frame_new ("");
  label = gtk_frame_get_label_widget (GTK_FRAME (frame2));
  gtk_label_set_text_with_mnemonic (GTK_LABEL (label), _("_Tab"));
  gtk_label_set_mnemonic_widget (GTK_LABEL (label), propdlg.spin_tab);
  hbox = gtk_hbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), SPACING);
  gtk_box_pack_start (GTK_BOX (hbox), propdlg.spin_tab, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), propdlg.check_code, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), propdlg.check_uline, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), propdlg.check_vline, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (frame2), hbox);

  frame3 = gtk_frame_new ("");
  label = gtk_frame_get_label_widget (GTK_FRAME (frame3));
  gtk_label_set_text_with_mnemonic (GTK_LABEL (label), _("Ed_it"));
  gtk_label_set_mnemonic_widget (GTK_LABEL (label), propdlg.check_autoindent);
  hbox = gtk_hbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), SPACING);
  gtk_box_pack_start (GTK_BOX(hbox),
                                    propdlg.check_autoindent, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), propdlg.check_tabconv, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (frame3), hbox);

  frame4 = gtk_frame_new ("");
  label = gtk_frame_get_label_widget (GTK_FRAME (frame4));
  gtk_label_set_text_with_mnemonic (GTK_LABEL (label), _("_Save"));
  gtk_label_set_mnemonic_widget (GTK_LABEL (label), propdlg.check_eof);
  tbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (tbox), SPACING);
  hbox = gtk_hbox_new (FALSE, SPACING);
  gtk_box_pack_start (GTK_BOX (hbox), propdlg.check_eof, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), propdlg.check_overwrite, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (tbox), hbox, FALSE, FALSE, 0);
  hbox = gtk_hbox_new (FALSE, SPACING);
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_box_pack_start (GTK_BOX (vbox), propdlg.radio_abort, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), propdlg.radio_question, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), propdlg.radio_decimal, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), propdlg.radio_hex, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), propdlg.radio_escape, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_box_pack_start (GTK_BOX (vbox), propdlg.radio_upper, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), propdlg.radio_lower, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (tbox), hbox, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (frame4), tbox);

  frame5 = gtk_frame_new ("");
  label = gtk_frame_get_label_widget (GTK_FRAME (frame5));
  gtk_label_set_text_with_mnemonic (GTK_LABEL (label), _("_Color"));
  gtk_label_set_mnemonic_widget (GTK_LABEL (label), propdlg.check_syscol);
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
  gtk_box_pack_start (GTK_BOX (vbox), scroll, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), propdlg.check_syscol, FALSE, FALSE, 0);
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox), propdlg.button_color, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (frame5), vbox);

  frame6 = gtk_frame_new ("");
  label = gtk_frame_get_label_widget (GTK_FRAME (frame6));
  gtk_label_set_text_with_mnemonic (GTK_LABEL (label),
                                                    _("Character _Encoding"));
#if GTK_CHECK_VERSION(2,4,0)
  gtk_label_set_mnemonic_widget (GTK_LABEL (label), propdlg.combo);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  gtk_label_set_mnemonic_widget (GTK_LABEL (label),
                                            GTK_COMBO (propdlg.combo)->entry);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
  gtk_box_pack_start (GTK_BOX (vbox), propdlg.combo, FALSE, FALSE, 0);
  if (button_auto)
    {
      hbox = gtk_hbox_new (FALSE, 0);
      gtk_box_pack_end (GTK_BOX (hbox), button_auto, FALSE, FALSE, 0);
      gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    }
  if (propdlg.charset)
    gtk_box_pack_start (GTK_BOX (vbox), propdlg.charset, FALSE, FALSE, 0);
  if (propdlg.check_html)
    gtk_box_pack_start (GTK_BOX (vbox), propdlg.check_html, FALSE, FALSE, 0);
  if (propdlg.check_esc)
    gtk_box_pack_start (GTK_BOX (vbox), propdlg.check_esc, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (frame6), vbox);

  tbox = gtk_hbox_new (FALSE, SPACING);
  if (propdlg.entry_text && propdlg.entry_ext)
    {
      hbox = gtk_hbox_new (FALSE, 0);
      gtk_box_pack_start (GTK_BOX (hbox),
                                gtk_label_new (_("Title")), FALSE, FALSE, 0);
      gtk_box_pack_start (GTK_BOX (hbox), propdlg.entry_text, TRUE, TRUE, 0);
      gtk_box_pack_start (GTK_BOX (tbox), hbox, TRUE, TRUE, 0);
      hbox = gtk_hbox_new (FALSE, 0);
      gtk_box_pack_start (GTK_BOX (hbox),
                            gtk_label_new (_("Extension")), FALSE, FALSE, 0);
      gtk_box_pack_start (GTK_BOX (hbox), propdlg.entry_ext, TRUE, TRUE, 0);
      gtk_box_pack_start (GTK_BOX (tbox), hbox, TRUE, TRUE, 0);
    }
  else
    {
      gtk_box_pack_start (GTK_BOX (tbox), propdlg.label, TRUE, TRUE, 0);
    }

  hbox = gtk_hbox_new (FALSE, SPACING);
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_box_pack_start (GTK_BOX (vbox), frame0, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), frame1, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), frame2, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), frame3, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), frame4, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_box_pack_start (GTK_BOX (vbox), frame5, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), frame6, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);

  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
  gtk_box_pack_start (GTK_BOX (vbox), tbox, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);
  hbox = gtk_hbox_new (FALSE, SPACING);
#if GTK_CHECK_VERSION(2,4,0)
  gtk_box_pack_start (GTK_BOX (hbox), propdlg.button_font, FALSE, FALSE, 0);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  gtk_box_pack_start (GTK_BOX (hbox), button_font, FALSE, FALSE, 0);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  if (button_associated)
    gtk_box_pack_start (GTK_BOX (hbox), button_associated, FALSE, FALSE, 0);
  if (propdlg.check_associate)
    gtk_box_pack_start (GTK_BOX (hbox),
                                    propdlg.check_associate, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (propdlg.dialog)->vbox), vbox);

  /* ja:表示 */
  gtk_widget_set_sensitive (propdlg.button_color, FALSE);
  gtk_widget_set_sensitive (propdlg.radio_upper, ft->writemode >= 3);
  gtk_widget_set_sensitive (propdlg.radio_lower, ft->writemode >= 3);
  gtk_widget_set_sensitive (propdlg.tview, !ft->syscol);
  gtk_dialog_set_response_sensitive (GTK_DIALOG (propdlg.dialog),
                                                    GTK_RESPONSE_OK, FALSE);
  gtk_widget_show_all (vbox);

  propdlg.ft_id = ft->ft_id;
  g_memmove (propdlg.gdk_color, ft->color, sizeof (GdkColor) * 12);
  propdlg.charset_list = button_auto ? g_strdup (ft->charset) : NULL;
#if ! GTK_CHECK_VERSION(2,4,0)
  propdlg.font_name = g_strdup (ft->font_name);
#endif /* not GTK_CHECK_VERSION(2,4,0) */

  result = gtk_dialog_run (GTK_DIALOG (propdlg.dialog)) == GTK_RESPONSE_OK;
  if (result)
    {
      gboolean success[12];

      ft->ft_id = propdlg.ft_id;
      if (propdlg.entry_text)
        {
          g_free (ft->text);
          ft->text
            = g_strdup (gtk_entry_get_text (GTK_ENTRY (propdlg.entry_text)));
        }
      if (propdlg.entry_ext)
        {
          g_free (ft->ext);
          ft->ext
            = g_strdup (gtk_entry_get_text (GTK_ENTRY (propdlg.entry_ext)));
        }
      ft->margin = gtk_spin_button_get_value_as_int
                                (GTK_SPIN_BUTTON (propdlg.spin_margin));
      ft->tab = gtk_spin_button_get_value_as_int
                                (GTK_SPIN_BUTTON (propdlg.spin_tab));
      ft->gline = gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg.check_gline));
      ft->space = gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg.check_space));
      ft->nline = gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg.check_nline));
      ft->mline = gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg.check_mline));
      ft->limit = gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg.check_limit));
      ft->code = gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg.check_code));
      ft->uline = gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg.check_uline));
      ft->vline = gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg.check_vline));
      ft->autoindent = gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg.check_autoindent));
      ft->tabconv = gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg.check_tabconv));
      ft->eof = gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg.check_eof));
      ft->overwrite = gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg.check_overwrite));
      ft->syscol = gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg.check_syscol));
      if (propdlg.check_html)
        ft->html = gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg.check_html));
      if (propdlg.check_esc)
        ft->esc = gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg.check_esc));
      if (propdlg.check_associate)
        ft->associate = gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg.check_associate));
      if (gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg.radio_question)))
        ft->writemode = 1;
      else if (gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg.radio_decimal)))
        ft->writemode = 2;
      else if (gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg.radio_hex)))
        ft->writemode = 3;
      else if (gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg.radio_escape)))
        ft->writemode = 4;
      else
        ft->writemode = 0;
      ft->hexcase = gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON (propdlg.radio_upper));
      if (propdlg.combo)
#if GTK_CHECK_VERSION(2,4,0)
        ft->crlf = gtk_combo_box_get_active (GTK_COMBO_BOX (propdlg.combo));
#else /* not GTK_CHECK_VERSION(2,4,0) */
        {
          const gchar *text;
          GList *glist;

          text = gtk_entry_get_text
                                (GTK_ENTRY (GTK_COMBO (propdlg.combo)->entry));
          glist = gtk_container_get_children
                            (GTK_CONTAINER (GTK_COMBO (propdlg.combo)->list));
          for (ft->crlf = g_list_length (glist) - 1; ft->crlf >= 0; ft->crlf--)
            if (g_strcmp (gtk_label_get_text (GTK_LABEL (gtk_bin_get_child
                (GTK_BIN (g_list_nth_data (glist, ft->crlf))))), text) == 0)
              break;
          g_list_free (glist);
          if (ft->crlf < 0)
            ft->crlf = 0;
        }
#endif /* not GTK_CHECK_VERSION(2,4,0) */
      g_free (ft->charset);
      if (propdlg.charset)
        ft->charset = orz_charset_get_charset (ORZ_CHARSET (propdlg.charset));
      if (button_auto)
        {
          ft->charset = propdlg.charset_list;
          propdlg.charset_list = NULL;
        }
      g_memmove (ft->color, propdlg.gdk_color, sizeof (GdkColor) * 12);
      gdk_colormap_alloc_colors (gdk_colormap_get_system (),
                                        ft->color, 12, FALSE, TRUE, success);
      g_free (ft->font_name);
#if GTK_CHECK_VERSION(2,4,0)
      ft->font_name = g_strdup (gtk_font_button_get_font_name
                                    (GTK_FONT_BUTTON (propdlg.button_font)));
#else /* not GTK_CHECK_VERSION(2,4,0) */
      ft->font_name = propdlg.font_name;
      propdlg.font_name = NULL;
#endif /* not GTK_CHECK_VERSION(2,4,0) */
    }
  gtk_widget_destroy (propdlg.dialog);

  g_free (propdlg.charset_list);
#if ! GTK_CHECK_VERSION(2,4,0)
  g_free (propdlg.font_name);
#endif /* not GTK_CHECK_VERSION(2,4,0) */

  return result;
}
