/*
    Text maid for Windows
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef __EDIT_H__
#define __EDIT_H__


#include "tmaid.h"


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */


/******************************************************************************
*                                                                             *
* ja:編集関数群                                                               *
*                                                                             *
******************************************************************************/
#define TMAID_CTYPE_ERROR      (-1)
#define TMAID_CTYPE_KASHIDA    512
#define TMAID_CTYPE_IDEOGRAPH  256
#define TMAID_CTYPE_KATAKANA   128
#define TMAID_CTYPE_HIRAGANA   64
#define TMAID_CTYPE_KANA       (TMAID_CTYPE_KATAKANA | TMAID_CTYPE_HIRAGANA)
#define TMAID_CTYPE_DIGIT      32
#define TMAID_CTYPE_ALPHA      16
#define TMAID_CTYPE_SPACE      8
#define TMAID_CTYPE_H_KATAKANA 4
#define TMAID_CTYPE_H_DIGIT    2
#define TMAID_CTYPE_H_ALPHA    1
#define TMAID_CTYPE_NULL       0


/*  ja:文字の種類を求める
     lpbType,文字の属性を格納するバッファ,0:ANK,1:SJIS1バイト目,2:SJIS2バイト目
    lpszText,文字列
     nLength,データサイズ                                                   */
void
mCharType (LPBYTE lpbType,
           LPCSTR lpszText,
           int    nLength);


/*  ja:ラインバッファを求める
    lpStart,ラインバッファ
     lpnOff,現在のy座標
          y,y座標
        RET,ラインバッファ                                                  */
LPLINEBUF
GetLineBuffer (LINEBUF **lpStart,
               int      *lpnOff,
               int       y);


/*  ja:桁数を取得する
    lpStart,ラインバッファ
     lpnOff,現在のy座標
          y,行
       nTab,タブ
        RET,桁数                                                            */
int
GetWidth (LINEBUF **lpStart,
          int      *lpnOff,
          int       y,
          int       nTab);


/*  ja:最大桁数を取得する
    ptw,TXTウインドウ情報
    RET,桁数                                                                */
int
GetWidthMax (LPTEXTWND ptw);


/*  ja:キャレットのアライメントされたx座標を求める
    lpStart,ラインバッファ
     lpnOff,現在のy座標
          x,x座標
          y,y座標
       nTab,タブ
     fAlign,TRUE:右に寄せる,FALSE:左に寄せる
        RET,座標                                                            */
int
GetAlignPos (LINEBUF **lpStart,
             int      *lpnOff,
             int       x,
             int       y,
             int       nTab,
             BOOL      fAlign);


/*  ja:画面上の座標からラインバッファ上のアドレスを求める
    lpStart,ラインバッファ
     lpnOff,現在のy座標
          x,x座標
          y,y座標
       nTab,タブ
     fAlign,TRUE:右に寄せる,FALSE:左に寄せる
        RET,アドレス                                                        */
int
GetDataPos (LINEBUF **lpStart,
            int      *lpnOff,
            int       x,
            int       y,
            int       nTab,
            BOOL      fAlign);


/*  ja:ラインバッファ上のアドレスから画面上の座標を求める
    lpStart,ラインバッファ
     lpnOff,現在のy座標
          x,x座標
          y,y座標
       nTab,タブ
        RET,座標                                                            */
int
GetScreenPos (LINEBUF **lpStart,
              int      *lpnOff,
              int       x,
              int       y,
              int       nTab);


/*  ja:文字の分類
    lpszText,文字列
     nLength,文字数
         RET,分類                                                           */
DWORD
GetCharType (LPCSTR lpszText,
             int    nLength);


/*  ja:区切りを求める
    lpStart,ラインバッファ
     lpnOff,現在のy座標
          x,x座標
          y,y座標
       nTab,タブ
      fMove,TRUE:右に移動する,FALSE:左に移動する
        RET,座標                                                            */
int
GetMovePos (LINEBUF **lpStart,
            int      *lpnOff,
            int       x,
            int       y,
            int       nTab,
            int       fMove);


/*  ja:指定範囲のバイト数を求める
      lpStart,ラインバッファ
       lpnOff,現在のy座標
    lpptStart,選択範囲
      lpptEnd,選択範囲
         nTab,タブ
          RET,バイト数                                                      */
int
GetSelByte (LINEBUF **lpStart,
            int      *lpnOff,
            LPPOINT   lpptStart,
            LPPOINT   lpptEnd,
            int       nTab);


/*  ja:指定範囲をメモリにコピーする
     lpStart,ラインバッファ
      lpnOff,現在のy座標
    pptStart,選択範囲
     lpptEnd,選択範囲
        nTab,タブ
    lpszText,メモリ                                                         */
void
CpySelMem (LINEBUF **lpStart,
           int      *lpnOff,
           LPPOINT   lpptStart,
           LPPOINT   lpptEnd,
           int       nTab,
           LPSTR     lpszText);


/*  ja:データを取得する
     lpStart,ラインバッファ
      lpnOff,現在のy座標
     lpptGet,取得をはじめるデータ座標
    lpszText,メモリ
     nLength,バイト数
         RET,取得できたバイト数                                             */
int
GetMem (LINEBUF **lpStart,
        int      *lpnOff,
        LPPOINT   lpptGet,
        LPSTR     lpszText,
        int       nLength);


/*  ja:メモリの内容を貼り付ける
     lpStart,ラインバッファ
      lpnOff,現在のy座標
     lpptPut,貼り付ける座標
     lpptNew,新しい座標
        nTab,タブ
    lpszText,メモリ
     nLength,バイト数
     nMargin,右マージン(0ならば折り返しなし)
         RET,挿入された行数                                                 */
int
PutMem (LINEBUF **lpStart,
        int      *lpnOff,
        LPPOINT   lpptPut,
        LPPOINT   lpptNew,
        int       nTab,
        LPCSTR    lpszText,
        int       nLength,
        int       nMargin);


/*  ja:指定範囲を削除する
      lpStart,ラインバッファ
       lpnOff,現在のy座標
    lpptStart,選択範囲
      lpptEnd,選択範囲
         nTab,タブ
     nMargin,右マージン(0ならば折り返しなし)
          RET,削除された行数                                                */
int
DelSelMem (LINEBUF **lpStart,
           int      *lpnOff,
           LPPOINT   lpptStart,
           LPPOINT   lpptEnd,
           int       nTab,
           int       nMargin);


#ifdef __cplusplus
}
#endif /* __cplusplus */


#endif /* __EDIT_H__ */
