/*
    Text maid for Windows
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "wndtext.h"
#include <commctrl.h>
#include <shlwapi.h>
#include "edit.h"
#include "file.h"
#include "find.h"
#include "general.h"
#include "history.h"
#include "jump.h"
#include "print.h"
#include "property.h"
#include "reload.h"
#include "resource.h"
#include "repinfo.h"
#include "replace.h"
#include "valchr.h"
#include "wcommon.h"


/******************************************************************************
*                                                                             *
* ja:テキストウインドウ関数群                                                 *
*                                                                             *
******************************************************************************/
static VOID
SetIcon (HWND    hWnd,
         LPCTSTR lpszFile)
{
  SHFILEINFO shfi;

  if (SHGetFileInfo (lpszFile, 0, &shfi, sizeof (SHFILEINFO), SHGFI_ICON
                                | SHGFI_LARGEICON | SHGFI_USEFILEATTRIBUTES))
    {
      HICON hIcon;

      hIcon = (HICON)SendMessage (hWnd, WM_SETICON,
                                    ICON_BIG, (LPARAM)CopyIcon (shfi.hIcon));
      if (hIcon)
        DestroyIcon (hIcon);
      DestroyIcon (shfi.hIcon);
    }
  if (SHGetFileInfo (lpszFile, 0, &shfi, sizeof (SHFILEINFO), SHGFI_ICON
                                | SHGFI_SMALLICON | SHGFI_USEFILEATTRIBUTES))
    {
      HICON hIcon;

      hIcon = (HICON)SendMessage (hWnd, WM_SETICON,
                                    ICON_SMALL, (LPARAM)CopyIcon (shfi.hIcon));
      if (hIcon)
        DestroyIcon (hIcon);
      DestroyIcon (shfi.hIcon);
    }
}


LRESULT CALLBACK
TextWndProc (HWND   hWnd,
             UINT   uMsg,
             WPARAM wParam,
             LPARAM lParam)
{
  static BOOL fDblClk = FALSE;

  switch (uMsg)
    {
      case WM_CREATE:/* ja:ウインドウが作られた時 */
        {
          HDC hDC;
          LOGFONT LogFont;
          LPTEXTWND ptw;
          TEXTMETRIC TextMetric;

          ptw = (LPTEXTWND)((LPMDICREATESTRUCT)
                        (((LPCREATESTRUCT)lParam)->lpCreateParams))->lParam;
          SetWindowLongPtr (hWnd, GWLP_USERDATA, (LONG_PTR)ptw);
          if (!ptw->fSysColor)
            {
              ptw->hBrushWindow = CreateSolidBrush(ptw->crColor[1]);
              ptw->hBrushSpace = CreateSolidBrush (ptw->crColor[3]);
              ptw->hBrushCrlf = CreateSolidBrush (ptw->crColor[4]);
              ptw->hBrushHighLight = CreateSolidBrush (ptw->crColor[9]);
              ptw->hPenTab = CreatePen (PS_SOLID, 0, ptw->crColor[5]);
              ptw->hPenMargin = CreatePen (PS_SOLID, 0, ptw->crColor[6]);
              ptw->hPenGrid = CreatePen (PS_SOLID, 0, ptw->crColor[7]);
            }
          LogFont = ptw->LogFont;
          LogFont.lfHeight /= 2;
          ptw->hFont = CreateFontIndirect (&ptw->LogFont);
          ptw->hFontsm = CreateFontIndirect (&LogFont);
          hDC = GetDC (hWnd);
          ptw->hFont = SelectObject (hDC, ptw->hFont);
          GetTextMetrics (hDC, &TextMetric);
          ptw->hFont = SelectObject (hDC, ptw->hFont);
          ReleaseDC (hWnd, hDC);
          ptw->nFontSize = (TextMetric.tmHeight + 1) / 2;
          SetIcon (hWnd, ptw->lpszFile);
          SetFocus (hWnd);
        }
        return 0;
      case WM_SIZE:/* ja:サイズが変更された */
        {
          int sx, sy, nMax;
          LPTEXTWND ptw;

          ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
          ptw->siWnd.cx = LOWORD (lParam);
          ptw->siWnd.cy = HIWORD (lParam);
          nMax = GetWidthMax (ptw);
          sx = max (ptw->siWnd.cx / ptw->nFontSize, 1);
          sy = max (ptw->siWnd.cy / (ptw->nFontSize * 2), 1);
          if (ptw->ptTop.x > nMax - sx + 1)
            ptw->ptTop.x = max (nMax - sx + 1, 0);
          if (ptw->ptTop.y > ptw->nMax - sy)
            ptw->ptTop.y = max (ptw->nMax - sy, 0);
          SetScrollBar (hWnd, SB_HORZ, 0, nMax, sx, ptw->ptTop.x);
          SetScrollBar (hWnd, SB_VERT, 0, ptw->nMax - 1, sy, ptw->ptTop.y);
          DrawCaret (hWnd);
        }
        break;
      case WM_PAINT:/* ja:描画 */
        {
          int y, nBkMode;
          COLORREF crTextColor, crText, crCtrl, crMbcs, crCrlf, crSelc;
          HBRUSH hBrush, hBrushOld;
          HPEN hPenOld;
          LPLINEBUF p;
          LPTEXTWND ptw;
          PAINTSTRUCT ps;
          POINT ptStart, ptEnd;

          ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
          BeginPaint (hWnd, &ps);
          if (ptw->ptSelect.x < 0)
            {
              /* ja:選択範囲がないとき */
              ptStart.x=-1;
            }
          else if (ptw->ptSelect.y < ptw->ptCursor.y
                                        || ptw->ptSelect.y == ptw->ptCursor.y
                                        && ptw->ptSelect.x < ptw->ptCursor.x)
            {
              /* ja:選択範囲があるとき */
              ptStart = ptw->ptSelect;
              ptEnd.x = GetAlignPos (&ptw->lpStart, &ptw->nOff,
                        ptw->ptCursor.x, ptw->ptCursor.y, ptw->nTab, FALSE);
              ptEnd.y = ptw->ptCursor.y;
            }
          else
            {
              /* ja:選択範囲があるとき */
              ptStart.x = GetAlignPos (&ptw->lpStart, &ptw->nOff,
                        ptw->ptCursor.x, ptw->ptCursor.y, ptw->nTab, FALSE);
              ptStart.y = ptw->ptCursor.y;
              ptEnd = ptw->ptSelect;
            }
          ptw->hFont = SelectObject (ps.hdc, ptw->hFont);
          nBkMode = SetBkMode (ps.hdc, TRANSPARENT);
          crTextColor = GetTextColor (ps.hdc);
          if (ps.fErase)/* ja:背景 */
            FillRect (ps.hdc, &ps.rcPaint,
                            ptw->fSysColor ? hBrushWindow : ptw->hBrushWindow);
          if (ptw->fGline)
            {
              /* ja:グリッド */
              int x, y;

              hPenOld = SelectObject (ps.hdc, ptw->fSysColor
                                                ? hPenGrid : ptw->hPenGrid);
              for (x = ptw->nFontSize - 1; x < ptw->siWnd.cx;
                                                        x += ptw->nFontSize)
                {
                  MoveToEx (ps.hdc, x, 0, NULL);
                  LineTo (ps.hdc, x, ptw->siWnd.cy);
                }
              for (y = ptw->nFontSize * 2 - 1;
                                    y < ptw->siWnd.cy; y += ptw->nFontSize * 2)
                {
                  MoveToEx (ps.hdc, 0, y, NULL);
                  LineTo (ps.hdc, ptw->siWnd.cx, y);
                }
              SelectObject (ps.hdc, hPenOld);
            }
          if (ptw->fVline)
            {
              /* ja:縦線 */
              int x;

              hPenOld = SelectObject (ps.hdc, ptw->fSysColor
                                                ? hPenGrid : ptw->hPenGrid);
              for (x = (ptw->nTab - ptw->ptTop.x % ptw->nTab)
                                                        * ptw->nFontSize - 1;
                            x < ptw->siWnd.cx; x += ptw->nFontSize * ptw->nTab)
                {
                  MoveToEx (ps.hdc, x, 0, NULL);
                  LineTo (ps.hdc, x, ptw->siWnd.cy);
                }
              SelectObject (ps.hdc, hPenOld);
            }
          if (ptw->fMline)
            {
              /* ja:右マージン */
              int x;

              hPenOld = SelectObject (ps.hdc, ptw->fSysColor
                                                ? hPenGray : ptw->hPenMargin);
              x = (ptw->nMargin - ptw->ptTop.x) * ptw->nFontSize;
              MoveToEx (ps.hdc, x, 0, NULL);
              LineTo (ps.hdc, x, ptw->siWnd.cy);
              SelectObject (ps.hdc, hPenOld);
            }
          if (ptw->fSysColor)
            {
              /* ja:システム色 */
              hBrush = hBrushHighLight;
              if (ptw->fCRLF)
                hBrushOld = SelectObject (ps.hdc, hBrushGray);
              hPenOld = SelectObject (ps.hdc, hPenGray);
              crText = crWindowText;
              crCtrl =
              crMbcs =
              crCrlf = crGrayText;
              crSelc = crHighLightText;
            }
          else
            {
              /* ja:カスタム色 */
              hBrush = ptw->hBrushHighLight;
              if (ptw->fCRLF)
                hBrushOld = SelectObject (ps.hdc, ptw->hBrushCrlf);
              hPenOld = SelectObject (ps.hdc, ptw->hPenTab);
              crText = ptw->crColor[0];
              crCtrl = ptw->crColor[2];
              crMbcs = ptw->crColor[3];
              crCrlf = ptw->crColor[4];
              crSelc = ptw->crColor[8];
            }
          for (y = 0,
                p = GetLineBuffer (&ptw->lpStart, &ptw->nOff, ptw->ptTop.y);
                                        y <= ps.rcPaint.bottom && p;
                                        y += ptw->nFontSize * 2, p = p->next)
            {
              int x, nDataPos = 0;

              if (y + ptw->nFontSize * 2 < ps.rcPaint.top)
                continue;
              x = -ptw->ptTop.x * ptw->nFontSize;
              while (x <= ps.rcPaint.right && nDataPos < p->nLength)
                {
                  RECT rc;

                  if (x + max (ptw->nTab, 2) * ptw->nFontSize
                                                            < ps.rcPaint.left)
                    {
                      /* ja:左側の表示されない部分を計算する */
                      if (p->lpszText[nDataPos] == '\t')
                        {
                          x = (((x / ptw->nFontSize + ptw->ptTop.x)
                                / ptw->nTab + 1) * ptw->nTab - ptw->ptTop.x)
                                                            * ptw->nFontSize;
                          nDataPos++;
                        }
                      else if (nDataPos + 1 == p->nLength
                                                || !IsDBCSLeadByteEx(CP_ACP,
                                                        p->lpszText[nDataPos]))
                        {
                          x += ptw->nFontSize;
                          nDataPos++;
                        }
                      else
                        {
                          x += ptw->nFontSize * 2;
                          nDataPos += 2;
                        }
                      continue;
                    }
                  if (ptStart.x >= 0 && (ptStart.y == ptEnd.y
                    ? ptStart.y == ptw->ptTop.y + y / (ptw->nFontSize * 2)
                        && ptStart.x <= ptw->ptTop.x + x / ptw->nFontSize
                        && ptw->ptTop.x + x / ptw->nFontSize < ptEnd.x
                    : ptStart.y < ptw->ptTop.y + y / (ptw->nFontSize * 2)
                        && ptw->ptTop.y + y / (ptw->nFontSize * 2) < ptEnd.y
                        || ptStart.y == ptw->ptTop.y + y / (ptw->nFontSize * 2)
                        && ptStart.x <= ptw->ptTop.x + x / ptw->nFontSize
                        || ptEnd.y == ptw->ptTop.y + y / (ptw->nFontSize * 2)
                        && ptw->ptTop.x + x / ptw->nFontSize < ptEnd.x))
                    {
                      /* ja:選択範囲内 */
                      rc.left = x;
                      rc.top = y;
                      rc.bottom = y + ptw->nFontSize * 2;
                      SetTextColor (ps.hdc, crSelc);
                    }
                  else
                    {
                      /* ja:選択範囲外 */
                      rc.left = -1;
                      SetTextColor (ps.hdc, crText);
                    }
                  if (p->lpszText[nDataPos] == '\t')
                    {
                      /* ja:タブ */
                      rc.right = (((x / ptw->nFontSize + ptw->ptTop.x)
                                                            / ptw->nTab + 1)
                                * ptw->nTab - ptw->ptTop.x) * ptw->nFontSize;
                      if (rc.left != -1)/* ja:選択 */
                        FillRect (ps.hdc, &rc, hBrush);
                      SetTextColor (ps.hdc, crCtrl);
                      if (ptw->fUline)
                        {
                          /* ja:下線 */
                          MoveToEx (ps.hdc,
                                        x, y + ptw->nFontSize * 2 - 1, NULL);
                          LineTo (ps.hdc,
                                        rc.right, y + ptw->nFontSize * 2 - 1);
                        }
                      if (ptw->fCode)
                        {
                          /* ja:コード文字 */
                          ptw->hFontsm = SelectObject (ps.hdc, ptw->hFontsm);
                          TextOutA (ps.hdc, x, y,
                                    ppszCode[(BYTE)p->lpszText[nDataPos]], 1);
                          TextOutA (ps.hdc,
                                    x + ptw->nFontSize / 2, y + ptw->nFontSize,
                                ppszCode[(BYTE)p->lpszText[nDataPos]] + 1, 1);
                          ptw->hFontsm = SelectObject (ps.hdc, ptw->hFontsm);
                        }
                      nDataPos++;
                    }
                  else if (nDataPos == p->nLength - 1
                        || !IsDBCSLeadByteEx (CP_ACP, p->lpszText[nDataPos]))
                    {
                      /* ja:1バイト文字 */
                      rc.right = x + ptw->nFontSize;
                      if (rc.left != -1)/* ja:選択 */
                        FillRect (ps.hdc, &rc, hBrush);
                      if (!IsCharControl (p->lpszText[nDataPos]))
                        {
                          /* ja:通常の1文字 */
                          TextOutA (ps.hdc, x, y, p->lpszText + nDataPos, 1);
                        }
                      else
                        {
                          /* ja:コントロールコード */
                          int nLength;
                          LPSTR lpszText;

                          SetTextColor (ps.hdc, crCtrl);/* ja:灰色 */
                          lpszText = ppszCode[(BYTE)p->lpszText[nDataPos]];
                          nLength = lstrlenA (lpszText);
                          if (nLength == 1)
                            {
                              /* ja:1バイト1文字 */
                              TextOutA (ps.hdc, x, y, lpszText, 1);
                            }
                          else
                            {
                              ptw->hFontsm = SelectObject (ps.hdc,
                                                                ptw->hFontsm);
                              if (nLength == 2)
                                {
                                  /* ja:1バイト2文字 */
                                  if (IsDBCSLeadByteEx (CP_ACP, *lpszText))
                                    {
                                      /* ja:全角1つ */
                                      TextOutA (ps.hdc, x,
                                                        y + ptw->nFontSize / 2,
                                                                lpszText, 2);
                                    }
                                  else
                                    {
                                      /* ja:半角2つ */
                                      TextOutA (ps.hdc, x, y, lpszText, 1);
                                      TextOutA (ps.hdc,x + ptw->nFontSize / 2,
                                        y + ptw->nFontSize, lpszText + 1, 1);
                                    }
                                }
                              else if (nLength == 3)
                                {
                                  /* ja:1バイト3文字 */
                                  if (!IsDBCSLeadByteEx (CP_ACP, lpszText[0])
                                    && IsDBCSLeadByteEx (CP_ACP, lpszText[1]))
                                    {
                                      /* ja:半角+全角 */
                                      TextOutA (ps.hdc, x, y, lpszText, 1);
                                      TextOutA (ps.hdc, x, y + ptw->nFontSize,
                                                            lpszText + 1, 2);
                                    }
                                  else
                                    {
                                      /* ja:全角+半角 半角+半角+半角 */
                                      TextOutA (ps.hdc, x, y, lpszText, 2);
                                      TextOutA (ps.hdc,x + ptw->nFontSize / 2,
                                        y + ptw->nFontSize, lpszText + 2, 1);
                                    }
                                }
                              else
                                {
                                  /* ja:1バイト4文字 */
                                  TextOutA (ps.hdc, x, y, lpszText,2 );
                                  TextOutA (ps.hdc, x, y + ptw->nFontSize,
                                                            lpszText + 2, 2);
                                }
                              ptw->hFontsm = SelectObject (ps.hdc,
                                                                ptw->hFontsm);
                            }
                        }
                      nDataPos++;
                    }
                  else
                    {
                      /* ja:2バイト文字 */
                      rc.right = x + ptw->nFontSize * 2;
                      if (rc.left != -1)/* ja:選択 */
                        FillRect (ps.hdc, &rc, hBrush);
                      if (rc.left == -1 && ptw->fSpace
                                    && GetCharType (p->lpszText + nDataPos, 2)
                                                        & TMAID_CTYPE_SPACE)
                        {
                          /* ja:全角スペース */
                          rc.left = x + 1;
                          rc.top = y + 1;
                          rc.bottom = y + ptw->nFontSize * 2;
                          FillRect (ps.hdc, &rc, ptw->fSysColor
                                            ? hBrushGray : ptw->hBrushSpace);
                        }
                      else
                        {
                          TextOutA (ps.hdc, x, y, p->lpszText + nDataPos, 2);
                        }
                      nDataPos += 2;
                    }
                  x = rc.right;
                }
              if (nDataPos == p->nLength && ptw->fCRLF)
                {
                  if (ptStart.x >= 0
                        && ptStart.y <= ptw->ptTop.y + y / (ptw->nFontSize * 2)
                        && ptw->ptTop.y + y / (ptw->nFontSize * 2) < ptEnd.y)
                    {
                      /* ja:選択 */
                      RECT rc;

                      rc.left = x;
                      rc.top = y;
                      rc.right = x + ptw->nFontSize;
                      rc.bottom = y + ptw->nFontSize * 2;
                      FillRect (ps.hdc, &rc, hBrush);
                    }
                  hPenNull = SelectObject (ps.hdc, hPenNull);
                  if (!p->next)
                    {
                      /* ja:ファイルの最後 */
                      Ellipse (ps.hdc, x + ptw->nFontSize / 6,
                                        y + ptw->nFontSize * 2 / 3,
                                        x + ptw->nFontSize * 5 / 6,
                                        y + ptw->nFontSize * 4 / 3);
                    }
                  else
                    {
                      int nLength;
                      POINT pt[4];

                      if (p->fMargin)
                        {
                          /* ja:右マージンによる改行 */
                          pt[0].x = pt[3].x = x + ptw->nFontSize / 6;
                          pt[0].y = pt[1].y = y + ptw->nFontSize * 2 / 3;
                          pt[1].x = pt[2].x = x + ptw->nFontSize * 5 / 6;
                          pt[2].y = pt[3].y = y + ptw->nFontSize * 4 / 3;
                          nLength = 4;
                        }
                      else
                        {
                          /* ja:通常の改行 */
                          pt[0].x = x + ptw->nFontSize / 6;
                          pt[1].x = x + ptw->nFontSize * 5 / 6;
                          pt[0].y = pt[1].y = y + ptw->nFontSize * 2 / 3;
                          pt[2].x = x + ptw->nFontSize / 2;
                          pt[2].y = y + ptw->nFontSize * 4 / 3;
                          nLength = 3;
                        }
                      Polygon (ps.hdc,pt, nLength);
                    }
                  hPenNull = SelectObject (ps.hdc, hPenNull);
                }
            }
          SetBkMode (ps.hdc, nBkMode);
          SetTextColor (ps.hdc, crTextColor);
          if (ptw->fCRLF)
            SelectObject (ps.hdc, hBrushOld);
          SelectObject (ps.hdc, hPenOld);
          ptw->hFont = SelectObject (ps.hdc, ptw->hFont);
          EndPaint (hWnd, &ps);
        }
        return 0;
      case WM_SETFOCUS:
        {
          LPTEXTWND ptw;

          ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
          SetImeFont (hWnd, &ptw->LogFont);
          CreateCaret (hWnd, (HBITMAP)0,
                        uIns != 0 ? ptw->nFontSize : 2, ptw->nFontSize * 2);
          DrawCaret (hWnd);
          ShowCaret (hWnd);
        }
        return 0;
      case WM_KILLFOCUS:
        DestroyCaret ();
        return 0;
      case WM_MDIACTIVATE:/* ja:MDIでアクティブになったとき */
        {
          LPTEXTWND ptw;

          ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
          if (hWnd == (HWND)lParam)
            {
              SetMenuBar (ptw);
            }
          else
            {
              SetMenuBar (NULL);
              SetImePos (hWnd, -1, -1);
              StatusBar_SetText (hWndStat, 0, 0, _T(""));
            }
        }
        return 0;
      case WM_MENUSELECT:/* ja:メニューが選択された時 */
        {
          UINT uIds[2] = {0, 0};

          MenuHelp (uMsg, wParam, lParam,
                    GetMenu (hWnd), GetModuleHandle (NULL), hWndStat, uIds);
        }
        return 0;
      case WM_HSCROLL:
        {
          int sx, nMax;
          LPTEXTWND ptw;
          POINT ptTop;

          ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
          ptTop = ptw->ptTop;
          sx = max (ptw->siWnd.cx / ptw->nFontSize, 1);
          switch (LOWORD (wParam))
            {
              case SB_LINELEFT:   ptw->ptTop.x--;     break;
              case SB_LINERIGHT:  ptw->ptTop.x++;     break;
              case SB_PAGELEFT:   ptw->ptTop.x -= sx; break;
              case SB_PAGERIGHT:  ptw->ptTop.x += sx; break;
              case SB_THUMBTRACK:
                {
                  SCROLLINFO si;

                  si.cbSize = sizeof (SCROLLINFO);
                  si.fMask = SIF_ALL;
                  GetScrollInfo (hWnd, SB_HORZ, &si);
                  ptw->ptTop.x = si.nTrackPos;
                }
                break;
              default: return 0;
            }
          nMax = GetWidthMax (ptw);
          if (ptw->ptTop.x < 0)
            ptw->ptTop.x = 0;
          else if (ptw->ptTop.x > nMax - sx + 1)
            ptw->ptTop.x = max (nMax - sx + 1, 0);
          HideCaret (hWnd);
          MoveTextWindow (hWnd, &ptTop);
          DrawCaret (hWnd);
          ShowCaret (hWnd);
        }
        return 0;
      case WM_VSCROLL:
      case WM_MOUSEWHEEL:
        {
          int sx, sy, nMax;
          LPTEXTWND ptw;
          POINT ptTop;

          ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
          ptTop = ptw->ptTop;
          sx = max (ptw->siWnd.cx / ptw->nFontSize, 1);
          sy = max (ptw->siWnd.cy / (ptw->nFontSize * 2), 1);
          if (uMsg == WM_VSCROLL)
            switch (LOWORD (wParam))
              {
                case SB_LINEUP:     ptw->ptTop.y--;     break;
                case SB_LINEDOWN:   ptw->ptTop.y++;     break;
                case SB_PAGEUP:     ptw->ptTop.y -= sy; break;
                case SB_PAGEDOWN:   ptw->ptTop.y += sy; break;
                case SB_THUMBTRACK:
                  {
                    SCROLLINFO si;

                    si.cbSize = sizeof (SCROLLINFO);
                    si.fMask = SIF_ALL;
                    GetScrollInfo (hWnd, SB_VERT, &si);
                    ptw->ptTop.y = si.nTrackPos;
                  }
                  break;
                default: return 0;
              }
          else/* en:WHEEL_DELTA=120 */
            ptw->ptTop.y -= (SHORT)HIWORD (wParam) * 4 / WHEEL_DELTA;
          if (ptw->ptTop.y < 0)
            ptw->ptTop.y = 0;
          else if (ptw->ptTop.y > ptw->nMax - sy)
            ptw->ptTop.y = max (ptw->nMax - sy, 0);
          nMax = GetWidthMax (ptw);
          if (ptw->ptTop.x > nMax - sx + 1)
            ptw->ptTop.x = max (nMax - sx + 1, 0);
          HideCaret (hWnd);
          MoveTextWindow (hWnd, &ptTop);
          DrawCaret (hWnd);
          ShowCaret (hWnd);
        }
        return 0;
      case WM_LBUTTONDOWN:
        {
          BOOL fShift;
          POINT ptCursor;
          LPTEXTWND ptw;

          ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
          ptCursor = ptw->ptCursor;
          ptw->ptCursor.x = ptw->ptTop.x + LOWORD (lParam) / ptw->nFontSize;
          ptw->ptCursor.y = ptw->ptTop.y + HIWORD (lParam)
                                                        / (ptw->nFontSize * 2);
          fShift = GetKeyState (VK_SHIFT) & 0x8000;
          if (ptw->ptCursor.y < 0)
            ptw->ptCursor.y = 0;
          else if (ptw->ptCursor.y > ptw->nMax - 1)
            ptw->ptCursor.y = ptw->nMax - 1;
          if (ptw->ptCursor.x < GetWidth (&ptw->lpStart, &ptw->nOff,
                                                ptw->ptCursor.y, ptw->nTab))
            ptw->ptCursor.x = GetAlignPos (&ptw->lpStart, &ptw->nOff,
                        ptw->ptCursor.x, ptw->ptCursor.y, ptw->nTab, FALSE);
          if (ptw->ptSelect.x >= 0 && (!fShift
                                        || ptw->ptCursor.x == ptw->ptSelect.x
                                        && ptw->ptCursor.y == ptw->ptSelect.y))
            {
              HideCaret (hWnd);
              ClearSel (hWnd, &ptCursor, &ptw->ptSelect);
              ptw->ptSelect.x = -1;
              ShowCaret (hWnd);
              SetMenuBar (ptw);
            }
          else if (fShift && (ptw->ptCursor.x != ptCursor.x
                                            || ptw->ptCursor.y != ptCursor.y))
            {
              HideCaret (hWnd);
              if (ptw->ptSelect.x < 0)
                {
                  ptw->ptSelect.x = GetAlignPos (&ptw->lpStart, &ptw->nOff,
                                    ptCursor.x, ptCursor.y, ptw->nTab, FALSE);
                  ptw->ptSelect.y = ptCursor.y;
                }
              ClearSel (hWnd, &ptCursor, &ptw->ptCursor);
              ShowCaret (hWnd);
              SetMenuBar (ptw);
            }
          DrawCaret (hWnd);
          SetCapture (hWnd);
        }
        return 0;
      case WM_LBUTTONUP:
        {
          int i = 0, j, nDataPos;
          DWORD ctype;
          LPLINEBUF p;
          LPTEXTWND ptw;
          MSG msg;

          while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
            {
              TranslateMessage (&msg);
              DispatchMessage (&msg);
            }
          ReleaseCapture ();
          if (!fDblClk)
            return 0;
          /* ja:ダブルクリックしたとき単語を選択する */
          fDblClk = FALSE;
          ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
          if (ptw->ptSelect.x >= 0)
            return 0;
          p = GetLineBuffer (&ptw->lpStart, &ptw->nOff, ptw->ptCursor.y);
          if (p->nLength <= 0)
            return 0;
          j = nDataPos = GetDataPos (&ptw->lpStart, &ptw->nOff,
                        ptw->ptCursor.x, ptw->ptCursor.y, ptw->nTab, FALSE);
          if (nDataPos > 0)
            {
              /* ja:選択開始位置 */
              LPBYTE lpbType;   /* ja:0:1バイト文字,1:1バイト目,2バイト目 */

              lpbType = MemoryAlloc (nDataPos * sizeof (BYTE));
              mCharType (lpbType, p->lpszText, nDataPos);
              if (nDataPos < p->nLength)
                {
                  int nLength;

                  nLength = nDataPos < p->nLength - 1 && IsDBCSLeadByteEx
                                    (CP_ACP, p->lpszText[nDataPos]) ? 2 : 1;
                  ctype = GetCharType (p->lpszText + nDataPos, nLength);
                  if (ctype == TMAID_CTYPE_ERROR)
                    {
                      MemoryFree (lpbType);
                      return 0;
                    }
                  j += nLength;
                }
              else
                {
                  int nLength;

                  nLength = lpbType[nDataPos - 1] == 2 ? 2 : 1;
                  nDataPos -= nLength;
                  ctype = GetCharType (p->lpszText + nDataPos, nLength);
                  if (ctype == TMAID_CTYPE_ERROR)
                    {
                      MemoryFree (lpbType);
                      return 0;
                    }
                }
              for (i = nDataPos - 1; i >= 0; i--)
                if (lpbType[i] < 2)
                  {
                    int nLength;
                    DWORD ct;

                    nLength = lpbType[i] == 1 ? 2 : 1;
                    ct = GetCharType (p->lpszText + i, nLength);
                    if (ct == TMAID_CTYPE_ERROR)
                      {
                        MemoryFree (lpbType);
                        return 0;
                      }
                    if (!(ctype & ct & TMAID_CTYPE_KANA) && ctype != ct)
                      {
                        i += nLength;
                        break;
                      }
                  }
              MemoryFree (lpbType);
            }
          else
            {
              int nLength;

              nLength = p->nLength > 1
                        && IsDBCSLeadByteEx (CP_ACP, p->lpszText[0]) ? 2 : 1;
              ctype = GetCharType (p->lpszText, nLength);
              if (ctype == TMAID_CTYPE_ERROR)
                return 0;
              j += nLength;
            }
          /* ja:選択終了位置 */
          while (j < p->nLength)
            {
              int nLength;
              DWORD ct;

              nLength = j < p->nLength - 1
                        && IsDBCSLeadByteEx (CP_ACP, p->lpszText[j]) ? 2 : 1;
              ct = GetCharType (p->lpszText + j, nLength);
              if (ct == TMAID_CTYPE_ERROR)
                return 0;
              if (!(ctype & ct & TMAID_CTYPE_KANA) && ctype != ct)
                break;
              j += nLength;
            }
          ptw->ptSelect.x = GetScreenPos (&ptw->lpStart, &ptw->nOff,
                                                i, ptw->ptCursor.y, ptw->nTab);
          ptw->ptCursor.x = GetScreenPos (&ptw->lpStart, &ptw->nOff,
                                                j, ptw->ptCursor.y, ptw->nTab);
          ptw->ptSelect.y = ptw->ptCursor.y;
          if (ptw->ptSelect.x == ptw->ptCursor.x)
            {
              ptw->ptSelect.x = -1;
            }
          else
            {
              SetMenuBar (ptw);
              DrawCaret (hWnd);
              ClearSel (hWnd, &ptw->ptSelect, &ptw->ptCursor);
            }
        }
        return 0;
      case WM_MOUSEMOVE:
        {
          int s, sx, sy;
          POINT ptCursor, ptTop;
          POINTS ps;
          LPTEXTWND ptw;

          if (!(wParam & MK_LBUTTON))
            return 0;
          ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
          ptCursor = ptw->ptCursor;
          ptTop = ptw->ptTop;
          sx = max (ptw->siWnd.cx / ptw->nFontSize, 1);
          sy = max (ptw->siWnd.cy / (ptw->nFontSize * 2), 1);
          ps = MAKEPOINTS (lParam);
          if (ps.x < 0)
            ps.x = 0;
          else if (ps.x > ptw->siWnd.cx)
            ps.x = (SHORT)ptw->siWnd.cx;
          if (ps.y < 0)
            ps.y = 0;
          else if (ps.y > ptw->siWnd.cy)
            ps.y = (SHORT)ptw->siWnd.cy;
          /* ja:カーソルが外にあるときスクロールする */
          if (sx > 1)
            {
              if ((SHORT)LOWORD (lParam) < 0 && ptw->ptTop.x > 0)
                ptw->ptTop.x--;
              else if ((SHORT)LOWORD (lParam) > ptw->siWnd.cx
                        && ptw->ptTop.x < GetWidth (&ptw->lpStart, &ptw->nOff,
                                        ptw->ptCursor.y, ptw->nTab) + 1 - sx)
                ptw->ptTop.x++;
            }
          if (sy > 1)
            {
              if ((SHORT)HIWORD (lParam) < 0 && ptw->ptTop.y > 0)
                ptw->ptTop.y--;
              else if ((SHORT)HIWORD (lParam) > ptw->siWnd.cy
                                            && ptw->ptTop.y < ptw->nMax - sy)
                ptw->ptTop.y++;
            }
          HideCaret (hWnd);
          MoveTextWindow (hWnd, &ptTop);
          s = ps.x - ((ptw->ptSelect.x < 0 ? ptw->ptCursor.x : ptw->ptSelect.x)
                                            - ptw->ptTop.x) * ptw->nFontSize;
          if (s < ptw->nFontSize / 2 || ptw->nFontSize < s
                || (ptw->ptSelect.x < 0 ? ptw->ptCursor.y : ptw->ptSelect.y)
                                != ps.y / (ptw->nFontSize * 2) + ptw->ptTop.y)
            {
              if (ptw->ptSelect.x < 0)
                {
                  /* ja:新たに選択する */
                  ptw->ptSelect.x = GetAlignPos (&ptw->lpStart, &ptw->nOff,
                        ptw->ptCursor.x, ptw->ptCursor.y, ptw->nTab, FALSE);
                  ptw->ptSelect.y = ptw->ptCursor.y;
                }
              ptw->ptCursor.x = ps.x / ptw->nFontSize + ptw->ptTop.x;
              ptw->ptCursor.y = ps.y / (ptw->nFontSize * 2) + ptw->ptTop.y;
              if (ptw->ptCursor.y < 0)
                ptw->ptCursor.y = 0;
              else if (ptw->ptCursor.y > ptw->nMax - 1)
                ptw->ptCursor.y = ptw->nMax - 1;
              ptw->ptCursor.x = GetAlignPos (&ptw->lpStart, &ptw->nOff,
                        ptw->ptCursor.x, ptw->ptCursor.y, ptw->nTab, FALSE);
            }
          else if (ptw->ptSelect.x >= 0)
            {
              /* ja:選択を解除する */
              ptw->ptCursor = ptw->ptSelect;
              ptw->ptSelect.x = -1;
            }
          if (ptw->ptSelect.x == ptw->ptCursor.x
                                        && ptw->ptSelect.y == ptw->ptCursor.y)
            ptw->ptSelect.x = -1;/* ja:選択の初めと終わりが同じとき */
          if (ptw->ptCursor.x != ptCursor.x || ptw->ptCursor.y != ptCursor.y
                        || ptw->ptTop.x != ptTop.x || ptw->ptTop.y != ptTop.y)
            {
              SetMenuBar (ptw);
              DrawCaret (hWnd);
              ClearSel (hWnd, &ptw->ptCursor, &ptCursor);
            }
          if (ptw->ptTop.x != ptTop.x || ptw->ptTop.y != ptTop.y)
            UpdateWindow (hWnd);
          ShowCaret (hWnd);
        }
        return 0;
      case WM_LBUTTONDBLCLK:
        fDblClk = TRUE;
        return 0;
      case WM_KEYDOWN:
        switch (wParam)
          {
            case VK_HOME:
            case VK_END:
            case VK_LEFT:
            case VK_RIGHT:
            case VK_UP:
            case VK_DOWN:
            case VK_PRIOR:
            case VK_NEXT:
              {
                int i, sx, sy, nScreenPos, nWidth;
                BOOL fCtrl;
                POINT ptCursor, ptTop, ptSelect;
                LPLINEBUF p;
                LPTEXTWND ptw;

                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                ptCursor = ptw->ptCursor;
                ptTop = ptw->ptTop;
                sx = max (ptw->siWnd.cx / ptw->nFontSize, 1);
                sy = max (ptw->siWnd.cy / (ptw->nFontSize * 2), 1);
                fCtrl = GetKeyState (VK_CONTROL) & 0x8000;
                switch (wParam)
                  {
                    case VK_HOME:
                      ptw->ptCursor.x = 0;
                      if (fCtrl)
                        ptw->ptCursor.y = 0;
                      break;
                    case VK_END:
                      if (fCtrl)
                        ptw->ptCursor.y = ptw->nMax - 1;
                      ptw->ptCursor.x = GetWidth (&ptw->lpStart, &ptw->nOff,
                                                ptw->ptCursor.y, ptw->nTab);
                      break;
                    case VK_LEFT:
                      ptw->ptCursor.x = GetAlignPos (&ptw->lpStart, &ptw->nOff,
                        ptw->ptCursor.x, ptw->ptCursor.y, ptw->nTab, FALSE);
                      if (ptw->ptCursor.x > 0)
                        {
                          ptw->ptCursor.x = fCtrl
                                    ? GetMovePos (&ptw->lpStart, &ptw->nOff,
                                            ptw->ptCursor.x, ptw->ptCursor.y,
                                                            ptw->nTab, FALSE)
                                    : GetAlignPos (&ptw->lpStart, &ptw->nOff,
                                        ptw->ptCursor.x - 1, ptw->ptCursor.y,
                                                            ptw->nTab, FALSE);
                        }
                      else if (ptw->ptCursor.y > 0)
                        {
                          /* ja:前の行の最後 */
                          ptw->ptCursor.y--;
                          ptw->ptCursor.x = GetWidth (&ptw->lpStart,
                                    &ptw->nOff, ptw->ptCursor.y, ptw->nTab);
                        }
                      break;
                    case VK_RIGHT:
                      nWidth = GetWidth (&ptw->lpStart, &ptw->nOff,
                                                ptw->ptCursor.y, ptw->nTab);
                      if (ptw->ptCursor.x < nWidth)
                        {
                          ptw->ptCursor.x = fCtrl
                                    ? GetMovePos (&ptw->lpStart, &ptw->nOff,
                                        ptw->ptCursor.x, ptw->ptCursor.y,
                                                            ptw->nTab, TRUE)
                                    : GetAlignPos (&ptw->lpStart, &ptw->nOff,
                                        ptw->ptCursor.x + 1, ptw->ptCursor.y,
                                                            ptw->nTab, TRUE);
                        }
                      else if (ptw->ptCursor.y < ptw->nMax - 1)
                        {
                          /* ja:次の行の最初 */
                          ptw->ptCursor.x = 0;
                          ptw->ptCursor.y++;
                        }
                      break;
                    case VK_UP:
                      if (!fCtrl)
                        {
                          ptw->ptCursor.y--;
                        }
                      else if (ptw->ptTop.y > 0)
                        {
                          ptw->ptTop.y--;
                          if (ptw->ptCursor.y - sy + 1 > ptw->ptTop.y)
                            ptw->ptCursor.y--;
                        }
                      break;
                    case VK_DOWN:
                      if (!fCtrl)
                        {
                          ptw->ptCursor.y++;
                        }
                      else if (ptw->ptTop.y < ptw->nMax - sy)
                        {
                          ptw->ptTop.y++;
                          if (ptw->ptCursor.y < ptw->ptTop.y)
                            ptw->ptCursor.y++;
                        }
                      break;
                    case VK_PRIOR:
                      if (fCtrl)
                        {
                          ptw->ptCursor.y = ptw->ptTop.y;
                        }
                      else
                        {
                          ptw->ptCursor.y -= sy;
                          ptw->ptTop.y -= sy;
                        }
                      break;
                    case VK_NEXT:
                      if (fCtrl)
                        {
                          ptw->ptCursor.y = ptw->ptTop.y + sy - 1;
                        }
                      else
                        {
                          ptw->ptCursor.y += sy;
                          ptw->ptTop.y += sy;
                        }
                  }
                if (ptw->ptCursor.y < 0)
                  ptw->ptCursor.y = 0;
                else if (ptw->ptCursor.y > ptw->nMax - 1)
                  ptw->ptCursor.y = ptw->nMax - 1;
                if (ptw->ptTop.y < 0)
                  ptw->ptTop.y = 0;
                else if (ptw->ptTop.y > ptw->nMax - sy)
                  ptw->ptTop.y = max (ptw->nMax - sy, 0);
                if (ptw->ptCursor.x < ptw->ptTop.x)
                  ptw->ptTop.x = ptw->ptCursor.x;
                else if (ptw->ptCursor.x - sx + 1 > ptw->ptTop.x)
                  ptw->ptTop.x = ptw->ptCursor.x - sx + 1;
                if (ptw->ptCursor.y < ptw->ptTop.y)
                  ptw->ptTop.y = ptw->ptCursor.y;
                else if (ptw->ptCursor.y - sy + 1 > ptw->ptTop.y)
                  ptw->ptTop.y = ptw->ptCursor.y - sy + 1;
                ptSelect = ptw->ptSelect;
                if (GetKeyState (VK_SHIFT) & 0x8000)
                  {
                    if (ptw->ptSelect.x < 0)
                      {
                        /* ja:新たに選択 */
                        ptw->ptSelect.x = GetAlignPos (&ptw->lpStart,
                                                                    &ptw->nOff,
                                    ptCursor.x, ptCursor.y, ptw->nTab, FALSE);
                        ptw->ptSelect.y = ptCursor.y;
                      }
                    if (ptw->ptSelect.y == ptw->ptCursor.y
                            && ptw->ptSelect.x == GetAlignPos (&ptw->lpStart,
                                &ptw->nOff, ptw->ptCursor.x, ptw->ptCursor.y,
                                                            ptw->nTab, FALSE))
                      ptw->ptSelect.x = -1;/* ja:選択範囲が同じとき */
                  }
                else
                  {
                    /* ja:選択解除 */
                    ptw->ptSelect.x = -1;
                  }
                if (ptw->ptCursor.x == ptCursor.x
                                            && ptw->ptCursor.y == ptCursor.y
                        && ptw->ptTop.x == ptTop.x && ptw->ptTop.y == ptTop.y
                                            && ptw->ptSelect.x == ptSelect.x
                                            && ptw->ptSelect.y == ptSelect.y)
                  return 0;
                HideCaret (hWnd);
                if (ptw->ptSelect.x != ptSelect.x)
                  SetMenuBar (ptw);
                DrawCaret (hWnd);
                MoveTextWindow (hWnd, &ptTop);
                if (ptSelect.x >= 0 && ptw->ptSelect.x < 0)
                  {
                    ClearSel (hWnd, &ptSelect, &ptCursor);
                  }
                else if (ptw->ptSelect.x >= 0)
                  {
                    ClearSel (hWnd, &ptw->ptCursor, &ptCursor);
                  }
                ShowCaret (hWnd);
              }
              return 0;
            case VK_BACK:
              {
                LPDOING d;
                LPTEXTWND ptw;
                POINT ptSelect;

                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                ptSelect = ptw->ptSelect;
                if (ptw->ptSelect.x < 0)
                  {
                    ptw->ptCursor.x = GetAlignPos (&ptw->lpStart, &ptw->nOff,
                        ptw->ptCursor.x, ptw->ptCursor.y, ptw->nTab, FALSE);
                    if (ptw->ptCursor.x > 0)
                      {
                        ptw->ptSelect.x = GetKeyState (VK_CONTROL) & 0x8000
                                    ? GetMovePos (&ptw->lpStart, &ptw->nOff,
                                            ptw->ptCursor.x, ptw->ptCursor.y,
                                                            ptw->nTab, FALSE)
                                    : GetAlignPos (&ptw->lpStart, &ptw->nOff,
                                        ptw->ptCursor.x - 1, ptw->ptCursor.y,
                                                            ptw->nTab, FALSE);
                        ptw->ptSelect.y=ptw->ptCursor.y;
                      }
                    else if (ptw->ptCursor.y > 0)
                      {
                        /* ja:2つの行を1つにする */
                        ptw->ptSelect.y = ptw->ptCursor.y - 1;
                        ptw->ptSelect.x = GetWidth (&ptw->lpStart, &ptw->nOff,
                                                ptw->ptSelect.y, ptw->nTab);
                      }
                    else
                      {
                        return 0;
                      }
                  }
                d = EditOperation (hWnd, NULL, 0, FALSE, FALSE);
                if (!d)
                  return 0;
                d->next = ptw->lpUndo;
                ptw->lpUndo = d;
                if (DeleteList (&ptw->lpRedo) > 0 || !d->next
                                                            || ptSelect.x >= 0)
                  {
                    HideCaret (hWnd);
                    SetMenuBar (ptw);
                    ShowCaret (hWnd);
                  }
                ptw->fEdit = TRUE;
              }
              return 0;
            case VK_DELETE:
              {
                LPDOING d;
                LPTEXTWND ptw;
                POINT ptSelect;

                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                ptSelect = ptw->ptSelect;
                if (ptw->ptSelect.x < 0)
                  {
                    int nWidth;

                    nWidth = GetWidth (&ptw->lpStart, &ptw->nOff,
                                                ptw->ptCursor.y, ptw->nTab);
                    ptw->ptCursor.x = GetAlignPos (&ptw->lpStart, &ptw->nOff,
                        ptw->ptCursor.x, ptw->ptCursor.y, ptw->nTab, FALSE);
                    if (ptw->ptCursor.x < nWidth)
                      {
                        ptw->ptSelect.x = GetKeyState (VK_CONTROL) & 0x8000
                                    ? GetMovePos (&ptw->lpStart, &ptw->nOff,
                                            ptw->ptCursor.x, ptw->ptCursor.y,
                                                            ptw->nTab, TRUE)
                                    : GetAlignPos (&ptw->lpStart, &ptw->nOff,
                                        ptw->ptCursor.x + 1, ptw->ptCursor.y,
                                                            ptw->nTab, TRUE);
                        ptw->ptSelect.y = ptw->ptCursor.y;
                      }
                    else if (ptw->ptCursor.y < ptw->nMax - 1)
                      {
                        /* ja:2つの行を1つにする */
                        ptw->ptSelect.x = 0;
                        ptw->ptSelect.y = ptw->ptCursor.y + 1;
                      }
                    else
                      {
                        return 0;
                      }
                  }
                d = EditOperation (hWnd, NULL, 0, FALSE, FALSE);
                if (!d)
                  return 0;
                d->next = ptw->lpUndo;
                ptw->lpUndo = d;
                if (DeleteList (&ptw->lpRedo) > 0 || !d->next
                                                            || ptSelect.x >= 0)
                  {
                    HideCaret (hWnd);
                    SetMenuBar (ptw);
                    ShowCaret (hWnd);
                  }
                ptw->fEdit = TRUE;
              }
              return 0;
            case VK_ESCAPE:
              {
                int sx, sy;
                POINT ptTop;
                LPTEXTWND ptw;

                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                if (ptw->ptSelect.x < 0)
                  return 0;
                ptTop = ptw->ptTop;
                sx = max (ptw->siWnd.cx / ptw->nFontSize, 1);
                sy = max (ptw->siWnd.cy / (ptw->nFontSize * 2), 1);
                if (ptw->ptCursor.x < ptw->ptTop.x)
                  ptw->ptTop.x = ptw->ptCursor.x;
                else if (ptw->ptCursor.x - sx + 1 > ptw->ptTop.x)
                  ptw->ptTop.x = ptw->ptCursor.x - sx + 1;
                if (ptw->ptCursor.y < ptw->ptTop.y)
                  ptw->ptTop.y = ptw->ptCursor.y;
                else if (ptw->ptCursor.y - sy + 1 > ptw->ptTop.y)
                  ptw->ptTop.y = ptw->ptCursor.y - sy + 1;
                HideCaret (hWnd);
                SetMenuBar (ptw);
                MoveTextWindow (hWnd, &ptTop);
                ClearSel (hWnd, &ptw->ptSelect, &ptw->ptCursor);
                ptw->ptSelect.x = -1;
                DrawCaret (hWnd);
                ShowCaret (hWnd);
              }
              return 0;
            case VK_TAB:
              {
                LPDOING d;
                LPTEXTWND ptw;
                POINT ptSelect;

                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                ptSelect = ptw->ptSelect;
                if (ptw->fTabConv ^ GetKeyState (VK_SHIFT) & 0x8000)
                  {
                    int n, nAlignPos;
                    LPSTR lpszText;

                    nAlignPos = GetAlignPos (&ptw->lpStart, &ptw->nOff,
                        ptw->ptCursor.x, ptw->ptCursor.y, ptw->nTab, FALSE);
                    n = (nAlignPos / ptw->nTab + 1) * ptw->nTab - nAlignPos;
                    lpszText = MemoryAlloc (n * sizeof (CHAR));
                    MemorySet (lpszText, ' ', n * sizeof (CHAR));
                    d = EditOperation (hWnd, lpszText, n, TRUE, FALSE);
                    MemoryFree (lpszText);
                  }
                else
                  {
                    d = EditOperation (hWnd, "\t", 1, TRUE, FALSE);
                  }
                if (!d)
                  return 0;
                d->next = ptw->lpUndo;
                ptw->lpUndo = d;
                if (DeleteList (&ptw->lpRedo) > 0 || !d->next
                                                            || ptSelect.x >= 0)
                  {
                    HideCaret (hWnd);
                    SetMenuBar (ptw);
                    ShowCaret (hWnd);
                  }
                ptw->fEdit = TRUE;
              }
              return 0;
            case VK_RETURN:
              {
                LPDOING d;
                LPTEXTWND ptw;
                POINT ptSelect;

                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                ptSelect = ptw->ptSelect;
                if (ptw->fAutoIndent)
                  {
                    int i, nDataPos;
                    LPLINEBUF p;
                    LPSTR lpszText;

                    nDataPos = GetDataPos (&ptw->lpStart, &ptw->nOff,
                        ptw->ptCursor.x, ptw->ptCursor.y, ptw->nTab, FALSE);
                    p = GetLineBuffer (&ptw->lpStart, &ptw->nOff,
                                                            ptw->ptCursor.y);
                    for (i = 0; i < nDataPos; i++)
                      if (p->lpszText[i] != '\t' && p->lpszText[i] != ' ')
                        break;
                    lpszText = MemoryAlloc ((i + 2) * sizeof (CHAR));
                    lpszText[0] = cCRLF[0];
                    lpszText[1] = cCRLF[1];
                    MemoryCopy (lpszText + 2, p->lpszText, i * sizeof (CHAR));
                    d = EditOperation (hWnd, lpszText, i + 2, TRUE, FALSE);
                    MemoryFree (lpszText);
                  }
                else
                  {
                    d = EditOperation (hWnd, cCRLF, 2, TRUE, FALSE);
                  }
                if (!d)
                  return 0;
                d->next = ptw->lpUndo;
                ptw->lpUndo = d;
                if (DeleteList (&ptw->lpRedo) > 0 || !d->next
                                                            || ptSelect.x >= 0)
                  {
                    HideCaret (hWnd);
                    SetMenuBar (ptw);
                    ShowCaret (hWnd);
                  }
                ptw->fEdit = TRUE;
              }
              return 0;
            case VK_INSERT:
              {
                LPTEXTWND ptw;

                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                uIns = (uIns == 0);
                CreateCaret (hWnd, (HBITMAP)0,
                        uIns != 0 ? ptw->nFontSize : 2, ptw->nFontSize * 2);
                DrawCaret (hWnd);
                ShowCaret (hWnd);
              }
          }
        return 0;
      case WM_CHAR:
        {
          static CHAR szText[2] = {'\0', '\0'};
          int nSize;
          LPDOING d;
          LPTEXTWND ptw;
          POINT ptSelect;
#ifndef UNICODE
          CHAR szCodePage[WCOMMON_BUFFER_SIZE];
          UINT uCodePage;
#endif /* not UNICODE */

          /* ja:入力ロケールの取得 */
#ifdef UNICODE
          nSize = WideCharToMultiByte (CP_ACP, 0, (LPCWSTR)&wParam, 1,
                                                        szText, 2, NULL, NULL);
          if (nSize <= 0 || nSize == 1 && IsCharControlA (szText[0]))
            return 0;
#else /* not UNICODE */
          GetLocaleInfo (MAKELCID (LOWORD (GetKeyboardLayout (0)),
                                                                SORT_DEFAULT),
                                            LOCALE_IDEFAULTANSICODEPAGE,
                                            szCodePage, WCOMMON_BUFFER_SIZE);
          uCodePage = StringToIntegerA (szCodePage, NULL, 0);
          if (szText[0] == '\0')
            {
              if (IsDBCSLeadByteEx (uCodePage, (BYTE)wParam))
                {
                  szText[0] = wParam;
                  return 0;
                }
              else if (IsCharControlA ((CHAR)wParam))
                {
                  return 0;
                }
            }
          if (szText[0] == '\0')
            {
              szText[0] = (CHAR)wParam;
              nSize = 1;
            }
          else
            {
              szText[1] = (CHAR)wParam;
              nSize = 2;
            }
          if (uCodePage != CP_SJIS)
            {
              /* ja:コードページを変換する */
              WCHAR wText;

              MultiByteToWideChar (uCodePage, 0, szText, nSize, &wText, 1);
              nSize = WideCharToMultiByte (CP_ACP, 0, &wText, 1,
                                                        szText, 2, NULL, NULL);
            }
#endif /* not UNICODE */
          if (nSize <= 0)
            {
              szText[0] = '\0';
              return 0;
            }
          ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
          ptSelect = ptw->ptSelect;
          ptw->ptCursor.x = GetAlignPos (&ptw->lpStart, &ptw->nOff,
                        ptw->ptCursor.x, ptw->ptCursor.y, ptw->nTab, FALSE);
          if (ptw->ptSelect.x < 0 && uIns != 0)
            {
              ptw->ptSelect.x = GetAlignPos (&ptw->lpStart, &ptw->nOff,
                        ptw->ptCursor.x + 1, ptw->ptCursor.y, ptw->nTab, TRUE);
              if (ptw->ptSelect.x <= ptw->ptCursor.x)
                ptw->ptSelect.x = -1;
              ptw->ptSelect.y = ptw->ptCursor.y;
            }
          d = EditOperation (hWnd, szText, nSize, TRUE, FALSE);
#ifndef UNICODE
          szText[0]='\0';
#endif /* not UNICODE */
          if (!d)
            return 1;
          d->next = ptw->lpUndo;
          ptw->lpUndo = d;
          if (DeleteList (&ptw->lpRedo) > 0 || !d->next || ptSelect.x >= 0)
            {
              HideCaret (hWnd);
              SetMenuBar (ptw);
              ShowCaret (hWnd);
            }
          ptw->fEdit = TRUE;
        }
        return 0;
      case WM_CONTEXTMENU:
        {
          LPTEXTWND ptw;

          ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
          EnableMenuItem (hMenuPopup, CM_UNDO, ptw->lpUndo
                                                    ? MF_ENABLED : MF_GRAYED);
          EnableMenuItem (hMenuPopup, CM_CUT, ptw->ptSelect.x >= 0
                                                    ? MF_ENABLED : MF_GRAYED);
          EnableMenuItem (hMenuPopup, CM_COPY, ptw->ptSelect.x >= 0
                                                    ? MF_ENABLED : MF_GRAYED);
          EnableMenuItem (hMenuPopup, CM_PASTE, fClipBoard
                                                    ? MF_ENABLED : MF_GRAYED);
          EnableMenuItem (hMenuPopup, CM_DELETE, ptw->ptSelect.x >= 0
                                                    ? MF_ENABLED : MF_GRAYED);
          TrackPopupMenu (hMenuPopup, TPM_RIGHTBUTTON,
                            LOWORD (lParam), HIWORD (lParam), 0, hWnd, NULL);
        }
        return 0;
      case WM_COMMAND:
        switch (LOWORD (wParam))
          {
            case CM_SAVE:
              {
                LPTEXTWND ptw;

                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                if (!ptw->fCreate)
                  {
                    BOOL fResult;

                    fResult = !ptw->fOverWrite
                                            || !PathFileExists (ptw->lpszFile);
                    if (!fResult)
                      {
                        LPTSTR lpszFormat, lpszText;

                        lpszFormat = LoadText (GetModuleHandle (NULL),
                                                            IDS_PROMPT_WRITE);
                        wasprintf (&lpszText, lpszFormat, ptw->lpszFile);
                        MemoryFree (lpszFormat);
                        fResult = MessageBox (hWnd, lpszText, APPLICATION,
                                    MB_YESNO | MB_ICONINFORMATION) == IDYES;
                        MemoryFree (lpszText);
                      }
                    if (fResult && SaveTextFile (ptw->lpszFile,ptw))
                      {
                        ptw->fEdit = FALSE;
                        if (DeleteList (&ptw->lpUndo)
                                            + DeleteList (&ptw->lpRedo) > 0)
                          SetMenuBar (ptw);
                      }
                    return 0;
                  }
              }
            case CM_SAVEAS:
              {
                int i, nPos = 0;
                LPTEXTWND ptw;
                LPTSTR lpszName, lpszFilter = NULL;
                OPENFILENAME ofn;
                TCHAR szFile[MAX_PATH];

                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                /* ja:フィルタの設定 */
                for (i = 0; i < nFileType; i++)
                  {
                    int nLength;

                    nLength = lstrlen (lpFileType[i].lpszText) + 1;
                    lpszFilter = MemoryReAlloc (lpszFilter,
                                        (nPos + nLength + 1) * sizeof (TCHAR));
                    MemoryCopy (lpszFilter + nPos, lpFileType[i].lpszText,
                                                    nLength * sizeof (TCHAR));
                    nPos += nLength;
                    nLength = lstrlen (lpFileType[i].lpszExt) + 1;
                    lpszFilter = MemoryReAlloc (lpszFilter,
                                        (nPos + nLength + 1) * sizeof (TCHAR));
                    MemoryCopy (lpszFilter + nPos, lpFileType[i].lpszExt,
                                                    nLength * sizeof (TCHAR));
                    nPos += nLength;
                  }
                lpszName = PathFindFileName (ptw->lpszFile);
                if (lpszName && lstrlen (lpszName) < MAX_PATH)
                  lstrcpy (szFile, lpszName);
                else
                  szFile[0] = '\0';
                ofn.lStructSize =  sizeof(OPENFILENAME);
                ofn.hwndOwner = hWndClient;
                ofn.lpstrFilter = lpszFilter;
                ofn.lpstrCustomFilter = NULL;
                ofn.nFilterIndex = dwFilter + 1;
                ofn.lpstrFile = szFile;
                ofn.nMaxFile = MAX_PATH;
                ofn.lpstrFileTitle = NULL;
                ofn.lpstrInitialDir = lpszSavePath;
                ofn.lpstrTitle = NULL;
                ofn.Flags = OFN_HIDEREADONLY | OFN_NOCHANGEDIR
                                    | OFN_OVERWRITEPROMPT | OFN_PATHMUSTEXIST;
                ofn.lpstrDefExt = NULL;
                if (GetSaveFileName (&ofn))
                  {
                    dwFilter = ofn.nFilterIndex - 1;
                    MemoryFree (lpszSavePath);
                    lpszSavePath = StringDuplicateEx (szFile, ofn.nFileOffset);
                    if (SaveTextFile (szFile, ptw))
                      {
                        LPTSTR lpszTitle;

                        ptw->fCreate = ptw->fEdit = FALSE;
                        DeleteEditFile (hWnd, ptw->lpszFile);
                        lpszTitle = AddEditFile (hWnd, szFile, &ptw->nSame);
                        ptw->lpszFile = StringDuplicate (szFile);
                        DeleteList (&ptw->lpUndo);
                        DeleteList (&ptw->lpRedo);
                        SetIcon (hWnd, ptw->lpszFile);
                        SetWindowText (hWnd, lpszTitle);
                        MemoryFree (lpszTitle);
                      }
                  }
                MemoryFree (lpszFilter);
              }
              return 0;
            case CM_RELOAD:
              {
                RELOADDLG ReloadDlg;
                HINSTANCE hInstance;
                LPTEXTWND ptw;

                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                hInstance = GetModuleHandle (NULL);
                if (ptw->fEdit)
                  {
                    int nResult;
                    LPTSTR lpszFormat, lpszText;

                    lpszFormat = LoadText (hInstance, IDS_PROMPT_RELOAD);
                    wasprintf (&lpszText, lpszFormat, ptw->lpszFile);
                    MemoryFree (lpszFormat);
                    nResult = MessageBox (hWnd, lpszText, APPLICATION,
                                                MB_YESNO | MB_ICONINFORMATION);
                    MemoryFree (lpszText);
                    if (nResult != IDYES)
                      return 0;
                  }
                ReloadDlg.uRetCode = ptw->uRetCode;
                ReloadDlg.uCharSet = ptw->uCharSet;
                if (DialogBoxParamGUI (hInstance, MAKEINTRESOURCE (DIALOG_2),
                            hWnd, ReloadDlgProc, (LPARAM)&ReloadDlg) == IDOK)
                  {
                    /* ja:バッファを解放する */
                    ptw->uRetCode = ReloadDlg.uRetCode;
                    ptw->uCharSet = ReloadDlg.uCharSet;
                    while (ptw->lpStart->prev)
                      ptw->lpStart = ptw->lpStart->prev;
                    while (ptw->lpStart)
                      {
                        LPLINEBUF p;

                        p = ptw->lpStart->next;
                        MemoryFree (ptw->lpStart->lpszText);
                        MemoryFree (ptw->lpStart);
                        ptw->lpStart = p;
                      }
                    /* ja:再読み込み */
                    if (!OpenTextFile (ptw,
                                    ptw->uCharSet + 1, FALSE, ptw->uRetCode))
                      {
                        LPLINEBUF p;

                        p = MemoryAlloc (sizeof (LINEBUF));
                        p->nLength = 0;
                        p->fMargin = FALSE;
                        p->lpszText = NULL;
                        p->prev = p->next = NULL;
                        ptw->nMax = 1;
                        ptw->nOff = 0;
                        ptw->lpStart = p;
                      }
                    DeleteList (&ptw->lpUndo);
                    DeleteList (&ptw->lpRedo);
                    ptw->ptCursor.x = ptw->ptCursor.y = 0;
                    ptw->ptTop.x = ptw->ptTop.y = 0;
                    ptw->ptSelect.x = -1;
                    ptw->fEdit = FALSE;
                    SetScrollBar (hWnd, SB_HORZ, 0, GetWidthMax (ptw),
                        max (ptw->siWnd.cx / ptw->nFontSize, 1), ptw->ptTop.x);
                    SetScrollBar (hWnd,SB_VERT, 0, ptw->nMax - 1,
                        max (ptw->siWnd.cy / (ptw->nFontSize * 2), 1),
                                                                ptw->ptTop.y);
                    SetMenuBar (ptw);
                    DrawCaret (hWnd);
                    InvalidateRect (hWnd, NULL, TRUE);
                  }
              }
              return 0;
            case CM_PROPERTY:
              {
                int i;
                LPTEXTWND ptw;
                FILETYPE stFileType;

                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                stFileType.nID = ptw->nID;
                stFileType.nMargin = ptw->nMargin;
                stFileType.nTab = ptw->nTab;
                stFileType.fAssociate = ptw->fAssociate;
                stFileType.fAutoIndent = ptw->fAutoIndent;
                stFileType.fCode = ptw->fCode;
                stFileType.fCRLF = ptw->fCRLF;
                stFileType.fEOF = ptw->fEOF;
                stFileType.fLimit = ptw->fLimit;
                stFileType.fOverWrite = ptw->fOverWrite;
                stFileType.fRecycle = ptw->fRecycle;
                stFileType.fSpace = ptw->fSpace;
                stFileType.fSysColor = ptw->fSysColor;
                stFileType.fTabConv = ptw->fTabConv;
                stFileType.fGline = ptw->fGline;
                stFileType.fMline = ptw->fMline;
                stFileType.fUline = ptw->fUline;
                stFileType.fVline = ptw->fVline;
                MemoryCopy (stFileType.crColor, ptw->crColor,
                                                    sizeof (COLORREF) * 12);
                stFileType.LogFont = ptw->LogFont;
                stFileType.uRetCode = ptw->uRetCode;
                stFileType.uCharSet = ptw->uCharSet;
                if (DialogBoxParamGUI (GetModuleHandle (NULL),
                                MAKEINTRESOURCE (DIALOG_5), hWnd,
                                PropertyDlgProc, (LPARAM)&stFileType) != IDOK)
                  return 0;
                ptw->uRetCode = stFileType.uRetCode;
                ptw->uCharSet = stFileType.uCharSet;
                if (stFileType.nID < 0)
                  {
                    /* ja:無所属のときにはこのウインドウだけ変更 */
                    ChangeProperty (hWnd, &stFileType);
                  }
                else
                  {
                    /* ja:同一ファイルタイプをすべて変更 */
                    ptw->nID = stFileType.nID;
                    for (i = 0; i < nFileType; i++)
                      if (lpFileType[i].nID == stFileType.nID)
                        {
                          lpFileType[i].nMargin = stFileType.nMargin;
                          lpFileType[i].nTab = stFileType.nTab;
                          lpFileType[i].fAutoIndent = stFileType.fAutoIndent;
                          lpFileType[i].fCode = stFileType.fCode;
                          lpFileType[i].fCRLF = stFileType.fCRLF;
                          lpFileType[i].fEOF = stFileType.fEOF;
                          lpFileType[i].fLimit = stFileType.fLimit;
                          lpFileType[i].fOverWrite = stFileType.fOverWrite;
                          lpFileType[i].fRecycle = stFileType.fRecycle;
                          lpFileType[i].fSpace = stFileType.fSpace;
                          lpFileType[i].fSysColor = stFileType.fSysColor;
                          lpFileType[i].fTabConv = stFileType.fTabConv;
                          lpFileType[i].fGline = stFileType.fGline;
                          lpFileType[i].fMline = stFileType.fMline;
                          lpFileType[i].fUline = stFileType.fUline;
                          lpFileType[i].fVline = stFileType.fVline;
                          MemoryCopy (lpFileType[i].crColor,
                                stFileType.crColor, sizeof (COLORREF) * 12);
                          lpFileType[i].LogFont = stFileType.LogFont;
                          break;
                        }
                    EnumChildWindows (hWndClient, PropertyEnumProc,
                                                        (LPARAM)&stFileType);
                  }
              }
              return 0;
            case CM_PRINT:
              PrintOut (hWnd);
              return 0;
            case CM_UNDO:
              HistoryOperation (hWnd, FALSE);
              return 0;
            case CM_REDO:
              HistoryOperation (hWnd, TRUE);
              return 0;
            case CM_CUT:
              SendMessage (hWnd, WM_COMMAND, CM_COPY, (LPARAM)NULL);
              SendMessage (hWnd, WM_COMMAND, CM_DELETE, (LPARAM)NULL);
              return 0;
            case CM_COPY:
              {
                int nLength;
                HGLOBAL hGlobalA, hGlobalW;
                LPSTR lpszTextA;
                LPWSTR lpszTextW;
                LPTEXTWND ptw;

                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                /* ja:選択範囲の文字数 */
                nLength = GetSelByte (&ptw->lpStart, &ptw->nOff,
                                    &ptw->ptSelect, &ptw->ptCursor, ptw->nTab);
                /* ja:メモリ確保 */
                hGlobalA = GlobalAlloc (GMEM_MOVEABLE,
                                                (nLength + 1) * sizeof (CHAR));
                if (!hGlobalA)
                  {
                    MessageBox (hWnd, _T("GlobalAlloc"),
                                        APPLICATION, MB_OK | MB_ICONSTOP);
                    DestroyWindow (hWndMain);
                    return 0;
                  }
                lpszTextA = GlobalLock (hGlobalA);
                if (!lpszTextA)
                  {
                    MessageBox (hWnd, _T("GlobalLock"),
                                        APPLICATION, MB_OK | MB_ICONSTOP);
                    GlobalFree (hGlobalA);
                    DestroyWindow (hWndMain);
                    return 0;
                  }
                /* ja:メモリに文字列をコピー */
                CpySelMem (&ptw->lpStart, &ptw->nOff,
                        &ptw->ptSelect, &ptw->ptCursor, ptw->nTab, lpszTextA);
                lpszTextA[nLength] = '\0';
                /* ja:UNICODEに変換 */
                nLength = MultiByteToWideChar (CP_ACP, 0,
                                                    lpszTextA, -1, NULL, 0);
                hGlobalW = GlobalAlloc (GMEM_MOVEABLE,
                                                    nLength * sizeof (WCHAR));
                if (!hGlobalW)
                  {
                    MessageBox (hWnd, _T("GlobalAlloc"),
                                        APPLICATION, MB_OK | MB_ICONSTOP);
                    GlobalUnlock (hGlobalA);
                    GlobalFree (hGlobalA);
                    DestroyWindow (hWndMain);
                    return 0;
                  }
                lpszTextW = GlobalLock (hGlobalW);
                if (!lpszTextW)
                  {
                    MessageBox (hWnd, _T("GlobalLock"),
                                        APPLICATION, MB_OK | MB_ICONSTOP);
                    GlobalUnlock (hGlobalA);
                    GlobalFree (hGlobalA);
                    GlobalFree (hGlobalW);
                    DestroyWindow (hWndMain);
                    return 0;
                  }
                MultiByteToWideChar (CP_ACP, 0, lpszTextA, -1,
                                                        lpszTextW, nLength);
                GlobalUnlock (hGlobalA);
                GlobalUnlock (hGlobalW);
                /* ja:クリップボード */
                if (!OpenClipboard (hWnd))
                  {
                    MessageBox (hWnd, _T("OpenClipboard"),
                                    APPLICATION, MB_OK | MB_ICONEXCLAMATION);
                    GlobalFree (hGlobalA);
                    GlobalFree (hGlobalW);
                    return 0;
                  }
                if (!EmptyClipboard ())
                  {
                    MessageBox (hWnd, _T("EmptyClipboard"),
                                        APPLICATION, MB_OK | MB_ICONSTOP);
                    GlobalFree (hGlobalA);
                    GlobalFree (hGlobalW);
                    DestroyWindow (hWndMain);
                    return 0;
                  }
                if (!SetClipboardData (CF_TEXT, hGlobalA)
                            || !SetClipboardData (CF_UNICODETEXT, hGlobalW))
                  {
                    MessageBox (hWnd, _T("SetClipboardData"),
                                        APPLICATION, MB_OK | MB_ICONSTOP);
                    GlobalFree (hGlobalA);
                    GlobalFree (hGlobalW);
                    DestroyWindow (hWndMain);
                    return 0;
                  }
                if (!CloseClipboard ())
                  {
                    MessageBox (hWnd, _T("CloseClipboard"),
                                        APPLICATION, MB_OK | MB_ICONSTOP);
                    DestroyWindow (hWndMain);
                  }
              }
              return 0;
            case CM_PASTE:
              {
                LPDOING d;
                HGLOBAL hGlobal;
                LPTEXTWND ptw;
                POINT ptSelect;

                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                ptSelect = ptw->ptSelect;
                if (!OpenClipboard (hWnd))
                  {
                    MessageBox (hWnd, _T("OpenClipboard"),
                                    APPLICATION, MB_OK | MB_ICONEXCLAMATION);
                    return 0;
                  }
                if (IsClipboardFormatAvailable (CF_UNICODETEXT))
                  {
                    /* en:UNICODE */
                    LPSTR lpszTextA;
                    LPWSTR lpszTextW;

                    hGlobal = GetClipboardData (CF_UNICODETEXT);
                    if (!hGlobal)
                      {
                        CloseClipboard ();
                        return 0;
                      }
                    lpszTextW = GlobalLock (hGlobal);
                    if (!lpszTextW)
                      {
                        CloseClipboard ();
                        MessageBox (hWnd, _T("GlobalLock"),
                                            APPLICATION, MB_OK | MB_ICONSTOP);
                        DestroyWindow (hWndMain);
                        return 0;
                      }
                    lpszTextA = StringWideCharToMultiByte (lpszTextW);
                    d = EditOperation (hWnd,
                                lpszTextA, lstrlenA (lpszTextA), TRUE, FALSE);
                    MemoryFree (lpszTextA);
                  }
                else
                  {
                    /* en:ANSI/OEM */
                    LPSTR lpszTextA;

                    hGlobal = GetClipboardData (CF_TEXT);
                    if (!hGlobal)
                      {
                        CloseClipboard ();
                        return 0;
                      }
                    lpszTextA = GlobalLock (hGlobal);
                    if (!lpszTextA)
                      {
                        CloseClipboard ();
                        MessageBox (hWnd, _T("GlobalLock"),
                                            APPLICATION, MB_OK | MB_ICONSTOP);
                        DestroyWindow (hWndMain);
                        return 0;
                      }
                    d = EditOperation (hWnd,
                                lpszTextA, lstrlenA (lpszTextA), TRUE, FALSE);
                  }
                GlobalUnlock (hGlobal);
                if (!CloseClipboard ())
                  {
                    MessageBox (hWnd, _T("CloseClipboard"),
                                        APPLICATION, MB_OK | MB_ICONSTOP);
                    DestroyWindow (hWndMain);
                    return 0;
                  }
                if (d)
                  {
                    d->next = ptw->lpUndo;
                    ptw->lpUndo = d;
                    if (DeleteList (&ptw->lpRedo) > 0 || !d->next
                                                            || ptSelect.x >= 0)
                      {
                        HideCaret (hWnd);
                        SetMenuBar (ptw);
                        ShowCaret (hWnd);
                      }
                    ptw->fEdit = TRUE;
                  }
              }
              return 0;
            case CM_DELETE:
              {
                LPDOING d;
                LPTEXTWND ptw;

                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                d = EditOperation (hWnd, NULL, 0, TRUE, FALSE);
                if (d)
                  {
                    d->next = ptw->lpUndo;
                    ptw->lpUndo = d;
                    DeleteList (&ptw->lpRedo);
                    HideCaret (hWnd);
                    SetMenuBar (ptw);
                    ShowCaret (hWnd);
                    ptw->fEdit = TRUE;
                  }
              }
              return 0;
            case CM_MARGIN:
              {
                LPTSTR lpszText;

                lpszText = LoadText (GetModuleHandle (NULL),
                                                            IDS_PROMPT_MARGIN);
                if (MessageBox (hWnd, lpszText, APPLICATION,
                                    MB_YESNO | MB_ICONINFORMATION) == IDYES)
                  MarginOperation (hWnd);
                MemoryFree (lpszText);
              }
              return 0;
            case CM_TAB:
              {
                LPTSTR lpszText;

                lpszText = LoadText (GetModuleHandle (NULL), IDS_PROMPT_TAB);
                if (MessageBox (hWnd, lpszText, APPLICATION,
                                    MB_YESNO | MB_ICONINFORMATION) == IDYES)
                  TabOperation (hWnd);
                MemoryFree (lpszText);
              }
              return 0;
            case CM_DATE:
              {
                int nLength;
                BOOL fCtrl;
                LPDOING d;
                LPTSTR lpszText;
#ifdef UNICODE
                LPSTR lpszTextA;
#endif /* UNICODE */
                LPTEXTWND ptw;
                POINT ptSelect;

                fCtrl = GetKeyState (VK_CONTROL) & 0x8000;
                nLength = GetDateFormat (LOCALE_USER_DEFAULT, fCtrl
                        ? DATE_LONGDATE : DATE_SHORTDATE, NULL, NULL, NULL, 0)
                    + GetTimeFormat (LOCALE_USER_DEFAULT, TIME_NOSECONDS
                                | TIME_FORCE24HOURFORMAT, NULL, NULL, NULL, 0);
                lpszText = MemoryAlloc (nLength * sizeof (TCHAR));
                if (GetKeyState (VK_MENU) & 0x8000)
                  {
                    int nPos;

                    nPos = GetDateFormat (LOCALE_USER_DEFAULT, fCtrl
                                            ? DATE_LONGDATE : DATE_SHORTDATE,
                                                NULL, NULL, lpszText, nLength);
                    lpszText[nPos - 1] = ' ';
                    GetTimeFormat (LOCALE_USER_DEFAULT,
                                    TIME_NOSECONDS | TIME_FORCE24HOURFORMAT,
                                NULL, NULL, lpszText + nPos, nLength - nPos);
                  }
                else
                  {
                    int nPos;

                    nPos = GetTimeFormat (LOCALE_USER_DEFAULT,
                                    TIME_NOSECONDS | TIME_FORCE24HOURFORMAT,
                                                NULL, NULL, lpszText, nLength);
                    lpszText[nPos - 1] = ' ';
                    GetDateFormat (LOCALE_USER_DEFAULT, fCtrl
                                ? DATE_LONGDATE : DATE_SHORTDATE, NULL, NULL,
                                            lpszText + nPos, nLength - nPos);
                  }
                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                ptSelect = ptw->ptSelect;
#ifdef UNICODE
                lpszTextA = StringWideCharToMultiByte (lpszText);
                d = EditOperation (hWnd,
                                lpszTextA, lstrlenA (lpszTextA), TRUE, FALSE);
                MemoryFree (lpszTextA);
#else /* not UNICODE */
                d = EditOperation (hWnd,
                                    lpszText, lstrlen (lpszText), TRUE, FALSE);
#endif /* not UNICODE */
                MemoryFree (lpszText);
                if (d)
                  {
                    d->next = ptw->lpUndo;
                    ptw->lpUndo = d;
                    if (DeleteList (&ptw->lpRedo) > 0 || !d->next
                                                            || ptSelect.x >= 0)
                      {
                        HideCaret (hWnd);
                        SetMenuBar (ptw);
                        ShowCaret (hWnd);
                      }
                  }
                ptw->fEdit = TRUE;
              }
              return 0;
            case CM_CODE:
              {
                int nCode;
                LPDOING d;
                LPTEXTWND ptw;
                POINT ptSelect;

                nCode = nCodeDef;
                if (DialogBoxParamGUI (GetModuleHandle (NULL),
                                        MAKEINTRESOURCE (DIALOG_6), hWnd,
                                        ValChrDlgProc, (LPARAM)&nCode) != IDOK)
                  return 0;
                nCodeDef = nCode;
                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                ptSelect = ptw->ptSelect;
                d = EditOperation (hWnd, (LPSTR)&nCode, 1, TRUE, FALSE);
                if (d)
                  {
                    d->next = ptw->lpUndo;
                    ptw->lpUndo = d;
                    if (DeleteList (&ptw->lpRedo) > 0 || !d->next
                                                            || ptSelect.x >= 0)
                      {
                        HideCaret (hWnd);
                        SetMenuBar (ptw);
                        ShowCaret (hWnd);
                      }
                  }
                ptw->fEdit = TRUE;
              }
              return 0;
            case CM_JUMP:
              {
                JUMPDLG JumpDlg;
                LPTEXTWND ptw;

                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                JumpDlg.nMax = ptw->nMax;
                JumpDlg.nCursor = ptw->ptCursor.y + 1;
                if (DialogBoxParamGUI (GetModuleHandle (NULL),
                                        MAKEINTRESOURCE (DIALOG_7), hWnd,
                                        JumpDlgProc, (LPARAM)&JumpDlg) == IDOK
                                    && ptw->ptCursor.y != JumpDlg.nCursor - 1)
                  JumpOperation (hWnd, JumpDlg.nCursor);
              }
              return 0;
            case CM_ALL:
              {
                LPTEXTWND ptw;

                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                if (ptw->nMax > 1 || ptw->lpStart->nLength > 0)
                  {
                    int sx, sy;
                    POINT ptTop;

                    ptTop = ptw->ptTop;
                    sx = max (ptw->siWnd.cx / ptw->nFontSize, 1);
                    sy = max (ptw->siWnd.cy / (ptw->nFontSize * 2), 1);
                    ptw->ptSelect.x = ptw->ptSelect.y = 0;
                    ptw->ptCursor.x = GetWidth (&ptw->lpStart, &ptw->nOff,
                                ptw->ptCursor.y = ptw->nMax - 1, ptw->nTab);
                    if (ptw->ptCursor.x < ptw->ptTop.x)
                      ptw->ptTop.x = ptw->ptCursor.x;
                    else if (ptw->ptCursor.x - sx + 1 > ptw->ptTop.x)
                      ptw->ptTop.x = ptw->ptCursor.x - sx + 1;
                    if (ptw->ptCursor.y < ptw->ptTop.y)
                      ptw->ptTop.y = ptw->ptCursor.y;
                    else if (ptw->ptCursor.y - sy + 1 > ptw->ptTop.y)
                      ptw->ptTop.y = ptw->ptCursor.y - sy + 1;
                    HideCaret (hWnd);
                    SetMenuBar (ptw);
                    DrawCaret (hWnd);
                    if (ptw->ptTop.x != ptTop.x)
                      SetScrollBar (hWnd, SB_HORZ, 0,
                                        GetWidthMax (ptw), sx, ptw->ptTop.x);
                    if (ptw->ptTop.y != ptTop.y)
                      SetScrollBar (hWnd, SB_VERT, 0,
                                        ptw->nMax - 1, sy, ptw->ptTop.y);
                    InvalidateRect (hWnd, NULL, TRUE);
                    ShowCaret (hWnd);
                  }
              }
              return 0;
            case CM_FIND:
              {
                int i;
                FINDDLG FindDlg;
                LPTEXTWND ptw;

                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                FindDlg.bArrow = bFindArrow;
                FindDlg.bCase = bFindCase;
                FindDlg.bWidth = bFindWidth;
                FindDlg.lpszText = NULL;
                if (ptw->ptSelect.x >= 0)
                  FindDlg.bArrow = ptw->ptCursor.y < ptw->ptSelect.y
                                        || ptw->ptCursor.y == ptw->ptSelect.y
                                        && ptw->ptCursor.x < ptw->ptSelect.x;
                if (DialogBoxParamGUI (GetModuleHandle (NULL),
                                        MAKEINTRESOURCE (DIALOG_8), hWnd,
                                        FindDlgProc, (LPARAM)&FindDlg) != IDOK)
                  return 0;
                SetMenuBar (ptw);
                bFindArrow = FindDlg.bArrow;
                bFindCase = FindDlg.bCase;
                bFindWidth = FindDlg.bWidth;
                for (i = 0; i < nFind; i++)
                  if (lstrcmp (FindDlg.lpszText, ppszFind[i]) == 0)
                    break;
                if (i == nFind)
                  {
                    if (nFind >= FINDNUM)
                      MemoryFree (ppszFind[FINDNUM - 1]);
                    else
                      nFind++;
                    i = nFind - 1;
                  }
                else
                  {
                    MemoryFree (ppszFind[i]);
                  }
                MemoryCopy (ppszFind + 1, ppszFind, i * sizeof (LPTSTR));
                ppszFind[0] = FindDlg.lpszText;
              }
            case CM_NEXT:
              {
                BOOL fResult;
                LPTEXTWND ptw;

                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                fResult = FindOperation (hWnd, &ptw->ptCursor, &ptw->ptSelect,
                                                                ppszFind[0],
                            GetKeyState (VK_SHIFT) & 0x8000
                            ? !bFindArrow : bFindArrow, bFindCase, bFindWidth);
                if (fResult && ptw->ptSelect.x < 0)
                  {
                    LPTSTR lpszFormat, lpszText;

                    lpszFormat = LoadText (GetModuleHandle (NULL),
                                                            IDS_REPINFO_ZERO);
                    wasprintf (&lpszText, lpszFormat, ppszFind[0]);
                    MemoryFree (lpszFormat);
                    MessageBox (hWnd, lpszText,
                                    APPLICATION, MB_OK | MB_ICONINFORMATION);
                    MemoryFree (lpszText);
                  }
                SetMenuBar (ptw);
              }
              return 0;
            case CM_REPLACE:
              {
                int i, nCount = 0, nResult, nDst;
                BOOL fReplace = FALSE;
                LPDOING d;
                HINSTANCE hInstance;
                LPTEXTWND ptw;
                POINT ptSelect;
                REPLACEDLG ReplaceDlg;
                REPINFODLG RepinfoDlg;
#ifdef UNICODE
                LPSTR lpszDst;
#endif /* not UNICODE */

                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                hInstance = GetModuleHandle (NULL);
                ReplaceDlg.bArrow = bReplaceArrow;
                ReplaceDlg.bCase = bReplaceCase;
                ReplaceDlg.bWidth = bReplaceWidth;
                ReplaceDlg.lpszSrc = ReplaceDlg.lpszDst = NULL;
                if (ptw->ptSelect.x >= 0)
                  ReplaceDlg.bArrow = ptw->ptCursor.y < ptw->ptSelect.y
                                        || ptw->ptCursor.y == ptw->ptSelect.y
                                        && ptw->ptCursor.x < ptw->ptSelect.x;
                nResult = DialogBoxParamGUI (hInstance,
                                        MAKEINTRESOURCE (DIALOG_9), hWnd,
                                        ReplaceDlgProc, (LPARAM)&ReplaceDlg);
                if (nResult == IDCANCEL)
                  return 0;
                bReplaceArrow = ReplaceDlg.bArrow;
                bReplaceCase = ReplaceDlg.bCase;
                bReplaceWidth = ReplaceDlg.bWidth;
                for (i = 0; i < nFind; i++)
                  if (lstrcmp (ReplaceDlg.lpszSrc, ppszFind[i]) == 0)
                    break;
                if (i == nFind)
                  {
                    if (nFind >= FINDNUM)
                      MemoryFree (ppszFind[FINDNUM - 1]);
                    else
                      nFind++;
                    MemoryCopy (ppszFind + 1, ppszFind,
                                                (nFind - 1) * sizeof (LPTSTR));
                    ppszFind[0] = StringDuplicate (ReplaceDlg.lpszSrc);
                  }
                else
                  {
                    LPTSTR lpszText;

                    lpszText = ppszFind[i];
                    MemoryCopy (ppszFind + 1, ppszFind, i * sizeof (LPTSTR));
                    ppszFind[0] = lpszText;
                  }
                for (i = 0; i < nReplace; i++)
                  if (lstrcmp (ReplaceDlg.lpszDst, ppszReplace[i]) == 0)
                    break;
                if (i == nReplace)
                  {
                    if (nReplace >= FINDNUM)
                      MemoryFree (ppszReplace[FINDNUM - 1]);
                    else
                      nReplace++;
                    MemoryCopy (ppszReplace + 1, ppszReplace,
                                            (nReplace - 1) * sizeof (LPTSTR));
                    ppszReplace[0] = StringDuplicate (ReplaceDlg.lpszDst);
                  }
                else
                  {
                    LPTSTR lpszText;

                    lpszText = ppszReplace[i];
                    MemoryCopy (ppszReplace + 1, ppszReplace,
                                                        i * sizeof (LPTSTR));
                    ppszReplace[0] = lpszText;
                  }
                ptSelect = ptw->ptSelect;
                RepinfoDlg.bArrow = ReplaceDlg.bArrow;
                RepinfoDlg.bCase = ReplaceDlg.bCase;
                RepinfoDlg.bWidth = ReplaceDlg.bWidth;
                RepinfoDlg.lpszDst = ReplaceDlg.lpszDst;
#ifdef UNICODE
                lpszDst = StringWideCharToMultiByte (ReplaceDlg.lpszDst);
                nDst = lstrlenA (lpszDst);
#else /* not UNICODE */
                nDst = lstrlenA (ReplaceDlg.lpszDst);
#endif /* not UNICODE */
                do
                  {
                    int nBytes, nDataPos;

                    FindOperation (hWnd, &ptw->ptCursor, &ptSelect,
                                                            ReplaceDlg.lpszSrc,
                                                            ReplaceDlg.bArrow,
                                                            ReplaceDlg.bCase,
                                                            ReplaceDlg.bWidth);
                    if (ptw->ptSelect.x < 0)
                      break;
                    fReplace = TRUE;
                    nBytes = GetSelByte (&ptw->lpStart, &ptw->nOff,
                                    &ptw->ptSelect, &ptw->ptCursor, ptw->nTab);
                    if (nResult < 100)
                      {
#ifdef UNICODE
                        LPSTR lpszSelect;
#endif /* UNICODE */
                        LPTSTR lpszSrc;

#ifdef UNICODE
                        lpszSelect
                                = MemoryAlloc ((nBytes + 1) * sizeof (CHAR));
                        CpySelMem (&ptw->lpStart, &ptw->nOff,
                                                &ptw->ptSelect, &ptw->ptCursor,
                                                        ptw->nTab, lpszSelect);
                        RepinfoDlg.lpszSrc = lpszSrc
                                    = StringMultiByteToWideChar (lpszSelect);
                        MemoryFree (lpszSelect);
#else /* not UNICODE */
                        RepinfoDlg.lpszSrc = lpszSrc
                                = MemoryAlloc ((nBytes + 1) * sizeof (CHAR));
                        CpySelMem (&ptw->lpStart, &ptw->nOff,
                                                &ptw->ptSelect, &ptw->ptCursor,
                                                ptw->nTab, RepinfoDlg.lpszSrc);
#endif /* not UNICODE */
                        RepinfoDlg.ptStart = ptw->ptSelect;
                        RepinfoDlg.ptEnd = ptw->ptCursor;
                        nResult = DialogBoxParamGUI (hInstance,
                                        MAKEINTRESOURCE (DIALOG_A), hWnd,
                                        RepinfoDlgProc, (LPARAM)&RepinfoDlg);
                        MemoryFree (lpszSrc);
                        if (nResult == IDNO || nResult == IDCANCEL)
                          continue;
                      }
                    if (ptSelect.x >= 0 && ptw->ptCursor.y == ptSelect.y
                                                        && ReplaceDlg.bArrow)
                      nDataPos = GetDataPos (&ptw->lpStart, &ptw->nOff,
                                    ptSelect.x, ptSelect.y, ptw->nTab, FALSE);
#ifdef UNICODE
                    d = EditOperation (hWnd, lpszDst, nDst,
                                                    ReplaceDlg.bArrow, TRUE);
#else /* not UNICODE */
                    d = EditOperation (hWnd, ReplaceDlg.lpszDst, nDst,
                                                    ReplaceDlg.bArrow, TRUE);
#endif /* not UNICODE */
                    if (ptSelect.x >= 0 && ptw->ptCursor.y == ptSelect.y
                                                        && ReplaceDlg.bArrow)
                      ptSelect.x = GetScreenPos (&ptw->lpStart, &ptw->nOff,
                            nDataPos + nDst - nBytes, ptSelect.y, ptw->nTab);
                    if (nResult >= 100)
                      nCount++;
                    d->next = ptw->lpUndo;
                    ptw->lpUndo = d;
                    if (DeleteList (&ptw->lpRedo) > 0 || !d->next)
                      {
                        HideCaret (hWnd);
                        SetMenuBar (ptw);
                        ShowCaret (hWnd);
                      }
                    ptw->fEdit = TRUE;
                  }
                while (nResult != IDCANCEL);
#ifdef UNICODE
                MemoryFree (lpszDst);
#endif /* UNICODE */
                if (!fReplace)
                  {
                    LPTSTR lpszFormat, lpszText;

                    lpszFormat = LoadText (hInstance, IDS_REPINFO_ZERO);
                    wasprintf (&lpszText, lpszFormat, ReplaceDlg.lpszSrc);
                    MemoryFree (lpszFormat);
                    MessageBox (hWnd, lpszText,
                                    APPLICATION, MB_OK | MB_ICONINFORMATION);
                    MemoryFree (lpszText);
                  }
                else if (nResult >= 100)
                  {
                    LPTSTR lpszFormat, lpszText;

                    lpszFormat = LoadText (hInstance, IDS_REPINFO_DONE);
                    wasprintf (&lpszText, lpszFormat, nCount);
                    MemoryFree (lpszFormat);
                    MessageBox (hWnd, lpszText,
                                    APPLICATION, MB_OK | MB_ICONINFORMATION);
                    MemoryFree (lpszText);
                  }
                MemoryFree (ReplaceDlg.lpszSrc);
                MemoryFree (ReplaceDlg.lpszDst);
                SetMenuBar (ptw);
              }
              return 0;
            case CM_LINE:
              {
                LPDOING d;
                LPTEXTWND ptw;
                POINT ptSelect;

                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                ptSelect = ptw->ptSelect;
                if (ptw->ptSelect.x < 0)
                  {
                    if (ptw->nMax - 1 <= ptw->ptCursor.y)
                      {
                        ptw->ptCursor.x = 0;
                        ptw->ptSelect.x = GetWidth (&ptw->lpStart, &ptw->nOff,
                                                ptw->ptCursor.y, ptw->nTab);
                        if (ptw->ptSelect.x <= 0)
                          {
                            if (ptw->ptCursor.y <= 0)
                              {
                                ptw->ptSelect.x = -1;
                                return 0;
                              }
                            ptw->ptSelect.y = ptw->ptCursor.y - 1;
                          }
                        else
                          {
                            ptw->ptSelect.y = ptw->ptCursor.y;
                          }
                      }
                    else
                      {
                        ptw->ptCursor.x = ptw->ptSelect.x = 0;
                        ptw->ptSelect.y = ptw->ptCursor.y + 1;
                      }
                  }
                d = EditOperation (hWnd, NULL, 0, FALSE, FALSE);
                if (d)
                  {
                    d->next = ptw->lpUndo;
                    ptw->lpUndo = d;
                    if (DeleteList (&ptw->lpRedo) > 0 || !d->next
                                                            || ptSelect.x >= 0)
                      {
                        HideCaret (hWnd);
                        SetMenuBar (ptw);
                        ShowCaret (hWnd);
                      }
                    ptw->fEdit = TRUE;
                  }
              }
              return 0;
            case CM_MENU:
              {
                int sx, sy;
                POINT ptScreen, ptTop;
                LPTEXTWND ptw;

                ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
                ptTop = ptw->ptTop;
                sx = max (ptw->siWnd.cx / ptw->nFontSize, 1);
                sy = max (ptw->siWnd.cy / (ptw->nFontSize * 2), 1);
                if (ptw->ptCursor.x < ptw->ptTop.x)
                  ptw->ptTop.x = ptw->ptCursor.x;
                else if (ptw->ptCursor.x - sx + 1 > ptw->ptTop.x)
                  ptw->ptTop.x = ptw->ptCursor.x - sx + 1;
                if (ptw->ptCursor.y < ptw->ptTop.y)
                  ptw->ptTop.y = ptw->ptCursor.y;
                else if (ptw->ptCursor.y - sy + 1 > ptw->ptTop.y)
                  ptw->ptTop.y = ptw->ptCursor.y - sy + 1;
                MoveTextWindow (hWnd, &ptTop);
                ptScreen.x = (GetAlignPos (&ptw->lpStart, &ptw->nOff,
                            ptw->ptCursor.x, ptw->ptCursor.y, ptw->nTab, FALSE)
                                            - ptw->ptTop.x) * ptw->nFontSize;
                ptScreen.y = (ptw->ptCursor.y - ptw->ptTop.y)
                                                        * ptw->nFontSize * 2;
                ClientToScreen (hWnd, &ptScreen);
                EnableMenuItem (hMenuPopup, CM_UNDO, ptw->lpUndo
                                                    ? MF_ENABLED : MF_GRAYED);
                EnableMenuItem (hMenuPopup, CM_CUT, ptw->ptSelect.x >= 0
                                                    ? MF_ENABLED : MF_GRAYED);
                EnableMenuItem (hMenuPopup, CM_COPY, ptw->ptSelect.x >= 0
                                                    ? MF_ENABLED : MF_GRAYED);
                EnableMenuItem (hMenuPopup, CM_PASTE, fClipBoard
                                                    ? MF_ENABLED : MF_GRAYED);
                EnableMenuItem (hMenuPopup, CM_DELETE, ptw->ptSelect.x >= 0
                                                    ? MF_ENABLED : MF_GRAYED);
                TrackPopupMenu (hMenuPopup, TPM_RIGHTBUTTON,
                                        ptScreen.x, ptScreen.y, 0, hWnd, NULL);
              }
          }
        return 0;
      case WM_QUERYENDSESSION:
      case WM_CLOSE:
        {
          LPTEXTWND ptw;

          ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
          if (ptw && ptw->fEdit)
            {
              int nResult;
              LPTSTR lpszFormat, lpszText;

              lpszFormat = LoadText (GetModuleHandle (NULL), IDS_PROMPT_SAVE);
              wasprintf (&lpszText, lpszFormat, ptw->lpszFile);
              MemoryFree (lpszFormat);
              nResult = MessageBox (hWnd, lpszText, APPLICATION,
                                        MB_YESNOCANCEL | MB_ICONINFORMATION);
              MemoryFree (lpszText);
              switch (nResult)
                {
                  case IDYES:
                    SendMessage (hWnd, WM_COMMAND, CM_SAVE, (LPARAM)NULL);
                    if (!ptw->fEdit)
                      break;
                  case IDCANCEL:
                    return 0;
                }
            }
        }
        break;
      case WM_DESTROY:
        {
          HICON hIcon;
          LPLINEBUF p;
          LPTEXTWND ptw;

          hIcon = (HICON)SendMessage (hWnd, WM_SETICON, ICON_BIG,
                                                                (LPARAM)NULL);
          if (hIcon)
            DestroyIcon (hIcon);
          hIcon = (HICON)SendMessage (hWnd, WM_SETICON, ICON_SMALL,
                                                                (LPARAM)NULL);
          if (hIcon)
            DestroyIcon (hIcon);
          ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
          DeleteEditFile (hWnd, ptw->lpszFile);
          MemoryFree (ptw->lpszFile);
          if (ptw->hBrushWindow)
            DeleteObject (ptw->hBrushWindow);
          if (ptw->hBrushSpace)
            DeleteObject (ptw->hBrushSpace);
          if (ptw->hBrushCrlf)
            DeleteObject (ptw->hBrushCrlf);
          if (ptw->hBrushHighLight)
            DeleteObject (ptw->hBrushHighLight);
          if (ptw->hPenTab)
            DeleteObject (ptw->hPenTab);
          if (ptw->hPenMargin)
            DeleteObject (ptw->hPenMargin);
          if (ptw->hPenGrid)
            DeleteObject (ptw->hPenGrid);
          if (ptw->hFont)
            DeleteObject (ptw->hFont);
          if (ptw->hFontsm)
            DeleteObject (ptw->hFontsm);
          while (ptw->lpStart->prev)
            ptw->lpStart = ptw->lpStart->prev;
          while (ptw->lpStart)
            {
              p = ptw->lpStart->next;
              if (ptw->lpStart->lpszText)
                MemoryFree (ptw->lpStart->lpszText);
              MemoryFree (ptw->lpStart);
              ptw->lpStart = p;
            }
          DeleteList (&ptw->lpUndo);
          DeleteList (&ptw->lpRedo);
          SetWindowLongPtr (hWnd, GWLP_USERDATA, (LONG_PTR)NULL);
          MemoryFree (ptw);
          fZoomed = IsZoomed (hWnd);
        }
        return 0;
    }
  return DefMDIChildProc (hWnd, uMsg, wParam, lParam);
}
