/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 *  Copyright (C) 2005 Takuro Ashie
 *  Copyright (C) 2006 Juernjakob Harder
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
  #include <config.h>
#endif

#include <stdlib.h>
#include <math.h>

#include "tomoe-canvas.h"

#define TOMOE_CANVAS_DEFAULT_SIZE 300
#define TOMOE_CANVAS_DEFAULT_RATE 0.7

enum {
    FIND_SIGNAL,
    CLEAR_SIGNAL,
    NORMALIZE_SIGNAL,
    STROKE_ADDED_SIGNAL,
    STROKE_REVERTED_SIGNAL,
    LAST_SIGNAL,
};

typedef struct _TomoeCanvasPriv TomoeCanvasPriv;
struct _TomoeCanvasPriv
{
    guint            size;
    gint             width;
    gint             height;

    GdkGC           *handwrite_line_gc;
    GdkGC           *adjust_line_gc;
    GdkGC           *annotate_gc;
    GdkGC           *axis_gc;

    GdkPixmap       *pixmap;
    gboolean         drawing;

    TomoeContext    *context;
    TomoeWriting    *writing;
    GList           *candidates;

    gint             auto_find_time;
    guint            auto_find_id;
    gboolean         locked;
};

#define TOMOE_CANVAS_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), TOMOE_TYPE_CANVAS, TomoeCanvasPriv))

G_DEFINE_TYPE (TomoeCanvas, tomoe_canvas, GTK_TYPE_WIDGET)
/* virtual functions for GtkObject class */
static void   dispose              (GObject           *object);
/* virtual functions for GtkWidget class */
static void   realize              (GtkWidget         *widget);
static void   size_allocate        (GtkWidget         *widget,
                                    GtkAllocation     *allocation);
static gint   expose_event         (GtkWidget         *widget,
                                    GdkEventExpose    *event);
static gint   button_press_event   (GtkWidget         *widget,
                                    GdkEventButton    *event);
static gint   button_release_event (GtkWidget         *widget,
                                    GdkEventButton    *event);
static gint   motion_notify_event  (GtkWidget         *widget,
                                    GdkEventMotion    *event);

static void   tomoe_canvas_real_find            (TomoeCanvas       *canvas);
static void   tomoe_canvas_real_clear           (TomoeCanvas       *canvas);
static void   tomoe_canvas_real_normalize       (TomoeCanvas       *canvas);
static void   tomoe_canvas_append_point         (TomoeCanvas       *canvas,
                                                 gint               x,
                                                 gint               y);
static void   tomoe_canvas_draw_line            (TomoeCanvas       *canvas,
                                                 TomoePoint        *point1,
                                                 TomoePoint        *point2,
                                                 GdkGC             *line_gc,
                                                 gboolean           draw);
static void   tomoe_canvas_draw_background      (TomoeCanvas       *canvas,
                                                 gboolean           draw);
static void   tomoe_canvas_draw_axis            (TomoeCanvas       *canvas);
static void   tomoe_canvas_resize_writing       (TomoeCanvas       *canvas,
                                                 gdouble            x_rate,
                                                 gdouble            y_rate);
static void   tomoe_canvas_move_writing         (TomoeCanvas       *canvas,
                                                 gint               dx,
                                                 gint               dy);

static void   _init_gc                          (TomoeCanvas       *canvas);
static void   draw_stroke                       (GList             *points,
                                                 TomoeCanvas       *canvas,
                                                 guint              index);
static void   draw_annotate                     (GList             *points,
                                                 TomoeCanvas       *canvas,
                                                 guint              index);

static GList*               instance_list = NULL;
static guint                canvas_signals[LAST_SIGNAL] = { 0 };

#define _g_list_free_all(l, free_func) \
{\
    g_list_foreach (l, (GFunc) free_func, NULL); \
    g_list_free (l); \
}

static void
tomoe_canvas_class_init (TomoeCanvasClass *klass)
{
    GObjectClass   *gobject_class = G_OBJECT_CLASS (klass);
    GtkWidgetClass *widget_class  = GTK_WIDGET_CLASS (klass);

    canvas_signals[FIND_SIGNAL] =
      g_signal_new ("find",
  		  G_TYPE_FROM_CLASS (klass),
  		  G_SIGNAL_RUN_LAST,
  		  G_STRUCT_OFFSET (TomoeCanvasClass, find),
  		  NULL, NULL,
  		  g_cclosure_marshal_VOID__VOID,
  		  G_TYPE_NONE, 0);
    canvas_signals[CLEAR_SIGNAL] =
      g_signal_new ("clear",
  		  G_TYPE_FROM_CLASS (klass),
  		  G_SIGNAL_RUN_LAST,
  		  G_STRUCT_OFFSET (TomoeCanvasClass, clear),
  		  NULL, NULL,
  		  g_cclosure_marshal_VOID__VOID,
  		  G_TYPE_NONE, 0);
    canvas_signals[NORMALIZE_SIGNAL] =
      g_signal_new ("normalize",
  		  G_TYPE_FROM_CLASS (klass),
  		  G_SIGNAL_RUN_LAST,
  		  G_STRUCT_OFFSET (TomoeCanvasClass, normalize),
  		  NULL, NULL,
  		  g_cclosure_marshal_VOID__VOID,
  		  G_TYPE_NONE, 0);
    canvas_signals[STROKE_ADDED_SIGNAL] =
      g_signal_new ("stroke-added",
  		  G_TYPE_FROM_CLASS (klass),
  		  G_SIGNAL_RUN_LAST,
  		  G_STRUCT_OFFSET (TomoeCanvasClass, stroke_added),
  		  NULL, NULL,
  		  g_cclosure_marshal_VOID__VOID,
  		  G_TYPE_NONE, 0);
    canvas_signals[STROKE_REVERTED_SIGNAL] =
      g_signal_new ("stroke-reverted",
  		  G_TYPE_FROM_CLASS (klass),
  		  G_SIGNAL_RUN_LAST,
  		  G_STRUCT_OFFSET (TomoeCanvasClass, stroke_reverted),
  		  NULL, NULL,
  		  g_cclosure_marshal_VOID__VOID,
  		  G_TYPE_NONE, 0);

    gobject_class->dispose             = dispose;
    widget_class->realize              = realize;
    widget_class->size_allocate        = size_allocate;
    widget_class->expose_event         = expose_event;
    widget_class->button_press_event   = button_press_event;
    widget_class->button_release_event = button_release_event;
    widget_class->motion_notify_event  = motion_notify_event;

    klass->find                        = tomoe_canvas_real_find;
    klass->clear                       = tomoe_canvas_real_clear;
    klass->normalize                   = tomoe_canvas_real_normalize;
    klass->stroke_added                = NULL;
    klass->stroke_reverted             = NULL;

    g_type_class_add_private (gobject_class, sizeof(TomoeCanvasPriv));
}

static void
tomoe_canvas_init (TomoeCanvas *canvas)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    
    priv->size              = TOMOE_CANVAS_DEFAULT_SIZE;
    priv->width             = TOMOE_CANVAS_DEFAULT_SIZE;
    priv->height            = TOMOE_CANVAS_DEFAULT_SIZE;

    priv->handwrite_line_gc = NULL;
    priv->adjust_line_gc    = NULL;
    priv->annotate_gc       = NULL;
    priv->axis_gc           = NULL;

    priv->pixmap            = NULL;
    priv->drawing           = FALSE;

    priv->context           = NULL;
    priv->writing           = tomoe_writing_new ();
    priv->candidates        = NULL;

    priv->auto_find_time    = 0;
    priv->auto_find_id      = 0;
    priv->locked            = FALSE;

    instance_list = g_list_append (instance_list, (gpointer) canvas);
}

static void
dispose (GObject *object)
{
    TomoeCanvas *canvas = TOMOE_CANVAS (object);
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);

    instance_list = g_list_remove (instance_list, (gpointer) canvas);

    if (priv->handwrite_line_gc) {
        g_object_unref (priv->handwrite_line_gc);
        priv->handwrite_line_gc = NULL;
    }

    if (priv->adjust_line_gc) {
        g_object_unref (priv->adjust_line_gc);
        priv->adjust_line_gc = NULL;
    }

    if (priv->annotate_gc) {
        g_object_unref (priv->annotate_gc);
        priv->annotate_gc = NULL;
    }

    if (priv->axis_gc) {
        g_object_unref (priv->axis_gc);
        priv->axis_gc = NULL;
    }

    if (priv->pixmap) {
        g_object_unref (priv->pixmap);
        priv->pixmap = NULL;
    }

    if (priv->candidates) {
        _g_list_free_all (priv->candidates, g_object_unref);
        priv->candidates     = NULL;
    }

    if (priv->writing) {
        g_object_unref (priv->writing);
        priv->writing = NULL;
    }

    if (priv->auto_find_id) {
        g_source_remove (priv->auto_find_id);
        priv->auto_find_id = 0;
    }

    if (G_OBJECT_CLASS(tomoe_canvas_parent_class)->dispose)
        G_OBJECT_CLASS(tomoe_canvas_parent_class)->dispose(object);
}

GtkWidget *
tomoe_canvas_new (void)
{
    return GTK_WIDGET(g_object_new (TOMOE_TYPE_CANVAS,
                                    NULL));
}

static void
realize (GtkWidget *widget)
{
    PangoFontDescription *font_desc;
    GdkWindowAttr attributes;

    GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

    attributes.window_type = GDK_WINDOW_CHILD;
    attributes.wclass = GDK_INPUT_OUTPUT;
    attributes.visual = gtk_widget_get_visual (widget);
    attributes.colormap = gtk_widget_get_colormap (widget);

    attributes.x = widget->allocation.x;
    attributes.y = widget->allocation.y;
    attributes.width = widget->allocation.width;
    attributes.height = widget->allocation.height;
    attributes.event_mask = GDK_EXPOSURE_MASK |
	    GDK_BUTTON_PRESS_MASK |
	    GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK |
	    GDK_POINTER_MOTION_HINT_MASK |
	    GDK_ENTER_NOTIFY_MASK |
	    GDK_LEAVE_NOTIFY_MASK;

    widget->window = gdk_window_new (gtk_widget_get_parent_window (widget),
		    &attributes,
		    GDK_WA_X | GDK_WA_Y |
		    GDK_WA_COLORMAP |
		    GDK_WA_VISUAL);
    gdk_window_set_user_data (widget->window, widget);
    widget->style = gtk_style_attach (widget->style, widget->window);

    gdk_window_set_background (widget->window, &widget->style->bg [GTK_STATE_NORMAL]);

    font_desc = pango_font_description_from_string ("Sans 12");
    gtk_widget_modify_font (widget, font_desc);
}

static void
size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (widget);

    priv->width = allocation->width;
    priv->height = allocation->height;
    priv->size = allocation->width;

    if (GTK_WIDGET_REALIZED (widget)) {
        if (priv->pixmap)
            g_object_unref(priv->pixmap);

        priv->pixmap = gdk_pixmap_new(widget->window,
                                      allocation->width,
                                      allocation->height,
                                      -1);
    	tomoe_canvas_refresh (TOMOE_CANVAS (widget));
    }
    if (GTK_WIDGET_CLASS (tomoe_canvas_parent_class)->size_allocate)
        GTK_WIDGET_CLASS (tomoe_canvas_parent_class)->size_allocate (widget, allocation);
}

static gint
expose_event (GtkWidget *widget, GdkEventExpose *event)
{
    TomoeCanvas *canvas = TOMOE_CANVAS (widget);
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    gboolean retval = FALSE;

    if (GTK_WIDGET_CLASS(tomoe_canvas_parent_class)->expose_event)
        retval = GTK_WIDGET_CLASS(tomoe_canvas_parent_class)->expose_event (widget,
                                                                            event);
    gdk_draw_drawable(widget->window,
                      widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
                      priv->pixmap,
                      event->area.x, event->area.y,
                      event->area.x, event->area.y,
                      event->area.width, event->area.height);

    return retval;
}

static gint
button_press_event (GtkWidget *widget, GdkEventButton *event)
{
    TomoeCanvas *canvas = TOMOE_CANVAS (widget);
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    gboolean retval = FALSE;

    if (priv->locked) return retval;

    if (priv->auto_find_id) {
        g_source_remove (priv->auto_find_id);
        priv->auto_find_id = 0;
    }

    /* tomoe_canvas_refresh (canvas); */

    if (event->button == 1) {
        priv->drawing = TRUE;
        tomoe_writing_move_to (priv->writing, (gint) event->x, (gint) event->y);
    }

    return retval;
}

static gboolean
timeout_auto_find (gpointer user_data)
{
    tomoe_canvas_find (TOMOE_CANVAS (user_data));
    return FALSE;
}

static gint
button_release_event (GtkWidget *widget, GdkEventButton *event)
{
    TomoeCanvas *canvas = TOMOE_CANVAS (widget);
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    gboolean retval = FALSE;

    if (priv->locked) return retval;
    if (!priv->drawing) return retval;

    /* tomoe_writing_line_to (priv->writing, (gint) event->x, (gint) event->y); */
    /* draw_annotate (points, canvas, tomoe_writing_get_n_strokes (priv->writing) + 1); */

    priv->drawing = FALSE;

    g_signal_emit (G_OBJECT (widget), canvas_signals[STROKE_ADDED_SIGNAL], 0);

    if (priv->auto_find_id) {
        g_source_remove (priv->auto_find_id);
        priv->auto_find_id = 0;
    }
    if (priv->auto_find_time >= 0) {
        priv->auto_find_id = g_timeout_add (priv->auto_find_time,
                                            timeout_auto_find,
                                            (gpointer)canvas);
    }

    return retval;
}

static gint
motion_notify_event (GtkWidget *widget, GdkEventMotion *event)
{
    TomoeCanvas *canvas = TOMOE_CANVAS (widget);
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    gint x, y;
    GdkModifierType state;
    gboolean retval = FALSE;

    if (priv->locked) return retval;
    if (!priv->drawing) return retval;

    if (event->is_hint) {
        gdk_window_get_pointer(event->window, &x, &y, &state);
    } else {
        x = (gint) event->x;
        y = (gint) event->y;
        state = (GdkModifierType) event->state;
    }
    
    tomoe_canvas_append_point (canvas, x, y);

    return retval;
}

static TomoeWriting *
_tomoe_writing_new_scale_writing (TomoeWriting *writing, gdouble sx, gdouble sy)
{
    const GList *strokes, *list;
    TomoeWriting *new = tomoe_writing_new ();

    strokes = tomoe_writing_get_strokes (writing);
    for (list= strokes; list; list = g_list_next (list)) {
        GList *points = (GList *) list->data;
        GList *point;
        gboolean first = TRUE;
        for ( point = points; point; point = g_list_next (point)) {
            TomoePoint *cp = (TomoePoint *) point->data;
            if (!first)
                tomoe_writing_line_to (new, cp->x * sx, cp->y * sy);
            else
                tomoe_writing_move_to (new, cp->x * sx, cp->y * sy);
            first = FALSE;
        }
    }
    return new;
}

static void
tomoe_canvas_real_find (TomoeCanvas *canvas)
{
    GtkWidget *widget = GTK_WIDGET (canvas);
    TomoeCanvasPriv *priv;
    TomoeWriting *writing;
    TomoeQuery *query;
    const GList *strokes, *list;
    guint i;

    g_return_if_fail (TOMOE_IS_CANVAS (canvas));

    priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    if (tomoe_writing_get_n_strokes (priv->writing) == 0)
        return;

    _init_gc (canvas);

    /* draw thin red lines and annotations for sparse points */
    strokes = tomoe_writing_get_strokes (priv->writing);
    for (list = strokes, i = 1; list; list = g_list_next (list), i++) {
        GList *points = (GList *) list->data;
 
        draw_annotate (points, canvas, i);
    }

    if (priv->candidates) {
        _g_list_free_all (priv->candidates, g_object_unref);
        priv->candidates     = NULL;
    }

    writing = _tomoe_writing_new_scale_writing (priv->writing,
                                                (gdouble) TOMOE_WRITING_WIDTH / priv->width,
                                                (gdouble) TOMOE_WRITING_WIDTH / priv->height);

    query = tomoe_query_new ();
    tomoe_query_set_writing (query, writing);
    priv->candidates = tomoe_context_search (priv->context, query);
    g_object_unref (query);
    g_object_unref (writing);
    gdk_draw_drawable(widget->window,
                      widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
                      priv->pixmap,
                      0, 0,
                      0, 0,
                      widget->allocation.width, widget->allocation.height);

}

static void
tomoe_canvas_real_clear (TomoeCanvas *canvas)
{
    TomoeCanvasPriv *priv;

    g_return_if_fail (TOMOE_IS_CANVAS (canvas));

    _init_gc (canvas);
    priv = TOMOE_CANVAS_GET_PRIVATE (canvas);

    tomoe_canvas_draw_background (canvas, TRUE);

    if (priv->candidates) {
        _g_list_free_all (priv->candidates, g_object_unref);
        priv->candidates     = NULL;
    }

    tomoe_writing_clear (priv->writing);

    tomoe_canvas_refresh (canvas);
}

TomoeWriting *
tomoe_canvas_get_writing (TomoeCanvas *canvas)
{
    TomoeWriting *writing;
    TomoeCanvasPriv *priv;

    g_return_val_if_fail (TOMOE_IS_CANVAS (canvas), NULL);

    priv = TOMOE_CANVAS_GET_PRIVATE (canvas);

    writing = _tomoe_writing_new_scale_writing (priv->writing,
                                                (gdouble) TOMOE_WRITING_WIDTH / priv->width,
                                                (gdouble) TOMOE_WRITING_WIDTH / priv->height);
    return writing;
}

void
tomoe_canvas_set_writing (TomoeCanvas *canvas, TomoeWriting *writing)
{
    TomoeCanvasPriv *priv;

    g_return_if_fail (TOMOE_IS_CANVAS (canvas));

    priv = TOMOE_CANVAS_GET_PRIVATE (canvas);

    if (priv->writing)
        g_object_unref (priv->writing);
    priv->writing = writing;

    if (GTK_WIDGET_REALIZED (GTK_WIDGET (canvas)))
    	tomoe_canvas_refresh (canvas);
}

void
tomoe_canvas_lock (TomoeCanvas *canvas, gboolean lock)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    priv->locked = lock;
}

static void
get_rectangle_for_stroke (GList *points, GdkRectangle *rect)
{
    GList *list;

    for (list = points; list; list = g_list_next (list)) {
        gint x = ((TomoePoint*)(list->data))->x;
        gint y = ((TomoePoint*)(list->data))->y;

        rect->x = MIN (rect->x, x);
        rect->y = MIN (rect->y, y);
        rect->width  = MAX (rect->width, x - rect->x);
        rect->height = MAX (rect->height, y - rect->y);
    }
}

static void
get_char_size (TomoeCanvas *canvas, GdkRectangle *rect)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    const GList *strokes, *list;
    TomoeWriting *g = priv->writing;

    rect->x = G_MAXINT;
    rect->y = G_MAXINT;
    rect->width = 0;
    rect->height = 0;

    strokes = tomoe_writing_get_strokes (g);
    for (list= strokes; list; list = g_list_next (list)) {
        GList *points = (GList *) list->data;
        GList *point;
        for (point = points; point; point = g_list_next (point)) {
            get_rectangle_for_stroke (points, rect);
        }
    }
}

static void
tomoe_canvas_resize_writing (TomoeCanvas *canvas, gdouble x_rate, gdouble y_rate)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    TomoeWriting *new_writing;

    new_writing = _tomoe_writing_new_scale_writing (priv->writing, x_rate, y_rate);

    tomoe_canvas_set_writing (canvas, new_writing);
}

static TomoeWriting *
_tomoe_writing_new_move_writing (TomoeWriting *writing, gint dx, gint dy)
{
    const GList *strokes, *list;
    TomoeWriting *new = tomoe_writing_new ();

    strokes = tomoe_writing_get_strokes (writing);
    for (list= strokes; list; list = g_list_next (list)) {
        GList *points = (GList *) list->data;
        GList *point;
        gboolean first = TRUE;
        for ( point = points; point; point = g_list_next (point)) {
            TomoePoint *cp = (TomoePoint *) point->data;
            if (!first)
                tomoe_writing_line_to (new, cp->x + dx, cp->y + dy);
            else
                tomoe_writing_move_to (new, cp->x + dx, cp->y + dy);
            first = FALSE;
        }
    }
    return new;
}

static void
tomoe_canvas_move_writing (TomoeCanvas *canvas, gint dx, gint dy)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    TomoeWriting *new_writing;

    new_writing = _tomoe_writing_new_move_writing (priv->writing, dx, dy);

    tomoe_canvas_set_writing (canvas, new_writing);
}

static void
tomoe_canvas_real_normalize (TomoeCanvas *canvas)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    GdkRectangle char_size;
    gdouble x_rate, y_rate;
    gint dx, dy;

    get_char_size (canvas, &char_size);
    x_rate = (priv->size * TOMOE_CANVAS_DEFAULT_RATE)
           / char_size.width;
    y_rate = (priv->size * TOMOE_CANVAS_DEFAULT_RATE)
           / char_size.height;

    tomoe_canvas_resize_writing (canvas, x_rate, y_rate);

    get_char_size (canvas, &char_size);
    dx = ((priv->size - char_size.width)  / 2) - char_size.x;
    dy = ((priv->size - char_size.height) / 2) - char_size.y;

    tomoe_canvas_move_writing (canvas, dx, dy);

    tomoe_canvas_refresh (canvas);
    tomoe_canvas_find (canvas);
}

void
tomoe_canvas_find (TomoeCanvas *canvas)
{
    g_return_if_fail (TOMOE_IS_CANVAS (canvas));
    g_signal_emit (G_OBJECT (canvas), canvas_signals[FIND_SIGNAL], 0);
}

guint
tomoe_canvas_get_number_of_candidates (TomoeCanvas *canvas)
{
    TomoeCanvasPriv *priv;

    g_return_val_if_fail (TOMOE_IS_CANVAS (canvas), 0);

    priv = TOMOE_CANVAS_GET_PRIVATE (canvas);

    return g_list_length (priv->candidates);
}

TomoeChar *
tomoe_canvas_get_nth_candidate (TomoeCanvas *canvas, guint nth)
{
    TomoeCanvasPriv *priv;

    g_return_val_if_fail (TOMOE_IS_CANVAS (canvas), NULL);

    priv = TOMOE_CANVAS_GET_PRIVATE (canvas);

    if (nth < g_list_length (priv->candidates)) {
        TomoeCandidate *cand;
        cand = g_list_nth_data (priv->candidates, nth);
        return tomoe_candidate_get_char (cand);
    } else {
        return NULL;
    }
}

void
tomoe_canvas_refresh (TomoeCanvas *canvas)
{
    TomoeCanvasPriv *priv;
    GtkWidget *widget;
    guint i;
    const GList *strokes, *list;

    g_return_if_fail (TOMOE_IS_CANVAS (canvas));

    _init_gc (canvas);

    priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    widget = GTK_WIDGET (canvas);

    gdk_draw_rectangle (priv->pixmap,
                        widget->style->white_gc,
                        TRUE,
                        0, 0,
                        widget->allocation.width,
                        widget->allocation.height);

    tomoe_canvas_draw_axis (canvas);

    if (priv->writing) {
        strokes = tomoe_writing_get_strokes (priv->writing);
        for (list = strokes, i = 1; list; list = g_list_next (list), i++) {
            GList *points = (GList *) list->data;
            draw_stroke (points, canvas, i);
        }
    }

    gdk_draw_drawable(widget->window,
                      widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
                      priv->pixmap,
                      0, 0,
                      0, 0,
                      widget->allocation.width, widget->allocation.height);
}

void
tomoe_canvas_revert (TomoeCanvas *canvas)
{
    TomoeCanvasPriv *priv;
    gboolean reverted = TRUE;

    g_return_if_fail (TOMOE_IS_CANVAS (canvas));

    priv = TOMOE_CANVAS_GET_PRIVATE (canvas);

    /* remove last line */
    tomoe_writing_remove_last_stroke (priv->writing);
    /* reverted = tomoe_writing_remove_last_stroke (priv->writing); */
    if (reverted) {
        tomoe_canvas_refresh (canvas);

        g_signal_emit (G_OBJECT (canvas),
                       canvas_signals[STROKE_REVERTED_SIGNAL], 0);

        if (tomoe_writing_get_n_strokes (priv->writing) == 0)
            g_signal_emit (G_OBJECT (canvas), canvas_signals[CLEAR_SIGNAL], 0);

    }
}

void
tomoe_canvas_clear (TomoeCanvas *canvas)
{
    g_return_if_fail (TOMOE_IS_CANVAS (canvas));
    g_signal_emit (G_OBJECT (canvas), canvas_signals[CLEAR_SIGNAL], 0);
}

void
tomoe_canvas_normalize (TomoeCanvas *canvas)
{
    g_return_if_fail (TOMOE_IS_CANVAS (canvas));
    g_signal_emit (G_OBJECT (canvas), canvas_signals[NORMALIZE_SIGNAL], 0);
}

gboolean
tomoe_canvas_has_stroke (TomoeCanvas *canvas)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);

    g_return_val_if_fail (TOMOE_IS_CANVAS (canvas), FALSE);

    return (tomoe_writing_get_n_strokes (priv->writing) > 0);
}

void
tomoe_canvas_set_size (TomoeCanvas *canvas, guint size)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);

    g_return_if_fail (TOMOE_IS_CANVAS (canvas));

    priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    priv->width = size;
    priv->height = size;

    gtk_widget_set_size_request (GTK_WIDGET (canvas), size, size);
    gtk_widget_queue_resize (GTK_WIDGET (canvas));
    if (GTK_WIDGET_REALIZED (GTK_WIDGET (canvas)))
        tomoe_canvas_refresh (canvas);
}

static void
tomoe_canvas_gc_set_foreground (GdkGC *gc, GdkColor *color)
{
    GdkColor default_color = { 0, 0x0000, 0x0000, 0x0000 };

    if (color) {
        gdk_colormap_alloc_color (gdk_colormap_get_system (), color,
                                  TRUE, TRUE);
        gdk_gc_set_foreground (gc, color);
    } else {
        gdk_colormap_alloc_color (gdk_colormap_get_system (), &default_color,
                                  TRUE, TRUE);
        gdk_gc_set_foreground (gc, &default_color);
    }
}

void
tomoe_canvas_set_handwrite_line_color (TomoeCanvas *canvas, GdkColor *color)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);

    g_return_if_fail (TOMOE_IS_CANVAS (canvas));

    priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    tomoe_canvas_gc_set_foreground (priv->handwrite_line_gc, color);
}

void
tomoe_canvas_set_adjust_line_color (TomoeCanvas *canvas, GdkColor *color)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);

    g_return_if_fail (TOMOE_IS_CANVAS (canvas));

    priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    tomoe_canvas_gc_set_foreground (priv->adjust_line_gc, color);
}

void
tomoe_canvas_set_annotate_color (TomoeCanvas *canvas, GdkColor *color)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);

    g_return_if_fail (TOMOE_IS_CANVAS (canvas));

    priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    tomoe_canvas_gc_set_foreground (priv->annotate_gc, color);
}

void
tomoe_canvas_set_axis_color (TomoeCanvas *canvas, GdkColor *color)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);

    g_return_if_fail (TOMOE_IS_CANVAS (canvas));

    priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    tomoe_canvas_gc_set_foreground (priv->axis_gc, color);
}

void
tomoe_canvas_set_auto_find_time (TomoeCanvas *canvas, gint time_msec)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);

    g_return_if_fail (TOMOE_IS_CANVAS (canvas));

    priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    if (time_msec < 0)
        priv->auto_find_time = -1;
    else
        priv->auto_find_time = time_msec;
}

gint
tomoe_canvas_get_auto_find_time (TomoeCanvas *canvas)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);

    g_return_val_if_fail (TOMOE_IS_CANVAS (canvas), -1);

    priv = TOMOE_CANVAS_GET_PRIVATE (canvas);

    return priv->auto_find_time;
}

void
tomoe_canvas_set_context (TomoeCanvas *canvas, TomoeContext *context)
{
    TomoeCanvasPriv *priv;
    g_return_if_fail (TOMOE_IS_CANVAS (canvas));
    priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    priv->context = context;
}

static void
tomoe_canvas_append_point (TomoeCanvas *canvas, gint x, gint y)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    TomoePoint pp, p;
    GList *strokes;
    GList *point = NULL;

    p.x = x;
    p.y = y;

    strokes = (GList *) tomoe_writing_get_strokes (priv->writing);
    strokes = g_list_last (strokes);
    g_return_if_fail (strokes);

    point = strokes->data;
    g_return_if_fail (point);

    point = g_list_last (point);
    g_return_if_fail (point->data);
    pp = *((TomoePoint*) point->data);

    _init_gc (canvas);
    tomoe_canvas_draw_line (canvas, &pp, &p,
                            priv->handwrite_line_gc,
                            TRUE);

    tomoe_writing_line_to (priv->writing, x, y);
}

static void
tomoe_canvas_draw_line (TomoeCanvas *canvas,
                        TomoePoint *p1, TomoePoint *p2,
                        GdkGC *line_gc,
                        gboolean draw)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    GtkWidget *widget = GTK_WIDGET (canvas);
    gint x, y, width, height;

    gdk_draw_line (priv->pixmap,
                   line_gc,
                   p1->x, p1->y,
                   p2->x, p2->y);
    if (draw) {
        x = (MIN (p1->x, p2->x) - 2);
        y = (MIN (p1->y, p2->y) - 2);
        width  = abs (p1->x - p2->x) + (2 * 2);
        height = abs (p1->y - p2->y) + (2 * 2);

        gtk_widget_queue_draw_area (widget, x, y, width, height);
    }
}

static void
tomoe_canvas_draw_background (TomoeCanvas *canvas, gboolean draw)
{
    TomoeCanvasPriv *priv;
    GtkWidget *widget;

    g_return_if_fail (TOMOE_IS_CANVAS (canvas));

    priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    widget = GTK_WIDGET (canvas);

    gdk_draw_rectangle (priv->pixmap,
                        widget->style->white_gc,
                        TRUE,
                        0, 0,
                        widget->allocation.width,
                        widget->allocation.height);

    tomoe_canvas_draw_axis (canvas);

    if (draw) {
        gdk_draw_drawable(widget->window,
                          widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
                          priv->pixmap,
                          0, 0,
                          0, 0,
                          widget->allocation.width, widget->allocation.height);
    }
}

static void
tomoe_canvas_draw_axis (TomoeCanvas *canvas)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    GtkWidget *widget = GTK_WIDGET (canvas);

    if (!priv->axis_gc) {
        /* FIXME: customize color */
        GdkColor color = { 0, 0x8000, 0x8000, 0x8000 };
        priv->axis_gc = gdk_gc_new (widget->window);
        tomoe_canvas_set_axis_color (canvas, &color);
        gdk_gc_set_line_attributes (priv->axis_gc, 1,
                                    GDK_LINE_ON_OFF_DASH,
                                    GDK_CAP_BUTT,
                                    GDK_JOIN_ROUND);
    }

    gdk_draw_line (priv->pixmap, priv->axis_gc,
                   (priv->size / 2), 0,
                   (priv->size / 2), priv->size);
    gdk_draw_line (priv->pixmap, priv->axis_gc,
                   0,          (priv->size / 2),
                   priv->size, (priv->size / 2));
}

static void
_init_gc (TomoeCanvas *canvas)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    GtkWidget *widget = GTK_WIDGET (canvas);

    if (!priv->adjust_line_gc) {
        GdkColor color = { 0, 0x8000, 0x0000, 0x0000 };
        priv->adjust_line_gc = gdk_gc_new (widget->window);
        tomoe_canvas_set_adjust_line_color (canvas, &color);
        gdk_gc_set_line_attributes (priv->adjust_line_gc, 1,
                                    GDK_LINE_SOLID,
                                    GDK_CAP_BUTT,
                                    GDK_JOIN_BEVEL);
    }

    if (!priv->handwrite_line_gc) {
        GdkColor color = { 0, 0x0000, 0x0000, 0x0000 };
        priv->handwrite_line_gc = gdk_gc_new (widget->window);
        tomoe_canvas_set_handwrite_line_color (canvas, &color);
        gdk_gc_set_line_attributes (priv->handwrite_line_gc, 4,
                                    GDK_LINE_SOLID,
                                    GDK_CAP_ROUND,
                                    GDK_JOIN_ROUND);
    }

    if (!priv->annotate_gc) {
        GdkColor color = { 0, 0x8000, 0x0000, 0x0000 };
        priv->annotate_gc = gdk_gc_new (widget->window);
        tomoe_canvas_set_annotate_color (canvas, &color);
    }
}

static void
draw_stroke (GList *points, TomoeCanvas *canvas, guint index)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    GList *node;

    _init_gc (canvas);

    for (node = points; node; node = g_list_next (node)) {
        GList *next = g_list_next (node);
        TomoePoint *p1, *p2;

        if (!next) break;

        p1 = (TomoePoint *) node->data;
        p2 = (TomoePoint *) next->data;

        tomoe_canvas_draw_line (canvas, p1, p2,
                                priv->handwrite_line_gc,
                                FALSE);
    }
    draw_annotate (points, canvas, index);
}

static void
draw_annotate (GList *points, TomoeCanvas *canvas, guint index)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);
    GtkWidget *widget = GTK_WIDGET (canvas);
    gchar *buffer; 
    PangoLayout *layout;
    gint width, height;
    gint x, y;
    gdouble r;
    gdouble dx,dy;
    gdouble dl;
    gint sign;

    x = ((TomoePoint*)(points->data))->x;
    y = ((TomoePoint*)(points->data))->y;

    if (g_list_length (points) == 1) {
        dx = x;
        dy = y;
    } else {
        dx = ((TomoePoint*)((g_list_last (points))->data))->x - x;
        dy = ((TomoePoint*)((g_list_last (points))->data))->y - y;
    }

    dl = sqrt (dx*dx + dy*dy);
    sign = (dy <= dx) ? 1 : -1;

    buffer = g_strdup_printf ("%d", index);
    layout = gtk_widget_create_pango_layout (widget, buffer);
    pango_layout_get_pixel_size (layout, &width, &height);

    r = sqrt (width*width + height*height);

    x += (0.5 + (0.5 * r * dx / dl) + (sign * 0.5 * r * dy / dl) - (width / 2));
    y += (0.5 + (0.5 * r * dy / dl) - (sign * 0.5 * r * dx / dl) - (height / 2));

    gdk_draw_layout (priv->pixmap,
                     priv->annotate_gc,
                     x, y, layout);

    g_free (buffer);
    g_object_unref (layout);
}
const GList *
tomoe_canvas_get_candidates (TomoeCanvas *canvas)
{
    TomoeCanvasPriv *priv = TOMOE_CANVAS_GET_PRIVATE (canvas);

    return priv->candidates;
}

/*
 * vi:ts=4:nowrap:ai:expandtab
 */
