/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 *  Copyright (C) 2005 Takuro Ashie
 *  Copyright (C) 2006 Juernjakob Harder
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
  #include <config.h>
#endif

#include <glib/gi18n.h>
#include <tomoe.h>

#include "tomoe-window.h"
#include "tomoe-canvas.h"
#include "tomoe-stroke-search.h"
#include "tomoe-char-table.h"
#include "tomoe-reading-search.h"

#define TOMOE_DATA_VIEW_ICON LIBTOMOEGTKDATADIR"/tomoe-data-view.png"
#define TOMOE_DICTIONARY_ICON LIBTOMOEGTKDATADIR"/tomoe-dictionary.png"
#define TOMOE_HANDWRITING_ICON LIBTOMOEGTKDATADIR"/tomoe-handwriting.png"
#define TOMOE_SEARCHING_ICON LIBTOMOEGTKDATADIR"/tomoe-search.png"

typedef struct _TomoeWindowPrivate	TomoeWindowPrivate;
struct _TomoeWindowPrivate
{
    TomoeContext *context;
    GtkWidget    *notebook;
    GtkWidget    *handwriting_page;
    GtkWidget    *reading_search;
    GtkWidget    *character_page;
};

#define TOMOE_WINDOW_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), TOMOE_TYPE_WINDOW, TomoeWindowPrivate))

G_DEFINE_TYPE (TomoeWindow, tomoe_window, GTK_TYPE_DIALOG)

static void dispose          (GObject *object);

static void
tomoe_window_class_init (TomoeWindowClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

    gobject_class->dispose = dispose;

    g_type_class_add_private (gobject_class, sizeof (TomoeWindowPrivate));
}

static void
tomoe_window_init (TomoeWindow *window)
{
    GtkWidget   *handwriting_page, *reading_search;
    GtkWidget   *notebook;
    TomoeCanvas *canvas;
    TomoeWindowPrivate *priv = TOMOE_WINDOW_GET_PRIVATE (window);

#ifdef ENABLE_NLS
    setlocale (LC_ALL, "");
    bindtextdomain (GETTEXT_PACKAGE, LIBTOMOE_GTK_LOCALEDIR);
    textdomain (GETTEXT_PACKAGE);
    bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
#endif

    /* load libtomoe */
    tomoe_init ();
    
    priv->context = tomoe_context_new ();
    /*tomoe_context_load_config (priv->context, g_getenv ("TOMOE_CONFIG_FILE"));*/
    tomoe_context_load_config (priv->context, NULL);

    gtk_window_set_title (GTK_WINDOW (window),
                          _("Handwriting recognition"));

    /* FIXME! */
    gtk_window_set_resizable (GTK_WINDOW (window), FALSE);

    /* create search pages view notebook */
    notebook = gtk_notebook_new ();
    priv->notebook = notebook;
    gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_LEFT);
    gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), TRUE);
    gtk_notebook_set_show_border (GTK_NOTEBOOK (notebook), FALSE);
    gtk_container_add (GTK_CONTAINER (GTK_DIALOG (window)->vbox), notebook);
    gtk_widget_show (notebook);

    handwriting_page = tomoe_stroke_search_new ();
    priv->handwriting_page = handwriting_page;
    canvas = TOMOE_CANVAS (tomoe_stroke_search_get_canvas (TOMOE_STROKE_SEARCH (handwriting_page)));
    tomoe_canvas_set_context (canvas, priv->context);
    gtk_widget_show (handwriting_page);
    tomoe_window_append_page (window, handwriting_page, 
                              gtk_image_new_from_file (TOMOE_HANDWRITING_ICON),
                              _("Handwriting recognition"));

    reading_search = tomoe_reading_search_new ();
    priv->reading_search = reading_search;
    tomoe_reading_search_set_context (TOMOE_READING_SEARCH (reading_search), priv->context);
    gtk_widget_show (reading_search);
    tomoe_window_append_page (window, reading_search, 
                              gtk_image_new_from_file (TOMOE_SEARCHING_ICON),
                              _("Search with reading "));
}

static void
dispose (GObject *object)
{
    TomoeWindowPrivate *priv = TOMOE_WINDOW_GET_PRIVATE (object);

    if (priv->context) {
        g_object_unref (priv->context);
        priv->context = NULL;
    }
    
    tomoe_quit ();

    if (G_OBJECT_CLASS(tomoe_window_parent_class)->dispose)
        G_OBJECT_CLASS(tomoe_window_parent_class)->dispose(object);
}

GtkWidget *
tomoe_window_new (void)
{
    return GTK_WIDGET(g_object_new (TOMOE_TYPE_WINDOW,
                                    NULL));
}

GtkWidget *
tomoe_window_get_canvas (TomoeWindow *window)
{
    TomoeWindowPrivate *priv;
    g_return_val_if_fail (TOMOE_IS_WINDOW (window), NULL);

    priv = TOMOE_WINDOW_GET_PRIVATE (window);
    g_return_val_if_fail (TOMOE_IS_STROKE_SEARCH (priv->handwriting_page), NULL);

    return tomoe_stroke_search_get_canvas (TOMOE_STROKE_SEARCH (priv->handwriting_page));
}
void
tomoe_window_append_page (TomoeWindow *window, GtkWidget *page, GtkWidget *label, const gchar *label_text)
{
    TomoeWindowPrivate *priv;
    GtkTooltips *tip;
    GtkWidget *event_box;

    g_return_if_fail (TOMOE_IS_WINDOW (window));

    if (!page || !GTK_IS_WIDGET (page)) return;

    priv = TOMOE_WINDOW_GET_PRIVATE (window);
    event_box = gtk_event_box_new ();
    gtk_event_box_set_visible_window (GTK_EVENT_BOX (event_box), FALSE);
    gtk_container_add (GTK_CONTAINER (event_box), label);
    gtk_widget_show (event_box);
    gtk_widget_show (label);

    gtk_notebook_append_page (GTK_NOTEBOOK (priv->notebook), page, event_box);

    if (label_text) {
        tip = gtk_tooltips_new ();
        gtk_tooltips_set_tip (tip, event_box, label_text, NULL);
    }
}

GtkWidget *
tomoe_window_get_handwriting_page (TomoeWindow *window)
{
    g_return_val_if_fail (TOMOE_IS_WINDOW (window), NULL);

    return TOMOE_WINDOW_GET_PRIVATE (window)->handwriting_page;
}

GtkWidget *
tomoe_window_get_notebook (TomoeWindow *window)
{
    g_return_val_if_fail (TOMOE_IS_WINDOW (window), NULL);

    return TOMOE_WINDOW_GET_PRIVATE (window)->notebook;
}

/*
 * vi:ts=4:nowrap:ai:expandtab
 */
