/*
 * tomoyo_capability_test.c
 *
 * Testing program for fs/tomoyo_capability.c
 *
 * Copyright (C) 2005-2006  NTT DATA CORPORATION
 *
 * Version: 1.0.2 2006/02/13
 *
 */
#include "include.h"

static const char *self = NULL;
static int is_enforce = 0;

static void ShowPrompt(const char *str) {
	printf("Testing %35s: (%s) ", str, is_enforce ? "must fail" : "should success");
	errno = 0;
}

static void ShowResult(int result) {
	if (is_enforce) {
		if (result == EOF) {
			if (errno == EPERM) printf("OK: Permission denied.\n");
			else printf("FAILED: %s\n", strerror(errno));
		} else {
			printf("BUG!\n");
		}
	} else {
		if (result != EOF) printf("OK\n");
		else printf("%s\n", strerror(errno));
	}
}

static void StageCapabilityTest1(void) {
	int fd;
	ShowPrompt("inet_tcp_create");
	fd = socket(AF_INET, SOCK_STREAM, 0);
	ShowResult(fd);
	if (fd != EOF) close(fd);
	
	// inet_tcp_listen and inet_tcp_connect are checked later.

	ShowPrompt("use_inet_udp");
	fd = socket(AF_INET, SOCK_DGRAM, 0);
	ShowResult(fd);
	if (fd != EOF) close(fd);
	
	ShowPrompt("use_inet_ip");
	fd = socket(AF_INET, SOCK_RAW, 0);
	ShowResult(fd);
	if (fd != EOF) close(fd);

	ShowPrompt("use_route");
	fd = socket(AF_ROUTE, 0, 0);
	ShowResult(fd);
	if (fd != EOF) close(fd);
	
	ShowPrompt("use_packet");
	fd = socket(AF_PACKET, 0, 0);
	ShowResult(fd);
	if (fd != EOF) close(fd);

	ShowPrompt("SYS_MOUNT");
	ShowResult(mount("/", "/", "nonexistent", 0, NULL));
	
	ShowPrompt("SYS_UMOUNT");
	ShowResult(umount("/"));
	
	ShowPrompt("SYS_REBOOT");
	ShowResult(reboot(LINUX_REBOOT_MAGIC1, LINUX_REBOOT_MAGIC2, 
					  0x0000C0DE /* Use invalid value so that the system won't reboot. */, NULL));
	
	ShowPrompt("SYS_CHROOT");
	ShowResult(chroot("/"));
		
	signal(SIGINT, SIG_IGN);
	ShowPrompt("SYS_KILL(kill())");
	ShowResult(kill(pid, SIGINT));
	ShowPrompt("SYS_KILL(tkill())");
	ShowResult(tkill(gettid(), SIGINT));
	if (is_kernel26) {
#ifdef tgkill
		ShowPrompt("SYS_KILL(tgkill())");
		ShowResult(tgkill(pid, gettid(), SIGINT));
#endif
	}
	signal(SIGINT, SIG_DFL);
	
	// SYS_VHANGUP is checked later.

	{
		struct timeval tv;
		struct timezone tz;
		struct timex buf;
		time_t now = time(NULL);
		ShowPrompt("SYS_TIME(stime())");
		ShowResult(stime(&now));
		gettimeofday(&tv, &tz);
		ShowPrompt("SYS_TIME(settimeofday())");
		ShowResult(settimeofday(&tv, &tz));
		memset(&buf, 0, sizeof(buf));
		buf.modes = 0x100; /* Use invalid value so that the clock won't change. */
		ShowPrompt("SYS_TIME(adjtimex())");
		ShowResult(adjtimex(&buf));
	}
	
	ShowPrompt("SYS_NICE(nice())");
	ShowResult(nice(0));
	ShowPrompt("SYS_NICE(setpriority())");
	ShowResult(setpriority(PRIO_PROCESS, pid, getpriority(PRIO_PROCESS, pid)));
	
	{
		char buffer[4096];
		memset(buffer, 0, sizeof(buffer));
		gethostname(buffer, sizeof(buffer) - 1);
		ShowPrompt("SYS_SETHOSTNAME(sethostname())");
		ShowResult(sethostname(buffer, strlen(buffer)));
		getdomainname(buffer, sizeof(buffer) - 1);
		ShowPrompt("SYS_SETHOSTNAME(setdomainname())");
		ShowResult(setdomainname(buffer, strlen(buffer)));
	}

	if (!is_kernel26) {
		ShowPrompt("use_kernel_module(create_module())");
		ShowResult((int) create_module("", 0));
	}
	ShowPrompt("use_kernel_module(init_module())");
	ShowResult(init_module("", NULL));
	ShowPrompt("use_kernel_module(delete_module())");
	ShowResult(delete_module(""));

	ShowPrompt("create_fifo");
	ShowResult(mknod("/", S_IFIFO, 0));

	ShowPrompt("create_block_dev");
	ShowResult(mknod("/", S_IFBLK, MKDEV(1, 0)));

	ShowPrompt("create_char_dev");
	ShowResult(mknod("/", S_IFCHR, MKDEV(1, 3)));

	ShowPrompt("create_unix_socket(mknod())");
	ShowResult(mknod("/", S_IFSOCK, 0));

	{
		struct sockaddr_un addr;
		int fd;
		memset(&addr, 0, sizeof(addr));
		addr.sun_family = AF_UNIX;
		strncpy(addr.sun_path, "/", sizeof(addr.sun_path) - 1);
		fd = socket(AF_UNIX, SOCK_STREAM, 0);
		ShowPrompt("create_unix_socket(bind())");
		ShowResult(bind(fd, (struct sockaddr *) &addr, sizeof(addr)));
		if (fd != EOF) close(fd);
	}

	ShowPrompt("SYS_LINK");
	ShowResult(link("/", "/"));
	
	ShowPrompt("SYS_SYMLINK");
	ShowResult(symlink("/", "/"));
	
	ShowPrompt("SYS_RENAME");
	ShowResult(rename("/", "/"));
		
	ShowPrompt("SYS_UNLINK");
	ShowResult(unlink("/"));
	
	ShowPrompt("SYS_CHMOD");
	ShowResult(chmod(self, 0));
	chmod(self, 0100);
	ShowPrompt("SYS_CHOWN");
	ShowResult(chown(self, 1, 1));
	chown(self, 0, 0);
	
	{
		int fd = open(self, O_RDONLY);
		ShowPrompt("SYS_IOCTL");
		ShowResult(ioctl(fd, 0 /* Use invalid value so that nothing happen. */));
		close(fd);
	}
#ifdef SPARC64
	/*
	if (!is_kernel26) {
	_syscall3(long, compat_sys_ioctl, unsigned int, fd, unsigned int, cmd, unsigned long, arg);
	compat_sys_ioctl(); // Sorry, I don't have SPARC64 environment to test.
	}
	*/
#endif
}

static void StageCapabilityTest2(void) {
	{
		struct sockaddr_in addr;
		int fd1, fd2;
		socklen_t size = sizeof(addr);
		memset(&addr, 0, sizeof(addr));
		addr.sin_family = AF_INET;
		addr.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
		addr.sin_port = htons(0);
		
		fd1 = socket(AF_INET, SOCK_STREAM, 0);
		bind(fd1, (struct sockaddr *) &addr, sizeof(addr));
		getsockname(fd1, (struct sockaddr *) &addr, &size);
		ShowPrompt("inet_tcp_listen");
		ShowResult(listen(fd1, 5));
		
		fd2 = socket(AF_INET, SOCK_STREAM, 0);
		ShowPrompt("inet_tcp_connect");
		ShowResult(connect(fd2, (struct sockaddr *) &addr, sizeof(addr)));

		if (fd2 != EOF) close(fd2);
		if (fd1 != EOF) close(fd1);
	}
	
	{
		int pty_fd = EOF, status = 0;
		int pipe_fd[2] = { EOF, EOF };
		pipe(pipe_fd);
		switch (forkpty(&pty_fd, NULL, NULL, NULL)) {
		case 0:
			vhangup();
			/* Unreachable if vhangup() succeeded. */
			status = errno;
			write(pipe_fd[1], &status, sizeof(status));
			_exit(0);
		case -1:
			fprintf(stderr, "forkpty() failed.\n");
			break;
		default:
			close(pipe_fd[1]);
			read(pipe_fd[0], &status, sizeof(status));
			wait(NULL);
			close(pty_fd);
			ShowPrompt("SYS_VHANGUP");
			errno = status;
			ShowResult(status ? EOF : 0);
		}
	}
}

static void SetCapabilityEnforce(int enforce) {
	FILE *fp = fopen("/proc/ccs/status", "r");
	static char buffer[4096];
	if (!fp) {
		fprintf(stderr, "Can't open /proc/ccs/status\n");
		return;
	}
	while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 10, fp)) {
		char *cp = strchr(buffer, '=');
		if (strncmp(buffer, "MAC_FOR_CAPABILITY::", 20) || !cp) continue;
		*cp = '\0';
		write(status_fd, buffer, strlen(buffer));
		sprintf(buffer, "=%d\n", enforce ? 3 : 2);
		write(status_fd, buffer, strlen(buffer));
	}
	fclose(fp);
}
	
int main(int argc, char *argv[]) {
	self = argv[0];
	Init();

	printf("***** Testing capability hooks in enforce mode. *****\n");
	is_enforce = 1;
	SetCapabilityEnforce(1);
	StageCapabilityTest1();
	// Disable inet_tcp_create SYS_IOCTL so that StageCapabilityTest2() can work.
	{
		const char *cmd = "MAC_FOR_CAPABILITY::inet_tcp_create=0\n"	"MAC_FOR_CAPABILITY::SYS_IOCTL=0\n";
		write(status_fd, cmd, strlen(cmd));
	}
	StageCapabilityTest2();
	printf("\n\n");

	printf("***** Testing capability hooks in permissive mode. *****\n");
	is_enforce = 0;
	SetCapabilityEnforce(0);
	StageCapabilityTest1();
	StageCapabilityTest2();
	printf("\n\n");

	ClearStatus();
	return 0;
}
