/*
 * checkpolicy.c
 *
 * A checker for domain policy on the disk.
 *
 * Copyright (C) 2005-2006  NTT DATA CORPORATION
 *
 * Version: 1.1   2006/04/01
 *
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/types.h>
#include <termios.h>

#define ROOT_NAME "<kernel>"

#define BYTE unsigned char
#define WORD unsigned short int

#define MAXBUFSIZE  8192

static void OutOfMemory(void) {
	fprintf(stderr, "Out of memory. Aborted.\n");
	exit(1);
}

#define PAGE_SIZE  4096

static const char *SaveName(const char *name) {
	static char *buf = NULL;
	static int buf_used_len = PAGE_SIZE;
	int i, len;
	char *saved_name = NULL;
	char **new_ptr = NULL;
	static char **search_list = NULL;
	static int search_list_count = 0;
	if (!name) return NULL;
	len = strlen(name) + 1;
	if (len > PAGE_SIZE) {
		printf("ERROR: Name too long for SaveName().\n");
		return NULL;
	}
	for (i = 0; i < search_list_count; i++) {
		if (strcmp(name, search_list[i]) == 0) return search_list[i];
	}
	if (buf_used_len + len > PAGE_SIZE) {
		if ((buf = malloc(PAGE_SIZE)) == NULL) OutOfMemory();
		memset(buf, 0, PAGE_SIZE);
		buf_used_len = 0;
	}
	saved_name = buf + buf_used_len;
	memmove(saved_name, name, len);
	if ((new_ptr = (char **) realloc(search_list, (search_list_count + 1) * sizeof(char *))) == NULL) OutOfMemory();
	search_list = new_ptr;
	search_list[search_list_count++] = saved_name;
	buf_used_len += len;
	return (const char *) saved_name;
}

/*
 * Check whether the given filename is patterened.
 * Returns nonzero if patterned, zero otherwise.
 */
static int PathContainsPattern(const char *filename) {
	if (filename) {
		char c, d, e;
		while ((c = *filename++) != '\0') {
			if (c != '\\') continue;
			switch (c = *filename++) {
			case '\\':  /* "\\" */
				continue;
			case '0':   /* "\ooo" */
			case '1':
			case '2':
			case '3':
				if ((d = *filename++) >= '0' && d <= '7' && (e = *filename++) >= '0' && e <= '7'
					&& (c != '0' || d != '0' || e != '0')) continue; /* pattern is not \000 */
			}
			return 1;
		}
	}
	return 0;
}

/*
 * Check whether the given filename follows the naming rules.
 * Returns nonzero if follows, zero otherwise.
 */
static int IsCorrectPath(const char *filename, const int may_contain_pattern) {
	if (filename && *filename == '/') {
		char c, d, e;
		while ((c = *filename++) != '\0') {
			if (c == '\\') {
				switch ((c = *filename++)) {
				case '\\':  /* "\\" */
					continue;
				case '$':   /* "\$" */
				case '+':   /* "\+" */
				case '?':   /* "\?" */
					if (may_contain_pattern) continue;
					break;
				case '*':   /* "\*" */
					if (may_contain_pattern) {
						while (*filename && *filename != '/') filename++;
						if (!*filename || *filename == '/') continue;
					}
					break;
				case '0':   /* "\ooo" */
				case '1':
				case '2':
				case '3':
					if ((d = *filename++) >= '0' && d <= '7' && (e = *filename++) >= '0' && e <= '7') {
						const unsigned char f =
							(((unsigned char) (c - '0')) << 6) +
							(((unsigned char) (d - '0')) << 3) +
							(((unsigned char) (e - '0')));
						if (f && (f <= ' ' || f >= 127)) continue; /* pattern is not \000 */
					}
				}
				return 0;
			} else if (c <= ' ' || c >= 127) {
				return 0;
			}
		}
		return 1;
	}
	return 0;
}

static int IsCorrectDomain(const unsigned char *domainname) {
	unsigned char c, d, e;
	if (!domainname || strncmp(domainname, ROOT_NAME, strlen(ROOT_NAME))) goto out;
	domainname += strlen(ROOT_NAME);
	if (!*domainname) return 1;
	do {
		if (*domainname++ != ' ') goto out;
		if (*domainname++ != '/') goto out;
		while ((c = *domainname) != '\0' && c != ' ') {
			domainname++;
			if (c == '\\') {
				switch ((c = *domainname++)) {
				case '\\':  /* "\\" */
					continue;
				case '0':   /* "\ooo" */
				case '1':
				case '2':
				case '3':
					if ((d = *domainname++) >= '0' && d <= '7' && (e = *domainname++) >= '0' && e <= '7') {
						const unsigned char f =
							(((unsigned char) (c - '0')) << 6) +
							(((unsigned char) (d - '0')) << 3) +
							(((unsigned char) (e - '0')));
						if (f && (f <= ' ' || f >= 127)) continue; /* pattern is not \000 */
					}
				}
				goto out;
			} else if (c < ' ' || c >= 127) {
				goto out;
			}
		}
	} while (*domainname);
	return 1;
 out:
	return 0;
}

static inline int strendswith(const char *name, const char *tail) {
	int len;
	if (!name || !tail) return 0;
	len = strlen(name) - strlen(tail);
	return len >= 0 && strcmp(name + len, tail) == 0;
}

static void SortPolicy(void);
static int AddFileACL(const char *filename, WORD perm, const int index);
static int FindOrAssignNewDomain(const char *domainname);
static int FindDomain(const char *domainname);
static void NormalizeLine(unsigned char *buffer);
static const char *DomainName(const int index);
static void DeleteDomain(const int index);
static int IsGloballyReadableFile(const char *filename);

typedef struct {
	const char *name;
	WORD perm;
} ACL_RECORD;

typedef struct domain_info {
	ACL_RECORD *acl_ptr;
	int acl_count;
	const char *domainname;
	const char **string_ptr;
	int string_count;
} DOMAIN_INFO;

static DOMAIN_INFO *domain_list = NULL;
static int domain_list_count = 0;

static const char **trusted_domainnames = NULL;
static int trusted_domainnames_count = 0;

static const char **initializers = NULL;
static int initializers_count = 0;

static const char **globally_readable_list = NULL;
static int globally_readable_list_count = 0;

static int acl_updated = 0;

static int AddFileACL(const char *filename, WORD perm, const int index) {
	ACL_RECORD *acl_ptr, *new_aclptr;
	const char *cp;
	int i;
	if (index < 0 || index >= domain_list_count) {
		printf("AddFileACL: ERROR: domain is out of range.\n");
		return -EINVAL;
	}
	if (!IsCorrectPath(filename, 1)) {
		fprintf(stderr, "AddFileACL: Invalid pathname '%s'\n", filename);
		return -EINVAL;
	}
	if (perm > 7) { /* Should I allow perm == 0, for somebody may wish to give '4 /etc/\*' and '0 /etc/shadow' ? */
		fprintf(stderr, "AddFileACL: Invalid permission '%d %s'\n", perm, filename);
		return -EINVAL;
	}
	if (strendswith(filename, "/")) {
		perm |= 5;  /* Always allow read and execute for dir. */
	} else if ((perm & 1) == 1 && PathContainsPattern(filename)) {
		perm ^= 1;  /* Never allow execute permission with patterns. */
		fprintf(stderr, "AddFileACL: Dropping execute permission for '%s'\n", filename);
	} else if (perm == 4 && IsGloballyReadableFile(filename)) {
		return 0;   /* Don't add if the file is globally readable files. */
	}
	
	acl_ptr = domain_list[index].acl_ptr;
	
	// Check if the file was accessed before.
	for (i = 0; i < domain_list[index].acl_count; i++) {
		const char *cp = acl_ptr[i].name;
		if (strcmp(filename, cp) == 0) {
			acl_ptr[i].perm |= perm;
			return 0;
		}
	}
	if ((cp = SaveName(filename)) == NULL) OutOfMemory();
	
	// Allocate space.
	i = domain_list[index].acl_count;
	if ((new_aclptr = (ACL_RECORD *) realloc(acl_ptr, (i + 1) * sizeof(ACL_RECORD))) == NULL) OutOfMemory();
	domain_list[index].acl_ptr = acl_ptr = new_aclptr;
	
	// Record the access log.
	acl_ptr[i].perm = perm;
	acl_ptr[i].name = cp;
	domain_list[index].acl_count++;
	return 0;
}

static int AddStringEntry(const char *entry, const int index) {
	const char **acl_ptr;
	int acl_count;
	const char *cp;
	int i;
	if (index < 0 || index >= domain_list_count) {
		printf("AddStringEntry: ERROR: domain is out of range.\n");
		return -EINVAL;
	}
	if (!entry || !*entry) return -EINVAL;
	acl_ptr = domain_list[index].string_ptr;
	acl_count = domain_list[index].string_count;
	
	// Check for the same entry.
	for (i = 0; i < acl_count; i++) {
		if (strcmp(entry, acl_ptr[i]) == 0) return 0;
	}

	if ((cp = SaveName(entry)) == NULL) OutOfMemory();
	if ((acl_ptr = (const char **) realloc(acl_ptr, (acl_count + 1) * sizeof(char *))) == NULL) OutOfMemory();
	acl_ptr[acl_count++] = cp;
	domain_list[index].string_ptr = acl_ptr;
	domain_list[index].string_count = acl_count;
	return 0;
}

static int IsGloballyReadableFile(const char *filename) {
	if (filename) {
		int i;
		for (i = 0; i < globally_readable_list_count; i++) {
			if (strcmp(filename, globally_readable_list[i]) == 0) return 1;
		}
	}
	return 0;
}

static int FindDomain(const char *domainname) {
	int i;
	for (i = 0; i < domain_list_count; i++) {
		if (strcmp(domainname, domain_list[i].domainname) == 0) {
			return i;
		}
	}
	return EOF;
}

static int FindOrAssignNewDomain(const char *domainname) {
	const char *saved_domainname;
	int index;
	if ((index = FindDomain(domainname)) == EOF) {
		if (IsCorrectDomain(domainname)) {
			if ((domain_list = (DOMAIN_INFO *) realloc(domain_list, (domain_list_count + 1) * sizeof(DOMAIN_INFO))) == NULL) OutOfMemory();
			memset(&domain_list[domain_list_count], 0, sizeof(DOMAIN_INFO));
			if ((saved_domainname = SaveName(domainname)) == NULL) OutOfMemory();
			domain_list[domain_list_count].domainname = saved_domainname;
			index = domain_list_count++;
		} else {
			fprintf(stderr, "FindOrAssignNewDomain: Invalid domainname '%s'\n", domainname);
		}
	}
	return index;
}

static const char *DomainName(const int index) {
	return domain_list[index].domainname;
}

static void DeleteDomain(const int index) {
	if (index > 0 && index < domain_list_count) {
		int i;
		for (i = index; i < domain_list_count - 1; i++) domain_list[i] = domain_list[i + 1];
		domain_list_count--;
	}
}

static int IsDomainDef(const unsigned char *buffer) {
	while (*buffer && (*buffer <= 32 || 127 <= *buffer)) buffer++;
	return strncmp(buffer, ROOT_NAME, strlen(ROOT_NAME)) == 0;
}

static void NormalizeLine(unsigned char *buffer) {
	unsigned char *sp = buffer, *dp = buffer;
	int first = 1;
	while (*sp && (*sp <= 32 || 127 <= *sp)) sp++;
	while (*sp) {
		if (!first) *dp++ = ' ';
		first = 0;
		while (32 < *sp && *sp < 127) *dp++ = *sp++;
		while (*sp && (*sp <= 32 || 127 <= *sp)) sp++;
	}
	*dp = '\0';
}

static void SortPolicy(void) {
	int i, j;
	for (i = 0; i < domain_list_count; i++) {
		for (j = i + 1; j < domain_list_count; j++) {
			if (strcmp(domain_list[i].domainname, domain_list[j].domainname) > 0) {
				DOMAIN_INFO tmp = domain_list[i]; domain_list[i] = domain_list[j]; domain_list[j] = tmp;
			}
		}
	}
}

static int SortAndUniq(const char **list, int list_count) {
	int i, j;
	for (i = 0; i < list_count; i++) {
		for (j = i + 1; j < list_count; j++) {
			if (strcmp(list[i], list[j]) > 0) {
				const char *tmp = list[i];
				list[i] = list[j];
				list[j] = tmp;
			}
		}
	}
	for (i = 0; i < list_count - 1; i++) {
		if (strcmp(list[i], list[i + 1])) continue;
		for (j = i + 1; j < list_count - 1; j++) list[j] = list[j + 1];
		list_count--; i--;
	}
	return list_count;
}

static char buffer[MAXBUFSIZE];

static void LoadPolicy(void) {
	FILE *fp;
	int index;
	
	FindOrAssignNewDomain(ROOT_NAME);
	
	// Load initializer list, trusted domain list, globally readable file list.
	initializers_count = 0;
	trusted_domainnames_count = 0;
	globally_readable_list = 0;
	
	if ((fp = fopen("exception_policy.txt", "r")) != NULL) {
		while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
			char *cp = strchr(buffer, '\n');
			const char **tmp;
			int i;
			if (cp) *cp = '\0';
			else if (!feof(fp)) break;
			NormalizeLine(buffer);
			if (strncmp(buffer, "initializer ", 12) == 0) {
				memmove(buffer, buffer + 12, strlen(buffer + 12) + 1);
				if (!IsCorrectPath(buffer, 0)) continue;
				for (i = 0; i < initializers_count; i++) {
					if (strcmp(buffer, initializers[i]) == 0) break;
				}
				if (i < initializers_count) continue;
				if ((cp = (char *) SaveName(buffer)) == NULL) continue;
				if ((tmp = (const char **) realloc(initializers, (initializers_count + 1) * sizeof(const char *))) == NULL) OutOfMemory();
				initializers = tmp;
				initializers[initializers_count++] = cp;
			} else if (strncmp(buffer, "trust_domain ", 13) == 0) {
				memmove(buffer, buffer + 13, strlen(buffer + 13) + 1);
				if (!IsCorrectDomain(buffer)) continue;
				for (i = 0; i < trusted_domainnames_count; i++) {
					if (strcmp(buffer, trusted_domainnames[i]) == 0) break;
				}
				if (i < trusted_domainnames_count) continue;
				if ((cp = (char *) SaveName(buffer)) == NULL) continue;
				if ((tmp = (const char **) realloc(trusted_domainnames, (trusted_domainnames_count + 1) * sizeof(const char *))) == NULL) OutOfMemory();
				trusted_domainnames = tmp;
				trusted_domainnames[trusted_domainnames_count++] = cp;
			} else if (strncmp(buffer, "allow_read ", 11) == 0) {
				memmove(buffer, buffer + 11, strlen(buffer + 11) + 1);
				if (!IsCorrectPath(buffer, 0)) continue;
				for (i = 0; i < globally_readable_list_count; i++) {
					if (strcmp(buffer, globally_readable_list[i]) == 0) break;
				}
				if (i < globally_readable_list_count) continue;
				if ((cp = (char *) SaveName(buffer)) == NULL) continue;
				if ((tmp = (const char **) realloc(globally_readable_list, (globally_readable_list_count + 1) * sizeof(const char *))) == NULL) OutOfMemory();
				globally_readable_list = tmp;
				globally_readable_list[globally_readable_list_count++] = cp;
			}
		}
		fclose(fp);
		initializers_count = SortAndUniq(initializers, initializers_count);
		trusted_domainnames_count = SortAndUniq(trusted_domainnames, trusted_domainnames_count);
		globally_readable_list_count = SortAndUniq(globally_readable_list, globally_readable_list_count);
	}
	
	// Load all domain list.
	if ((fp = fopen("domain_policy.txt", "r")) == NULL) {
		fprintf(stderr, "Can't open domain_policy.txt\n"); sleep(3); exit(1);
	}
	index = EOF;
	while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
		char *cp = strchr(buffer, '\n');
		int perm;
		if (cp) *cp = '\0';
		else if (!feof(fp)) break;
		NormalizeLine(buffer);
		if (IsDomainDef(buffer)) {
			index = FindOrAssignNewDomain(buffer);
		} else if (index >= 0 && sscanf(buffer, "%d", &perm) == 1 && (cp = strchr(buffer, ' ')) != NULL) {
			AddFileACL(cp + 1, (WORD) perm, index);
		} else if (index >= 0 && strncmp(buffer, "allow_", 6) == 0) {
			AddStringEntry(buffer, index);
		}
	}
	fclose(fp);
	
	SortPolicy();
}

static void CheckPolicy(void) {
	int index;
	// Check if a domain contains initializers.
	for (index = 1; index < domain_list_count; index++) {
		static char buffer2[MAXBUFSIZE];
		char *exe;
		if ((exe = strchr(domain_list[index].domainname, ' ')) == NULL) continue;
		if ((exe = strchr(exe + 1, ' ')) == NULL) continue;
		memset(buffer2, 0, sizeof(buffer2));
		strncpy(buffer2, exe, sizeof(buffer2) - 1);
		while ((exe = strrchr(buffer2, ' ')) != NULL) {
			int i, c;
			*exe++ = '\0';
			for (i = 0; i < initializers_count; i++) {
				if (strcmp(exe, initializers[i]) == 0) break;
			}
			if (i == initializers_count) continue;
			snprintf(buffer, sizeof(buffer) - 1, ROOT_NAME " %s", exe);
			fprintf(stderr, "Domain '%s' contains initializer '%s'.\n", domain_list[index].domainname, exe);
			fprintf(stderr, "Rename/Delete/Ignore? ");
			do { c = getc(stdin); } while (!(c == 'R' || c == 'r' || c == 'D' || c == 'd' || c == 'I' || c == 'i' || c == EOF));
			fprintf(stderr, "%c\n", c);
			if (c == 'R' || c == 'r') {
				fprintf(stderr, "'%s' -> '%s' renamed.\n", domain_list[index].domainname, buffer);
				domain_list[index].domainname = SaveName(buffer);
				acl_updated = 1;
			} else if (c == 'D' || c == 'd') {
				fprintf(stderr, "'%s' deleted.\n", domain_list[index].domainname);
				DeleteDomain(index); index--;
				acl_updated = 1;
			} else {
				fprintf(stderr, "Ignored.\n");
			}
			buffer2[0] = '\0';
			fprintf(stderr, "\n");
			break;
		}
	}
	SortPolicy();
	
	// Check if a domain contains a trusted pattern.
	for (index = 1; index < domain_list_count; index++) {
		const char *domainname = DomainName(index);
		int i;
		for (i = 0; i < trusted_domainnames_count; i++) {
			const char *p = trusted_domainnames[i];
			const int len = strlen(p);
			int c;
			if (strncmp(domainname, p, len) != 0 || domainname[len] != ' ') continue;
			fprintf(stderr, "Domain '%s' is unreachable because it contains trusted pattern '%s'.\n", domainname, p);
			fprintf(stderr, "Delete/Ignore? ");
			do { c = getc(stdin); } while (!(c == 'D' || c == 'd' || c == 'I' || c == 'i' || c == EOF));
			fprintf(stderr, "%c\n", c);
			if (c == 'D' || c == 'd') {
				fprintf(stderr, "'%s' deleted.\n", domainname);
				DeleteDomain(index); index--;
				acl_updated = 1;
			}
			fprintf(stderr, "\n");
			break;
		}
	}
	SortPolicy();

	// Merge ACLs for same domains.
	for (index = 1; index < domain_list_count; index++) {
		ACL_RECORD *acl_ptr;
		const char **string_ptr;
		int i;
		if (strcmp(domain_list[index - 1].domainname, domain_list[index].domainname)) continue;
		// Merge file policy.
		acl_ptr = domain_list[index].acl_ptr;
		for (i = 0; i < domain_list[index].acl_count; i++) AddFileACL(acl_ptr[i].name, acl_ptr[i].perm, index - 1);
		// Merge string policy.
		string_ptr = domain_list[index].string_ptr;
		for (i = 0; i < domain_list[index].string_count; i++) AddStringEntry(string_ptr[i], index - 1);
		// Delete duplicated domain.
		DeleteDomain(index);
		index = 0;
	}
	SortPolicy();
}

static void SavePolicy(void) {
	int c;
	if (!acl_updated) return;
	fprintf(stderr, "Policy has changed. Save?(Y/N)");
	do { c = getc(stdin); } while (!(c == 'Y' || c == 'y' || c == 'N' || c == 'n' || c == EOF));
	fprintf(stderr, "%c\n", c);
	if (c == 'Y' || c == 'y') {
		static char filename[1024];
		time_t now = time(NULL);
		struct tm *tm = localtime(&now);
		FILE *fp;
		int index;
		memset(filename, 0, sizeof(filename));
		snprintf(filename, sizeof(filename) - 1, "domain_policy.%02d-%02d-%02d.%02d:%02d:%02d.txt", tm->tm_year % 100, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
		if ((fp = fopen(filename, "w")) == NULL) {
			fprintf(stderr, "Can't open %s\n", filename); return;
		}
		for (index = 0; index < domain_list_count; index++) {
			int i;
			ACL_RECORD *acl_ptr = domain_list[index].acl_ptr;
			const char **string_ptr = domain_list[index].string_ptr;
			fprintf(fp, "%s\n\n", domain_list[index].domainname);
			for (i = 0; i < domain_list[index].acl_count; i++) fprintf(fp, "%d %s \n", acl_ptr[i].perm, acl_ptr[i].name);
			for (i = 0; i < domain_list[index].string_count; i++) fprintf(fp, "%s\n", string_ptr[i]);
			fprintf(fp, "\n");
		}
		fclose(fp);
		unlink("domain_policy.txt");
		symlink(filename, "domain_policy.txt");
	}
}

int main(int argc, char *argv[]) {
	struct termios tp, tp0;
	if (argc > 1) {
		if (chdir(argv[1]) != 0) {
			fprintf(stderr, "Can't chdir %s\n", argv[1]);
			return 1;
		}
	} else {
		if (chdir("/root/security") != 0) {
			fprintf(stderr, "Can't chdir /root/security\n");
			return 1;
		}
	}
	tcgetattr(0, &tp0); tp = tp0;
	tp.c_lflag &= ~(ICANON | ECHO);
	tp.c_cc[VTIME] = 0;
	tp.c_cc[VMIN] = 1;
	LoadPolicy();
	tcsetattr(0, TCSANOW, &tp);
	CheckPolicy();
	tcsetattr(0, TCSANOW, &tp0);
	SavePolicy();
	return 0;
}
