/*
 * editpolicy.c
 *
 * An editor for editing TOMOYO Linux's policy.
 *
 * Copyright (C) 2005-2006  NTT DATA CORPORATION
 *
 * Version: 1.1   2006/04/01
 *
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <curses.h>

#define ROOT_NAME "<kernel>"

#define WORD unsigned short int

#define MAXBUFSIZE  8192

static void OutOfMemory(void) {
	fprintf(stderr, "Out of memory. Aborted.\n");
	exit(127);
}

#define PAGE_SIZE  4096

static const char *SaveName(const char *name) {
	static char *buf = NULL;
	static int buf_used_len = PAGE_SIZE;
	int i, len;
	char *saved_name = NULL;
	char **new_ptr = NULL;
	static char **search_list = NULL;
	static int search_list_count = 0;
	if (!name) return NULL;
	len = strlen(name) + 1;
	if (len > PAGE_SIZE) {
		printf("ERROR: Name too long for SaveName().\n");
		return NULL;
	}
	for (i = 0; i < search_list_count; i++) {
		if (strcmp(name, search_list[i]) == 0) return search_list[i];
	}
	if (buf_used_len + len > PAGE_SIZE) {
		if ((buf = malloc(PAGE_SIZE)) == NULL) OutOfMemory();
		memset(buf, 0, PAGE_SIZE);
		buf_used_len = 0;
	}
	saved_name = buf + buf_used_len;
	memmove(saved_name, name, len);
	if ((new_ptr = (char **) realloc(search_list, (search_list_count + 1) * sizeof(char *))) == NULL) OutOfMemory();
	search_list = new_ptr;
	search_list[search_list_count++] = saved_name;
	buf_used_len += len;
	return (const char *) saved_name;
}

static int StrStr(const char *source, const char *target) {
	const char *cp;
	if ((cp = strstr(source, target)) != NULL) {
		const int len = strlen(target);
		if (cp[len] == ' ' || cp[len] == '\0') {
			if (cp == source || *(cp - 1) == ' ') return 1;
		}
	}
	return 0;
}

/*
 * Check whether the given filename is patterened.
 * Returns nonzero if patterned, zero otherwise.
 */
static int PathContainsPattern(const char *filename) {
	if (filename) {
		char c, d, e;
		while ((c = *filename++) != '\0') {
			if (c != '\\') continue;
			switch (c = *filename++) {
			case '\\':  /* "\\" */
				continue;
			case '0':   /* "\ooo" */
			case '1':
			case '2':
			case '3':
				if ((d = *filename++) >= '0' && d <= '7' && (e = *filename++) >= '0' && e <= '7'
					&& (c != '0' || d != '0' || e != '0')) continue; /* pattern is not \000 */
			}
			return 1;
		}
	}
	return 0;
}

/*
 * Check whether the given filename follows the naming rules.
 * Returns nonzero if follows, zero otherwise.
 */
static int IsCorrectPath(const char *filename, const int may_contain_pattern) {
	if (filename && *filename == '/') {
		char c, d, e;
		while ((c = *filename++) != '\0') {
			if (c == '\\') {
				switch ((c = *filename++)) {
				case '\\':  /* "\\" */
					continue;
				case '$':   /* "\$" */
				case '+':   /* "\+" */
				case '?':   /* "\?" */
					if (may_contain_pattern) continue;
					break;
				case '*':   /* "\*" */
					if (may_contain_pattern) {
						while (*filename && *filename != '/') filename++;
						if (!*filename || *filename == '/') continue;
					}
					break;
				case '0':   /* "\ooo" */
				case '1':
				case '2':
				case '3':
					if ((d = *filename++) >= '0' && d <= '7' && (e = *filename++) >= '0' && e <= '7') {
						const unsigned char f =
							(((unsigned char) (c - '0')) << 6) +
							(((unsigned char) (d - '0')) << 3) +
							(((unsigned char) (e - '0')));
						if (f && (f <= ' ' || f >= 127)) continue; /* pattern is not \000 */
					}
				}
				return 0;
			} else if (c <= ' ' || c >= 127) {
				return 0;
			}
		}
		return 1;
	}
	return 0;
}

static int IsCorrectDomain(const unsigned char *domainname) {
	unsigned char c, d, e;
	if (!domainname || strncmp(domainname, ROOT_NAME, strlen(ROOT_NAME))) goto out;
	domainname += strlen(ROOT_NAME);
	if (!*domainname) return 1;
	do {
		if (*domainname++ != ' ') goto out;
		if (*domainname++ != '/') goto out;
		while ((c = *domainname) != '\0' && c != ' ') {
			domainname++;
			if (c == '\\') {
				switch ((c = *domainname++)) {
				case '\\':  /* "\\" */
					continue;
				case '0':   /* "\ooo" */
				case '1':
				case '2':
				case '3':
					if ((d = *domainname++) >= '0' && d <= '7' && (e = *domainname++) >= '0' && e <= '7') {
						const unsigned char f =
							(((unsigned char) (c - '0')) << 6) +
							(((unsigned char) (d - '0')) << 3) +
							(((unsigned char) (e - '0')));
						if (f && (f <= ' ' || f >= 127)) continue; /* pattern is not \000 */
					}
				}
				goto out;
			} else if (c < ' ' || c >= 127) {
				goto out;
			}
		}
	} while (*domainname);
	return 1;
 out:
	return 0;
}

static inline int strendswith(const char *name, const char *tail) {
	int len;
	if (!name || !tail) return 0;
	len = strlen(name) - strlen(tail);
	return len >= 0 && strcmp(name + len, tail) == 0;
}

static void ReadSystemPolicy(void);
static void ReadExceptionPolicy(void);
static void ReadDomainPolicy(void);
static void SortDomainPolicyByName(void);
static void SortACLByName(const int index);
static void UpdateDomainPolicy(const int index);
static void SaveSystemPolicy(void);
static void SaveExceptionPolicy(void);
static void SaveDomainPolicy(void);
static int AddFileACL(const char *filename, WORD perm, const int index);
static int DelFileACL(const char *filename, const int index);
static int FindOrAssignNewDomain(const char *domainname);
static int FindDomain(const char *domainname);
static void NormalizeLine(unsigned char *buffer);
static const char *DomainName(const int index);
static int FindParent(const char *key);
static void DeleteDomain(const int index);
static int IsTrustedDomain(const int index);
static int IsInitializer(const char *filename);
static int IsInitializerDomain(const int index);
static int IsVirtualDomain(const int index);
static int IsGloballyReadableFile(const char *filename);

static int edit_on_memory = 0;
static const char *SYSTEM_POLICY_FILE[2] = {
	"system_policy.txt",
	"/proc/ccs/policy/system_policy"
};
static const char *EXCEPTION_POLICY_FILE[2] = {
	"exception_policy.txt",
	"/proc/ccs/policy/exception_policy"
};
static const char *DOMAIN_POLICY_FILE[2] = {
	"domain_policy.txt",
	"/proc/ccs/policy/domain_policy"
};

// List for system policy.
static const char **system_list = NULL;
static int system_list_count = 0;

// List for exception policy.
static const char **exception_list = NULL;
static int exception_list_count = 0;

// List for domain policy.
typedef struct {
	const char *name;
	WORD perm;
} ACL_RECORD;

typedef struct domain_info {
	ACL_RECORD *acl_ptr;
	int acl_count;
	const char *domainname;
	const char **string_ptr;
	int string_count;
	int is_virtual;
} DOMAIN_INFO;

static DOMAIN_INFO *domain_list = NULL;
static int domain_list_count = 0;

static const char **trusted_domainnames = NULL;
static int trusted_domainnames_count = 0;

static const char **initializers = NULL;
static int initializers_count = 0;

static const char **globally_readable_list = NULL;
static int globally_readable_list_count = 0;

static int delete_domain_fd = EOF, update_domain_fd = EOF;

///////////////////////////  ACL HANDLER  //////////////////////////////

static int AddFileACL(const char *filename, WORD perm, const int index) {
	ACL_RECORD *acl_ptr, *new_aclptr;
	const char *cp;
	int i;
	if (index < 0 || index >= domain_list_count) {
		fprintf(stderr, "%s: ERROR: domain is out of range.\n", __FUNCTION__);
		return -EINVAL;
	}
	if (!IsCorrectPath(filename, 1)) {
		fprintf(stderr, "%s: Invalid pathname '%s'\n",  __FUNCTION__, filename);
		return -EINVAL;
	}
	if (perm > 7 || !perm) {
		fprintf(stderr, "%s: Invalid permission '%d %s'\n",  __FUNCTION__, perm, filename);
		return -EINVAL;
	}
	if (strendswith(filename, "/")) {
		perm |= 5;  /* Always allow read and execute for dir. */
	} else if ((perm & 1) == 1 && PathContainsPattern(filename)) {
		perm ^= 1;  /* Never allow execute permission with patterns. */
		fprintf(stderr, "%s: Dropping execute permission for '%s'\n", __FUNCTION__, filename);
	} else if (perm == 4 && IsGloballyReadableFile(filename)) {
		return 0;   /* Don't add if the file is globally readable files. */
	}
	
	acl_ptr = domain_list[index].acl_ptr;
	
	// Check if the file was accessed before.
	for (i = 0; i < domain_list[index].acl_count; i++) {
		const char *cp = acl_ptr[i].name;
		if (strcmp(filename, cp) == 0) {
			acl_ptr[i].perm |= perm;
			return 0;
		}
	}
	if ((cp = SaveName(filename)) == NULL) OutOfMemory();
	
	// Allocate space.
	i = domain_list[index].acl_count;
	if ((new_aclptr = (ACL_RECORD *) realloc(acl_ptr, (i + 1) * sizeof(ACL_RECORD))) == NULL) OutOfMemory();
	domain_list[index].acl_ptr = acl_ptr = new_aclptr;
	
	// Record the access log.
	acl_ptr[i].perm = perm;
	acl_ptr[i].name = cp;
	domain_list[index].acl_count++;
	return 0;
}

static int DelFileACL(const char *filename, const int index) {
	ACL_RECORD *acl_ptr;
	int i;
	if (index < 0 || index >= domain_list_count) {
		fprintf(stderr, "%s: ERROR: domain is out of range.\n",  __FUNCTION__);
		return -EINVAL;
	}
	acl_ptr = domain_list[index].acl_ptr;
	for (i = 0; i < domain_list[index].acl_count; i++) {
		const char *cp = acl_ptr[i].name;
		if (strcmp(filename, cp) == 0) {
			int j;
			domain_list[index].acl_count--;
			for (j = i; j < domain_list[index].acl_count; j++) {
				acl_ptr[j] = acl_ptr[j + 1];
			}
			return 0;
		}
	}
	return -ENOENT;
}

static int AddStringEntry(const char *entry, const int index) {
	const char **acl_ptr;
	int acl_count;
	const char *cp;
	int i;
	if (index < 0 || index >= domain_list_count) {
		fprintf(stderr, "%s: ERROR: domain is out of range.\n",  __FUNCTION__);
		return -EINVAL;
	}
	if (!entry || !*entry) return -EINVAL;
	acl_ptr = domain_list[index].string_ptr;
	acl_count = domain_list[index].string_count;
	
	// Check for the same entry.
	for (i = 0; i < acl_count; i++) {
		if (strcmp(entry, acl_ptr[i]) == 0) return 0;
	}

	if ((cp = SaveName(entry)) == NULL) OutOfMemory();
	if ((acl_ptr = (const char **) realloc(acl_ptr, (acl_count + 1) * sizeof(char *))) == NULL) OutOfMemory();
	acl_ptr[acl_count++] = cp;
	domain_list[index].string_ptr = acl_ptr;
	domain_list[index].string_count = acl_count;
	return 0;
}

static int DelStringEntry(const char *entry, const int index) {
	const char **acl_ptr;
	int acl_count;
	int i;
	if (index < 0 || index >= domain_list_count) {
		fprintf(stderr, "%s: ERROR: domain is out of range.\n",  __FUNCTION__);
		return -EINVAL;
	}
	if (!entry || !*entry) return -EINVAL;
	acl_ptr = domain_list[index].string_ptr;
	acl_count = domain_list[index].string_count;
	for (i = 0; i < acl_count; i++) {
		int j;
		if (strcmp(entry, acl_ptr[i])) continue;
		acl_count--;
		for (j = i; j < acl_count; j++) {
			acl_ptr[j] = acl_ptr[j + 1];
		}
		domain_list[index].string_count--;
		return 0;
	}
	return -ENOENT;
}

static int IsTrustedDomain(const int index) {
	int i;
	for (i = 0; i < trusted_domainnames_count; i++) {
		if (StrStr(domain_list[index].domainname, trusted_domainnames[i])) return 1;
	}
	return 0;
}

static int IsInitializer(const char *filename) {
	if (filename) {
		int i;
		for (i = 0; i < initializers_count; i++) {
			if (strcmp(filename, initializers[i]) == 0) return 1;
		}
	}
	return 0;
}

static int IsInitializerDomain(const int index) {
	char *cp = strchr(domain_list[index].domainname, ' ');
	if (cp) return IsInitializer(cp + 1);
	return 0;
}

static int IsVirtualDomain(const int index) {
	return domain_list[index].is_virtual;
}

static int IsGloballyReadableFile(const char *filename) {
	if (filename) {
		int i;
		for (i = 0; i < globally_readable_list_count; i++) {
			if (strcmp(filename, globally_readable_list[i]) == 0) return 1;
		}
	}
	return 0;
}

////////////////////////////  DOMAIN HANDLER  ////////////////////////////

static int FindDomain(const char *domainname) {
	int i;
	for (i = 0; i < domain_list_count; i++) {
		if (strcmp(domainname, domain_list[i].domainname) == 0) {
			return i;
		}
	}
	return EOF;
}

static int FindOrAssignNewDomain(const char *domainname) {
	const char *saved_domainname;
	int index;
	if ((index = FindDomain(domainname)) == EOF) {
		if (IsCorrectDomain(domainname)) {
			if ((domain_list = (DOMAIN_INFO *) realloc(domain_list, (domain_list_count + 1) * sizeof(DOMAIN_INFO))) == NULL) OutOfMemory();
			memset(&domain_list[domain_list_count], 0, sizeof(DOMAIN_INFO));
			if ((saved_domainname = SaveName(domainname)) == NULL) OutOfMemory();
			domain_list[domain_list_count].domainname = saved_domainname;
			index = domain_list_count++;
		} else {
			fprintf(stderr, "%s: Invalid domainname '%s'\n",  __FUNCTION__, domainname);
		}
	}
	return index;
}

static const char *DomainName(const int index) {
	return domain_list[index].domainname;
}

static int FindParent(const char *key) {
	int index = EOF;
	char *parent = strdup(key);
	char *cp;
	if (!parent) OutOfMemory();
	cp = strrchr(parent, ' ');
	if (cp != NULL) {
		int i;
		*cp++ = '\0';
		for (i = 0; i < domain_list_count; i++) {
			if (strcmp(domain_list[i].domainname, parent) == 0) {
				index = i;
				break;
			}
		}
	}
	free(parent);
	return index;
}

static void DeleteDomain(const int index) {
	if (index > 0 && index < domain_list_count) {
		int i;
		const int len = strlen(domain_list[index].domainname);
		write(delete_domain_fd, domain_list[index].domainname, len);
		write(delete_domain_fd, "\n", 1);
		for (i = index; i < domain_list_count - 1; i++) domain_list[i] = domain_list[i + 1];
		domain_list_count--;
	}
}

///////////////////////////  UTILITY FUNCTIONS  //////////////////////////////

static int IsDomainDef(const unsigned char *buffer) {
	while (*buffer && (*buffer <= 32 || 127 <= *buffer)) buffer++;
	return strncmp(buffer, ROOT_NAME, strlen(ROOT_NAME)) == 0;
}

static void NormalizeLine(unsigned char *buffer) {
	unsigned char *sp = buffer, *dp = buffer;
	int first = 1;
	while (*sp && (*sp <= 32 || 127 <= *sp)) sp++;
	while (*sp) {
		if (!first) *dp++ = ' ';
		first = 0;
		while (32 < *sp && *sp < 127) *dp++ = *sp++;
		while (*sp && (*sp <= 32 || 127 <= *sp)) sp++;
	}
	*dp = '\0';
}

static void SortDomainPolicyByName(void) {
	int i, j;
	for (i = 0; i < domain_list_count; i++) {
		for (j = i + 1; j < domain_list_count; j++) {
			if (strcmp(domain_list[i].domainname, domain_list[j].domainname) > 0) {
				DOMAIN_INFO tmp = domain_list[i]; domain_list[i] = domain_list[j]; domain_list[j] = tmp;
			}
		}
	}
	for (i = 0; i < domain_list_count; i++) SortACLByName(i);
}

static int SortAndUniq(const char **list, int list_count) {
	int i, j;
	for (i = 0; i < list_count; i++) {
		for (j = i + 1; j < list_count; j++) {
			if (strcmp(list[i], list[j]) > 0) {
				const char *tmp = list[i];
				list[i] = list[j];
				list[j] = tmp;
			}
		}
	}
	for (i = 0; i < list_count - 1; i++) {
		if (strcmp(list[i], list[i + 1])) continue;
		for (j = i + 1; j < list_count - 1; j++) list[j] = list[j + 1];
		list_count--; i--;
	}
	return list_count;
}

static void AddSystemPolicy(char *buffer) {
	char *cp;
	NormalizeLine(buffer);
	if ((cp = (char *) SaveName(buffer)) == NULL) OutOfMemory();
	if ((system_list = (const char **) realloc(system_list, (system_list_count + 1) * sizeof(const char *))) == NULL) OutOfMemory();
	system_list[system_list_count++] = cp;
}

static void DelSystemPolicy(const char *buffer) {
	int i;
	for (i = 0; i < system_list_count; i++) {
		if (strcmp(buffer, system_list[i])) continue;
		for (system_list_count--; i < system_list_count; i++) system_list[i] = system_list[i + 1];
		break;
	}
}

static void ReadSystemPolicy(void) {
	static char buffer[MAXBUFSIZE];
	FILE *fp;
	system_list_count = 0;
	if ((fp = fopen(SYSTEM_POLICY_FILE[edit_on_memory], "r")) != NULL) {
		while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
			char *cp = strchr(buffer, '\n');
			if (cp) *cp = '\0';
			else if (!feof(fp)) break;
			AddSystemPolicy(buffer);
		}
		fclose(fp);
	}
	system_list_count = SortAndUniq(system_list, system_list_count);
}

static void AddExceptionPolicy(char *buffer) {
	char *cp;
	NormalizeLine(buffer);
	if ((cp = (char *) SaveName(buffer)) == NULL) OutOfMemory();
	if ((exception_list = (const char **) realloc(exception_list, (exception_list_count + 1) * sizeof(const char *))) == NULL) OutOfMemory();
	exception_list[exception_list_count++] = cp;
}

static void DelExceptionPolicy(const char *buffer) {
	int i;
	for (i = 0; i < exception_list_count; i++) {
		if (strcmp(buffer, exception_list[i])) continue;
		for (exception_list_count--; i < exception_list_count; i++) exception_list[i] = exception_list[i + 1];
		break;
	}
}

static void ReadExceptionPolicy(void) {
	static char buffer[MAXBUFSIZE];
	FILE *fp;
	exception_list_count = 0;
	if ((fp = fopen(EXCEPTION_POLICY_FILE[edit_on_memory], "r")) != NULL) {
		while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
			char *cp = strchr(buffer, '\n');
			if (cp) *cp = '\0';
			else if (!feof(fp)) break;
			AddExceptionPolicy(buffer);
		}
		fclose(fp);
	}
	exception_list_count = SortAndUniq(exception_list, exception_list_count);
}

static void ReadDomainPolicy(void) {
	static char buffer[MAXBUFSIZE];
	FILE *fp;
	int index;
	FindOrAssignNewDomain(ROOT_NAME);

	// Load initializer list, trusted domain list, globally readable file list.
	initializers_count = 0;
	trusted_domainnames_count = 0;
	globally_readable_list = 0;

	if ((fp = fopen(EXCEPTION_POLICY_FILE[edit_on_memory], "r")) != NULL) {
		while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
			char *cp = strchr(buffer, '\n');
			if (cp) *cp = '\0';
			else if (!feof(fp)) break;
			NormalizeLine(buffer);
			if (strncmp(buffer, "initializer ", 12) == 0) {
				memmove(buffer, buffer + 12, strlen(buffer + 12) + 1);
				if (!IsCorrectPath(buffer, 0)) continue;
				if ((cp = (char *) SaveName(buffer)) == NULL) OutOfMemory();
				if ((initializers = (const char **) realloc(initializers, (initializers_count + 1) * sizeof(const char *))) == NULL) OutOfMemory();
				initializers[initializers_count++] = cp;
			} else if (strncmp(buffer, "trust_domain ", 13) == 0) {
				memmove(buffer, buffer + 13, strlen(buffer + 13) + 1);
				if (!IsCorrectDomain(buffer)) continue;
				if ((cp = (char *) SaveName(buffer)) == NULL) OutOfMemory();
				if ((trusted_domainnames = (const char **) realloc(trusted_domainnames, (trusted_domainnames_count + 1) * sizeof(const char *))) == NULL) OutOfMemory();
				trusted_domainnames[trusted_domainnames_count++] = cp;
			} else if (strncmp(buffer, "allow_read ", 11) == 0) {
				memmove(buffer, buffer + 11, strlen(buffer + 11) + 1);
				if (!IsCorrectPath(buffer, 0)) continue;
				if ((cp = (char *) SaveName(buffer)) == NULL) OutOfMemory();
				if ((globally_readable_list = (const char **) realloc(globally_readable_list, (globally_readable_list_count + 1) * sizeof(const char *))) == NULL) OutOfMemory();
				globally_readable_list[globally_readable_list_count++] = cp;
			}
		}
		fclose(fp);
		initializers_count = SortAndUniq(initializers, initializers_count);
		trusted_domainnames_count = SortAndUniq(trusted_domainnames, trusted_domainnames_count);
		globally_readable_list_count = SortAndUniq(globally_readable_list, globally_readable_list_count);
	}
	
	// Load all domain list.
	if ((fp = fopen(DOMAIN_POLICY_FILE[edit_on_memory], "r")) != NULL) {
		index = EOF;
		while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
			char *cp = strchr(buffer, '\n');
			int perm;
			if (cp) *cp = '\0';
			else if (!feof(fp)) break;
			NormalizeLine(buffer);
			if (IsDomainDef(buffer)) {
				index = FindOrAssignNewDomain(buffer);
			} else if (index >= 0 && sscanf(buffer, "%d", &perm) == 1 && (cp = strchr(buffer, ' ')) != NULL) {
				AddFileACL(cp + 1, (WORD) perm, index);
			} else if (index >= 0 && strncmp(buffer, "allow_", 6) == 0) {
				AddStringEntry(buffer, index);
			}
		}
		fclose(fp);
	}
	SortDomainPolicyByName();

	// Create virtual domains.
	for (index = 1; index < domain_list_count; index++) {
		ACL_RECORD *acl_ptr = domain_list[index].acl_ptr;
		int i;
		for (i = 0; i < domain_list[index].acl_count; i++) {
			int virtual_index;
			const char *cp = acl_ptr[i].name;
			if ((acl_ptr[i].perm & 1) == 0 || PathContainsPattern(cp) || strendswith(cp, "/") || !IsInitializer(cp)) continue;
			snprintf(buffer, sizeof(buffer) - 1, "%s %s", domain_list[index].domainname, cp);
			NormalizeLine(buffer);
			if ((virtual_index = FindOrAssignNewDomain(buffer)) == EOF) {
				fprintf(stderr, "ERROR: Can't create domain for '%s'.\n", buffer);
				exit(127);
			}
			domain_list[virtual_index].is_virtual = 1;
		}
	}
	SortDomainPolicyByName();
}

static void UpdateDomainPolicy(const int index) {
	FILE *fp = NULL;
	ACL_RECORD *acl_ptr = domain_list[index].acl_ptr;
	const char **string_ptr = domain_list[index].string_ptr;
	int i;
	if (!edit_on_memory) {
		SaveDomainPolicy();
		return;
	}
	if ((fp = fopen(DOMAIN_POLICY_FILE[edit_on_memory], "w")) == NULL) return;
	{
		const int len = strlen(domain_list[index].domainname);
		write(delete_domain_fd, domain_list[index].domainname, len);
		write(delete_domain_fd, "\n", 1);
	}
	fprintf(fp, "%s\n\n", domain_list[index].domainname);
	for (i = 0; i < domain_list[index].acl_count; i++) {
		fprintf(fp, "%d %s \n", acl_ptr[i].perm, acl_ptr[i].name);
	}
	for (i = 0; i < domain_list[index].string_count; i++) {
		fprintf(fp, "%s\n", string_ptr[i]);
	}
	fprintf(fp, "\n");
	fclose(fp);
	{
		const int len = strlen(domain_list[index].domainname);
		write(update_domain_fd, domain_list[index].domainname, len);
		write(update_domain_fd, "\n", 1);
	}
}

static void SaveSystemPolicy(void) {
	FILE *fp;
	int i;
	static char filename[1024], prev_filename[1024] = "";
	time_t now = time(NULL);
	struct tm *tm = localtime(&now);
	memset(filename, 0, sizeof(filename));
	snprintf(filename, sizeof(filename) - 1, "system_policy.%02d-%02d-%02d.%02d:%02d:%02d.txt", tm->tm_year % 100, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
	if ((fp = fopen(filename, "w")) == NULL) {
		fprintf(stderr, "Can't open %s\n", filename); return;
	}
	for (i = 0; i < system_list_count; i++) {
		fprintf(fp, "%s\n", system_list[i]);
	}
	fclose(fp);
	if (strcmp(filename, prev_filename) == 0) return;
	unlink("system_policy.txt");
	symlink(filename, "system_policy.txt");
	unlink(prev_filename);
	memmove(prev_filename, filename, sizeof(prev_filename));
}

static void SaveExceptionPolicy(void) {
	FILE *fp;
	int i;
	static char filename[1024], prev_filename[1024] = "";
	time_t now = time(NULL);
	struct tm *tm = localtime(&now);
	memset(filename, 0, sizeof(filename));
	snprintf(filename, sizeof(filename) - 1, "exception_policy.%02d-%02d-%02d.%02d:%02d:%02d.txt", tm->tm_year % 100, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
	if ((fp = fopen(filename, "w")) == NULL) {
		fprintf(stderr, "Can't open %s\n", filename); return;
	}
	for (i = 0; i < exception_list_count; i++) {
		fprintf(fp, "%s\n", exception_list[i]);
	}
	fclose(fp);
	if (strcmp(filename, prev_filename) == 0) return;
	unlink("exception_policy.txt");
	symlink(filename, "exception_policy.txt");
	unlink(prev_filename);
	memmove(prev_filename, filename, sizeof(prev_filename));
}

static void SaveDomainPolicy(void) {
	FILE *fp;
	int index;
	static char filename[1024], prev_filename[1024] = "";
	time_t now = time(NULL);
	struct tm *tm = localtime(&now);
	memset(filename, 0, sizeof(filename));
	snprintf(filename, sizeof(filename) - 1, "domain_policy.%02d-%02d-%02d.%02d:%02d:%02d.txt", tm->tm_year % 100, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
	if ((fp = fopen(filename, "w")) == NULL) {
		fprintf(stderr, "Can't open %s\n", filename); return;
	}
	for (index = 0; index < domain_list_count; index++) {
		int i;
		ACL_RECORD *acl_ptr = domain_list[index].acl_ptr;
		const char **string_ptr = domain_list[index].string_ptr;
		if (domain_list[index].is_virtual == 1) continue;
		fprintf(fp, "%s\n\n", domain_list[index].domainname);
		for (i = 0; i < domain_list[index].acl_count; i++) fprintf(fp, "%d %s \n", acl_ptr[i].perm, acl_ptr[i].name);
		for (i = 0; i < domain_list[index].string_count; i++) fprintf(fp, "%s\n", string_ptr[i]);
		fprintf(fp, "\n");
	}
	fclose(fp);
	if (strcmp(filename, prev_filename) == 0) return;
	unlink("domain_policy.txt");
	symlink(filename, "domain_policy.txt");
	unlink(prev_filename);
	memmove(prev_filename, filename, sizeof(prev_filename));
}

static void SortACLByName(const int index) {
	DOMAIN_INFO *ptr = &domain_list[index];
	ACL_RECORD *acl_ptr = ptr->acl_ptr;
	const char **string_ptr = ptr->string_ptr;
	int i, j;
	for (i = 0; i < ptr->acl_count; i++) {
		for (j = i + 1; j < ptr->acl_count; j++) {
			if (strcmp(acl_ptr[i].name, acl_ptr[j].name) > 0) {
				ACL_RECORD tmp = acl_ptr[i]; acl_ptr[i] = acl_ptr[j]; acl_ptr[j] = tmp;
			}
		}
	}
	for (i = 0; i < ptr->string_count; i++) {
		for (j = i + 1; j < ptr->string_count; j++) {
			if (strcmp(string_ptr[i], string_ptr[j]) > 0) {
				const char *tmp = string_ptr[i]; string_ptr[i] = string_ptr[j]; string_ptr[j] = tmp;
			}
		}
	}
}

////////////////////////////  UI HANDLER  ////////////////////////////

static int SystemListLoop(void);
static int ExceptionListLoop(void);
static int DomainListLoop(void);
static void ACLListLoop(void);
static void ShowDomainList(void);
static void ShowACLList(void);
static void ShowExceptionList(void);
static void ShowSystemList(void);
static void ShowList(const int screen);

static void ResizeWindow(const int screen);
static void UpArrowKey(const int screen);
static void DownArrowKey(const int screen);
static void PageUpKey(const int screen);
static void PageDownKey(const int screen);
static void ShowCurrent(const int screen);
static void UpdateCursorPos(const int screen, const int item_count);
static int GetCurrent(const int screen);

#define SCREEN_SYSTEM_LIST    0
#define SCREEN_EXCEPTION_LIST 1
#define SCREEN_DOMAIN_LIST    2
#define SCREEN_ACL_LIST       3
#define MAXSCREEN             4

static int window_width = 0, window_height = 0;
static int current_y[MAXSCREEN], current_item_index[MAXSCREEN], list_item_count[MAXSCREEN];
static int pipe_fd[2] = { EOF, EOF };

static struct {
	int current_y[MAXSCREEN];
	int current_item_index[MAXSCREEN];
	char search_buffer[4096];
} history;

static const int header_lines = 3;
static int body_lines = 0;

static void ShowSystemList(void) {
	int i;
	const int offset = current_item_index[SCREEN_SYSTEM_LIST];
	static char buffer[MAXBUFSIZE];
	list_item_count[SCREEN_SYSTEM_LIST] = system_list_count;
	clear();
	if (window_height < header_lines + 1) {
		mvprintw(0, 0, "Please resize window. This program needs at least %d lines.\n", header_lines + 1);
		refresh();
		return;
	}
	memset(buffer, 0, sizeof(buffer));
	snprintf(buffer, sizeof(buffer) - 1, "<<< System Policy Editor >>>      %d entr%s", list_item_count[SCREEN_SYSTEM_LIST], list_item_count[SCREEN_SYSTEM_LIST] > 1 ? "ies" : "y");
	if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
	mvprintw(0, 0, "%s", buffer);
	snprintf(buffer, sizeof(buffer) - 1, "Commands =  Q:quit  A:append  D:delete  R:refresh  Tab:switch");
	if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
	mvprintw(1, 0, "%s", buffer);
	for (i = 0; i < body_lines; i++) {
		if (offset + i >= list_item_count[SCREEN_SYSTEM_LIST]) break;
		memset(buffer, 0, sizeof(buffer));
		snprintf(buffer, sizeof(buffer) - 1, "%5d: %s", offset + i, system_list[offset + i]);
		if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
		mvprintw(header_lines + i, 0, "%s", buffer);
	}
	ShowCurrent(SCREEN_SYSTEM_LIST);
}

static void ShowExceptionList(void) {
	int i;
	const int offset = current_item_index[SCREEN_EXCEPTION_LIST];
	static char buffer[MAXBUFSIZE];
	list_item_count[SCREEN_EXCEPTION_LIST] = exception_list_count;
	clear();
	if (window_height < header_lines + 1) {
		mvprintw(0, 0, "Please resize window. This program needs at least %d lines.\n", header_lines + 1);
		refresh();
		return;
	}
	memset(buffer, 0, sizeof(buffer));
	snprintf(buffer, sizeof(buffer) - 1, "<<< Exception Policy Editor >>>      %d entr%s", list_item_count[SCREEN_EXCEPTION_LIST], list_item_count[SCREEN_EXCEPTION_LIST] > 1 ? "ies" : "y");
	if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
	mvprintw(0, 0, "%s", buffer);
	snprintf(buffer, sizeof(buffer) - 1, "Commands =  Q:quit  A:append  D:delete  R:refresh  Tab:switch");
	if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
	mvprintw(1, 0, "%s", buffer);
	for (i = 0; i < body_lines; i++) {
		if (offset + i >= list_item_count[SCREEN_EXCEPTION_LIST]) break;
		memset(buffer, 0, sizeof(buffer));
		snprintf(buffer, sizeof(buffer) - 1, "%5d: %s", offset + i, exception_list[offset + i]);
		if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
		mvprintw(header_lines + i, 0, "%s", buffer);
	}
	ShowCurrent(SCREEN_EXCEPTION_LIST);
}

static void ShowDomainList(void) {
	int i;
	const int offset = current_item_index[SCREEN_DOMAIN_LIST];
	static char buffer[MAXBUFSIZE];
	list_item_count[SCREEN_DOMAIN_LIST] = domain_list_count;
	clear();
	if (window_height < header_lines + 1) {
		mvprintw(0, 0, "Please resize window. This program needs at least %d lines.\n", header_lines + 1);
		refresh();
		return;
	}
	memset(buffer, 0, sizeof(buffer));
	i = list_item_count[SCREEN_DOMAIN_LIST];
	snprintf(buffer, sizeof(buffer) - 1, "<<< Domain Transition Editor >>>      %d domain%c", i, i > 1 ? 's' : ' ');
	if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
	mvprintw(0, 0, "%s", buffer);
	snprintf(buffer, sizeof(buffer) - 1, "Commands =  Q:quit  D:delete  R:refresh  Enter:edit  @:search  Tab:switch");
	if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
	mvprintw(1, 0, "%s", buffer);
	for (i = 0; i < body_lines; i++) {
		const char *sp;
		char *dp, *cp;
		char flag1, flag2;
		int count = 0;
		if (offset + i >= list_item_count[SCREEN_DOMAIN_LIST]) break;
		flag1 = IsTrustedDomain(offset + i) ? '#' : ' ';
		flag2 = IsInitializerDomain(offset + i) ? '*' : ' ';
		memset(buffer, 0, sizeof(buffer));
		snprintf(buffer, sizeof(buffer) - 1, "%5d: %c%c ", offset + i, flag1, flag2);
		sp = DomainName(offset + i);
		while ((cp = strchr(sp, ' ')) != NULL) {
			count += 4; sp = cp + 1;
		}
		dp = strchr(buffer, '\0');
		while (count) {
			if (dp - buffer >= sizeof(buffer)) exit(127);
			*dp++ = ' '; count--;
		}
		while (*sp) {
			if (dp - buffer >= sizeof(buffer)) exit(127);
			*dp++ = *sp++;
		}
		if (IsVirtualDomain(offset + i)) {
			static char buf[MAXBUFSIZE];
			char *sp = buf;
			int redirect_index;
			memset(buf, 0, sizeof(buf));
			snprintf(buf, sizeof(buf) - 1, ROOT_NAME "%s", strrchr(DomainName(offset + i), ' '));
			redirect_index = FindDomain(buf);
			snprintf(buf, sizeof(buf) - 1, " ( -> %d )", redirect_index);
			while (*sp) {
				if (dp - buffer >= sizeof(buffer)) exit(127);
				*dp++ = *sp++;
			}
		}
		if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
		mvprintw(header_lines + i, 0, "%s", buffer);
	}
	ShowCurrent(SCREEN_DOMAIN_LIST);
}

static void ShowACLList(void) {
	const int current = GetCurrent(SCREEN_DOMAIN_LIST), offset = current_item_index[SCREEN_ACL_LIST];
	static char buffer[MAXBUFSIZE];
	DOMAIN_INFO *ptr = &domain_list[current];
	ACL_RECORD *acl_ptr = ptr->acl_ptr;
	const char **string_ptr = ptr->string_ptr;
	int i;
	list_item_count[SCREEN_ACL_LIST] = ptr->acl_count + ptr->string_count;
	clear();
	if (window_height < header_lines + 1) {
		mvprintw(0, 0, "Please resize window. This program needs at least %d lines.\n", header_lines + 1);
		refresh();
		return;
	}
	memset(buffer, 0, sizeof(buffer));
	i = list_item_count[SCREEN_ACL_LIST];
	snprintf(buffer, sizeof(buffer) - 1, "<<< Domain ACL Editor >>>      %d acl%c", i, i > 1 ? 's' : ' ');
	if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
	mvprintw(0, 0, "%s", buffer);
	snprintf(buffer, sizeof(buffer) - 1, "Commands =  A:append  D:delete  Enter:cancel  S:save");
	if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
	mvprintw(1, 0, "%s", buffer);
	snprintf(buffer, sizeof(buffer) - 1, "%s", DomainName(current));
	if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
	mvprintw(2, 0, "%s", buffer);
	for (i = 0; i < body_lines; i++) {
		int index = offset + i;
		if (index >= list_item_count[SCREEN_ACL_LIST]) break;
		if (index < ptr->acl_count) {
			mvprintw(header_lines + i, 0, "%5d: %d %s", offset + i, acl_ptr[index].perm, acl_ptr[index].name);
		} else {
			index -= ptr->acl_count;
			mvprintw(header_lines + i, 0, "%5d: %s", offset + i, string_ptr[index]);
		}
	}
	ShowCurrent(SCREEN_ACL_LIST);
}

static void ShowList(const int screen) {
	switch (screen) {
	case SCREEN_DOMAIN_LIST:
		ShowDomainList();
		break;
	case SCREEN_ACL_LIST:
		ShowACLList();
		break;
	case SCREEN_EXCEPTION_LIST:
		ShowExceptionList();
		break;
	case SCREEN_SYSTEM_LIST:
		ShowSystemList();
		break;
	}
}

static void ResizeWindow(const int screen) {
	getmaxyx(stdscr, window_height, window_width);
	body_lines = window_height - header_lines;
	if (body_lines <= current_y[screen]) current_y[screen] = body_lines - 1;
	if (current_y[screen] < 0) current_y[screen] = 0;
}

static void UpArrowKey(const int screen) {
	if (current_y[screen] > 0) {
		current_y[screen]--;
		ShowCurrent(screen);
	} else if (current_item_index[screen] > 0) {
		current_item_index[screen]--;
		ShowList(screen);
	}
}

static void DownArrowKey(const int screen) {
	if (current_y[screen] < body_lines - 1) {
		if (current_item_index[screen] + current_y[screen] < list_item_count[screen] - 1) {
			current_y[screen]++;
			ShowCurrent(screen);
		}
	} else if (current_item_index[screen] + current_y[screen] < list_item_count[screen] - 1) {
		current_item_index[screen]++;
		ShowList(screen);
	}
}

static void PageUpKey(const int screen) {
	if (current_item_index[screen] + current_y[screen] > body_lines) {
		current_item_index[screen] -= body_lines;
		if (current_item_index[screen] < 0) current_item_index[screen] = 0;
		ShowList(screen);
	} else if (current_item_index[screen] + current_y[screen] > 0) {
		current_item_index[screen] = 0;
		current_y[screen] = 0;
		ShowList(screen);
	}
}

static void PageDownKey(const int screen) {
	if (list_item_count[screen] - current_item_index[screen] > body_lines) {
		current_item_index[screen] += body_lines;
		if (current_item_index[screen] + current_y[screen] > list_item_count[screen] - 1) current_y[screen] = list_item_count[screen] - 1 - current_item_index[screen];
		ShowList(screen);
	} else if (current_item_index[screen] + current_y[screen] < list_item_count[screen] - 1) {
		current_y[screen] = list_item_count[screen] - current_item_index[screen] - 1;
		ShowCurrent(screen);
	}
}

static int GetCurrent(const int screen) {
	if (list_item_count[screen] == 0) return EOF;
	if (current_item_index[screen] + current_y[screen] < 0 || current_item_index[screen] + current_y[screen] >= list_item_count[screen]) {
		fprintf(stderr, "ERROR: current_item_index=%d current_y=%d\n", current_item_index[screen], current_y[screen]);
		exit(127);
	}
	return current_item_index[screen] + current_y[screen];
}

static void ShowCurrent(const int screen) {
	if (screen == SCREEN_DOMAIN_LIST) {
		static char buffer[MAXBUFSIZE];
		int current = GetCurrent(screen);
		memset(buffer, 0, sizeof(buffer));
		snprintf(buffer, sizeof(buffer) - 1, "%s", DomainName(current));
		if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
		move(2, 0);
		clrtoeol();
		printw("%s", buffer);
	}
	move(header_lines + current_y[screen], 0);
	refresh();
}

static void LoadCursorPos(const int screen) {
	current_item_index[screen] = history.current_item_index[screen];
	current_y[screen] = history.current_y[screen];
}

static void AdjustCursorPos(const int screen, const int item_count) {
	if (item_count == 0) {
		current_item_index[screen] = current_y[screen] = 0;
	} else {
		while (current_item_index[screen] + current_y[screen] >= item_count) {
			if (current_y[screen] > 0) current_y[screen]--;
			else if (current_item_index[screen] > 0) current_item_index[screen]--;
		}
	}
}

static void SaveCursorPos(const int screen) {
	history.current_item_index[screen] = current_item_index[screen];
	history.current_y[screen] = current_y[screen];
	write(pipe_fd[1], &history, sizeof(history));
}

static void UpdateCursorPos(const int screen, const int item_count) {
	list_item_count[screen] = item_count;
	AdjustCursorPos(screen, item_count);
}

static int SystemListLoop(void) {
	LoadCursorPos(SCREEN_SYSTEM_LIST);
 start:
	ReadSystemPolicy();
	AdjustCursorPos(SCREEN_SYSTEM_LIST, system_list_count);
	ShowList(SCREEN_SYSTEM_LIST);
	while (1) {
		const int c = getch();
		SaveCursorPos(SCREEN_SYSTEM_LIST); 
		if (c == 'q' || c == 'Q') break;
		if (c == '\t') return SCREEN_EXCEPTION_LIST;
		if (c == ERR) continue; // Ignore invalid key.
		switch(c) {
		case KEY_RESIZE:
			ResizeWindow(SCREEN_SYSTEM_LIST);
			ShowList(SCREEN_SYSTEM_LIST);
			break;
		case KEY_UP:
			UpArrowKey(SCREEN_SYSTEM_LIST);
			break;
		case KEY_DOWN:
			DownArrowKey(SCREEN_SYSTEM_LIST);
			break;
		case KEY_PPAGE:
		case KEY_DC:
			PageUpKey(SCREEN_SYSTEM_LIST);
			break;
		case KEY_NPAGE:
			PageDownKey(SCREEN_SYSTEM_LIST);
			break;
		case 'd':
		case 'D':
			{
				const int index = GetCurrent(SCREEN_SYSTEM_LIST);
				if (index >= 0) {
					if (edit_on_memory) {
						int fd = open(SYSTEM_POLICY_FILE[edit_on_memory], O_WRONLY);
						if (fd != EOF) {
							write(fd, "delete ", 7); write(fd, system_list[index], strlen(system_list[index])); write(fd, "\n", 1); close(fd);
							close(fd);
						}
					} else {
						DelSystemPolicy(system_list[index]);
						SaveSystemPolicy();
					}
					goto start;
				}
			}
			break;
		case 'a':
		case 'A':
			{
				static char buffer[4096];
				mvprintw(window_height - 1, 0, "Enter new entry> "); clrtoeol(); refresh(); echo();
				memset(buffer, 0, sizeof(buffer));
				getnstr(buffer, sizeof(buffer) - 1); noecho();
				NormalizeLine(buffer);
				if (edit_on_memory) {
					int fd = open(SYSTEM_POLICY_FILE[edit_on_memory], O_WRONLY);
					if (fd != EOF) {
						write(fd, buffer, strlen(buffer)); write(fd, "\n", 1); close(fd);
					}
				} else {
					AddSystemPolicy(buffer);
					system_list_count = SortAndUniq(system_list, system_list_count);
					SaveSystemPolicy();
				}
				goto start;
			}
			break;
		case 'r':
		case 'R':
			return SCREEN_SYSTEM_LIST;
		}
	}
	return MAXSCREEN + 1;
}

static int ExceptionListLoop(void) {
	LoadCursorPos(SCREEN_EXCEPTION_LIST);
 start:
	ReadExceptionPolicy();
	AdjustCursorPos(SCREEN_EXCEPTION_LIST, exception_list_count);
	ShowList(SCREEN_EXCEPTION_LIST);
	while (1) {
		const int c = getch();
		SaveCursorPos(SCREEN_EXCEPTION_LIST);
		if (c == 'q' || c == 'Q') break;
		if (c == '\t') return SCREEN_DOMAIN_LIST;
		if (c == ERR) continue; // Ignore invalid key.
		switch (c) {
		case KEY_RESIZE:
			ResizeWindow(SCREEN_EXCEPTION_LIST);
			ShowList(SCREEN_EXCEPTION_LIST);
			break;
		case KEY_UP:
			UpArrowKey(SCREEN_EXCEPTION_LIST);
			break;
		case KEY_DOWN:
			DownArrowKey(SCREEN_EXCEPTION_LIST);
			break;
		case KEY_PPAGE:
		case KEY_DC:
			PageUpKey(SCREEN_EXCEPTION_LIST);
			break;
		case KEY_NPAGE:
			PageDownKey(SCREEN_EXCEPTION_LIST);
			break;
		case 'd':
		case 'D':
			{
				const int index = GetCurrent(SCREEN_EXCEPTION_LIST);
				if (index >= 0) {
					if (edit_on_memory) {
						int fd = open(EXCEPTION_POLICY_FILE[edit_on_memory], O_WRONLY);
						if (fd != EOF) {
							write(fd, "delete ", 7); write(fd, exception_list[index], strlen(exception_list[index])); write(fd, "\n", 1); close(fd);
							close(fd);
						}
					} else {
						DelExceptionPolicy(exception_list[index]);
						SaveExceptionPolicy();
					}
					goto start;
				}
			}
			break;
		case 'a':
		case 'A':
			{
				static char buffer[4096];
				mvprintw(window_height - 1, 0, "Enter new entry> "); clrtoeol(); refresh(); echo();
				memset(buffer, 0, sizeof(buffer));
				getnstr(buffer, sizeof(buffer) - 1); noecho();
				NormalizeLine(buffer);
				if (edit_on_memory) {
					int fd = open(EXCEPTION_POLICY_FILE[edit_on_memory], O_WRONLY);
					if (fd != EOF) {
						write(fd, buffer, strlen(buffer)); write(fd, "\n", 1); close(fd);
					}
				} else {
					AddExceptionPolicy(buffer);
					exception_list_count = SortAndUniq(exception_list, exception_list_count);
					SaveExceptionPolicy();
				}
				goto start;
			}
			break;
		case 'r':
		case 'R':
			return SCREEN_EXCEPTION_LIST;
		}
	}
	return MAXSCREEN + 1;
}

static int DomainListLoop(void) {
	LoadCursorPos(SCREEN_DOMAIN_LIST);
 start: ;
	ReadDomainPolicy();
	AdjustCursorPos(SCREEN_DOMAIN_LIST, domain_list_count);
	ShowList(SCREEN_DOMAIN_LIST);
	while (1) {
		const int c = getch();
		SaveCursorPos(SCREEN_DOMAIN_LIST);
		if (c == 'q' || c == 'Q') break;
		if (c == '\t') return SCREEN_SYSTEM_LIST; 
		if (c == ERR) continue; // Ignore invalid key.
		switch (c) {
		case KEY_RESIZE:
			ResizeWindow(SCREEN_DOMAIN_LIST);
			ShowList(SCREEN_DOMAIN_LIST);
			break;
		case KEY_UP:
			UpArrowKey(SCREEN_DOMAIN_LIST);
			break;
		case KEY_DOWN:
			DownArrowKey(SCREEN_DOMAIN_LIST);
			break;
		case KEY_PPAGE:
		case KEY_DC:
			PageUpKey(SCREEN_DOMAIN_LIST);
			break;
		case KEY_NPAGE:
			PageDownKey(SCREEN_DOMAIN_LIST);
			break;
		case '@':
			{
				int i;
			input_path:
				mvprintw(window_height - 1, 0, "Search> "); clrtoeol(); refresh(); echo();
				memset(history.search_buffer, 0, sizeof(history.search_buffer));
				getnstr(history.search_buffer, sizeof(history.search_buffer) - 1); noecho();
				NormalizeLine(history.search_buffer);
				if (history.search_buffer[0]) {
					for (i = 0; i < list_item_count[SCREEN_DOMAIN_LIST]; i++) {
						const char *cp = DomainName(i);
						if (strchr(history.search_buffer, '/')) {
							if (strrchr(cp, ' ')) cp = strrchr(cp, ' ') + 1;
						} else {
							if (strrchr(cp, '/')) cp = strrchr(cp, '/') + 1;
						}
						if (strcmp(cp, history.search_buffer)) continue;
						while (i < current_y[SCREEN_DOMAIN_LIST] + current_item_index[SCREEN_DOMAIN_LIST]) {
							if (current_y[SCREEN_DOMAIN_LIST] > 0) current_y[SCREEN_DOMAIN_LIST]--;
							else current_item_index[SCREEN_DOMAIN_LIST]--;
						}
						while (i > current_y[SCREEN_DOMAIN_LIST] + current_item_index[SCREEN_DOMAIN_LIST]) {
							if (current_y[SCREEN_DOMAIN_LIST] < body_lines - 1) current_y[SCREEN_DOMAIN_LIST]++;
							else current_item_index[SCREEN_DOMAIN_LIST]++;
						}
						break;
					}
				}
				ShowList(SCREEN_DOMAIN_LIST);
			}
			break;
		case KEY_LEFT:
			{
				int i;
				if (!history.search_buffer[0]) goto input_path;
				for (i = GetCurrent(SCREEN_DOMAIN_LIST) - 1; i >= 0; i--) {
					const char *cp = DomainName(i);
					if (strchr(history.search_buffer, '/')) {
						if (strrchr(cp, ' ')) cp = strrchr(cp, ' ') + 1;
					} else {
						if (strrchr(cp, '/')) cp = strrchr(cp, '/') + 1;
					}
					if (strcmp(cp, history.search_buffer)) continue;
					while (i < current_y[SCREEN_DOMAIN_LIST] + current_item_index[SCREEN_DOMAIN_LIST]) {
						if (current_y[SCREEN_DOMAIN_LIST] > 0) current_y[SCREEN_DOMAIN_LIST]--;
						else current_item_index[SCREEN_DOMAIN_LIST]--;
					}
					ShowList(SCREEN_DOMAIN_LIST);
					break;
				}
			}
			break;
		case KEY_RIGHT:
			{
				int i;
				if (!history.search_buffer[0]) goto input_path;
				for (i = GetCurrent(SCREEN_DOMAIN_LIST) + 1; i < list_item_count[SCREEN_DOMAIN_LIST]; i++) {
					const char *cp = DomainName(i);
					if (strchr(history.search_buffer, '/')) {
						if (strrchr(cp, ' ')) cp = strrchr(cp, ' ') + 1;
					} else {
						if (strrchr(cp, '/')) cp = strrchr(cp, '/') + 1;
					}
					if (strcmp(cp, history.search_buffer)) continue;
					while (i > current_y[SCREEN_DOMAIN_LIST] + current_item_index[SCREEN_DOMAIN_LIST]) {
						if (current_y[SCREEN_DOMAIN_LIST] < body_lines - 1) current_y[SCREEN_DOMAIN_LIST]++;
						else current_item_index[SCREEN_DOMAIN_LIST]++;
					}
					ShowList(SCREEN_DOMAIN_LIST);
					break;
				}
			}
			break;
		case 'd':
		case 'D':
			{
				const int current = GetCurrent(SCREEN_DOMAIN_LIST);
				if (current > 0) { // Never delete ROOT_NAME
					const char *key = DomainName(current);
					int c;
					mvprintw(header_lines, 0, "Delete '%s' and its descendant domains? (Y/N):", key);
					clrtoeol();
					refresh();
					do {
						c = getch();
					} while (!(c == 'Y' || c == 'y' || c == 'N' || c == 'n' || c == EOF));
					ResizeWindow(SCREEN_DOMAIN_LIST);
					if (c == 'Y' || c == 'y') {
						const int key_len = strlen(key);
						const char *exe = strrchr(key, ' ') + 1;
						const int parent = FindParent(key);
						if (IsVirtualDomain(current)) {
							if (parent >= 0 && DelFileACL(exe, parent) == 0) {
								UpdateDomainPolicy(parent);
								return SCREEN_DOMAIN_LIST;
							}
						} else {
							int index;
							for (index = 0; index < domain_list_count; index++) {
								const char *cp = DomainName(index);
								if (strncmp(cp, key, key_len) || (cp[key_len] != '\0' && cp[key_len] != ' ')) continue;
								DeleteDomain(index);
								index = 0;
							}
							if (!edit_on_memory) SaveDomainPolicy();
						}
						UpdateCursorPos(SCREEN_DOMAIN_LIST, domain_list_count);
					}
					ShowList(SCREEN_DOMAIN_LIST);
				}
			}
			break;
		case 'r':
		case 'R':
			return SCREEN_DOMAIN_LIST;
		case '\r':
			{
				const int current = GetCurrent(SCREEN_DOMAIN_LIST);
				if (IsVirtualDomain(current)) {
					static char buf[MAXBUFSIZE];
					int redirect_index;
					memset(buf, 0, sizeof(buf));
					snprintf(buf, sizeof(buf) - 1, ROOT_NAME "%s", strrchr(DomainName(current), ' '));
					redirect_index = FindDomain(buf);
					if (redirect_index != EOF) {
						current_item_index[SCREEN_DOMAIN_LIST] = redirect_index - current_y[SCREEN_DOMAIN_LIST];
						while (current_item_index[SCREEN_DOMAIN_LIST] < 0) {
							current_item_index[SCREEN_DOMAIN_LIST]++; current_y[SCREEN_DOMAIN_LIST]--;
						}
						ShowList(SCREEN_DOMAIN_LIST);
					}
				} else {
					ACLListLoop();
					goto start;
				}
			}
			break;
		}
	}
	return MAXSCREEN + 1;
}

static void ACLListLoop(void) {
	int domain_acl_updated = 0;
	current_y[SCREEN_ACL_LIST] = current_item_index[SCREEN_ACL_LIST] = 0;
	ShowList(SCREEN_ACL_LIST);
	while (1) {
		const int c = getch();
		if (c == '\r') break;
		if (c == ERR) continue; // Ignore invalid key.
		switch (c) {
		case KEY_RESIZE:
			ResizeWindow(SCREEN_ACL_LIST);
			ShowList(SCREEN_ACL_LIST);
			break;
		case KEY_UP:
			UpArrowKey(SCREEN_ACL_LIST);
			break;
		case KEY_DOWN:
			DownArrowKey(SCREEN_ACL_LIST);
			break;
		case KEY_PPAGE:
		case KEY_DC:
			PageUpKey(SCREEN_ACL_LIST);
			break;
		case KEY_NPAGE:
			PageDownKey(SCREEN_ACL_LIST);
			break;
		case 'a':
		case 'A':
			{
				static char buffer[MAXBUFSIZE];
				const int domain = GetCurrent(SCREEN_DOMAIN_LIST);
				int perm = 0;
				char *cp;
				mvprintw(window_height - 1, 0, "Enter new entry> "); clrtoeol(); refresh(); echo();
				memset(buffer, 0, sizeof(buffer));
				getnstr(buffer, sizeof(buffer) - 1); noecho();
				NormalizeLine(buffer);
				if (sscanf(buffer, "%d", &perm) == 1 && (cp = strchr(buffer, ' ')) != NULL && IsCorrectPath(cp + 1, 1)) {
					if (AddFileACL(cp + 1, (WORD) perm, domain) == 0) domain_acl_updated = 1;
				} else if (strncmp(buffer, "allow_", 6) == 0) {
					if (AddStringEntry(buffer, domain) == 0) domain_acl_updated = 1;
				}
				SortACLByName(domain);
				ShowList(SCREEN_ACL_LIST);
			}
			break;
		case 'd':
		case 'D':
			{
				const int current = GetCurrent(SCREEN_ACL_LIST);
				if (current >= 0) {
					const int domain = GetCurrent(SCREEN_DOMAIN_LIST);
					DOMAIN_INFO *ptr = &domain_list[domain];
					int index = current;
					if (index < ptr->acl_count) {
						if (DelFileACL(ptr->acl_ptr[index].name, domain) == 0) domain_acl_updated = 1;
					} else {
						index -= ptr->acl_count;
						if (DelStringEntry(ptr->string_ptr[index], domain) == 0) domain_acl_updated = 1;
					}
					UpdateCursorPos(SCREEN_ACL_LIST, ptr->acl_count + ptr->string_count);
					ShowList(SCREEN_ACL_LIST);
				}
			}
			break;
		case 's':
		case 'S':
			if (!domain_acl_updated) break;
			UpdateDomainPolicy(GetCurrent(SCREEN_DOMAIN_LIST));
			_exit(SCREEN_DOMAIN_LIST);
			break;
		}
	}
	_exit(SCREEN_DOMAIN_LIST);		
}

int main(int argc, char *argv[]) {
	int screen = SCREEN_DOMAIN_LIST;
	int current_x;
	memset(&history, 0, sizeof(history));
	memset(current_y, 0, sizeof(current_y));
	memset(current_item_index, 0, sizeof(current_item_index));
	memset(list_item_count, 0, sizeof(list_item_count));
	if (argc > 1) {
		if (strcmp(argv[1], "s") == 0) screen = SCREEN_SYSTEM_LIST;
		else if (strcmp(argv[1], "e") == 0) screen = SCREEN_EXCEPTION_LIST;
		else if (strcmp(argv[1], "d") == 0) screen = SCREEN_DOMAIN_LIST;
		else {
			printf("Usage: %s [s|e|d]\n\n", argv[0]);
			printf("This is a policy editor for TOMOYO Linux.\n");
			printf("If this program is invoked as 'editpolicy_offline', "
				   "this program behaves as an off-line editor that edits files in /root/security/ directory.\n"
				   "If this program is invoked as other than 'editpolicy_offline', "
				   "this program behaves as an on-line editor that edits files in /proc/ccs/policy/ directory.\n");
			return 1;
		}
	}
	if (strendswith(argv[0], "/editpolicy_offline") || strcmp(argv[0], "editpolicy_offline") == 0) {
		edit_on_memory = 0;
		if (chdir("/root/security/")) {
			fprintf(stderr, "Can't chdir to /root/security/\n");
			return 1;
		}
	} else {
		edit_on_memory = 1;
		if (chdir("/proc/ccs/policy/")) {
			fprintf(stderr, "You can't use this editor for this kernel.\n");
			return 1;
		}
	}
	delete_domain_fd = open("/proc/ccs/policy/delete_domain", O_WRONLY);
	update_domain_fd = open("/proc/ccs/policy/update_domain", O_WRONLY);
	if ((delete_domain_fd != EOF && write(delete_domain_fd, "", 0) != 0) || (update_domain_fd != EOF && write(update_domain_fd, "", 0) != 0)) {
	out: ;
		fprintf(stderr, "You need to register this program to /root/security/manager.txt and reboot to run this program.\n");
		return 1;
	}
	{
		int fd = open(SYSTEM_POLICY_FILE[edit_on_memory], O_RDWR);
		if (fd != EOF && write(fd, "", 0) != 0) goto out;
		close(fd);
		fd = open(EXCEPTION_POLICY_FILE[edit_on_memory], O_RDWR);
		if (fd != EOF && write(fd, "", 0) != 0) goto out;
		close(fd);
		fd = open(DOMAIN_POLICY_FILE[edit_on_memory], O_RDWR);
		if (fd != EOF && write(fd, "", 0) != 0) goto out;
		close(fd);
	}
	setenv("TERM", "linux", 1);
	initscr();
	cbreak();
	noecho();
	nonl();
	intrflush(stdscr, FALSE);
	keypad(stdscr, TRUE);
	getyx(stdscr, current_y[SCREEN_DOMAIN_LIST], current_x);
	current_y[SCREEN_ACL_LIST] = current_y[SCREEN_EXCEPTION_LIST] = current_y[SCREEN_SYSTEM_LIST] = current_y[SCREEN_DOMAIN_LIST];
	getmaxyx(stdscr, window_height, window_width);
	while (1) {
		int status = 0;
		if (edit_on_memory) {
			if (delete_domain_fd == EOF && update_domain_fd == EOF) screen = SCREEN_SYSTEM_LIST;
			else if (screen == SCREEN_SYSTEM_LIST && access(SYSTEM_POLICY_FILE[edit_on_memory], F_OK)) screen = SCREEN_EXCEPTION_LIST;
		}
		if (pipe(pipe_fd)) break;
		switch(fork()) {
		case 0:
			close(pipe_fd[0]);
			ResizeWindow(screen);
			switch (screen) {
			case SCREEN_SYSTEM_LIST:
				_exit(SystemListLoop());
				break;
			case SCREEN_EXCEPTION_LIST:
				_exit(ExceptionListLoop());
				break;
			case SCREEN_DOMAIN_LIST:
				_exit(DomainListLoop());
				break;
			}
		case -1:
			close(pipe_fd[0]);
			break;
		}
		close(pipe_fd[1]);
		while (read(pipe_fd[0], &history, sizeof(history)) > 0);
		close(pipe_fd[0]);
		while (wait(&status) == EOF && errno == EINTR);
		if (!WIFEXITED(status)) break;
		status = WEXITSTATUS(status);
		if (status == MAXSCREEN) continue;
		if (status > MAXSCREEN) break;
		screen = status;
	}
	clear();
	move(0, 0);
	refresh();
	endwin();
	return 0;
}
