/*
 * savepolicy.c
 *
 * Save the current policy in the kernel onto the disk.
 *
 * Copyright (C) 2005-2006  NTT DATA CORPORATION
 *
 * Version: 1.1   2006/04/01
 *
 * This program saves policies in /proc/ccs/policy/ directory.
 * You can call this program to save policies just before the system halts
 * (i.e. at the end of "halt" script).
 * This program remounts root fs as read/write if necessary.
 *
 */

#include "policy.h"
#include <sys/mount.h>
#include <time.h>

DOMAIN_INFO *domain_list[4] = { NULL, NULL, NULL, NULL };
int domain_list_count[4] = { 0, 0, 0, 0 };

static char *ReadFile(const char *filename) {
	char *buffer = NULL;
	int fd;
	if ((fd = open(filename, O_RDONLY)) != EOF) {
		int buffer_len = 0;
		while (1) {
			char *cp = realloc(buffer, buffer_len + 4096);
			int len;
			if (!cp) {
				free(buffer);
				return NULL;
			}
			buffer = cp;
			len = read(fd, buffer + buffer_len, 4095);
			if (len <= 0) break;
			buffer_len += len;
		}
		close(fd);
		buffer[buffer_len] = '\0';
	}
	return buffer;
}

int main(int argc, char *argv[]) {
	int remount_root = 0;
	static char filename[1024];
	int save_system_policy = 0;
	int save_exception_policy = 0;
	int save_domain_policy = 0;
	int force_save = 0;
	int repeat;
	time_t now = time(NULL);
	struct tm *tm = localtime(&now);
	memset(filename, 0, sizeof(filename));
	if (access("/proc/self/", F_OK)) mount("/proc", "/proc", "proc", 0, NULL);
	if (access("/proc/ccs/policy/", F_OK)) {
		fprintf(stderr, "You can't run this program for this kernel.\n");
		return 0;
	}
	if (argc == 1) {
		force_save = save_domain_policy = 1;
	} else {
		int i;
		for (i = 1; i < argc; i++) {
			char *p = argv[i];
			char *s = strchr(p, 's');
			char *e = strchr(p, 'e');
			char *d = strchr(p, 'd');
			char *a = strchr(p, 'a');
			char *f = strchr(p, 'f');
			if (s || a) save_system_policy = 1;
			if (e || a) save_exception_policy = 1;
			if (d || a) save_domain_policy = 1;
			if (f) force_save = 1;
			if (strcspn(p, "sedaf")) {
				printf("%s [s][e][d][a][f]\n"
					   "s : Save system_policy.\n"
					   "e : Save exception_policy.\n"
					   "d : Save domain_policy.\n"
					   "a : Save all policies.\n"
					   "f : Save even if on-disk policy and on-memory policy are the same.\n\n"
					   "If no options given, this program assumes 'd' and 'f' are given.\n", argv[0]);
				return 0;
			}
		}
	}
	if (chdir("/root/security/")) {
		printf("Directory /root/security/ doesn't exist.\n");
		return 1;
	}
	if (access(".", W_OK) == EOF) {
		if (errno != EROFS || mount("/", "/", "rootfs", MS_REMOUNT, NULL) == EOF) {
			printf("Can't remount for read-write. (%s)\n", strerror(errno));
			return 1;
		}
		remount_root = 1;
	}

	/* Repeat twice so that necessary permissions for this program are included in domain policy. */
	for (repeat = 0; repeat < 2; repeat++) {

		if (save_system_policy) {
			char *new_policy = ReadFile("/proc/ccs/policy/system_policy");
			char *old_policy = ReadFile("system_policy.txt");
			if (new_policy && (force_save || !old_policy || strcmp(new_policy, old_policy))) {
				int fd;
				snprintf(filename, sizeof(filename) - 1, "system_policy.%02d-%02d-%02d.%02d:%02d:%02d.txt", tm->tm_year % 100, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
				if ((fd = open(filename, O_WRONLY | O_CREAT | O_TRUNC, 0600)) != EOF) {
					write(fd, new_policy, strlen(new_policy));
					close(fd);
					unlink("system_policy.txt");
					symlink(filename, "system_policy.txt");
				} else {
					printf("Can't create %s\n", filename);
				}
			}
			free(old_policy);
			free(new_policy);
		}
		
		if (save_exception_policy) {
			char *new_policy = ReadFile("/proc/ccs/policy/exception_policy");
			char *old_policy = ReadFile("exception_policy.txt");
			if (new_policy && (force_save || !old_policy || strcmp(new_policy, old_policy))) {
				int fd;
				snprintf(filename, sizeof(filename) - 1, "exception_policy.%02d-%02d-%02d.%02d:%02d:%02d.txt", tm->tm_year % 100, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
				if ((fd = open(filename, O_WRONLY | O_CREAT | O_TRUNC, 0600)) != EOF) {
					write(fd, new_policy, strlen(new_policy));
					close(fd);
					unlink("exception_policy.txt");
					symlink(filename, "exception_policy.txt");
				} else {
					printf("Can't create %s\n", filename);
				}
			}
			free(old_policy);
			free(new_policy);
		}
		
		if (save_domain_policy) {
			const int src = repeat ? 2 : 0;
			const int dest = repeat ? 3 : 1;
			ReadDomainPolicy("/proc/ccs/policy/domain_policy", src);
			ReadDomainPolicy("domain_policy.txt", dest);
			if (force_save || !IsSameDomainList(src, dest)) {
				int fd;
				snprintf(filename, sizeof(filename) - 1, "domain_policy.%02d-%02d-%02d.%02d:%02d:%02d.txt", tm->tm_year % 100, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
				if ((fd = open(filename, O_WRONLY | O_CREAT | O_TRUNC, 0600)) != EOF) {
					WriteDomainPolicy(fd, src);
					close(fd);
					unlink("domain_policy.txt");
					symlink(filename, "domain_policy.txt");
				} else {
					printf("Can't create %s\n", filename);
				}
			}
		}
	}
	if (remount_root) mount("/", "/", "rootfs", MS_REMOUNT | MS_RDONLY, NULL);
	return 0;
}
