/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.base.ssh.ui.wizards.newproject;

import java.io.File;

import jp.sourceforge.tomoyo.base.ssh.Activator;
import jp.sourceforge.tomoyo.base.ssh.connection.SSHConnectionAdapter;
import jp.sourceforge.tomoyo.core.PersistentPropertyManager;
import jp.sourceforge.tomoyo.core.ProjectProperty;
import jp.sourceforge.tomoyo.core.server.ConnectionManager;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.custom.CTabItem;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

public class ConnectionSettingPage extends AbstractWizardPage {

	private static final String DIALOG_SETTINGS_KEY_HOSTLIST = "host.list"; //$NON-NLS-1$

	private static final String PAGE_DESCRPTION = Messages.ConnectionSettingPage_PageDescription;
	
	private Combo hostCombo = null;
	private Text portText = null;
	private Combo userCombo = null;
	private Text pswdText = null;
	private Text pkeyText = null;
	private Button useKnownHostsCheck;
	private Text knownHostsText;
	private Button knownHostsButton;
	private Button pkeyButton;
	private Button connectButton = null;
	
	public ConnectionSettingPage() {
		super(Messages.ConnectionSettingPage_PageName);
		setTitle(Messages.ConnectionSettingPage_PageTitle);
		setDescription(PAGE_DESCRPTION);
		setPageComplete(false);
	}
	
	protected String getHelpResourceName() {
		return "connection_settings"; //$NON-NLS-1$
	}

	//--------------------------------------------------------------------------------------------
	// Creating controls
	//--------------------------------------------------------------------------------------------
	
	private CTabFolder tabFolder;
	
	public void createControl(Composite parent) {
		Composite composite = new Composite(parent, SWT.NONE);

		GridLayout layout = new GridLayout(1, false);
		composite.setLayout(layout);

		createHostGroup(composite);
		
		tabFolder = new CTabFolder(composite, SWT.BORDER);
		tabFolder.setSimple(false);
		tabFolder.setTabPosition(SWT.TOP);
		tabFolder.setSelectionForeground(Display.getCurrent().getSystemColor(SWT.COLOR_TITLE_FOREGROUND));
		tabFolder.setSelectionBackground(
			new Color[] {
					Activator.getStandardDisplay().getSystemColor(SWT.COLOR_TITLE_BACKGROUND),
					Activator.getStandardDisplay().getSystemColor(SWT.COLOR_TITLE_BACKGROUND_GRADIENT),
			},
			new int[] {100}, true
		);
		tabFolder.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));
		tabFolder.addSelectionListener(new SelectionListener() {
			public void widgetDefaultSelected(SelectionEvent e) {
			}
			public void widgetSelected(SelectionEvent e) {
				tabFolderSelectionChanged();
			}
		});

		createPublicKeyAuthFolder(tabFolder);
		createPasswordAuthFolder(tabFolder);

		if (tabFolder.getItemCount() > 0)
			tabFolder.setSelection(0);

		createConnectionGroup(composite);
		
		hostCombo.setFocus();
		
		restoreHistory();
		setupDefault();
		setControl(composite);
	}
	
	private void tabFolderSelectionChanged() {
		inputChanged();
	}

	private void setupDefault() {
		if (hostCombo.getItemCount() > 0) {
			hostCombo.select(0);
		}
		userCombo.setText("root"); //$NON-NLS-1$
	}
	
	public IWizardPage getPreviousPage() {
		if (ConnectionManager.getInstance().connect(getProject()))
			return null;
		else
			return super.getPreviousPage();
	}

	private void restoreHistory() {
		IDialogSettings settings = Activator.getDefault().getDialogSettings(getClass().getName());
		if (settings == null) {
			return;
		}
		String[] hosts = settings.getArray(DIALOG_SETTINGS_KEY_HOSTLIST);
		if (hosts != null) {
			hostCombo.setItems(hosts);
		}
	}
    
	private void saveHistory() {
    	IDialogSettings settings = Activator.getDefault().getDialogSettings(getClass().getName());
    	String newHost = hostCombo.getText();
    	String[] hosts;
    	if (settings.getArray(DIALOG_SETTINGS_KEY_HOSTLIST) == null) {
    		hosts = new String[0];
    	} else {
    		hosts = settings.getArray(DIALOG_SETTINGS_KEY_HOSTLIST);
    		if (includesHosts(hosts, newHost))
    			return;
    	}
    	String[] newHosts = new String[hosts.length + 1];
    	newHosts[0] = newHost;
    	System.arraycopy(hostCombo.getItems(), 0, newHosts, 1, hostCombo.getItemCount());
    	settings.put(DIALOG_SETTINGS_KEY_HOSTLIST, newHosts);
	}
    
    private boolean includesHosts(String[] hosts, String newHost) {
    	if (hosts == null)
    		return false;
    	for (int cnt = 0; cnt < hosts.length; cnt++) {
    		if (hosts[cnt].equals(newHost))
    			return true;
    	}
    	return false;
    }

	private void createHostGroup(Composite parent) {
		Composite container = new Composite(parent, SWT.NULL);
		container.setLayout(new GridLayout(3, false));
		GridData data = new GridData(SWT.FILL, SWT.FILL, true, false, 1, 1);
		container.setLayoutData(data);
		{
			Label ipLabel = new Label(container, SWT.NONE);
			ipLabel.setText(Messages.ConnectionSettingPage_HostLabelCaption);
			
			GridData griddata = new GridData();
			griddata.widthHint = 150;
			hostCombo = new Combo(container, SWT.SINGLE | SWT.BORDER);
			hostCombo.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
			hostCombo.addModifyListener(new ModifyListener() {
				public void modifyText(ModifyEvent e) {
					inputChanged();
				}
			});
			hostCombo.setLayoutData(griddata);
			
			new Label(container, SWT.NONE);
		}
		{
			Label portLabel = new Label(container, SWT.NONE);
			portLabel.setText(Messages.ConnectionSettingPage_PortLabelCaption);
			
			GridData portGrid = new GridData();
			portGrid.widthHint = 50;
			portText = new Text(container, SWT.BORDER);
			portText.setText("22"); //$NON-NLS-1$
			portText.setTextLimit(5);
			portText.setLayoutData(portGrid);
			portText.addModifyListener(new ModifyListener() {
				public void modifyText(ModifyEvent e) {
					inputChanged();
				}
			});
			
			new Label(container, SWT.NONE);
		}
		
		createKnownHostGroup(container);

    }
	
	private void createPublicKeyAuthFolder(CTabFolder tabFolder) {
		CTabItem tabItem = new CTabItem(tabFolder, SWT.NULL);
		tabItem.setText(Messages.ConnectionSettingPage_PublicKeyAuthTabTitle);
		tabItem.setControl(createPublicKeyControl(tabFolder));
	}

	private void createPasswordAuthFolder(CTabFolder tabFolder) {
		CTabItem tabItem = new CTabItem(tabFolder, SWT.NULL);
		tabItem.setText(Messages.ConnectionSettingPage_PasswordAuthTabTitle);
		tabItem.setControl(createPasswordAuthControl(tabFolder));
	}
	
	private Composite createPublicKeyControl(Composite parent) {
		Composite container = new Composite(parent, SWT.NULL);
		container.setLayout(new GridLayout(3, false));
		GridData data = new GridData(SWT.FILL, SWT.FILL, true, false, 1, 1);
		container.setLayoutData(data);
		
		{
			Label pathLabel = new Label(container, SWT.NONE);
			pathLabel.setText(Messages.ConnectionSettingPage_PrivateKeyLabelCaption);
		}
		{
			pkeyText = new Text(container, SWT.BORDER);
			pkeyText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
			pkeyText.setText(System.getProperty("user.home") + System.getProperty("file.separator") + ".ssh" + System.getProperty("file.separator") + "id_rsa"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
			pkeyText.addModifyListener(new ModifyListener() {
				public void modifyText(ModifyEvent e) {
					inputChanged();
				}
			});
		}
		{
			pkeyButton = new Button(container, SWT.PUSH);
			pkeyButton.setText(Messages.ConnectionSettingPage_PrivateKeyFileSelectionButtonTitle);
			pkeyButton.setEnabled(true);
			pkeyButton.setLayoutData(createButtonGridData());
			pkeyButton.addSelectionListener(new SelectionListener() {
				public void widgetSelected(SelectionEvent e) {
					FileDialog dialog = new FileDialog(Display.getCurrent().getActiveShell(), SWT.OPEN);
					dialog.setText(pkeyText.getText());
			        String file = dialog.open();
			        if (file != null)
			        	pkeyText.setText(file);
				}
				public void widgetDefaultSelected(SelectionEvent e) {
				}
			});
		}
		return container;
	}
	
	private Button rememberPassword;
	
	private Composite createPasswordAuthControl(Composite parent) {
		Composite container = new Composite(parent, SWT.NULL);
		container.setLayout(new GridLayout(2, false));
		GridData data = new GridData(SWT.FILL, SWT.FILL, true, false, 1, 1);
		container.setLayoutData(data);

		{
			Label userLabel = new Label(container, SWT.NONE);
			userLabel.setText(Messages.ConnectionSettingPage_UsernameLabelCaption);
		}
		{
			GridData gridData = new GridData();
			gridData.widthHint = 200;
			userCombo = new Combo(container, SWT.DROP_DOWN);
			userCombo.setLayoutData(gridData);
			userCombo.addModifyListener(new ModifyListener() {
				public void modifyText(ModifyEvent e) {
					inputChanged();
				}
			});
		}
		{
			Label pswdLabel = new Label(container, SWT.NONE);
			pswdLabel.setText(Messages.ConnectionSettingPage_PasswordLabelCaption);
		}
		{
			GridData gridData = new GridData();
			gridData.widthHint = 200;
			pswdText = new Text(container, SWT.SINGLE | SWT.PASSWORD | SWT.BORDER);
			pswdText.setLayoutData(gridData);
			pswdText.addModifyListener(new ModifyListener() {
				public void modifyText(ModifyEvent e) {
					inputChanged();
				}
			});
		}
		{
			GridData gridData = new GridData();
			rememberPassword = new Button(container, SWT.CHECK);
			rememberPassword.setText(Messages.ConnectionSettingPage_RememberPassword);
			rememberPassword.setSelection(false);
			rememberPassword.setLayoutData(gridData);
			rememberPassword.addSelectionListener(new SelectionListener() {
				public void widgetSelected(SelectionEvent e) {
					inputChanged();
				}
				public void widgetDefaultSelected(SelectionEvent e) {
				}
			});		
		}
		return container;
	}

	private void createKnownHostGroup(Composite parent) {
		{
			GridData gridData = new GridData();
			useKnownHostsCheck = new Button(parent, SWT.CHECK);
			useKnownHostsCheck.setText(Messages.ConnectionSettingPage_UseKnownHostFileCheckButtonTitle);
			useKnownHostsCheck.setSelection(true);
			useKnownHostsCheck.setLayoutData(gridData);
			useKnownHostsCheck.addSelectionListener(new SelectionListener() {
				public void widgetSelected(SelectionEvent e) {
					inputChanged();
				}
				public void widgetDefaultSelected(SelectionEvent e) {
				}
			});		
		}
		{
			knownHostsText = new Text(parent, SWT.BORDER);
			knownHostsText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
			knownHostsText.setText(System.getProperty("user.home") + System.getProperty("file.separator") + ".ssh" + System.getProperty("file.separator") + "known_hosts"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
			knownHostsText.setEnabled(false);
			knownHostsText.addModifyListener(new ModifyListener() {
				public void modifyText(ModifyEvent e) {
					inputChanged();
				}
			});

			knownHostsButton = new Button(parent, SWT.PUSH);
			knownHostsButton.setText(Messages.ConnectionSettingPage_KnownHostFileSelectionButtonTitle);
			knownHostsButton.setEnabled(false);
			GridData buttonGrid = createButtonGridData();
			knownHostsButton.setLayoutData(buttonGrid);
			knownHostsButton.addSelectionListener(new SelectionListener() {
				public void widgetSelected(SelectionEvent e) {
					FileDialog dialog = new FileDialog(Display.getCurrent().getActiveShell(), SWT.OPEN);
					dialog.setText(knownHostsText.getText());
					String file = dialog.open();
					if (file != null)
						knownHostsText.setText(file);
				}
				public void widgetDefaultSelected(SelectionEvent e) {
				}
			});
		}
    }
	
	private GridData createButtonGridData() {
		GridData gridData = new GridData(GridData.END);
		gridData.widthHint = 50;
		return gridData;
	}
    
    private void createConnectionGroup(Composite container) {
		Composite conButtonComposite = new Composite(container, SWT.NONE);
		conButtonComposite.setLayout(new GridLayout(2, false));
		GridData gridData = new GridData();
		gridData.horizontalSpan = 3;
		gridData.horizontalAlignment = SWT.FILL;
		conButtonComposite.setLayoutData(gridData);

        Label blank = new Label(conButtonComposite, SWT.NONE);
        GridData lineData = new GridData(GridData.FILL_HORIZONTAL);
        lineData.horizontalSpan = 1;
        blank.setLayoutData(lineData);

		connectButton = new Button(conButtonComposite, SWT.PUSH);
		connectButton.setText(Messages.ConnectionSettingPage_ConnectionTestButtonTitle);
		connectButton.setEnabled(false);
		connectButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_END));
		connectButton.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				handleConnectionButtonPressed();
			}
			public void widgetDefaultSelected(SelectionEvent e) {
			}
		});
    }
    
	//--------------------------------------------------------------------------------------------
	// Connection check
	//--------------------------------------------------------------------------------------------

	private void handleConnectionButtonPressed() {
	    try {
	    	setErrorMessage(null);
	    	
			IProject project = getProject();
	    	storeProperties(project);
	    	if (ConnectionManager.getInstance().connect(project)) {
			    setMessage(Messages.ConnectionSettingPage_ConnectionEstablishedMessage, INFORMATION);
			    
			    disableControls();
			    getWizard().getContainer().updateButtons();

			    saveHistory();
			    setPageComplete(true);
	    	} else {
	    		setErrorMessage(ConnectionManager.getInstance().getErrorMessage(project));
	    	}
		} catch (CoreException e1) {
			Activator.logException(e1);
		}
	}

    private void disableControls() {
		// enable not to be changed any more.
		hostCombo.setEnabled(false);
		userCombo.setEnabled(false);
		portText.setEnabled(false);
		
		knownHostsText.setEnabled(false);
		knownHostsButton.setEnabled(false);
		
		useKnownHostsCheck.setEnabled(false);
		
		pkeyText.setEnabled(false);
		pkeyButton.setEnabled(false);
		pswdText.setEnabled(false);

		connectButton.setEnabled(false);
    }

    private boolean usePublicKeyAuth() {
    	if (tabFolder == null)
    		return true;
    	return tabFolder.getSelectionIndex() == 0;
    }
    
	private void storeProperties(IProject project) throws CoreException {
		ProjectProperty properties = new ProjectProperty(project);
		properties.setProperty(PersistentPropertyManager.PROPERTY_CONNECTION_ADAPTER, SSHConnectionAdapter.ADAPTER_ID);
		properties.setProperty(PersistentPropertyManager.PROPERTY_SERVER_HOSTNAME, hostCombo.getText().trim());
		properties.setProperty(PersistentPropertyManager.PROPERTY_SERVER_PORT, portText.getText().trim());
		properties.setProperty(PersistentPropertyManager.PROPERTY_LOGIN_USERNAME, userCombo.getText().trim());
		if (useKnownHostsCheck.getSelection())
			properties.setProperty(PersistentPropertyManager.PROPERTY_KNOWN_HOSTS, knownHostsText.getText().trim());
		else
			properties.setProperty(PersistentPropertyManager.PROPERTY_KNOWN_HOSTS, null);
		if (usePublicKeyAuth()) {
			properties.setProperty(PersistentPropertyManager.PROPERTY_PRIVATE_KEY, pkeyText.getText().trim());
			properties.setProperty(PersistentPropertyManager.PROPERTY_LOGIN_PASSWORD, null);
		} else {
			properties.setProperty(PersistentPropertyManager.PROPERTY_PRIVATE_KEY, null);
			if (rememberPassword.getSelection())
				properties.setProperty(PersistentPropertyManager.PROPERTY_LOGIN_PASSWORD, pswdText.getText().trim());
			else
				properties.setProperty(PersistentPropertyManager.PROPERTY_LOGIN_PASSWORD, null);
		}
		
		properties.setProperty(PersistentPropertyManager.PROPERTY_MANAGER_CONFIRM_CONNECT, "true"); //$NON-NLS-1$
	}

	private void inputChanged() {
		knownHostsText.setEnabled(useKnownHostsCheck.getSelection());
		knownHostsButton.setEnabled(useKnownHostsCheck.getSelection());
		pswdText.setEnabled(rememberPassword.getSelection());
		// check input values.
		if (hostCombo.getText().trim().length() == 0) {
			setInputErrorMessage(Messages.ConnectionSettingPage_ErrorMessageInputHostname);
			return;
		}
		if (portText.getText().trim().length() == 0) {
			setInputErrorMessage(Messages.ConnectionSettingPage_ErrorMessageInputPortNo);
			return;
		} else {
			try {
				int port = Integer.parseInt(portText.getText().trim());
				if (port < 1 || 65535 < port) {
					setInputErrorMessage(Messages.ConnectionSettingPage_ErrorMessagePortNoRange);
					return;
				}
			} catch (NumberFormatException e) {
				setInputErrorMessage(Messages.ConnectionSettingPage_ErrorMessagePortNoMustBeInteger);
				return;
			}
		}
		if (usePublicKeyAuth()) {
			if (pkeyText.getText().trim().length() == 0) {
				setInputErrorMessage(Messages.ConnectionSettingPage_ErrorMessageInputPrivateKeyFile);
				return;
			}
			if (!new File(pkeyText.getText().trim()).exists()) {
				setInputErrorMessage(Messages.ConnectionSettingPage_ErrorMessageInvalidPrivateKeyFile);
				return;
			}
		} else {
			if (userCombo.getText().trim().length() == 0) {
				setInputErrorMessage(Messages.ConnectionSettingPage_ErrorMessageInputUsername);
				return;
			}
		}
		if (useKnownHostsCheck.getSelection()) {
			if (knownHostsText.getText().trim().length() == 0) {
				setInputErrorMessage(Messages.ConnectionSettingPage_ErrorMessageInputKnownHotsFile);
				return;
			}
			if (!new File(knownHostsText.getText().trim()).exists()) {
				setInputErrorMessage(Messages.ConnectionSettingPage_ErrorMessageInvalidKnownHotsFile);
				return;
			}
		}

		connectButton.setEnabled(true);
		setErrorMessage(null);		
		setMessage(Messages.ConnectionSettingPage_ConnectionOK, INFORMATION);
	}
	
	private void setInputErrorMessage(String message) {
		connectButton.setEnabled(false);
		setErrorMessage(message);		
	}

}
