/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.core.local.model.status;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Hashtable;
import java.util.Observable;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import jp.sourceforge.tomoyo.core.TomoyoCorePlugin;
import jp.sourceforge.tomoyo.core.local.resource.ProcResourceManager;
import jp.sourceforge.tomoyo.core.local.resource.Status;
import jp.sourceforge.tomoyo.core.server.CommandManager;
import jp.sourceforge.tomoyo.core.server.ConcreteCommand;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;

public class ProjectProfile extends Observable {
	
	private Hashtable<String, ProfileGroup> profileGroups = new Hashtable<String, ProfileGroup>();
	private IProject project;

	public ProjectProfile(IProject project) {
		this.project = project;
	}
	
	public void load() {
		updateCache();
	}
	
	private void updateCache() {
		try {
			IFile file = ProcResourceManager.getInstance().getProcResourceSet(project).getStatus().getFile();
			updateCache(new InputStreamReader(file.getContents()));
		} catch (CoreException e) {
			TomoyoCorePlugin.logException(e);
		}
	}
	
	private void updateCache(String stdout) {
		updateCache(new InputStreamReader(new ByteArrayInputStream(stdout.getBytes())));
	}
	
	private void updateCache(InputStreamReader reader) {
		profileGroups.clear();
		BufferedReader br = new BufferedReader(reader);
		try {
			String line = null;
			Pattern pattern = Pattern.compile("^([0-9]+)-([a-zA-Z0-9:_]+)=(.+)$"); //$NON-NLS-1$
			while ((line = br.readLine()) != null) {
				Matcher matcher = pattern.matcher(line);
				if (matcher.matches()) {
					String strGroupNo = matcher.group(1);
					String strProfileName = matcher.group(2);
					String strProfileValue = matcher.group(3);
					ProfileGroup group = profileGroups.get(strGroupNo);
					if (group == null) {
						group = new ProfileGroup(Integer.parseInt(strGroupNo));
						profileGroups.put(strGroupNo, group);
					}
					group.storeProfileValue(strProfileName, strProfileValue);
				}
			}
		} catch (IOException e) {
			TomoyoCorePlugin.logException(e);
		}
	}
	
	public ProfileGroup getGroup(int groupNo) {
		return profileGroups.get(String.valueOf(groupNo));
	}

	public ProfileGroup createGroup(int groupNo) {
		ProfileGroup group = profileGroups.get(String.valueOf(groupNo));
		if (group == null) {
			group = new ProfileGroup(groupNo);
			profileGroups.put(String.valueOf(groupNo), group);
		}		
		return group;
	}
	
	public int getGroupCount() {
		return profileGroups.keySet().size();
	}
	
	public ArrayList<ProfileGroup> toList(boolean sort) {
		ArrayList<ProfileGroup> groupList = new ArrayList<ProfileGroup>();
		groupList.addAll(profileGroups.values());
		if (sort) {
			Collections.sort(groupList, new GroupComparator());
		}
		return groupList;
	}
	
	public ProfileGroup[] toArray(boolean sort) {
		ArrayList<ProfileGroup> groupList = toList(sort);
		return groupList.toArray(new ProfileGroup[groupList.size()]);
	}
	
	private class GroupComparator implements Comparator<ProfileGroup> {
		/*
		public int compare(Object arg0, Object arg1) {
			ProfileGroup group1 = (ProfileGroup)arg0;
			ProfileGroup group2 = (ProfileGroup)arg1;
			return group1.getNo() - group2.getNo();
		}
	*/
		@Override
		public int compare(ProfileGroup arg0, ProfileGroup arg1) {
			return arg0.getNo() - arg1.getNo();
		}
	}

	public ProfileGroup getGroupByIndex(int index) {
		return toArray(true)[index];
	}

	public boolean save() {
		ProfileGroup[] groups = toArray(false);
		ArrayList<String> saveLines = new ArrayList<String>();
		for (int cnt = 0; cnt < groups.length; cnt++) {
			ProfileGroup group = groups[cnt];
			Profile[] profiles = group.listProfile();
			for (int pcnt = 0; pcnt < profiles.length; pcnt++) {
				Profile profile = profiles[pcnt];
				if (profile == null)
					continue;
				if (profile.isDirty()) {
					ProfileMetaData metaData = ProfileDefinition.getMetaData(profile.getIndex());
					if (profile.needQuate())
						saveLines.add(group.getNo() + "-" + metaData.getName() + "=" +  "\"" + profile.getValue() + "\""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
					else
						saveLines.add(group.getNo() + "-" + metaData.getName() + "=" +  profile.getValue()); //$NON-NLS-1$ //$NON-NLS-2$
				}
			}
		}
		if (saveLines.size() == 0)
			return false;
		
		boolean isSaved = save((String[])saveLines.toArray(new String[saveLines.size()]));
		
		return isSaved;
	}

	private boolean save(String[] lines) {
		ConcreteCommand command = CommandManager.getInstance().exec(project,
				CommandManager.createCCSToolPath(project, CommandManager.CMD_SETLEVEL),
				null, lines);
		if (command.isSuccessful()) {
			Status status = ProcResourceManager.getInstance().getProcResourceSet(project).getStatus();
			ProcResourceManager.getInstance().synchronize(project, status);
		} else {
			return false;
		}

		notifyUpdate();
		
		updateCache(command.getSTDOut());
		return true;
	}
	
	private void notifyUpdate() {
		setChanged();
		notifyObservers(createNotificationObject());
	}

	private Object createNotificationObject() {
		Hashtable<Class<IProject>, IProject> table = new Hashtable<Class<IProject>, IProject>();
		table.put(IProject.class, project);
		return table;
	}

	public void setUpdate() {
		notifyUpdate();
	}

	public boolean hasDirty() {
		ProfileGroup[] groups = toArray(false);
		for (int cnt = 0; cnt < groups.length; cnt++) {
			ProfileGroup group = groups[cnt];
			Profile[] profiles = group.listProfile();
			for (int pcnt = 0; pcnt < profiles.length; pcnt++) {
				Profile profile = profiles[pcnt];
				if (profile == null)
					continue;
				if (profile.isDirty()) {
					return true;
				}
			}
		}
		return false;
	}

	public int getNewGroupNo() {
		int newNo = 0;
		ProfileGroup[] groups = toArray(false);
		for (int cnt = 0; cnt < groups.length; cnt++) {
			ProfileGroup group = groups[cnt];
			if (newNo <= group.getNo())
				newNo = group.getNo() + 1;
		}
		return newNo;
	}

	public void rollback() {
		ProfileGroup[] groups = toArray(false);
		for (int cnt = 0; cnt < groups.length; cnt++) {
			ProfileGroup group = groups[cnt];
			Profile[] profiles = group.listProfile();
			for (int pcnt = 0; pcnt < profiles.length; pcnt++) {
				Profile profile = profiles[pcnt];
				if (profile == null)
					continue;
				profile.rollback();
			}
		}
	}

}
